/* e-contact-store.c - A GtkTreeModel implementation that stores EContacts
 *
 * Copyright (C) 2008 Nokia Corp.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Joergen Scheibengruber <jorgen.scheibengruber AT nokia.com>
 */

#include "e-contact-store.h"

typedef struct _EContactStorePrivate EContactStorePrivate;
struct _EContactStorePrivate
{
    int stamp;

    EBookView *book_view;

    GSequence *contacts;

    GHashTable *uid_to_seqiter;

    EContactCompareFunc compare_func;
};

enum {
    PROP_0,

    PROP_BOOK_VIEW
};

#define E_CONTACT_STORE_PRIVATE(o) \
(G_TYPE_INSTANCE_GET_PRIVATE ((o), E_CONTACT_STORE_TYPE, EContactStorePrivate))

static void e_contact_store_class_init (EContactStoreClass *klass);
static void e_contact_store_init       (EContactStore *self);
static void e_contact_store_dispose    (GObject *object);
static void e_contact_store_finalize   (GObject *object);
static void e_contact_store_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec);
static void e_contact_store_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec);
static void e_contact_store_tree_model_init (GtkTreeModelIface *iface);

G_DEFINE_TYPE_WITH_CODE (EContactStore, e_contact_store, G_TYPE_OBJECT,
             G_IMPLEMENT_INTERFACE (GTK_TYPE_TREE_MODEL,
                        e_contact_store_tree_model_init));


static GtkTreeModelFlags e_contact_store_get_flags (GtkTreeModel *tree_model);
static gint         e_contact_store_get_n_columns   (GtkTreeModel      *tree_model);
static GType        e_contact_store_get_column_type (GtkTreeModel      *tree_model,
                                                     gint               index);
static gboolean     e_contact_store_get_iter        (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreePath       *path);
static GtkTreePath* e_contact_store_get_path        (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);
static void         e_contact_store_get_value       (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     gint               column,
                                                     GValue            *value);
static gboolean     e_contact_store_iter_next       (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);
static gboolean     e_contact_store_iter_children   (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *parent);
static gboolean     e_contact_store_iter_has_child  (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);
static gint         e_contact_store_iter_n_children (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter);
static gboolean     e_contact_store_iter_nth_child  (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *parent,
                                                     gint               n);
static gboolean     e_contact_store_iter_parent     (GtkTreeModel      *tree_model,
                                                     GtkTreeIter       *iter,
                                                     GtkTreeIter       *child);

static void on_book_view_contacts_added   (EBookView *book_view, GList *contacts, gpointer user_data);
static void on_book_view_contacts_changed (EBookView *book_view, GList *contacts, gpointer user_data);
static void on_book_view_contacts_removed (EBookView *book_view, GList     *uids, gpointer user_data);



static void
e_contact_store_class_init (EContactStoreClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (EContactStorePrivate));

    object_class->set_property = e_contact_store_set_property;
    object_class->get_property = e_contact_store_get_property;
    object_class->dispose = e_contact_store_dispose;
    object_class->finalize = e_contact_store_finalize;

    g_object_class_install_property
        (object_class,
         PROP_BOOK_VIEW,
         g_param_spec_object
            ("book-view",
             "BookView",
             "The tracked book-view",
             E_TYPE_BOOK_VIEW,
             G_PARAM_READWRITE | G_PARAM_STATIC_NAME | G_PARAM_CONSTRUCT));
}

static void
e_contact_store_tree_model_init (GtkTreeModelIface *iface)
{
    iface->get_flags = e_contact_store_get_flags;
    iface->get_n_columns = e_contact_store_get_n_columns;
    iface->get_column_type = e_contact_store_get_column_type;
    iface->get_iter = e_contact_store_get_iter;
    iface->get_path = e_contact_store_get_path;
    iface->get_value = e_contact_store_get_value;
    iface->iter_next = e_contact_store_iter_next;
    iface->iter_children = e_contact_store_iter_children;
    iface->iter_has_child = e_contact_store_iter_has_child;
    iface->iter_n_children = e_contact_store_iter_n_children;
    iface->iter_nth_child = e_contact_store_iter_nth_child;
    iface->iter_parent = e_contact_store_iter_parent;
}

static void
e_contact_store_init (EContactStore *self)
{
    EContactStorePrivate *priv;

    priv = E_CONTACT_STORE_PRIVATE (self);

    priv->stamp = 0;
    priv->book_view = NULL;
    priv->uid_to_seqiter = NULL;
    priv->contacts = NULL;
}

EContactStore*
e_contact_store_new (void)
{
    return g_object_new (E_CONTACT_STORE_TYPE,
                         NULL);
}

EContactStore*
e_contact_store_new_with_book_view (EBookView *book_view)
{
    return g_object_new (E_CONTACT_STORE_TYPE,
                         "book-view", book_view,
                         NULL);
}
static void
e_contact_store_dispose (GObject *object)
{
    EContactStorePrivate *priv;

    priv = E_CONTACT_STORE_PRIVATE (object);

    e_contact_store_set_book_view (E_CONTACT_STORE (object), NULL);

    G_OBJECT_CLASS (e_contact_store_parent_class)->dispose (object);
}

static void
e_contact_store_finalize (GObject *object)
{
    EContactStorePrivate *priv;

    priv = E_CONTACT_STORE_PRIVATE (object);

    g_assert (priv->contacts == NULL);
    g_assert (priv->uid_to_seqiter == NULL);

    G_OBJECT_CLASS (e_contact_store_parent_class)->finalize (object);
}

static void
e_contact_store_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec)
{
    switch (property_id) {
    case PROP_BOOK_VIEW:
        e_contact_store_set_book_view (E_CONTACT_STORE (object), g_value_get_object (value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

static void
e_contact_store_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec)
{
    EContactStorePrivate *priv;

    priv = E_CONTACT_STORE_PRIVATE (object);

    switch (property_id) {
    case PROP_BOOK_VIEW:
        g_value_set_object (value, priv->book_view);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

static void
e_contact_store_unset_book_view (EContactStore* store)
{
    EContactStorePrivate *priv;

    priv = E_CONTACT_STORE_PRIVATE (store);

    if (NULL == priv->book_view)
        return;

    g_object_disconnect (priv->book_view,
                         "any_signal::contacts-added",   G_CALLBACK (on_book_view_contacts_added), store,
                         "any_signal::contacts-changed", G_CALLBACK (on_book_view_contacts_changed), store,
                         "any_signal::contacts-removed", G_CALLBACK (on_book_view_contacts_removed), store,
                         NULL);

    g_object_unref (priv->book_view);
    g_sequence_free (priv->contacts);
    g_hash_table_destroy (priv->uid_to_seqiter);

    priv->stamp = 0;
    priv->book_view = NULL;
    priv->contacts = NULL;
    priv->uid_to_seqiter = NULL;
}

void
e_contact_store_set_book_view (EContactStore* store, EBookView *book_view)
{
    EContactStorePrivate *priv;

    g_return_if_fail (IS_E_CONTACT_STORE (store));
    g_return_if_fail (E_IS_BOOK_VIEW (book_view) || book_view == NULL);
    priv = E_CONTACT_STORE_PRIVATE (store);

    e_contact_store_unset_book_view (store);

    priv->book_view = book_view;
    priv->stamp = GPOINTER_TO_INT (book_view);
    if (book_view) {
        g_object_ref (priv->book_view);

        priv->contacts = g_sequence_new (g_object_unref);
        priv->uid_to_seqiter = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);

        g_object_connect (priv->book_view,
                         "signal::contacts-added",   G_CALLBACK (on_book_view_contacts_added), store,
                         "signal::contacts-changed", G_CALLBACK (on_book_view_contacts_changed), store,
                         "signal::contacts-removed", G_CALLBACK (on_book_view_contacts_removed), store,
                         NULL);
    }
}

EBookView*
e_contact_store_get_book_view (EContactStore* store)
{
    EContactStorePrivate *priv;

    g_return_val_if_fail (IS_E_CONTACT_STORE (store), NULL);
    priv = E_CONTACT_STORE_PRIVATE (store);

    return priv->book_view;
}

void
e_contact_store_set_sort_func (EContactStore *store, EContactCompareFunc compare_func)
{
    EContactStorePrivate *priv;

    g_return_if_fail (IS_E_CONTACT_STORE (store));
    priv = E_CONTACT_STORE_PRIVATE (store);

    priv->compare_func = compare_func;
}

static GtkTreeModelFlags
e_contact_store_get_flags (GtkTreeModel *tree_model)
{
    g_return_val_if_fail (IS_E_CONTACT_STORE (tree_model), 0);

    return GTK_TREE_MODEL_ITERS_PERSIST | GTK_TREE_MODEL_LIST_ONLY;
}

static gint
e_contact_store_get_n_columns (GtkTreeModel *tree_model)
{
    g_return_val_if_fail (IS_E_CONTACT_STORE (tree_model), 0);

    return 1;
}

static GType
e_contact_store_get_column_type (GtkTreeModel *tree_model,
                                 gint          index)
{
    g_return_val_if_fail (IS_E_CONTACT_STORE (tree_model), G_TYPE_INVALID);

    switch (index) {
        case 0:
            return E_TYPE_CONTACT;
            break;
        default:
            break;
    }
    return G_TYPE_INVALID;
}

static gboolean
e_contact_store_get_iter (GtkTreeModel *tree_model,
                          GtkTreeIter  *iter,
                          GtkTreePath  *path)
{
    EContactStorePrivate *priv;
    gint n;

    g_return_val_if_fail (IS_E_CONTACT_STORE (tree_model), FALSE);
    g_return_val_if_fail (iter, FALSE);

    priv = E_CONTACT_STORE_PRIVATE (tree_model);

    n = gtk_tree_path_get_indices (path)[0];

    if (n < 0 || n >= g_sequence_get_length (priv->contacts))
        return FALSE;

    iter->stamp = priv->stamp;
    iter->user_data = g_sequence_get_iter_at_pos (priv->contacts, n);;

    return TRUE;
}

static GtkTreePath*
e_contact_store_get_path (GtkTreeModel *tree_model,
                          GtkTreeIter  *iter)
{
    EContactStorePrivate *priv;
    GtkTreePath *path;

    g_return_val_if_fail (IS_E_CONTACT_STORE (tree_model), NULL);
    g_return_val_if_fail (iter, NULL);
    priv = E_CONTACT_STORE_PRIVATE (tree_model);
    g_return_val_if_fail (iter->stamp == priv->stamp, NULL);

    if (g_sequence_iter_is_end (iter->user_data))
        return NULL;

    path = gtk_tree_path_new ();
    gtk_tree_path_append_index (path, g_sequence_iter_get_position (iter->user_data));

    return path;
}

static void
e_contact_store_get_value (GtkTreeModel *tree_model,
                           GtkTreeIter  *iter,
                           gint          column,
                           GValue       *value)
{
    EContactStorePrivate *priv;

    g_return_if_fail (IS_E_CONTACT_STORE (tree_model));
    g_return_if_fail (column == 0);

    priv = E_CONTACT_STORE_PRIVATE (tree_model);
    g_return_if_fail (iter->stamp == priv->stamp);

    g_value_init (value, E_TYPE_CONTACT);
    g_value_set_object (value, g_sequence_get (iter->user_data));
}

static gboolean
e_contact_store_iter_next (GtkTreeModel *tree_model,
                           GtkTreeIter  *iter)
{
    EContactStorePrivate *priv;
    GSequenceIter *next;

    g_return_val_if_fail (IS_E_CONTACT_STORE (tree_model), FALSE);
    g_return_val_if_fail (iter, FALSE);

    priv = E_CONTACT_STORE_PRIVATE (tree_model);
    g_return_val_if_fail (iter->stamp == priv->stamp, FALSE);

    next = g_sequence_iter_next (iter->user_data);
    if (g_sequence_iter_is_end (next))
        return FALSE;

    iter->user_data = next;

    return TRUE;
}

static gboolean
e_contact_store_iter_children (GtkTreeModel *tree_model,
                               GtkTreeIter  *iter,
                               GtkTreeIter  *parent)
{
    iter->stamp = 0;
    return FALSE;
}

static gboolean
e_contact_store_iter_has_child (GtkTreeModel *tree_model,
                                GtkTreeIter  *iter)
{
    return FALSE;
}

static gint
e_contact_store_iter_n_children (GtkTreeModel *tree_model,
                           GtkTreeIter  *iter)
{
    EContactStorePrivate *priv;

    g_return_val_if_fail (IS_E_CONTACT_STORE (tree_model), 0);

    if (iter) {
        return 0;
    }

    priv = E_CONTACT_STORE_PRIVATE (tree_model);

    return g_sequence_get_length (priv->contacts);
}

static gboolean
e_contact_store_iter_nth_child  (GtkTreeModel *tree_model,
                                 GtkTreeIter  *iter,
                                 GtkTreeIter  *parent,
                                 gint          n)
{
    EContactStorePrivate *priv;

    g_return_val_if_fail (IS_E_CONTACT_STORE (tree_model), FALSE);
    g_return_val_if_fail (iter, FALSE);
    priv = E_CONTACT_STORE_PRIVATE (tree_model);

    if (parent)
        return FALSE;

    if (n < 0 || n >= g_sequence_get_length (priv->contacts))
        return FALSE;
    iter->user_data = g_sequence_get_iter_at_pos (priv->contacts, n);
    return TRUE;
}

static gboolean
e_contact_store_iter_parent (GtkTreeModel *tree_model,
                             GtkTreeIter  *iter,
                             GtkTreeIter  *child)
{
    iter->stamp = 0;

    return FALSE;
}

static void
on_book_view_contacts_added (EBookView *book_view, GList *contacts, gpointer user_data)
{
    EContactStorePrivate *priv = E_CONTACT_STORE_PRIVATE (user_data);

    for (; contacts; contacts = contacts->next) {
        EContact *contact = contacts->data;
        const char *uid;
        GtkTreeIter iter;
        GtkTreePath *path;

        uid = e_contact_get_const (contact, E_CONTACT_UID);

        iter.stamp = priv->stamp;
        if (priv->compare_func) {
            iter.user_data = g_sequence_insert_sorted (priv->contacts, g_object_ref (contact), (GCompareDataFunc)priv->compare_func, NULL);
        } else {
            iter.user_data = g_sequence_append (priv->contacts, g_object_ref (contact));
        }
        g_hash_table_insert (priv->uid_to_seqiter, g_strdup (uid), iter.user_data);

        path = e_contact_store_get_path (GTK_TREE_MODEL (user_data), &iter);
        gtk_tree_model_row_inserted (GTK_TREE_MODEL (user_data), path, &iter);
        gtk_tree_path_free (path);
    }
}

static GHashTable*
store_old_order (GSequence *seq)
{
    GHashTable *iters = NULL;
    GSequenceIter *iter;
    int i;

    iters = g_hash_table_new (g_direct_hash, g_direct_equal);

    iter = g_sequence_get_begin_iter (seq);
    for (i = 0; !g_sequence_iter_is_end (iter); i++, iter = g_sequence_iter_next (iter)) {
        g_hash_table_insert (iters, iter, GINT_TO_POINTER (i));
    }
    return iters;
}

static int*
old_to_new_order (GHashTable* iters, GSequence *seq)
{
    int i, *new_order;
    GSequenceIter *iter;

    new_order = g_malloc0 (sizeof (int) * g_hash_table_size (iters));

    iter = g_sequence_get_begin_iter (seq);
    for (i = 0; !g_sequence_iter_is_end (iter); i++, iter = g_sequence_iter_next (iter)) {
        int old_pos;

        old_pos = GPOINTER_TO_INT (g_hash_table_lookup (iters, iter));
        new_order[i] = old_pos;
    }
    return new_order;
}

static void
on_book_view_contacts_changed (EBookView *book_view, GList *contacts, gpointer user_data)
{
    EContactStorePrivate *priv = E_CONTACT_STORE_PRIVATE (user_data);
    GHashTable *iters = NULL;

    if (priv->compare_func) {
        iters = store_old_order (priv->contacts);
    }

    for (; contacts; contacts = contacts->next) {
        EContact *contact = contacts->data;
        const char *uid;
        GtkTreeIter iter;
        GtkTreePath *path;

        uid = e_contact_get_const (contact, E_CONTACT_UID);

        iter.stamp = priv->stamp;
        iter.user_data = g_hash_table_lookup (priv->uid_to_seqiter, uid);
        g_sequence_set (iter.user_data, g_object_ref (contact));
        if (priv->compare_func) {
            g_sequence_sort_changed (iter.user_data, (GCompareDataFunc)priv->compare_func, NULL);
        }

        path = e_contact_store_get_path (GTK_TREE_MODEL (user_data), &iter);
        gtk_tree_model_row_changed (GTK_TREE_MODEL (user_data), path, &iter);
        gtk_tree_path_free (path);
    }

    if (priv->compare_func) {
        int *new_order;
        GtkTreePath *path;

        new_order = old_to_new_order (iters, priv->contacts);
        g_hash_table_destroy (iters);

        path = gtk_tree_path_new ();
        gtk_tree_model_rows_reordered (GTK_TREE_MODEL (user_data), path, NULL, new_order);
        gtk_tree_path_free (path);

        g_free (new_order);
    }
}

static void
on_book_view_contacts_removed (EBookView *book_view, GList *uids, gpointer user_data)
{
    EContactStorePrivate *priv = E_CONTACT_STORE_PRIVATE (user_data);

    for (; uids; uids = uids->next) {
        const char *uid = uids->data;
        GtkTreeIter iter;
        GtkTreePath *path;

        iter.stamp = priv->stamp;
        iter.user_data = g_hash_table_lookup (priv->uid_to_seqiter, uid);
        path = e_contact_store_get_path (GTK_TREE_MODEL (user_data), &iter);
        g_sequence_remove (iter.user_data);

        gtk_tree_model_row_deleted (GTK_TREE_MODEL (user_data), path);
        gtk_tree_path_free (path);
    }
}

