/* util.c - Some misc libebook utils
 *
 * Copyright (C) 2008 Nokia Corp.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Joergen Scheibengruber <jorgen.scheibengruber AT nokia.com>
 */

#include "util.h"

EBook*
_e_book_new_from_argv (const char **args, GError **error)
{
    EBook *book;
    if (args && args[0]) {
        if (args[1]) {
            ESource *source;
            ESourceGroup *group;

            group = e_source_group_new ("", args[0]);
            source = e_source_new ("", "");
            e_source_set_group (source, group);
            e_source_set_relative_uri (source, NULL);

            args++;
            while (*args) {
                char **key_value;

                key_value = g_strsplit (*args, "=", 2);
                if (key_value && key_value[0] && key_value[1]) {
                    e_source_set_property (source, key_value[0], key_value[1]);
                } else {
                    g_set_error (error, G_OPTION_ERROR,
                                        G_OPTION_ERROR_BAD_VALUE,
                                        "Property '%s' does not have the format key=value",
                                        *args);
                }
                g_strfreev (key_value);
                if (*error) {
                    break;
                }
                args++;
            }
            if (*error) {
                book = NULL;
            } else {
                book = e_book_new (source, error);
            }

            g_object_unref (source);
            g_object_unref (group);
        } else {
            book = e_book_new_from_uri (args [0], error);
        }
    } else {
        book = e_book_new_system_addressbook (error);
    }
    return book;
}

static inline gboolean
is_not_empty (char *str)
{
    if (!str) {
        return FALSE;
    }
    if (!str[0]) {
        g_free (str);
        return FALSE;
    }
    return TRUE;
}

const char *
_e_contact_get_name (EContact *contact)
{
    EContactName* name;
    char *result;
    char *cached_name;

    cached_name = g_object_get_data (G_OBJECT (contact), "cached::name");
    if (cached_name) {
        return cached_name;
    }

    name = e_contact_get (contact, E_CONTACT_NAME);
    if (name) {
        result = e_contact_name_to_string (name);
        e_contact_name_free (name);
        if (is_not_empty (result)) {
            goto have_name;
        }
    }
    result = e_contact_get (contact, E_CONTACT_FULL_NAME);
    if (is_not_empty (result)) {
        goto have_name;
    }
    result = e_contact_get (contact, E_CONTACT_NICKNAME);
    if (is_not_empty (result)) {
        goto have_name;
    }
    result = e_contact_get (contact, E_CONTACT_ORG);
    if (is_not_empty (result)) {
        goto have_name;
    }
    result = g_strdup ("no name");

have_name:
    g_object_set_data_full (G_OBJECT (contact), "cached::name", result, g_free);

    return result;
}

#ifndef MAEMO_CHANGES
gboolean
e_book_add_contacts (EBook *book, GList *contacts, GError **error)
{
    gboolean rv = TRUE;
    while (contacts) {
        rv = e_book_add_contact (book, E_CONTACT (contacts->data), error);
        if (!rv) {
            return rv;
        }
        contacts = contacts->next;
    }
    return TRUE;
}

gboolean
e_book_commit_contacts (EBook *book, GList *contacts, GError **error)
{
    gboolean rv = TRUE;
    while (contacts) {
        rv = e_book_commit_contact (book, E_CONTACT (contacts->data), error);
        if (!rv) {
            return rv;
        }
        contacts = contacts->next;
    }
    return TRUE;
}
#endif

