/* ebook-editor.c - A UI to edit the vcards from an Ebook
 *
 * Copyright (C) 2008 Nokia Corp.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Joergen Scheibengruber <jorgen.scheibengruber AT nokia.com>
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <libebook/e-book.h>
#include "util.h"
#include "book-editor.h"

static gboolean verbose = FALSE;
static char **args = NULL;
static char *query = NULL;

static GOptionEntry entries[] =
{
    { "verbose", 'v', 0, G_OPTION_ARG_NONE, &verbose, "Be verbose", NULL },
    { "query",   'q', 0, G_OPTION_ARG_STRING, &query, "An EBookQuery as a string", "sexp-query" },
    { G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_STRING_ARRAY, &args, NULL, "[uri [key1=value1 ...] " },
    { 0, 0, 0, 0, 0, 0, NULL }
};

static void
on_book_auth_required (EBook *book, gpointer user_data)
{
    ESource *source;
    const char *username, *password;
    GError *error = NULL;

    source = e_book_get_source (book);
    username = e_source_get_property (source, "username");
    password = e_source_get_property (source, "password");
    if (verbose) {
        g_debug ("Book requires authorization");
    }

    e_book_authenticate_user (book, username ? username : "", password ? password : "", "plain/password", &error);
    if (error) {
        g_printerr ("Authentication for user '%s' failed: %s\n", username, error->message);
        g_clear_error (&error);
        gtk_main_quit ();
    }
}

static void
on_book_backend_died (EBook *book, gpointer user_data)
{
    g_critical ("Backend is buggy");
    gtk_main_quit ();
}

static void
on_book_connection_status (EBook *book, gboolean connected, gpointer user_data)
{
    if (verbose) {
        g_debug ("Book is now %s", connected ? "connected" : "disconnected");
    }
}

static void
on_book_writable_status (EBook *book, gboolean writable, gpointer user_data)
{
    if (verbose) {
        g_debug ("Book is now %s", writable ? "writable" : "read only");
    }
}

static void
on_book_view_sequence_complete (EBookView *book_view, int status, gpointer user_data)
{
    GTimer *timer = user_data;
    if (verbose) {
        g_debug ("Sequence-complete with status %d after %.2f seconds",
                 status, g_timer_elapsed (timer, NULL));
    }
}

static EBookView*
open_book_view (const char **args, const char *query_string, GError **error)
{
    EBook *book;
    EBookQuery *query;
    EBookView *book_view = NULL;

    book = _e_book_new_from_argv (args, error);
    if (*error) {
        return NULL;
    }

    g_object_connect (G_OBJECT (book),
                      "signal::auth-required", G_CALLBACK (on_book_auth_required), NULL,
                      "signal::backend-died", G_CALLBACK (on_book_backend_died), NULL,
                      "signal::connection-status", G_CALLBACK (on_book_connection_status), NULL,
                      "signal::writable-status", G_CALLBACK (on_book_writable_status), NULL,
                      NULL);

    e_book_open (book, FALSE, error);
    if (*error) {
        return NULL;
    }

    if (NULL == query_string) {
        query_string = "(contains \"x-evolution-any-field\" \"\")";
    }
    g_debug ("Query is '%s'", query_string);
    query = e_book_query_from_string (query_string);
    if (NULL == query) {
        g_set_error (error,
                     E_BOOK_ERROR,
                     E_BOOK_ERROR_INVALID_ARG,
                     "invalid query: %s", query_string);
        return NULL;
    }
    e_book_get_book_view (book, query, NULL, 0, &book_view, error);
    g_object_unref (book);
    e_book_query_unref (query);

    if (*error) {
        return NULL;
    }

    return book_view;
}
#if 0

#define MAX_VALUES_LEN 100
#define MAX_PARAMS_LEN 100

static void
vcard_values_cell_data_func (GtkTreeViewColumn *tree_column,
                             GtkCellRenderer *cell,
                             GtkTreeModel *tree_model,
                             GtkTreeIter *iter,
                             gpointer data)
{
    GList *values;
    GString *s;
    gboolean editable = TRUE;

    gtk_tree_model_get (tree_model, iter, 3, &values, -1);

    s = g_string_new ("");

    for (;values; values = values->next) {
        g_string_append (s, values->data);
        if (values->next) {
            g_string_append (s, ";");
        }
    }

    if (s->len > MAX_VALUES_LEN) {
        g_string_truncate (s, MAX_VALUES_LEN);
        g_string_append (s, " ...");
        editable = FALSE;
    }

    g_object_set (cell, "text", s->str, "editable", editable, NULL);
    g_string_free (s, TRUE);
}

static void
vcard_params_cell_data_func (GtkTreeViewColumn *tree_column,
                             GtkCellRenderer *cell,
                             GtkTreeModel *tree_model,
                             GtkTreeIter *iter,
                             gpointer data)
{
    GList *params;
    GString *s;
    gboolean editable = TRUE;

    gtk_tree_model_get (tree_model, iter, 2, &params, -1);

    s = g_string_new ("");

    for (;params; params = params->next) {
        EVCardAttributeParam *param = params->data;
        const char *name = e_vcard_attribute_param_get_name (param);
        GList *values = e_vcard_attribute_param_get_values (param);
        if (name) {
            g_string_append (s, name);
            if (values)
                g_string_append (s, "=");
        }
        for (;values; values = values->next) {
            g_string_append (s, values->data);
            if (values->next) {
                g_string_append (s, ",");
            }
        }
        if (params->next) {
            g_string_append (s, ";");
        }
    }

    if (s->len > MAX_PARAMS_LEN) {
        g_string_truncate (s, MAX_PARAMS_LEN);
        g_string_append (s, " ...");
        editable = FALSE;
    }

    g_object_set (cell, "text", s->str, "editable", editable, NULL);
    g_string_free (s, TRUE);
}

static void
set_vcard_store_from_contact (GtkTreeView *vcard_treeview, EContact *contact)
{
    EContact *copy;
    EVCardStore *vcard_store;

    copy = e_contact_duplicate (contact);
    vcard_store = e_vcard_store_new_from_vcard (E_VCARD (copy));
    gtk_tree_view_set_model (vcard_treeview, GTK_TREE_MODEL (vcard_store));
    g_object_unref (vcard_store);

    g_object_unref (copy);
}

static void
on_contacts_treeview_cursor_changed (GtkTreeView *contact_treeview, gpointer user_data)
{
    GtkTreeView *vcard_treeview = user_data;
    GtkTreePath *path;

    gtk_tree_view_get_cursor (contact_treeview, &path, NULL);

    if (path) {
        GtkTreeIter iter;
        EContact *contact;

        gtk_tree_model_get_iter (gtk_tree_view_get_model (contact_treeview), &iter, path);
        gtk_tree_path_free (path);

        gtk_tree_model_get (gtk_tree_view_get_model (contact_treeview), &iter, 0, &contact, -1);

        set_vcard_store_from_contact (vcard_treeview, contact);
        g_object_unref (contact);
    }
}

static void
on_contacts_store_row_changed (GtkTreeModel *contact_store, GtkTreePath *changed_path, GtkTreeIter *iter, gpointer user_data)
{
    GtkTreeView *contact_treeview = user_data;
    GtkTreeView *vcard_treeview;
    GtkTreePath *selected_path;

    gtk_tree_view_get_cursor (contact_treeview, &selected_path, NULL);

    if (selected_path) {
        if (0 == gtk_tree_path_compare (changed_path, selected_path)) {
            EContact *contact;

            gtk_tree_model_get (contact_store, iter, 0, &contact, -1);
            vcard_treeview = g_object_get_data (G_OBJECT (contact_treeview), "vcard-treeview");

            set_vcard_store_from_contact (vcard_treeview, contact);
            g_object_unref (contact);
        }
        gtk_tree_path_free (selected_path);
    }
}

static void
on_vcard_attribute_values_edited (GtkCellRendererText *renderer,
                                  char                *path_string,
                                  char                *new_text,
                                  gpointer             user_data)
{
    GtkTreeView *vcard_treeview = user_data;
    GtkTreeModel *model;
    GtkTreeIter iter;
    GtkTreePath *path;
    char **vals;
    GList *values;
    int i;

    model = gtk_tree_view_get_model (vcard_treeview);

    vals = g_strsplit (new_text, ";", -1);

    for (i = 0, values = NULL; vals[i]; i++) {
        values = g_list_append (values, g_strdup (vals[i]));
    }
    g_strfreev (vals);

    path = gtk_tree_path_new_from_string (path_string);
    gtk_tree_model_get_iter (model, &iter, path);
    gtk_tree_path_free (path);
    e_vcard_store_set_attribute_values (E_VCARD_STORE (model), &iter, values);
}

static void
on_vcard_attribute_params_edited (GtkCellRendererText *renderer,
                                 char                *path_string,
                                 char                *new_text,
                                 gpointer             user_data)
{
    GtkTreeView *vcard_treeview = user_data;
    GtkTreeModel *model;
    GtkTreeIter iter;
    GtkTreePath *path;
    char **strs;
    GList *params;
    int i;

    model = gtk_tree_view_get_model (vcard_treeview);

    strs = g_strsplit (new_text, ";", -1);

    for (i = 0, params = NULL; strs[i]; i++) {
        EVCardAttributeParam *param;
        char **name_rest;

        name_rest = g_strsplit (strs[i], "=", 2);

        if (NULL == name_rest)
            continue;
        g_assert (name_rest[0]);

        param = e_vcard_attribute_param_new (name_rest[0]);
        if (name_rest[1]) {
            char **values;
            int j;

            values = g_strsplit (name_rest[1], ",", -1);
            for (j = 0; values[j]; j++) {
                e_vcard_attribute_param_add_value (param, values[j]);
            }
            g_strfreev (values);
        }
        g_strfreev (name_rest);
        params = g_list_append (params, param);
    }
    g_strfreev (strs);

    path = gtk_tree_path_new_from_string (path_string);
    gtk_tree_model_get_iter (model, &iter, path);
    gtk_tree_path_free (path);
    e_vcard_store_set_attribute_params (E_VCARD_STORE (model), &iter, params);
}

static void
commit_selected_contact (GtkTreeView *contact_treeview)
{
    GtkTreeView *vcard_treeview;
    GtkTreeModel *vcard_store, *contact_store;
    EBookView *book_view;
    EBook *book;
    EVCard *contact;
    GError *error = NULL;
    const char *name;

    vcard_treeview = g_object_get_data (G_OBJECT (contact_treeview), "vcard-treeview");
    vcard_store = gtk_tree_view_get_model (vcard_treeview);
    contact = e_vcard_store_get_vcard (E_VCARD_STORE (vcard_store));

    contact_store = gtk_tree_view_get_model (contact_treeview);
    book_view = e_contact_store_get_book_view (E_CONTACT_STORE (contact_store));
    book = e_book_view_get_book (book_view);
    name = e_contact_get_const (E_CONTACT (contact), E_CONTACT_FULL_NAME);

    if (verbose) {
        g_debug ("Commiting contact %s:", name);
        e_vcard_dump_structure (contact);
    }
    e_book_commit_contact (book, E_CONTACT (contact), &error);
    if (error) {
        g_printerr ("Could not commit contact %s: %s\n", name, error->message);
        g_clear_error (&error);

        /* This will update the vcard treeview to show the original again */
        on_contacts_treeview_cursor_changed (contact_treeview, vcard_treeview);
    }
}

static void
delete_selected_vcard_attribute (GtkTreeView *vcard_treeview)
{
    GtkTreePath *path;
    gtk_tree_view_get_cursor (vcard_treeview, &path, NULL);

    if (path) {
        GtkTreeModel *vcard_store;
        GtkTreeIter iter;

        vcard_store = gtk_tree_view_get_model (vcard_treeview);
        gtk_tree_model_get_iter (vcard_store, &iter, path);
        gtk_tree_path_free (path);

        e_vcard_store_remove_attribute (E_VCARD_STORE (vcard_store), &iter);
    }
}

static void
on_add_vcard_attribute_dialog_response (GtkDialog *dialog, int response, gpointer user_data)
{
    GtkTreeView *vcard_treeview = user_data;
    GtkTreeModel *vcard_store;
    GtkEntry *entry;
    const char *text;

    vcard_store = gtk_tree_view_get_model (vcard_treeview);

    if (response == GTK_RESPONSE_OK) {
        entry = g_object_get_data (G_OBJECT (dialog), "entry");

        text = gtk_entry_get_text (GTK_ENTRY (entry));
        e_vcard_store_append_attribute (E_VCARD_STORE (vcard_store), text, "");
    }

    gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
add_vcard_attribute (GtkTreeView *vcard_treeview)
{
    GtkWidget *dialog;
    GtkWidget *entry;
    
    dialog = gtk_dialog_new_with_buttons ("Enter Attribute-name", NULL,
                                          GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
                                          GTK_STOCK_OK, GTK_RESPONSE_OK,
                                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                          NULL);
    gtk_window_set_default_size (GTK_WINDOW (dialog), 300, -1);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);

    entry = gtk_entry_new ();
    gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), entry, TRUE, TRUE, 10);
    g_object_set_data (G_OBJECT (dialog), "entry", entry);

    gtk_widget_show_all (dialog);
    
    g_signal_connect (dialog, "response", G_CALLBACK (on_add_vcard_attribute_dialog_response), vcard_treeview);
}

static gboolean
on_contact_treeview_key_press_event (GtkWidget *contact_treeview, GdkEventKey *event, gpointer user_data)
{
    switch (event->keyval) {
        case GDK_a:
        case GDK_A:
            add_new_contact (GTK_TREE_VIEW (contact_treeview));
        break;
        case GDK_d:
        case GDK_D:
            delete_selected_contact (GTK_TREE_VIEW (contact_treeview));
        break;
        case GDK_q:
        case GDK_Q:
        case GDK_Escape:
            gtk_main_quit ();
        break;
        default:
            return FALSE;
    }

    return TRUE;
}

static gboolean
on_vcard_treeview_key_press_event (GtkWidget *vcard_treeview, GdkEventKey *event, gpointer user_data)
{
    GtkTreeView *contact_treeview = user_data;

    switch (event->keyval) {
        case GDK_a:
        case GDK_A:
            add_vcard_attribute (GTK_TREE_VIEW (vcard_treeview));
        break;
        case GDK_d:
        case GDK_D:
            delete_selected_vcard_attribute (GTK_TREE_VIEW (vcard_treeview));
        break;
        case GDK_s:
        case GDK_S:
            commit_selected_contact (GTK_TREE_VIEW (contact_treeview));
        break;
        case GDK_q:
        case GDK_Q:
        case GDK_Escape:
            gtk_main_quit ();
        break;
        default:
            return FALSE;
    }

    return TRUE;
}
#endif

int main (int argc, char* argv[])
{
    GError *error = NULL;
    GOptionContext *context;
    EBookView *book_view;
    GtkWidget *book_editor;
    GTimer *timer = NULL;
    char *display_uri, *window_title;

    g_set_application_name ("Ebook Editor");

    context = g_option_context_new ("- a small libebook application with UI");
    g_option_context_add_main_entries (context, entries, "");
    g_option_context_parse (context, &argc, &argv, &error);
    g_option_context_free (context);

    if (error) {
        g_printerr ("Could not parse arguments: %s\n", error->message);
        g_clear_error (&error);

        return 1;
    }

    gtk_init (&argc, &argv);

    display_uri = g_strdup (args ? args[0] : "system addressbook");

    book_view = open_book_view ((const char**)args, query, &error);
    if (error) {
        g_printerr ("Could not open %s: %s\n", display_uri, error->message);
        g_clear_error (&error);

        return 1;
    }
    g_strfreev (args);

    book_editor = book_editor_new (book_view);
    g_object_unref (book_view);

    gtk_window_set_default_size (GTK_WINDOW (book_editor), 800, 400);
    window_title = g_strdup_printf ("EBook Editor - %s", display_uri);
    gtk_window_set_title (GTK_WINDOW (book_editor), window_title);
    g_free (window_title);
    g_free (display_uri);

#if 0
    /* VCard pane */

    vcard_treeview = gtk_tree_view_new ();
    g_object_set_data (G_OBJECT (contact_treeview), "vcard-treeview", vcard_treeview);
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (vcard_treeview), FALSE);

    column = gtk_tree_view_column_new ();
    gtk_tree_view_append_column (GTK_TREE_VIEW (vcard_treeview), column);
    name_renderer = gtk_cell_renderer_text_new ();
    gtk_tree_view_column_pack_start (column, name_renderer, FALSE);
    gtk_tree_view_column_add_attribute (column, name_renderer, "text", 0);

    values_renderer = gtk_cell_renderer_text_new ();
    g_object_set (values_renderer, "editable", TRUE, NULL);
    gtk_tree_view_column_pack_start (column, values_renderer, FALSE);
    gtk_tree_view_column_set_cell_data_func (column, values_renderer, vcard_values_cell_data_func, NULL, NULL);

    params_renderer = gtk_cell_renderer_text_new ();
    g_object_set (params_renderer, "editable", TRUE, NULL);
    gtk_tree_view_column_pack_start (column, params_renderer, FALSE);
    gtk_tree_view_column_set_cell_data_func (column, params_renderer, vcard_params_cell_data_func, NULL, NULL);

    sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_add (GTK_CONTAINER (sw), vcard_treeview);
    gtk_paned_add2 (GTK_PANED (paned), sw);
#endif

    gtk_widget_show_all (book_editor);

    g_signal_connect (book_editor, "delete-event", gtk_main_quit, NULL);
#if 0
    g_signal_connect (contact_treeview, "key-press-event", G_CALLBACK (on_contact_treeview_key_press_event), NULL);
    g_signal_connect (vcard_treeview, "key-press-event", G_CALLBACK (on_vcard_treeview_key_press_event), contact_treeview);
    g_signal_connect (contact_treeview, "cursor-changed", G_CALLBACK (on_contacts_treeview_cursor_changed), vcard_treeview);
    g_signal_connect (contact_store, "row-changed", G_CALLBACK (on_contacts_store_row_changed), contact_treeview);
    g_signal_connect (values_renderer, "edited", G_CALLBACK (on_vcard_attribute_values_edited), vcard_treeview);
    g_signal_connect (params_renderer, "edited", G_CALLBACK (on_vcard_attribute_params_edited), vcard_treeview);
#endif

    timer = g_timer_new ();
    g_signal_connect (book_view, "sequence-complete", G_CALLBACK (on_book_view_sequence_complete), timer);

    gtk_main ();

    g_timer_destroy (timer);

    return 0;
}
