/* ebook-client.c - A small general purpose ebook cmdline client
 *
 * Copyright (C) 2008 Nokia Corp.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Joergen Scheibengruber <jorgen.scheibengruber AT nokia.com>
 */

#include "config.h"
#include <string.h>
#include <libebook/e-book.h>
#include "util.h"

static char **args = NULL;
static char *query_string = NULL;
static char *live_query_string = NULL;
static char *add_filename = NULL;
static char *modify_filename = NULL;
static char *delete_uids = NULL;
static char *get_uids = NULL;
static char *change_id = NULL;
static char print_uids = FALSE;

static GOptionEntry entries[] =
{
    { "query",      'q', 0, G_OPTION_ARG_STRING, &query_string, "Either \"all\", a valid EBookQuery string or a free string", "John" },
    { "live-query", 'l', 0, G_OPTION_ARG_STRING, &live_query_string, "Either \"all\", a valid EBookQuery string or a free string", "John" },
    { "get",        'g', 0, G_OPTION_ARG_STRING, &get_uids, "A list of EContact uids, or \"-\" to read this list from stdin", "1 2 3" },
    { "add",        'a', 0, G_OPTION_ARG_FILENAME, &add_filename, "An file containing vcards, or \"-\" for stdin", "sample.vcf" },
    { "modify",     'm', 0, G_OPTION_ARG_FILENAME, &modify_filename, "An file containing vcards, or \"-\" for stdin", "sample.vcf" },
    { "delete",     'd', 0, G_OPTION_ARG_STRING, &delete_uids, "A list of EContact uids, or \"-\" to read this list from stdin", "1 2 3" },
    { "print-uids", 'u', 0, G_OPTION_ARG_NONE, &print_uids, "Print only the EContact uid, not the whole vcard", "" },
    { "get-changes",'c', 0, G_OPTION_ARG_STRING, &change_id, "Get the list of changes for the provided change-id", "change-track-1" },
    { G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_STRING_ARRAY, &args, NULL, "[uri property=value ...] " },
    { 0, 0, 0, 0, 0, 0, NULL }
};

static EBookQuery *
query_string_to_ebookquery (const char *query_string)
{
    EBookQuery *query;
    char *query_string2;

    if (0 == strcmp (query_string, "all")) {
        query = e_book_query_from_string ("(contains \"x-evolution-any-field\" \"\")");
    } else {
        if (query_string[0] == '(') {
            query = e_book_query_from_string (query_string);
        } else {
            query_string2 = g_strdup_printf ("(contains \"x-evolution-any-field\" \"%s\")", query_string);
            query = e_book_query_from_string (query_string2);
            g_free (query_string2);
        }
    }
    if (NULL == query) {
        return FALSE;
    }

    query_string2 = e_book_query_to_string (query);
    g_printerr ("Running query %s\n", query_string2);
    g_free (query_string2);

    return query;
}

static void
print_contact (EContact *contact)
{
    if (print_uids) {
        const char *uid;
        uid = e_contact_get_const (contact, E_CONTACT_UID);
        g_print ("%s\n", uid);
    } else {
        char *vcard;
        vcard = e_vcard_to_string (E_VCARD (contact), EVC_FORMAT_VCARD_30);
        g_print ("%s\n", vcard);
        g_free (vcard);
    }
}

static void
print_contacts (GList *contacts, gboolean free_them)
{
    while (contacts) {
        EContact *contact = contacts->data;

        print_contact (contact);
        if (free_them) {
            contacts = g_list_delete_link (contacts, contacts);
            g_object_unref (contact);
        } else {
            contacts = contacts->next;
        }
    }
}

gboolean
query_contacts (EBook *book, const char *query_string, int *count, GError **error)
{
    EBookQuery *query;
    GList *contacts;
    gboolean rv;

    query = query_string_to_ebookquery (query_string);

    rv = e_book_get_contacts (book, query, &contacts, error);
    e_book_query_unref (query);
    if (FALSE == rv) {
        return FALSE;
    }

    if (count) {
        *count = g_list_length (contacts);
    }
    print_contacts (contacts, TRUE);
    return TRUE;
}

static void
on_contacts_added (EBookView *book_view, GList *contacts, gpointer user_data)
{
    g_printerr ("Contacts added: %d\n", g_list_length (contacts));
    print_contacts (contacts, FALSE);
}

static void
on_contacts_changed (EBookView *book_view, GList *contacts, gpointer user_data)
{
    g_printerr ("Contacts changed: %d\n", g_list_length (contacts));
    print_contacts (contacts, FALSE);
}

static void
on_contacts_removed (EBookView *book_view, GList *contact_uids, gpointer user_data)
{
    g_printerr ("Contacts removed: %d\n", g_list_length (contact_uids));
    for (;contact_uids; contact_uids = contact_uids->next) {
        g_print ("%s", (const char*)contact_uids->data);
    }
}

static void
on_sequence_complete (EBookView *book_view, EBookViewStatus status, gpointer user_data)
{
    g_printerr ("Sequence complete: %d\n", status);
}

gboolean
start_book_view (EBook *book, const char *query_string, GError **error)
{
    EBookQuery *query;
    EBookView *book_view = NULL;
    gboolean rv;
    GMainLoop *loop = NULL;

    query = query_string_to_ebookquery (query_string);

    rv = e_book_get_book_view (book, query, NULL, 0, &book_view, error);
    e_book_query_unref (query);
    if (FALSE == rv) {
        return FALSE;
    }

    g_signal_connect (G_OBJECT (book_view), "contacts-added", G_CALLBACK (on_contacts_added), NULL);
    g_signal_connect (G_OBJECT (book_view), "contacts-changed", G_CALLBACK (on_contacts_changed), NULL);
    g_signal_connect (G_OBJECT (book_view), "contacts-removed", G_CALLBACK (on_contacts_removed), NULL);
    g_signal_connect (G_OBJECT (book_view), "sequence-complete", G_CALLBACK (on_sequence_complete), NULL);

    loop = g_main_loop_new (NULL, FALSE);

    e_book_view_start (book_view);
    g_main_loop_run (loop);

    return TRUE;
}

gboolean
add_modify_contacts (EBook *book, gboolean create, const char *filename, int *count, GError **error)
{
    char *contents = NULL;
    char **vcards = NULL;
    GList *contacts = NULL;
    gboolean rv;
    int i = -1;

    if (0 == strcmp (filename, "-")) {
        filename = "/dev/stdin";
    }
    rv = g_file_get_contents (filename, &contents, NULL, error);
    if (FALSE == rv) {
        goto out;
    }

    vcards = g_strsplit (contents, "END:VCARD\n", -1);
    g_free (contents);
    for (i = 0; vcards[i] && vcards[i][0]; i++) {
        EContact *contact;
        char *vcard;

        vcard = g_strdup_printf ("%sEND:VCARD\n", vcards[i]);
        contact = e_contact_new_from_vcard (vcard);
        if (contact) {
            if (!create) {
                const char *uid, *name_or_org;
                uid = e_contact_get_const (contact, E_CONTACT_UID);
                name_or_org = e_contact_get_const (contact, E_CONTACT_NAME_OR_ORG);

                if (!uid) {
                    g_printerr ("Contact %s did not have a uid\n", name_or_org);
                    g_object_unref (contact);
                    continue;
                }
            }
            contacts = g_list_prepend (contacts, contact);
        }
    }
    contacts = g_list_reverse (contacts);
    if (create) {
        rv = e_book_add_contacts (book, contacts, error);
    } else {
        rv = e_book_commit_contacts (book, contacts, error);
    }
    g_list_foreach (contacts, (GFunc)g_object_unref, NULL);
    g_list_free (contacts);
out:
    if (count) {
        *count = i;
    }
    g_strfreev (vcards);

    return rv;
}

gboolean
add_contacts (EBook *book, const char *filename, int *count, GError **error)
{
    return add_modify_contacts (book, TRUE, filename, count, error);
}

gboolean
modify_contacts (EBook *book, const char *filename, int *count, GError **error)
{
    return add_modify_contacts (book, FALSE, filename, count, error);
}

static GList*
obtain_uids (const char* uid_string, GError **error)
{
    GList *l = NULL;
    char **uids = NULL;
    gboolean rv;
    int i;

    if (0 == strcmp (uid_string, "-")) {
        char *contents = NULL;
        rv = g_file_get_contents ("/dev/stdin", &contents, NULL, error);
        if (FALSE == rv) {
            return NULL;
        }
        uids = g_strsplit_set (contents, " \n", -1);
        g_free (contents);
    } else {
        uids = g_strsplit_set (uid_string, " \n", -1);
    }

    if (NULL == uids) {
        return NULL;
    }

    for (i = 0; uids[i]; i++) {
        if (uids[i][0]) {
            l = g_list_prepend (l, g_strdup (uids[i]));
        }
    }
    g_strfreev (uids);

    return l;
}

gboolean
get_contacts (EBook *book, const char *uid_string, int *count, GError **error)
{
    GList *l, *uids = NULL, *contacts = NULL;
    gboolean rv = FALSE;
    int nr = -1;

    uids = obtain_uids (uid_string, error);

    for (l = uids; l; l = l->next) {
        EContact *contact;

        rv = e_book_get_contact (book, l->data, &contact, error);
        if (FALSE == rv) {
            goto out;
        }
        contacts = g_list_append (contacts, contact);
    }
    nr = g_list_length (contacts);
out:
    if (count) {
        *count = nr;
    }
    if (contacts) {
        print_contacts (contacts, TRUE);
    }
    g_list_foreach (uids, (GFunc)g_free, NULL);
    g_list_free (uids);

    return rv;
}

gboolean
get_changes (EBook *book, const char *change_id, GError **error)
{
    GList *l, *changes = NULL;
    gboolean rv = FALSE;

    rv = e_book_get_changes (book, change_id, &changes, error);
    if (FALSE == rv) {
        return FALSE;
    }
    for (l = changes; l; l = l->next) {
        EBookChange *change = l->data;

        switch (change->change_type) {
            case E_BOOK_CHANGE_CARD_ADDED:
                g_printerr ("--ADD--\n");
                break;
            case E_BOOK_CHANGE_CARD_DELETED:
                g_printerr ("--DELETED--\n");
                break;
            case E_BOOK_CHANGE_CARD_MODIFIED:
                g_printerr ("--MODIFIED--\n");
                break;
            default:
                break;
        }
        print_contact (change->contact);
    }

    return rv;
}

gboolean
delete_contacts (EBook *book, const char *uid_string, int *count, GError **error)
{
    GList *uids = NULL;
    gboolean rv = FALSE;
    int nr_del = -1;


    uids = obtain_uids (uid_string, error);
    if (FALSE == uids) {
        goto out;
    }

    rv = e_book_remove_contacts (book, uids, error);
    if (FALSE == rv) {
        goto out;
    }
    nr_del = g_list_length (uids);
out:
    if (count) {
        *count = nr_del;
    }
    g_list_foreach (uids, (GFunc)g_free, NULL);
    g_list_free (uids);

    return rv;
}


int main (int argc, char *argv [])
{
    GError *error = NULL;
    GOptionContext *context;
    EBook *book;
    int count;

    g_type_init ();

    context = g_option_context_new ("- a small libebook commandline client");
    g_option_context_add_main_entries (context, entries, "");
    g_option_context_parse (context, &argc, &argv, &error);
    g_option_context_free (context);

    if (error) {
        g_printerr ("Could not parse arguments: %s\n", error->message);
        g_clear_error (&error);

        return 1;
    }

    book = _e_book_new_from_argv ((const char**)args, &error);
    if (error) {
        g_printerr ("Could not find %s: %s\n", args ? args[0] : "system addressbook", error->message);
        g_clear_error (&error);

        return 1;
    }

    e_book_open (book, TRUE, &error);
    if (error) {
        g_printerr ("Could not open %s: %s\n", args ? args[0] : "system addressbook", error->message);
        g_clear_error (&error);
        return 1;
    }
    g_strfreev (args);

    if (query_string) {
        query_contacts (book, query_string, &count, &error);
        g_free (query_string);
        if (error) {
            g_printerr ("Could not get contacts: %s\n", error->message);
            g_clear_error (&error);
            return 1;
        } else {
            g_printerr ("Found: %d contacts\n", count);
        }
    }
    if (get_uids) {
        get_contacts (book, get_uids, &count, &error);
        g_free (get_uids);
        if (error) {
            g_printerr ("Could not get contacts: %s\n", error->message);
            g_clear_error (&error);
            return 1;
        } else {
            g_printerr ("Got %d contacts\n", count);
        }
    }
    if (add_filename) {
        add_contacts (book, add_filename, &count, &error);
        g_free (add_filename);
        if (error) {
            g_printerr ("Could not add contacts: %s\n", error->message);
            g_clear_error (&error);
            return 1;
        } else {
            g_printerr ("Successfully added %d contacts\n", count);
        }
    }
    if (modify_filename) {
        modify_contacts (book, modify_filename, &count, &error);
        g_free (modify_filename);
        if (error) {
            g_printerr ("Could not modify contacts: %s\n", error->message);
            g_clear_error (&error);
            return 1;
        } else {
            g_printerr ("Successfully modified %d contacts\n", count);
        }
    }
    if (delete_uids) {
        delete_contacts (book, delete_uids, &count, &error);
        g_free (delete_uids);
        if (error) {
            g_printerr ("Could not delete contacts: %s\n", error->message);
            g_clear_error (&error);
            return 1;
        } else {
            g_printerr ("Successfully deleted %d contacts\n", count);
        }
    }
    if (change_id) {
        get_changes (book, change_id, &error);
        if (error) {
            g_printerr ("Could not get changes for id '%s': %s\n", change_id, error->message);
            g_free (change_id);
            g_clear_error (&error);
            return 1;
        } else {
            g_free (change_id);
        }
    }
    if (live_query_string) {
        start_book_view (book, live_query_string, &error);
        g_free (live_query_string);
        if (error) {
            g_printerr ("Could not get contacts: %s\n", error->message);
            g_clear_error (&error);
            return 1;
        }
    }

    g_object_unref (book);

    return 0;
}
