/* e-vard-store.c - A GtkTreeModel implementation that stores the attributes
 * of a EVcard
 *
 * Copyright (C) 2008 Nokia Corp.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Joergen Scheibengruber <jorgen.scheibengruber AT nokia.com>
 */

#include "e-vcard-store.h"

typedef struct _EVCardStorePrivate EVCardStorePrivate;
struct _EVCardStorePrivate
{
    int stamp;

    EVCard *vcard;
};

enum {
    PROP_0,

    PROP_VCARD
};

#define E_VCARD_STORE_PRIVATE(o) \
(G_TYPE_INSTANCE_GET_PRIVATE ((o), E_VCARD_STORE_TYPE, EVCardStorePrivate))

static void e_vcard_store_class_init (EVCardStoreClass *klass);
static void e_vcard_store_init       (EVCardStore *self);
static void e_vcard_store_dispose    (GObject *object);
static void e_vcard_store_finalize   (GObject *object);
static void e_vcard_store_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec);
static void e_vcard_store_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec);
static void e_vcard_store_tree_model_init (GtkTreeModelIface *iface);

G_DEFINE_TYPE_WITH_CODE (EVCardStore, e_vcard_store, G_TYPE_OBJECT,
             G_IMPLEMENT_INTERFACE (GTK_TYPE_TREE_MODEL,
                        e_vcard_store_tree_model_init));


static GtkTreeModelFlags e_vcard_store_get_flags (GtkTreeModel *tree_model);
static gint         e_vcard_store_get_n_columns   (GtkTreeModel      *tree_model);
static GType        e_vcard_store_get_column_type (GtkTreeModel      *tree_model,
                                                   gint               index);
static gboolean     e_vcard_store_get_iter        (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter,
                                                   GtkTreePath       *path);
static GtkTreePath* e_vcard_store_get_path        (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter);
static void         e_vcard_store_get_value       (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter,
                                                   gint               column,
                                                   GValue            *value);
static gboolean     e_vcard_store_iter_next       (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter);
static gboolean     e_vcard_store_iter_children   (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter,
                                                   GtkTreeIter       *parent);
static gboolean     e_vcard_store_iter_has_child  (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter);
static gint         e_vcard_store_iter_n_children (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter);
static gboolean     e_vcard_store_iter_nth_child  (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter,
                                                   GtkTreeIter       *parent,
                                                   gint               n);
static gboolean     e_vcard_store_iter_parent     (GtkTreeModel      *tree_model,
                                                   GtkTreeIter       *iter,
                                                   GtkTreeIter       *child);


static void
e_vcard_store_class_init (EVCardStoreClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (EVCardStorePrivate));

    object_class->set_property = e_vcard_store_set_property;
    object_class->get_property = e_vcard_store_get_property;
    object_class->dispose = e_vcard_store_dispose;
    object_class->finalize = e_vcard_store_finalize;

    g_object_class_install_property
        (object_class,
         PROP_VCARD,
         g_param_spec_object
            ("vcard",
             "VCard",
             "The wrapped vcard",
             E_TYPE_VCARD,
             G_PARAM_READWRITE | G_PARAM_STATIC_NAME | G_PARAM_CONSTRUCT));
}

static void
e_vcard_store_tree_model_init (GtkTreeModelIface *iface)
{
    iface->get_flags = e_vcard_store_get_flags;
    iface->get_n_columns = e_vcard_store_get_n_columns;
    iface->get_column_type = e_vcard_store_get_column_type;
    iface->get_iter = e_vcard_store_get_iter;
    iface->get_path = e_vcard_store_get_path;
    iface->get_value = e_vcard_store_get_value;
    iface->iter_next = e_vcard_store_iter_next;
    iface->iter_children = e_vcard_store_iter_children;
    iface->iter_has_child = e_vcard_store_iter_has_child;
    iface->iter_n_children = e_vcard_store_iter_n_children;
    iface->iter_nth_child = e_vcard_store_iter_nth_child;
    iface->iter_parent = e_vcard_store_iter_parent;
}

static void
e_vcard_store_init (EVCardStore *self)
{
    EVCardStorePrivate *priv;

    priv = E_VCARD_STORE_PRIVATE (self);

    priv->stamp = 0;
    priv->vcard = NULL;
}

EVCardStore*
e_vcard_store_new (void)
{
    return g_object_new (E_VCARD_STORE_TYPE,
                         NULL);
}

EVCardStore*
e_vcard_store_new_from_vcard (EVCard *vcard)
{
    return g_object_new (E_VCARD_STORE_TYPE,
                         "vcard", vcard,
                         NULL);
}

static void
e_vcard_store_dispose (GObject *object)
{
    EVCardStorePrivate *priv;

    priv = E_VCARD_STORE_PRIVATE (object);

    e_vcard_store_set_vcard (E_VCARD_STORE (object), NULL);

    G_OBJECT_CLASS (e_vcard_store_parent_class)->dispose (object);
}

static void
e_vcard_store_finalize (GObject *object)
{
    EVCardStorePrivate *priv;

    priv = E_VCARD_STORE_PRIVATE (object);

    G_OBJECT_CLASS (e_vcard_store_parent_class)->finalize (object);
}

static void
e_vcard_store_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec)
{
    switch (property_id) {
    case PROP_VCARD:
        e_vcard_store_set_vcard (E_VCARD_STORE (object), g_value_get_object (value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

static void
e_vcard_store_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec)
{
    EVCardStorePrivate *priv;

    priv = E_VCARD_STORE_PRIVATE (object);

    switch (property_id) {
    case PROP_VCARD:
        g_value_set_object (value, priv->vcard);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

void
e_vcard_store_set_vcard (EVCardStore* store, EVCard *vcard)
{
    EVCardStorePrivate *priv;

    g_return_if_fail (IS_E_VCARD_STORE (store));
    g_return_if_fail (E_IS_VCARD (vcard) || vcard == NULL);
    priv = E_VCARD_STORE_PRIVATE (store);

    if (priv->vcard) {
        g_object_unref (priv->vcard);
        priv->stamp = 0;
    }

    priv->vcard = vcard;
    if (vcard) {
        g_object_ref (priv->vcard);
        priv->stamp = GPOINTER_TO_INT (priv->vcard);
    }
}

EVCard*
e_vcard_store_get_vcard (EVCardStore* store)
{
    EVCardStorePrivate *priv;

    g_return_val_if_fail (IS_E_VCARD_STORE (store), NULL);
    priv = E_VCARD_STORE_PRIVATE (store);

    return priv->vcard;
}

static void
e_vcard_store_attribute_inserted (EVCardStore *store, GtkTreeIter *iter)
{
    GtkTreePath *path;

    path = e_vcard_store_get_path (GTK_TREE_MODEL (store), iter);
    gtk_tree_model_row_inserted (GTK_TREE_MODEL (store), path, iter);
    gtk_tree_path_free (path);
}

static void
e_vcard_store_attribute_changed (EVCardStore *store, GtkTreeIter *iter)
{
    GtkTreePath *path;

    path = e_vcard_store_get_path (GTK_TREE_MODEL (store), iter);
    gtk_tree_model_row_changed (GTK_TREE_MODEL (store), path, iter);
    gtk_tree_path_free (path);
}

static void
e_vcard_store_attribute_deleted (EVCardStore *store, GtkTreeIter *iter)
{
    GtkTreePath *path;

    path = e_vcard_store_get_path (GTK_TREE_MODEL (store), iter);
    gtk_tree_model_row_deleted (GTK_TREE_MODEL (store), path);
    gtk_tree_path_free (path);
}

void
e_vcard_store_set_attribute_params (EVCardStore *store, GtkTreeIter *iter, GList *params)
{
    EVCardAttribute *attr;
    g_return_if_fail (IS_E_VCARD_STORE (store));
    g_return_if_fail (iter && iter->stamp != 0);

    attr = ((GList*)iter->user_data)->data;;
    e_vcard_attribute_remove_params (attr);
    params = g_list_last (params);
    while (params) {
        e_vcard_attribute_add_param (attr, params->data);
        params = params->prev;
    }

    e_vcard_store_attribute_changed (store, iter);
}

void
e_vcard_store_set_attribute_values (EVCardStore *store, GtkTreeIter *iter, GList *values)
{
    EVCardAttribute *attr;
    g_return_if_fail (IS_E_VCARD_STORE (store));
    g_return_if_fail (iter && iter->stamp != 0);

    attr = ((GList*)iter->user_data)->data;;
    e_vcard_attribute_remove_values (attr);
    while (values) {
        e_vcard_attribute_add_value (attr, values->data);
        values = values->next;
    }

    e_vcard_store_attribute_changed (store, iter);
}

void e_vcard_store_append_attribute (EVCardStore *store, const char *name, const char *group)
{
    EVCardStorePrivate *priv;
    EVCardAttribute *attr;
    GtkTreeIter iter;
    GList *attrs;

    g_return_if_fail (IS_E_VCARD_STORE (store));
    g_return_if_fail (name != NULL && name[0] != 0);

    priv = E_VCARD_STORE_PRIVATE (store);

    attr = e_vcard_attribute_new (group, name);
    e_vcard_add_attribute (priv->vcard, attr);

    attrs = e_vcard_get_attributes (priv->vcard);
    iter.stamp = priv->stamp;
    iter.user_data = g_list_last (attrs);
    iter.user_data2 = GINT_TO_POINTER (g_list_length (attrs) - 1);

    e_vcard_store_attribute_inserted (store, &iter);
}

void e_vcard_store_remove_attribute (EVCardStore *store, GtkTreeIter *iter)
{
    EVCardStorePrivate *priv;
    EVCardAttribute *attr;

    g_return_if_fail (IS_E_VCARD_STORE (store));
    g_return_if_fail (iter && iter->stamp != 0);

    priv = E_VCARD_STORE_PRIVATE (store);

    attr = ((GList*)iter->user_data)->data;;
    e_vcard_remove_attribute (priv->vcard, attr);

    e_vcard_store_attribute_deleted (store, iter);
    iter->stamp = 0;
    iter->user_data = 0;
}

static GtkTreeModelFlags
e_vcard_store_get_flags (GtkTreeModel *tree_model)
{
    g_return_val_if_fail (IS_E_VCARD_STORE (tree_model), 0);

    return GTK_TREE_MODEL_ITERS_PERSIST | GTK_TREE_MODEL_LIST_ONLY;
}

static gint
e_vcard_store_get_n_columns (GtkTreeModel *tree_model)
{
    g_return_val_if_fail (IS_E_VCARD_STORE (tree_model), 0);

    return 4;
}

static GType
e_vcard_store_get_column_type (GtkTreeModel *tree_model,
                               gint          index)
{
    g_return_val_if_fail (IS_E_VCARD_STORE (tree_model), G_TYPE_INVALID);

    switch (index) {
        case EVCARD_STORE_COLUMN_NAME:
        case EVCARD_STORE_COLUMN_GROUP:
            return G_TYPE_STRING;
            break;
        case EVCARD_STORE_COLUMN_PARAMS:
        case EVCARD_STORE_COLUMN_VALUES:
            return G_TYPE_POINTER;
            break;
        default:
            break;
    }
    return G_TYPE_INVALID;
}

static gboolean
e_vcard_store_get_iter (GtkTreeModel *tree_model,
                        GtkTreeIter  *iter,
                        GtkTreePath  *path)
{
    EVCardStorePrivate *priv;
    GList *attrs;
    gint n;

    g_return_val_if_fail (IS_E_VCARD_STORE (tree_model), FALSE);
    g_return_val_if_fail (iter, FALSE);

    priv = E_VCARD_STORE_PRIVATE (tree_model);

    n = gtk_tree_path_get_indices (path)[0];

    attrs = e_vcard_get_attributes (priv->vcard);
    iter->stamp = 0;
    if (n < 0 || n >= g_list_length (attrs))
        return FALSE;

    iter->stamp = priv->stamp;
    iter->user_data = g_list_nth (attrs, n);
    iter->user_data2 = GINT_TO_POINTER (n);

    return TRUE;
}

static GtkTreePath*
e_vcard_store_get_path (GtkTreeModel *tree_model,
                        GtkTreeIter  *iter)
{
    EVCardStorePrivate *priv;
    GtkTreePath *path;

    g_return_val_if_fail (IS_E_VCARD_STORE (tree_model), NULL);
    g_return_val_if_fail (iter, NULL);
    priv = E_VCARD_STORE_PRIVATE (tree_model);
    g_return_val_if_fail (iter->stamp == priv->stamp, NULL);

    path = gtk_tree_path_new ();
    gtk_tree_path_append_index (path, GPOINTER_TO_INT (iter->user_data2));

    return path;
}

static void
e_vcard_store_get_value (GtkTreeModel *tree_model,
                         GtkTreeIter  *iter,
                         gint          column,
                         GValue       *value)
{
    EVCardStorePrivate *priv;
    EVCardAttribute *attr;
    //int n;

    g_return_if_fail (IS_E_VCARD_STORE (tree_model));
    g_return_if_fail (column >= 0 && column < 4);

    priv = E_VCARD_STORE_PRIVATE (tree_model);
    g_return_if_fail (iter->stamp == priv->stamp);

//    n = GPOINTER_TO_INT (iter->user_data2);
    attr = ((GList*)iter->user_data)->data;

    switch (column) {
        case 0:
            g_value_init (value, G_TYPE_STRING);
            g_value_set_string (value, e_vcard_attribute_get_name (attr));
        break;
        case 1:
            g_value_init (value, G_TYPE_STRING);
            g_value_set_string (value, e_vcard_attribute_get_group (attr));
        break;
        case 2:
            g_value_init (value, G_TYPE_POINTER);
            g_value_set_pointer (value, e_vcard_attribute_get_params (attr));
        break;
        case 3:
            g_value_init (value, G_TYPE_POINTER);
            g_value_set_pointer (value, e_vcard_attribute_get_values (attr));
        break;
        default:
            g_value_init (value, G_TYPE_INVALID);
        break;
    }
}

static gboolean
e_vcard_store_iter_next (GtkTreeModel *tree_model,
                           GtkTreeIter  *iter)
{
    EVCardStorePrivate *priv;
    GList *next;

    g_return_val_if_fail (IS_E_VCARD_STORE (tree_model), FALSE);
    g_return_val_if_fail (iter, FALSE);

    priv = E_VCARD_STORE_PRIVATE (tree_model);
    g_return_val_if_fail (iter->stamp == priv->stamp, FALSE);

    next = ((GList*)iter->user_data)->next;
    if (NULL == next) {
        iter->stamp = 0;
        return FALSE;
    }
    iter->user_data = next;
    iter->user_data2 = GINT_TO_POINTER (GPOINTER_TO_INT (iter->user_data2) + 1);

    return TRUE;
}

static gboolean
e_vcard_store_iter_children (GtkTreeModel *tree_model,
                               GtkTreeIter  *iter,
                               GtkTreeIter  *parent)
{
    iter->stamp = 0;
    return FALSE;
}

static gboolean
e_vcard_store_iter_has_child (GtkTreeModel *tree_model,
                                GtkTreeIter  *iter)
{
    return FALSE;
}

static gint
e_vcard_store_iter_n_children (GtkTreeModel *tree_model,
                           GtkTreeIter  *iter)
{
    EVCardStorePrivate *priv;
    GList *attrs;

    g_return_val_if_fail (IS_E_VCARD_STORE (tree_model), 0);

    if (iter) {
        return 0;
    }

    priv = E_VCARD_STORE_PRIVATE (tree_model);
    attrs = e_vcard_get_attributes (priv->vcard);

    return g_list_length (attrs);
}

static gboolean
e_vcard_store_iter_nth_child  (GtkTreeModel *tree_model,
                               GtkTreeIter  *iter,
                               GtkTreeIter  *parent,
                               gint          n)
{
    EVCardStorePrivate *priv;
    GList *attrs;

    g_return_val_if_fail (IS_E_VCARD_STORE (tree_model), FALSE);
    g_return_val_if_fail (iter, FALSE);
    priv = E_VCARD_STORE_PRIVATE (tree_model);

    if (parent)
        return FALSE;

    attrs = e_vcard_get_attributes (priv->vcard);
    if (n < 0 || n >= g_list_length (attrs))
        return FALSE;
    iter->user_data = g_list_nth (attrs, n);
    iter->user_data2 = GINT_TO_POINTER (n);

    return TRUE;
}

static gboolean
e_vcard_store_iter_parent (GtkTreeModel *tree_model,
                           GtkTreeIter  *iter,
                           GtkTreeIter  *child)
{
    iter->stamp = 0;

    return FALSE;
}
