#include "e-contact-store.h"
#include "book-editor.h"
#include "vcard-editor.h"
#include "util.h"

enum {
    PROP_NONE,
    PROP_BOOK_VIEW
};

typedef struct _BookEditorPrivate BookEditorPrivate;
struct _BookEditorPrivate
{
    GtkWidget *tree_view;
    EContactStore *contact_store;
    EBookView *book_view;
    gboolean update_cursor;
};

#define BOOK_EDITOR_PRIVATE(o) \
(G_TYPE_INSTANCE_GET_PRIVATE ((o), BOOK_EDITOR_TYPE, BookEditorPrivate))

static void book_editor_class_init (BookEditorClass *klass);
static void book_editor_init       (BookEditor *self);
static void book_editor_dispose    (GObject *object);
static void book_editor_finalize   (GObject *object);
static void book_editor_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec);
static void book_editor_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec);
static void book_editor_set_book_view (BookEditor *editor, EBookView *book_view);

static void on_book_view_sequence_complete (EBookView *book_view, int status, gpointer user_data);

static void on_contacts_store_row_inserted (GtkTreeModel *contact_store, GtkTreePath *new_path, GtkTreeIter *iter, gpointer user_data);
static void on_contacts_store_row_changed (GtkTreeModel *contact_store, GtkTreePath *path, GtkTreeIter *iter, gpointer user_data);
static void on_contacts_store_row_deleted (GtkTreeModel *contact_store, GtkTreePath *path, gpointer user_data);

static void on_tree_view_row_activated (GtkTreeView *tree_view, GtkTreePath *path, GtkTreeViewColumn *column, gpointer user_data);

static void contact_photo_cell_data_func (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell, GtkTreeModel *tree_model, GtkTreeIter *iter, gpointer data);
static void contact_name_cell_data_func (GtkTreeViewColumn *tree_column, GtkCellRenderer *cell, GtkTreeModel *tree_model, GtkTreeIter *iter, gpointer data);

static void on_add_contact_btn_clicked (GtkToolButton *button, gpointer user_data);
static void on_delete_contact_btn_clicked (GtkToolButton *button, gpointer user_data);

static int compare_contacts_by_name (EContact *contact_a, EContact *contact_b, gpointer user_data);
static void set_progress_indicator (GtkWindow *window, gboolean ongoing);

G_DEFINE_TYPE (BookEditor, book_editor, WINDOW_TYPE);

static void
book_editor_class_init (BookEditorClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (BookEditorPrivate));

    object_class->set_property = book_editor_set_property;
    object_class->get_property = book_editor_get_property;
    object_class->dispose = book_editor_dispose;
    object_class->finalize = book_editor_finalize;

    g_object_class_install_property
            (object_class, PROP_BOOK_VIEW,
             g_param_spec_object
                    ("book-view",
                     "EBookView",
                     "The EBookView from which the contacts are loaded",
                     E_TYPE_BOOK_VIEW,
                     G_PARAM_READWRITE | G_PARAM_STATIC_NAME |
                     G_PARAM_CONSTRUCT_ONLY |
                     G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
}

static void
book_editor_init (BookEditor *self)
{
    BookEditorPrivate *priv;
    GtkWidget *vbox;
    GtkWidget *sw;
    GtkTreeViewColumn *column;
    GtkCellRenderer *name_renderer;
    GtkCellRenderer *photo_renderer;
    GtkWidget *toolbar;
    GtkToolItem *toolitem;

    priv = BOOK_EDITOR_PRIVATE (self);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (self), vbox);

    /* Contacts pane */
    sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
                                    GTK_POLICY_NEVER,
                                    GTK_POLICY_AUTOMATIC);

    priv->contact_store = e_contact_store_new ();
    e_contact_store_set_sort_func (priv->contact_store,
                                   compare_contacts_by_name);
    g_signal_connect_after (priv->contact_store, "row-inserted",
                      G_CALLBACK (on_contacts_store_row_inserted), self);
    g_signal_connect (priv->contact_store, "row-changed",
                      G_CALLBACK (on_contacts_store_row_changed), self);
    g_signal_connect (priv->contact_store, "row-deleted",
                      G_CALLBACK (on_contacts_store_row_deleted), self);

    priv->tree_view = gtk_tree_view_new ();
    gtk_container_add (GTK_CONTAINER (sw), priv->tree_view);
    gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (priv->tree_view),
                                       FALSE);
    g_signal_connect (priv->tree_view, "row-activated",
                      G_CALLBACK (on_tree_view_row_activated), self);

    name_renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new ();

    photo_renderer = gtk_cell_renderer_pixbuf_new ();
    g_object_set (photo_renderer, "height", 26, "width", 26, NULL);
    gtk_tree_view_column_pack_start (column, photo_renderer, FALSE);
    gtk_tree_view_column_set_cell_data_func (column, photo_renderer,
                                             contact_photo_cell_data_func,
                                             self, NULL);

    gtk_tree_view_column_pack_start (column, name_renderer, TRUE);
    gtk_tree_view_column_set_cell_data_func (column, name_renderer,
                                             contact_name_cell_data_func,
                                             self, NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (priv->tree_view), column);

    /* Toolbar */
    toolbar = gtk_toolbar_new ();
    gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);

    toolitem = gtk_tool_button_new (NULL, NULL);
    gtk_tool_button_set_icon_name (GTK_TOOL_BUTTON (toolitem), ADD_ICON);
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    g_signal_connect (toolitem, "clicked",
                      G_CALLBACK (on_add_contact_btn_clicked), self);

    toolitem = gtk_tool_button_new (NULL, NULL);
    gtk_tool_button_set_icon_name (GTK_TOOL_BUTTON (toolitem), DELETE_ICON);
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    g_signal_connect (toolitem, "clicked",
                      G_CALLBACK (on_delete_contact_btn_clicked), self);

    gtk_widget_show_all (vbox);
}

static void
book_editor_dispose (GObject *object)
{
    BookEditor *editor = BOOK_EDITOR (object);
    BookEditorPrivate *priv;

    priv = BOOK_EDITOR_PRIVATE (editor);

    if (priv->book_view) {
        e_book_view_stop (priv->book_view);
        g_object_unref (priv->book_view);
        priv->book_view = NULL;
    }

    G_OBJECT_CLASS (book_editor_parent_class)->dispose (object);
}

static void
book_editor_finalize (GObject *object)
{
    G_OBJECT_CLASS (book_editor_parent_class)->finalize (object);
}

static void
book_editor_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec)
{
    switch (property_id) {
    case PROP_BOOK_VIEW:
        book_editor_set_book_view (BOOK_EDITOR (object), g_value_get_object (value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

static void
book_editor_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec)
{
    BookEditor *editor = BOOK_EDITOR (object);
    BookEditorPrivate *priv;

    priv = BOOK_EDITOR_PRIVATE (editor);

    switch (property_id) {
    case PROP_BOOK_VIEW:
        g_value_set_object (value, priv->book_view);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

GtkWidget* book_editor_new (EBookView *book_view)
{
    return g_object_new (BOOK_EDITOR_TYPE, "book-view", book_view, NULL);
}

static void
book_editor_set_book_view (BookEditor *editor, EBookView *book_view)
{
    BookEditorPrivate *priv;

    priv = BOOK_EDITOR_PRIVATE (editor);

    g_return_if_fail (priv->book_view == NULL);
    priv->book_view = g_object_ref (book_view);

    g_signal_connect (priv->book_view, "sequence-complete",
                      G_CALLBACK (on_book_view_sequence_complete),
                      editor);
    e_contact_store_set_book_view (priv->contact_store,
                                   priv->book_view);
    e_book_view_start (priv->book_view);
    set_progress_indicator (GTK_WINDOW (editor), TRUE);
}

static EContact*
book_editor_get_contact_at_cursor (BookEditor *editor)
{
    BookEditorPrivate *priv;
    GtkTreePath *path;
    EContact *contact = NULL;

    priv = BOOK_EDITOR_PRIVATE (editor);
    gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->tree_view), &path, NULL);

    if (path) {
        EBook *book;
        GtkTreeIter iter;
        GError *error = NULL;
        GtkTreeModel *model = GTK_TREE_MODEL (priv->contact_store);

        gtk_tree_model_get_iter (model, &iter, path);
        gtk_tree_path_free (path);

        gtk_tree_model_get (model, &iter, 0, &contact, -1);
    }

    return contact;
}

static void
on_book_view_sequence_complete (EBookView *book_view, int status, gpointer user_data)
{
    BookEditor *editor = user_data;
    BookEditorPrivate *priv;

    priv = BOOK_EDITOR_PRIVATE (editor);

    gtk_tree_view_set_model (GTK_TREE_VIEW (priv->tree_view),
                             GTK_TREE_MODEL (priv->contact_store));

    set_progress_indicator (GTK_WINDOW (editor), FALSE);
    g_object_disconnect (G_OBJECT (priv->book_view), "any_signal::sequence-complete",
                         on_book_view_sequence_complete, user_data, NULL);
}

static gboolean
on_vcard_editor_delete_event (GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
#if MAEMO_CHANGES
    HildonWindowStack *window_stack;

    window_stack = hildon_window_stack_get_default ();
    hildon_window_stack_pop_1 (window_stack);
#else
    gtk_grab_remove (widget);
#endif

    return FALSE;
}

static void
on_tree_view_row_activated (GtkTreeView *tree_view, GtkTreePath *path,
                            GtkTreeViewColumn *column, gpointer user_data)
{
    BookEditor *editor = user_data;
    BookEditorPrivate *priv;
    EContact *contact;

    priv = BOOK_EDITOR_PRIVATE (editor);

    contact = book_editor_get_contact_at_cursor (editor);
    if (contact) {
        GtkWidget *vcard_editor;
        EBook *book;
        GError *error = NULL;
#if MAEMO_CHANGES
        HildonWindowStack *window_stack;
#endif

        book = e_book_view_get_book (priv->book_view);
        vcard_editor = vcard_editor_new (GTK_WINDOW (editor));
        gtk_window_set_default_size (GTK_WINDOW (vcard_editor), 800, 400);
        g_object_set (vcard_editor, "contact", contact, "book", book, NULL);
        g_object_unref (contact);
#if MAEMO_CHANGES
        window_stack = hildon_window_stack_get_default ();
        hildon_window_stack_push_1 (window_stack,
                                    HILDON_STACKABLE_WINDOW (vcard_editor));
#else
        gtk_widget_show_all (vcard_editor);
        gtk_grab_add (vcard_editor);
#endif
        g_signal_connect (vcard_editor, "delete-event",
                          G_CALLBACK (on_vcard_editor_delete_event), NULL);
        priv->update_cursor = TRUE;
    }
}

static void
on_add_contact_btn_clicked (GtkToolButton *button, gpointer user_data)
{
    BookEditor *editor = user_data;
    BookEditorPrivate *priv;
    EContact *contact;
    GError *error = NULL;
    EBook *book;

    priv = BOOK_EDITOR_PRIVATE (editor);

    book = e_book_view_get_book (priv->book_view);

    contact = e_contact_new ();
    e_contact_set (contact, E_CONTACT_FULL_NAME, "1234567");
    e_contact_set (contact, E_CONTACT_PHONE_MOBILE, "1234567");

    e_book_add_contact (book, contact, &error);
    if (error) {
        g_printerr ("Could not add contact: %s\n", error->message);
        g_clear_error (&error);
    }
    g_object_unref (contact);
    priv->update_cursor = TRUE;
}

static void
on_delete_contact_btn_clicked (GtkToolButton *button, gpointer user_data)
{
    BookEditor *editor = user_data;
    BookEditorPrivate *priv;
    EContact *contact;

    priv = BOOK_EDITOR_PRIVATE (editor);

    contact = book_editor_get_contact_at_cursor (editor);
    if (contact) {
        EBook *book;
        GError *error = NULL;

        book = e_book_view_get_book (priv->book_view);
        e_book_remove_contact (book, e_contact_get_const (contact, E_CONTACT_UID), &error);
        if (error) {
            g_printerr ("Could not delete contact %s: %s\n",
                        _e_contact_get_name (contact),
                        error->message);
            g_clear_error (&error);
        }
        g_object_unref (contact);
    }
}

static int
compare_contacts_by_name (EContact *contact_a, EContact *contact_b, gpointer user_data)
{
    const char *name_a, *name_b;

    name_a = _e_contact_get_name (contact_a);
    name_b = _e_contact_get_name (contact_b);

    return name_a ? (name_b ? g_utf8_collate (name_a, name_b) : -1 ) : 1;
}


static void
on_contacts_store_row_inserted (GtkTreeModel *contact_store, GtkTreePath *new_path,
                                GtkTreeIter *iter, gpointer user_data)
{
    BookEditor *editor = user_data;
    BookEditorPrivate *priv;

    priv = BOOK_EDITOR_PRIVATE (editor);

    if (priv->update_cursor) {
        gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->tree_view), new_path, NULL, FALSE);
        gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (priv->tree_view), new_path, NULL, TRUE, 0.5f, 0.0f);
        priv->update_cursor = FALSE;
    }
}

static void
on_contacts_store_row_changed (GtkTreeModel *contact_store, GtkTreePath *path,
                               GtkTreeIter *iter, gpointer user_data)
{
    BookEditor *editor = user_data;
    BookEditorPrivate *priv;

    priv = BOOK_EDITOR_PRIVATE (editor);

    if (priv->update_cursor) {
        gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW (priv->tree_view), path, NULL, TRUE, 0.5f, 0.0f);
        priv->update_cursor = FALSE;
    }
}

static void
on_contacts_store_row_deleted (GtkTreeModel *contact_store, GtkTreePath *path, gpointer user_data)
{
    BookEditor *editor = user_data;
    BookEditorPrivate *priv;
    GtkTreePath *cursor_path;

    priv = BOOK_EDITOR_PRIVATE (editor);

    gtk_tree_view_get_cursor (GTK_TREE_VIEW (priv->tree_view),
                              &cursor_path, NULL);
    if (cursor_path) {
        if (gtk_tree_path_compare (path, cursor_path) == 0) {
            gtk_tree_path_next (cursor_path);
            gtk_tree_view_set_cursor (GTK_TREE_VIEW (priv->tree_view), cursor_path, NULL, FALSE);
        }
        gtk_tree_path_free (cursor_path);
    }
}

static void
contact_name_cell_data_func (GtkTreeViewColumn *tree_column,
                             GtkCellRenderer *cell,
                             GtkTreeModel *tree_model,
                             GtkTreeIter *iter,
                             gpointer data)
{
    EContact *contact;
    const char *name;

    gtk_tree_model_get (tree_model, iter, 0, &contact, -1);

    name = _e_contact_get_name (contact);

    g_object_set (cell, "text", name, NULL);

    g_object_unref (contact);
}

static gboolean
on_idle_load_photo (gpointer user_data)
{
    EContact *contact = user_data;
    GtkTreeRowReference *row;
    GtkTreeModel *tree_model;
    GtkTreePath *path;
    GtkTreeIter iter;

    g_return_val_if_fail (E_IS_CONTACT (contact), FALSE);
    row = g_object_get_data (G_OBJECT (contact), "row");
    g_return_val_if_fail (row != NULL, FALSE);

    tree_model = gtk_tree_row_reference_get_model (row);
    path = gtk_tree_row_reference_get_path (row);

    g_object_set_data (G_OBJECT (contact), "idle-load-completed", GINT_TO_POINTER (TRUE));
    if (gtk_tree_model_get_iter (tree_model, &iter, path)) {
        EContactPhoto *photo;
        GdkPixbuf *pixbuf = NULL;

        photo = e_contact_get (contact, E_CONTACT_PHOTO);

        if (photo) {
            if (photo->type == E_CONTACT_PHOTO_TYPE_INLINED) {
                GdkPixbufLoader *loader;

                loader = gdk_pixbuf_loader_new ();
                if (loader) {
                    gdk_pixbuf_loader_set_size (loader, 24, 24);
                    gdk_pixbuf_loader_write (loader, photo->data.inlined.data,
                                                     photo->data.inlined.length,
                                                     NULL);
                    gdk_pixbuf_loader_close (loader, NULL);
                    pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
                    if (pixbuf) {
                        g_object_ref (pixbuf);
                    }
                    g_object_unref (loader);
                }
            } else {
                const char *uri;
                char *filename;

                uri = photo->data.uri;

                if (uri && uri[0]) {
                    if (g_str_has_prefix (uri, FILE_URI_PREFIX)) {
                        filename = g_strdup (uri + g_utf8_strlen (FILE_URI_PREFIX, -1));
                    } else {
                        filename = g_strdup (uri);
                    }

                    pixbuf = gdk_pixbuf_new_from_file_at_size (filename, -1,
                                                               24, NULL);
                }
            }
        }

        if (pixbuf) {
            g_object_set_data_full (G_OBJECT (contact), "pixbuf", pixbuf, g_object_unref);
            gtk_tree_model_row_changed (tree_model, path, &iter);
        }

        e_contact_photo_free (photo);
    }
    gtk_tree_row_reference_free (row);
    g_object_unref (contact);

    return FALSE;
}

static void
contact_photo_cell_data_func (GtkTreeViewColumn *tree_column,
                              GtkCellRenderer *cell,
                              GtkTreeModel *tree_model,
                              GtkTreeIter *iter,
                              gpointer data)
{
    EContact *contact;
    GdkPixbuf *pixbuf = NULL;
    gboolean idle_load_completed;

    gtk_tree_model_get (tree_model, iter, 0, &contact, -1);

    idle_load_completed = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (contact),
                                                              "idle-load-completed"));

    if (idle_load_completed) {
        pixbuf = g_object_get_data (G_OBJECT (contact), "pixbuf");
    } else {
        GtkTreePath *path;
        GtkTreeRowReference *row;
        gboolean idle_load_started;

        idle_load_started = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (contact),
                                                                "idle-load-started"));
        if (FALSE == idle_load_started) {
            g_object_set_data (G_OBJECT (contact), "idle-load-started",
                               GINT_TO_POINTER (TRUE));
            path = gtk_tree_model_get_path (tree_model, iter);
            row = gtk_tree_row_reference_new (tree_model, path);
            gtk_tree_path_free (path);
            g_object_set_data (G_OBJECT (contact), "row", row);
            g_object_ref (contact);
            g_idle_add (on_idle_load_photo, contact);
        }
    }
    if (pixbuf) {
        g_object_set (cell, "pixbuf", pixbuf, NULL);
    } else {
        g_object_set (cell, "icon-name", AVATAR_ICON, NULL);
    }
    g_object_unref (contact);
}

static void
set_progress_indicator (GtkWindow *window, gboolean ongoing)
{
#ifdef MAEMO_CHANGES
    hildon_gtk_window_set_progress_indicator (window, ongoing);
#else
    GdkCursor *cursor;

    if (ongoing) {
        cursor = gdk_cursor_new (GDK_WATCH);
        gdk_window_set_cursor (GTK_WIDGET (window)->window,
                               cursor);
        gdk_cursor_unref (cursor);
    } else {
        gdk_window_set_cursor (GTK_WIDGET (window)->window, NULL);
    }
#endif
}

