#!/usr/bin/env python

##########################################################################
# EasyPlayer
#
# a mp3/ogg/aac/wav (whatever gstreamer supports) audio player
# for Maemo 5 on the N900
# written 2010 by Klaus Rotter (klaus at rotters dot de)
# (c) 2010 by Klaus Rotter, distribution GPL
#
# History
#
# 0.1	2010-03-09	First steps, based on a gstreamer+python example
#			added a treeview widget with code to scan and
#			display /home/usr/MyDocs/.sounds
#
# 0.2	2010-03-22	added code to load/save/display prefs, added timeout
#			bug fixing
#
# 0.3	2010-03-25	fixed some bugs (timeout never did not work)			
#			added slider
#
# 0.4	2010-04-11	display Metatags as an alternate to Filenames while playing (Progressbar)
#			rework of preferences and file scanning code
#			added .flac type
#			now recognices all types .mp3/.MP3/.Mp3 etc
#			Bug reported and fixed by Frank Boer (Thanks!)
#			- pause eating cpu cycles
#			- no sound output in silent mode
# 0.5	2010-08-13	Added some features mentioned in the gui thread in maemo.talk
#			removed Stop button - Now Player stops in pause mode but remembers the position
#			Prev button now jumps to beginning of song if duration is more than 10s
#
# TODO:
#
# * rearange buttons in portrait mode
# * do a backround rescan every startup (seperate thread) and then update playlist
# * option to manually delete files on the device (long press in directory browser?)
#   delete complete directories?
# * show file which is playing in the browser  
# * add volume bar / button which shows volume bar and automatically hides it
# * delete files from playlist which don't play
# * Support for sleeper and headphoned over D-Bus. See
#   http://talk.maemo.org/showthread.php?t=48414&highlight=easyplayer
#
# * KISS Mode (just buttons: prev / play_stop / next and progressbar), big buttons
# * use controls via bluetooth
# * more sublevels in browser
# 
##########################################################################

import sys, os, thread, time, string, pickle
import pygtk, gtk, gobject
import pygst
pygst.require("0.10")
import gst
import hildon
from portrait import FremantleRotation

version = "0.5-4 - 2011-01-03"
# Enable/disable debugging
dbg = 0

class GTK_Main:
  def __init__(self):
    # Some global variables are defined here
    self.pl_dirs = []  	# list to store the directorys containing music files
    self.dc = 0        	# actual number of directory
    self.pl_files = [] 	# list to store filenames of music files
    self.max_files = 0	# number of entries stored in playlist	
    
    self.play_name = ""		# Name of actual playing audio file
    self.act_file_id = 0	# id of above
    self.act_file_len = 0	# len in nano-seconds of above

    self.act_pos = 0;

    self.clear_tags()
    
    # Where to look for audio files / books
    self.base_media_dir = "/home/user/MyDocs/"

    #self.prefs_file = "/home/user/MyDocs/.easyplayer/preferences"
    #self.pl_dirs_file = "/home/user/MyDocs/.easyplayer/playlistdirs"
    #self.pl_files_file = "/home/user/MyDocs/.easyplayer/playlistfiles"
 
    home = os.environ["HOME"] # some people change their user name from "user"
    self.prefs_file = os.path.join(home, ".easyplayer", "preferences")
    self.pl_dirs_file = os.path.join(home, ".easyplayer", "playlistdirs")
    self.pl_files_file = os.path.join(home, ".easyplayer", "playlistfiles")
 
    # Try to load the preferences
    #if os.path.isfile(self.prefs_file):
    #  try:
    #    fd = open(self.prefs_file)
    #    self.prefs = pickle.load(fd) # overwrite default values
    #    fd.close()
    #  except:
    #    if (dbg): print "Can't load settings"


    try:
      # doing everything in one function call means we don't have to close
      # the file
      self.prefs = pickle.load(open(self.prefs_file))
    except IOError:
      self.init_prefs()
      self.save_prefs()
      try:
        os.mkdir(os.path.join(home, ".easyplayer"))
      except OSError:
        pass

    # Look, if the preferences are new enough
    # If not (or some tag "version" just don't exists), overwrite with default
    try:
      if self.prefs["version"] < 5:
        self.init_prefs()
        self.save_prefs()    
    except:
      self.init_prefs()
      self.save_prefs()
    
    if self.prefs["bigbuttons"] == 1: 
      self.std_size = gtk.HILDON_SIZE_AUTO|gtk.HILDON_SIZE_THUMB_HEIGHT
    else:
      self.std_size = gtk.HILDON_SIZE_AUTO|gtk.HILDON_SIZE_FINGER_HEIGHT
        
    program = hildon.Program.get_instance()

    if self.prefs["silenthack"] == 1: 
      gtk.set_application_name("FMRadio")
    else:
      gtk.set_application_name("EasyPlayer")
                  
    self.window = hildon.StackableWindow()
    self.window.show()
    self.window.connect("destroy", gtk.main_quit, "WM destroy")

    # Try to load the playlist 
    try:
      if (dbg): print "Trying to load dirs..."
      # Load list containing directories
      self.pl_dirs = pickle.load(open(self.pl_dirs_file))

      if (dbg): print "Trying to load files..."      
      # Load list containing filenames
      self.pl_files = pickle.load(open(self.pl_files_file))

      self.max_files = len(self.pl_files)	
      if (dbg): print "(load) Max Files:",self.max_files        
    except:
      if (dbg): print "Can't load playlist. Rescanning..."
      self.window.set_title("Scanning for music files...")
      hildon.hildon_gtk_window_set_progress_indicator(self.window, 1)
      #banner = hildon.hildon_banner_show_information(self.window, "",\
      #         "Can't load playlist. Rescanning for music files...")
      
      #win = hildon.StackableWindow()
      #win.set_title("Scanning for audiofiles")
      #win.show_all()  
      #if self.window <> None:
      #  hildon.hildon_gtk_window_set_progress_indicator(self.window, 1)
      self.pl_dirs = []  	# list to store the directorys containing music files
      self.dc = 0        	# actual number of directory
      self.pl_files = [] 	# list to store filenames of music files
      self.max_files = 0	# number of entries stored in playlist	
      self.rescan(self.base_media_dir)
      self.save_playlist()
      if (dbg): print "(save) Max Files:",self.max_files        
      hildon.hildon_gtk_window_set_progress_indicator(self.window, 0)        
    program.add_window(self.window)

    # Add a menu
    menu = hildon.AppMenu()

    self.pref_btn = hildon.GtkButton(self.std_size)
    self.pref_btn.set_label("Settings");
    self.pref_btn.connect("clicked",self.show_pref_win_cb)
    menu.append(self.pref_btn)

    self.scan_btn = hildon.GtkButton(self.std_size)
    self.scan_btn.set_label("Rescan Files");
    self.scan_btn.connect("clicked",self.menu_rescan_cb, self.scan_btn)
    menu.append(self.scan_btn)

    self.about_btn = hildon.GtkButton(self.std_size)
    self.about_btn.set_label("About");
    self.about_btn.connect("clicked",self.menu_about_cb, self.about_btn)
    menu.append(self.about_btn)

    self.info_btn = hildon.GtkButton(self.std_size)
    self.info_btn.set_label("File Tags");
    self.info_btn.connect("clicked",self.menu_display_tags_cb, self.info_btn)
    menu.append(self.info_btn)

    menu.show_all()
    self.window.set_app_menu(menu)
        
    vbox = gtk.VBox()
    self.window.add(vbox)
		
    panarea = hildon.PannableArea()
    #hildon.hildon_pannable_area_new_full(mode, enabled, vel_min, vel_max, decel, sps)	
    
    # create a TreeStore with one string column to use as the model
    self.treestore = gtk.TreeStore(str)

    # we'll add some data now - 4 rows with 3 child rows each
    #for parent in range(4):
    #  piter = self.treestore.append(None, ['parent %i' % parent])
    #  for child in range(3):
    #    self.treestore.append(piter, ['child %i of parent %i' %
    #    	(child, parent)])
    
    last_dir = -1
    parent = None   	
    for i in self.pl_files:
      if i[1] > last_dir: # must we add a new parent dir?
        last_dir = i[1]
        d = self.pl_dirs[last_dir]
        if d == self.base_media_dir or self.get_parent_dir(d) == ".sounds": # don't add dir if it is base dir
          parent = None
        else: # add a new parent  
          parent = self.treestore.append(None, [ self.get_parent_dir(d) ])    
      # add new entry    
      self.treestore.append(parent,[ i[0] ])

    # create the TreeView using treestore
    self.treeview = gtk.TreeView(self.treestore)
    # create the TreeViewColumn to display the data
    self.tvcolumn = gtk.TreeViewColumn('Song')

    # add tvcolumn to treeview
    self.treeview.append_column(self.tvcolumn)

    # create a CellRendererText to render the data
    self.cell = gtk.CellRendererText()
    # add the cell to the tvcolumn and allow it to expand
    self.tvcolumn.pack_start(self.cell, True)
    # set the cell "text" attribute to column 0 - retrieve text
    # from that column in treestore
    self.tvcolumn.add_attribute(self.cell, 'text', 0)

    # make it searchable
    self.treeview.set_search_column(0)
    # Allow sorting on the column
    self.tvcolumn.set_sort_column_id(0)

    # Allow drag and drop reordering of rows
    self.treeview.set_reorderable(True)

    self.treeview.get_selection().set_mode(gtk.SELECTION_SINGLE) 
    self.treeview.connect("row-activated", self.tv_row_activated_cb)

    panarea.add(self.treeview)
    vbox.pack_start(panarea, True, True)
			
    # Create the ProgressBar
    self.pbar_eb = gtk.EventBox()
    self.pbar_eb.set_events(gtk.gdk.BUTTON_PRESS_MASK)
    self.pbar_eb.connect('button-press-event', self.pbar_eb_press_cb)
    self.pbar = gtk.ProgressBar()
    self.pbar.set_size_request(-1,85)
    self.pbar_eb.add(self.pbar)
    vbox.pack_start(self.pbar_eb, False, True, 0)
    self.pbar.show()
    self.act_file_id = self.prefs["lastfile"]      
    self.play_name = self.pl_files[self.act_file_id][0]
    self.act_pos = self.prefs["lastpos"]
    self.update_pbar_s(self.prefs["lastpos"],self.prefs["lastdur"])
    
    buttonbox = gtk.HBox()
    prev_btn = self.new_button(gtk.STOCK_MEDIA_PREVIOUS)
    prev_btn.connect("clicked", self.prev_cb)
    buttonbox.add(prev_btn)
		
    rew_btn = self.new_button(gtk.STOCK_MEDIA_REWIND)
    rew_btn.connect("clicked", self.rewind_cb)
    buttonbox.add(rew_btn)

    # Use own images, allow us to switch
    # Play images and button nmust be public
    self.play_img = gtk.Image()
    self.set_play_img()

    self.play_btn = hildon.GtkButton(self.std_size)
    self.play_btn.add(self.play_img)
    self.play_btn.connect("clicked", self.play_stop_cb)
    buttonbox.add(self.play_btn)

    #stop_btn = self.new_button(gtk.STOCK_MEDIA_STOP)
    #stop_btn.connect("clicked", self.stop_cb)
    #buttonbox.add(stop_btn)

    forward_btn = self.new_button(gtk.STOCK_MEDIA_FORWARD)
    forward_btn.connect("clicked", self.forward_cb)
    buttonbox.add(forward_btn)

    next_btn = self.new_button(gtk.STOCK_MEDIA_NEXT)
    next_btn.connect("clicked", self.next_cb)
    buttonbox.add(next_btn)

    #self.lbl_time = gtk.Label()
    #self.lbl_time.set_text("00:00 / 00:00")
    #buttonbox.add(self.lbl_time)

    vbox.pack_start(buttonbox,False,True)	
    self.window.set_title("EasyPlayer")
    self.window.show_all()

    self.window.connect("delete_event", self.delete_event)
		
    self.player = gst.element_factory_make("playbin2", "player")
    fakesink = gst.element_factory_make("fakesink", "fakesink")
    self.player.set_property("video-sink", fakesink)
    bus = self.player.get_bus()
    bus.add_signal_watch()
    bus.connect("message", self.on_message)
    self.running = False;
    self.time_format = gst.Format(gst.FORMAT_TIME)
    if self.prefs["resumeplay"] == 1:
      self.play_stop_h()
    self.timeout = 0;
    self.reset_timer() # Reset TimeOut Timer
    self.rot_obj = FremantleRotation("EasyPlayer",self.window,"1.0",FremantleRotation.AUTOMATIC)

  # 'delete_event' is called when the window is to be closed
  # save preferences file
  def delete_event(self, widget, event, data=None):
    self.save_prefs()
    if dbg: print("delete event")
    return False # 'false' destroys Main Window

  # update progress bar in nano seconds
  #
  def update_pbar_ns(self, pos_int, dur_int):
    self.update_pbar_s(pos_int/1000000000, dur_int/1000000000)
  
  # update progress bar in seconds
  #
  def update_pbar_s(self, pos_int, dur_int):
    pos_str = self.convert_s(pos_int)
    dur_str = self.convert_s(dur_int)

    try:
      dt = self.prefs["displaytimeout"]
      to = self.timeout
    except:
      dt = 0
      to = -1

    if (to == -1) or (dt == 0):
      self.pbar.set_text(self.play_name + " - " + pos_str + " / " + dur_str)
    else:
      to_str = self.convert_s(self.timeout)   
      self.pbar.set_text(self.play_name + " - " + pos_str + " / " + dur_str + " (" + to_str +")")
    if (dur_int > 0) and (pos_int > 0):  
      self.pbar.set_fraction(float(pos_int) / float(dur_int))

  # Thread to display time / update process bar while running
  # copied from pygst tutorial
  def play_thread(self):
    play_thread_id = self.play_thread_id
    oldtime = time.time()
    gtk.gdk.threads_enter()
    gtk.gdk.threads_leave()
    while play_thread_id == self.play_thread_id:
      try:
        time.sleep(0.2)
        if dbg: print ("Actual pos:",self.act_pos)
        if self.act_pos > 0:
          seek_ns = self.act_pos * 1000000000
          self.player.seek_simple(self.time_format, gst.SEEK_FLAG_FLUSH, seek_ns) 
          time.sleep(0.2)
          if (self.player.query_position(self.time_format, None)[0] < seek_ns):
            time.sleep(0.4)
            self.player.seek_simple(self.time_format, gst.SEEK_FLAG_FLUSH, seek_ns) 
        dur_int = self.player.query_duration(self.time_format, None)[0]
        self.prefs["lastdur"] = dur_int / 1000000000 
	self.act_file_len = dur_int
        gtk.gdk.threads_enter()
	self.update_pbar_ns(self.act_pos*1000000000,dur_int)
        gtk.gdk.threads_leave()
	break
      except:
        pass
    time.sleep(0.2)
    #
    # Main Thread for displaying time infos etc.
    #
    while play_thread_id == self.play_thread_id:
      try: 
        pos_int = self.player.query_position(self.time_format, None)[0]
        self.prefs["lastpos"] = pos_int / 1000000000 
        # Do we use timeout for suspend?
        if self.timeout <> -1:
          if self.timeout > 0: 
            newtime = time.time()
            self.timeout = self.timeout - int(newtime-oldtime)
            oldtime = newtime
          else: # Timeout is reached, suspend player
            self.save_prefs()
            self.stop_player() # prepare stopping of threat
            break; 	# leave threat
            
        if self.prefs["metatags"] == 1: # show metatags instead of filename
          if self.tags["title"] <> "":
            self.play_name = self.tags["artist"] + " - " + self.tags["title"]                          
             
        if play_thread_id == self.play_thread_id:
          gtk.gdk.threads_enter()
          self.update_pbar_ns(pos_int,dur_int)
          gtk.gdk.threads_leave()
        time.sleep(1)
      except:
        time.sleep(0.2)

  # Extracs the parent directory name of a given path
  #
  def get_parent_dir(self, path):
    (rest, name) = os.path.split(path)
    if name == '': # if the path ends with '/', we get an empty string
      name = os.path.split(rest)[1] # repeat to get the name
    return name        

  # Recursive rescan given dir and subdirs for audio files.
  # 
  def rescan(self, dir):
    if (dbg): print "Scanning dir ",dir,self.dc
    if not dir.endswith(os.sep):
      dir += os.sep
    self.pl_dirs.append(dir)
    idx = self.dc
    d = os.listdir(dir)
    for f in d:
      if os.path.isdir(dir+f) == 1: # f is a directory
        self.dc += 1 
        self.rescan(dir+f)
      elif os.path.isfile(dir+f) == 1: # f is a file
        ls = string.lower(f)
        if ls.endswith(".mp3") == 1 or \
          ls.endswith(".wav") == 1 or \
          ls.endswith(".aac") == 1 or \
          ls.endswith(".ogg") == 1 or \
          ls.endswith(".flac") == 1:
            self.pl_files.append((f,idx))
            self.max_files += 1

  # Rescan dir for music/sound files
  #
  def menu_rescan_cb(self, btn, lbl):
    # clear fileslist
    hildon.hildon_banner_show_information(self.window, "", "Scanning for new audio files...")
    self.pl_dirs = []      # list to store the directorys containing music files
    self.dc = 0          # actual number of directory
    self.pl_files = []     # list to store filenames of music files
    self.max_files = 0
    self.rescan(self.base_media_dir)
    self.save_playlist()
    hildon.hildon_banner_show_information(self.window, "", "Restart Easyplayer to get new playlist!")

  # row acivated callback is activated if a row is
  # double clicked by the user
  #
  def tv_row_activated_cb(self, tv, path, view_colum):
    self.reset_timer()
    tv.collapse_row(path)
    iter = self.treestore.get_iter(path)
    value = self.treestore.get(iter, 0)
    id = self.get_id(value[0])
    #print "Path: ",path," Value:",value," id:",id
    if id <> None:
      self.play_new_file(id) # Start new file at pos 0

  # Get the file_id to a given name
  #
  def get_id(self, name):
    for i in range(self.max_files):
      if name == self.pl_files[i][0]:
        return i
    return None      

  # Create a button with a default stock image   
  #
  def new_button(self, stock_img):
    image = gtk.Image()
    image.set_from_stock(stock_img,gtk.ICON_SIZE_LARGE_TOOLBAR)
    image.show()
    button = hildon.GtkButton(self.std_size)
    button.add(image)
    return button

  # Menu "About" callback - display About...
  #
  def menu_about_cb(self, btn, lbl):
    note = hildon.hildon_note_new_information(self.window, \
           "EasyPlayer was written 2010 for Maemo 5\n" \
           "by Klaus Rotter (klaus at rotters dot de)\n" \
           "Version " + version)   
    response = gtk.Dialog.run(note)
    #if response == gtk.RESPONSE_DELETE_EVENT:
    #  print "show_information_note: gtk.RESPONSE_DELETE_EVENT"

  # Menu "Display Tags" callback - display mp3 tags of actual playing file
  #
  def menu_display_tags_cb(self, btn, lbl):
    note = hildon.hildon_note_new_information(self.window, \
          "Titel: "+self.tags["title"]+"\n"  \
          "Artist: "+self.tags["artist"]+"\n" \
          "Album: "+self.tags["album"]+"\n"  \
          "Track: "+str(self.tags["track-number"])+"\n" \
          "Encoder: "+self.tags["encoder"]+"\n" \
          "Bitrate: "+str(self.tags["bitrate"]/1000)+" kbit/s")
          # "genre","date","encoder-version","audio-codec","nominal-bitrate"
    gtk.Dialog.run(note)

  # Resets the time-out timer to its starting value (eg. 5 min to 60 min)
  #
  def reset_timer(self):
    to = self.prefs["timeout"]
    if to == 0:
      self.timeout = 5*60	# 5 minutes 
    elif to == 1:
      self.timeout = 10*60	# 10 minutes 
    elif to == 2:
      self.timeout = 20*60	# 20 minutes 
    elif to == 3:
      self.timeout = 30*60	# 30 minutes 
    elif to == 4:
      self.timeout = 60*60	# 60 minutes 
    elif to > 4:	
      self.timeout = -1		# Never

  # Progressbar callback, set new
  #
  def pbar_eb_press_cb(self, widget, event):
    if event.type == gtk.gdk.BUTTON_PRESS:
      frac = event.x/float(widget.get_allocation().width)
      if (self.running == True):
        self.player.seek_simple(self.time_format, gst.SEEK_FLAG_FLUSH, int(self.act_file_len*frac))
        self.pbar.set_fraction(frac)
      else:
        self.act_pos = int(self.act_file_len*frac)/1000000000 # to round to 1 sec
	self.update_pbar_ns(self.act_pos*1000000000,self.act_file_len)

  # Set Image of Button Play to "Play"
  #  
  def set_play_img(self):
    self.play_img.set_from_stock(gtk.STOCK_MEDIA_PLAY, gtk.ICON_SIZE_BUTTON)
    self.play_img.show()

  # Set Image of Button Play to "Stop"
  #  
  def set_stop_img(self):
    self.play_img.set_from_stock(gtk.STOCK_MEDIA_STOP, gtk.ICON_SIZE_BUTTON)
    self.play_img.show()

  # Resets file position to 0 and starts new file
  # 
  def play_new_file(self, id):
    self.stop_player() 		# Stop current file if one is playing
    self.act_file_id = id	# Select new id
    self.act_pos = 0		# Start at pos 0
    self.play_stop_h()  	# let's go...
		
  # Start / Stop Playback
  #
  def play_stop_cb(self, w):
    self.reset_timer() # Reset TimeOut Timer
    self.play_stop_h()
  
  # Start / Stop Handler
  #
  def play_stop_h(self):
    if self.running == False: 
      file_id = self.act_file_id; #which file to play?
      dir_id = self.pl_files[file_id][1]
      self.play_name = self.pl_files[file_id][0]
      filepath = self.pl_dirs[dir_id]+self.play_name
      self.clear_tags()
      #print filepath
      if os.path.isfile(filepath):
        self.set_stop_img()
        self.player.set_property("uri", "file://" + filepath)
        self.player.set_state(gst.STATE_PLAYING)
	# Start thread
	self.play_thread_id = thread.start_new_thread(self.play_thread, ())
        self.running = True
        self.prefs["lastfile"] = file_id
        self.save_prefs()
    else:
      # We are running, so stop player
      self.act_pos = self.player.query_position(self.time_format, None)[0]/1000000000
      if dbg: print("Stopping player: act_pos=",self.act_pos)
      self.stop_player()
      self.prefs["lastpos"] = self.act_pos
      self.save_prefs()

  # "Stop" Handler
  # 
  def stop_player(self):
    if self.running == True:
      self.set_play_img()
      self.running = False
      self.play_thread_id = None
      self.player.set_state(gst.STATE_NULL)

  # Parse any messages from gstreamer
  #    
  def on_message(self, bus, message):
    t = message.type
    if t == gst.MESSAGE_EOS:
      if self.act_file_id < self.max_files: # Is there at least one file more to play
        self.play_new_file(self.act_file_id+1) 	# Start new file  
      else: 			# No more files, just stop
        self.stop_player()
      
    elif t == gst.MESSAGE_ERROR:
      self.stop_player()
      err, debug = message.parse_error()
      if (dbg): print "Error: %s" % err, debug
    elif t == gst.MESSAGE_TAG:
      for key in message.parse_tag().keys():
        self.tags[key] = message.structure[key]
        #print "Tags:",key,message.structure[key]

  # Skip backward 10 s callback
  #
  def rewind_cb(self, w):
    self.reset_timer() # Reset TimeOut Timer
    pos_int = self.player.query_position(self.time_format, None)[0]
    seek_ns = pos_int - (10 * 1000000000)
    self.player.seek_simple(self.time_format, gst.SEEK_FLAG_FLUSH, seek_ns)

  # Skip forward 10 s callback
  #            
  def forward_cb(self, w):
    self.reset_timer() # Reset TimeOut Timer
    pos_int = self.player.query_position(self.time_format, None)[0]
    seek_ns = pos_int + (10 * 1000000000)
    self.player.seek_simple(self.time_format, gst.SEEK_FLAG_FLUSH, seek_ns)

  # Select Previous file to play
  #
  def prev_cb(self, w):
    self.reset_timer() # Reset TimeOut Timer
    pos_int = self.player.query_position(self.time_format, None)[0]
    if pos_int > 10*1000000000: # Jump back to beginning of song
      self.player.seek_simple(self.time_format, gst.SEEK_FLAG_FLUSH, 0)
    elif self.act_file_id > 0: # Is there at least one previous file to play
      self.play_new_file(self.act_file_id-1) # Start new file  
      
  # Select next file to play
  #
  def next_cb(self, w):
    self.reset_timer() # Reset TimeOut Timer
    if self.act_file_id < self.max_files: # Is there at least one file more to play
      self.play_new_file(self.act_file_id+1) # Start new file  

  # Convert a time in secondes to min:sec
  # taken from the pygst tutorial 
  #
  def convert_s(self, time_int):
    time_str = ""
    if time_int >= 3600:
      _hours = time_int / 3600
      time_int = time_int - (_hours * 3600)
      time_str = str(_hours) + ":"
    if time_int >= 600:
      _mins = time_int / 60
      time_int = time_int - (_mins * 60)
      time_str = time_str + str(_mins) + ":"
    elif time_int >= 60:
      _mins = time_int / 60
      time_int = time_int - (_mins * 60)
      time_str = time_str + "0" + str(_mins) + ":"
    else:
      time_str = time_str + "00:"
    if time_int > 9:
      time_str = time_str + str(time_int)
    else:
      time_str = time_str + "0" + str(time_int)
            
    return time_str
        
  # Convert medias streams time in ns to min:sec eg 01:27
  #
  def convert_ns(self, time_int):
    time_int = time_int / 1000000000
    return self.convert_s(time_int)

  # Create and display the preferences window
  #
  def show_pref_win_cb(self, widget):
    # Create the main window
    prefwin = hildon.StackableWindow()
    prefwin.set_title("Settings")
   
    vbox = gtk.VBox(False,0)
    prefwin.add(vbox)
 
    pref_size = gtk.HILDON_SIZE_AUTO|gtk.HILDON_SIZE_FINGER_HEIGHT
 
    cb1 = hildon.CheckButton(pref_size)
    cb1.set_label("Resume playing last file after startup")
    cb1.set_active(self.prefs["resumeplay"])
    cb1.connect("toggled", self.cb_toggled_resume)
    vbox.pack_start(cb1, False, True)

    cb2 = hildon.CheckButton(pref_size)
    cb2.set_label("Use big (thumb) buttons")
    cb2.set_active(self.prefs["bigbuttons"])
    cb2.connect("toggled", self.cb_toggled_big_btn)
    vbox.pack_start(cb2, False, True)
    
    cb3 = hildon.CheckButton(pref_size)
    cb3.set_label("Output sound in silent mode")
    cb3.set_active(self.prefs["silenthack"])
    cb3.connect("toggled", self.cb_toggled_silent_hack)
    vbox.pack_start(cb3, False, True)

    cb4 = hildon.CheckButton(pref_size)
    cb4.set_label("Display Metatags")
    cb4.set_active(self.prefs["metatags"])
    cb4.connect("toggled", self.cb_toggled_metatags)
    vbox.pack_start(cb4, False, True)

    cb5 = hildon.CheckButton(pref_size)
    cb5.set_label("Display remaining time until timeout")
    cb5.set_active(self.prefs["displaytimeout"])
    cb5.connect("toggled", self.cb_toggled_timeout)
    vbox.pack_start(cb5, False, True)

    timeouts = [ "5 min", "10 min", "20 min", "30 min", "60 min", "Never" ]
    # Create a picker button
    timeout_pk = hildon.PickerButton(self.std_size, hildon.BUTTON_ARRANGEMENT_HORIZONTAL) 
    # Set a title to the button 
    timeout_pk.set_title("Stop playing after: ") 
    # Create a touch selector entry
    selector = hildon.TouchSelectorEntry(text=True)
    # Populate the selector
    for timeout in timeouts:
      selector.append_text(timeout)
    # Attach the touch selector to the picker button
    timeout_pk.set_selector(selector)
    timeout_pk.set_active(self.prefs["timeout"])
    timeout_pk.connect("value-changed", self.timeout_changed)

    # Add button to main window
    vbox.pack_start(timeout_pk, False, True, 0)	
    prefwin.show_all()

  # Callback for changing displaying of timeout 
  #
  def cb_toggled_timeout(self, cb):
    self.prefs["displaytimeout"] = cb.get_active()
    self.save_prefs()

  # Callback for changing resume playing 
  #
  def cb_toggled_resume(self, cb):
    self.prefs["resumeplay"] = cb.get_active()
    self.save_prefs()

  # Callback for changing button size 
  #
  def cb_toggled_big_btn(self, cb):
    self.prefs["bigbuttons"] = cb.get_active()
    self.save_prefs()
    hildon.hildon_banner_show_information(self.window, "", "Restart Easyplayer to see this feature")

  # Callback for changing beviour with silent profile
  #
  def cb_toggled_silent_hack(self, cb):
    self.prefs["silenthack"] = cb.get_active()
    self.save_prefs()
    hildon.hildon_banner_show_information(self.window, "", "Restart Easyplayer to see this feature")

    # Changing name once started does not work. Restart player for this to work.
    #if cb.get_active():
    #    gtk.set_application_name("FMRadio")
    #else:
    #    gtk.set_application_name("EasyPlayer")

  # Callback for changing displaying of Filename or Metatags
  #
  def cb_toggled_metatags(self, cb):
    self.prefs["metatags"] = cb.get_active()
    self.save_prefs()

  # Callback for changing timeout/suspend 
  #
  def timeout_changed(self, picker):
    self.prefs["timeout"] = picker.get_active()
    self.reset_timer()
    self.save_prefs()

  # Clear the taglist
  # 
  def clear_tags(self):  
    self.tags = { 
      "title":"", "artist":"", "album":"", "track-number":"", "genre":"", 
      "date":"", "encoder":"", "encoder-version":"", "audio-codec":"", 
      "nominal-bitrate":"", "bitrate":"" 
    }

  # Set default preferences
  # 
  def init_prefs(self):  
    self.prefs = {
      "version":5,		# Preferences version 0.4
      "lastfile":0,		# id of last played file
      "lastpos":0,		# position of last played file in seconds
      "lastdur":0,		# duration of last played file
      "hideext":0,		# Hide filename extention, eg. ".mp3",".ogg"
      "displaytimeout":1,	# Show the remaining time until timeout
      "resumeplay":0,		# resume playing last file
      "timeout":3,		# timeout in minutes, 3 is 30 minutes
      "bigbuttons":0,		# Use HILDON_THUMB_SIZE if 1 (make Buttons even bigger)
      "silenthack":0,		# Do not mute audio in silent profile if 1 (requires restart)
      "metatags":0		# Display audio stream's metatags instead of filename	
    }

  # Save preferences file
  #
  def save_prefs(self):
    prefdir = os.path.dirname(self.prefs_file)
    if os.path.exists(prefdir) == 0: # pref dir doesn't exists
      try:
        os.mkdir(prefdir)
      except:
        if (dbg):print "Can't create preferences directory!"
    try: 
      fd = open(self.prefs_file,"w")
      pickle.dump(self.prefs, fd)
      fd.close()
    except:
      if (dbg):print "Can't save preferences file"

  # Save playlist files: dirs and files
  #
  def save_playlist(self):
    if (dbg):print "Save playlist..."
    pldir = os.path.dirname(self.pl_dirs_file)
    if os.path.exists(pldir) == 0: # playlist dir doesn't exists
      try:
        os.mkdir(pldir)
      except:
        if (dbg):print "Can't create playlist directory!"
    try: 
      # Save directories
      if (dbg):print "Save dirs..."
      fd = open(self.pl_dirs_file,"w")
      pickle.dump(self.pl_dirs, fd)
      fd.close()      
      # Save filenames
      if (dbg):print "Save filenames..."
      fd = open(self.pl_files_file,"w")
      pickle.dump(self.pl_files, fd)
      fd.close()
    except:
      if (dbg):print "Can't save playlist file"
  
  # Helperfunction for makeing a menu item
  #       
  def make_menu_item(self, menu, name, callback, data=None):
    item = gtk.MenuItem(name)
    item.connect("activate", callback, data)
    item.show()
    menu.append(item)
    return item

###################################################
# MAIN
###################################################
    
GTK_Main()
gtk.gdk.threads_init()
gtk.main()
