/*
 */

#include <stdio.h>
#include <stdlib.h>

#include <SDL.h>
#include <SDL_image.h>

#ifdef MAEMO5
#include <gdk/gdkx.h>
#include <X11/Xatom.h>
#endif

#ifndef MAEMO
#include <GL/gl.h>
#include <GL/glu.h>
#define DATA_PATH "data/"
#else
#include <SDL_syswm.h>
#include <SDL_gles.h>
#include <GLES/gl.h>
#include "glues/glu.h"
#define DATA_PATH "/opt/earth/"
#include <pthread.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <math.h>
#endif

/* screen width, height, and bit depth */
#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     16

/* Set up some booleans */
#define TRUE  1
#define FALSE 0

/* Number of textures to load */
#define NUM_TEXTURES 2

/* This is our SDL surface */
SDL_Surface *surface;
#ifdef MAEMO
SDL_GLES_Context *context;
#endif

/* Whether or not lighting is on */
int light = FALSE;

GLfloat xrot = 0.0f;      /* X Rotation            */
GLfloat yrot = 0.0f;      /* Y Rotation            */
GLfloat yspeed = 0.1f;    /* Y Rotation Speed      */
GLfloat z = -4.0f; /* Depth Into The Screen */
int order = 0;

/* Ambient Light Values */
GLfloat LightAmbient[]  = { 0.01f, 0.01f, 0.01f, 1.0f };
/* Diffuse Light Values */
GLfloat LightDiffuse[]  = { 0.8f, 0.8f, 0.8f, 1.0f };
/* Specular Light Values */
GLfloat LightSpecular[]  = { 5.0f, 5.0f, 5.0f, 1.0f };
/* Light Position */
GLfloat LightPosition[] = { 4.0f, 4.0f, 1.0f, 1.0f };

GLUquadricObj *quadratic;     /* Storage For Our Quadratic Objects */

GLuint texture[NUM_TEXTURES]; /* Storage for textures            */

#ifdef MAEMO
/* accelerometer stuff */
pthread_t thread;
pthread_mutex_t acc_mutex;
#define AVG  8
int acc_axis_list[3][AVG];
int acc_axis[3];

static void *acc_thread(void *data) {
  /* run as long as there's another client besides the thread itself */
  /* and as long as the file descriptor is valid */
  while(1) {
    int acc_fd = open("/sys/class/i2c-adapter/i2c-3/3-001d/coord", O_RDONLY);
    if(acc_fd >= 0) {
      char buffer[32];
      int n;

      if((n = read(acc_fd, buffer, sizeof(buffer))) >= 0) {
        buffer[n] == 0;
	int i, j;
	for(i=0;i<3;i++)
	  memmove(&acc_axis_list[i][1], &acc_axis_list[i][0], 
		  (AVG-1) * sizeof(int));

        sscanf(buffer, "%d %d %d\n", 
            &acc_axis_list[0][0], &acc_axis_list[1][0], &acc_axis_list[2][0]);

	pthread_mutex_lock(&acc_mutex);
	for(i=0;i<3;i++) {
	  acc_axis[i] = 0;
	  for(j=0;j<AVG;j++)
	    acc_axis[i] += acc_axis_list[i][j];
	  acc_axis[i] /= AVG;
	}
	pthread_mutex_unlock(&acc_mutex);
      }
      close(acc_fd);
    }
    usleep(20000);
  }
}

static void acc_init(void) {
  pthread_mutex_init(&acc_mutex, NULL);

  /* fork a handler thread */
  if ( pthread_create(&thread, NULL, acc_thread, NULL) != 0 ) {
    fprintf(stderr, "acc: Creation of thread failed: %s(%d)\n",
            strerror(errno), errno);
  }
}
#endif

/* function to release/destroy our resources and restoring the old desktop */
void Quit( int returnCode )
{
    /* Clean up our quadratic */
    gluDeleteQuadric( quadratic );

    /* Clean up our textures */
    glDeleteTextures( NUM_TEXTURES, texture);

#ifdef MAEMO
    SDL_GLES_DeleteContext(context);  
#endif

    /* clean up the window */
    SDL_Quit( );

    /* and exit appropriately */
    exit( returnCode );
}

/* function to load in bitmap as a GL texture */
int LoadGLTextures( )
{
  /* Status indicator */
  int status = TRUE;
  int i;

  const char *TextureNames[] = {
    DATA_PATH "texture.jpg",
    DATA_PATH "clouds.jpg"
  };

  /* Create The Texture */
  glGenTextures( NUM_TEXTURES, texture );
  
  for(i=0;i<NUM_TEXTURES;i++) {
    /* Create storage space for the texture */
    SDL_Surface *TextureImage = NULL;
  
    /* Load The Bitmap, Check For Errors, If Bitmap's Not Found Quit */
    if ( ( TextureImage = IMG_Load( TextureNames[i] ) ) ) {
      
      /* Typical Texture Generation Using Data From The Bitmap */
      glBindTexture( GL_TEXTURE_2D, texture[i] );
      
      /* Mipmapped Filtering */
      glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
		       GL_LINEAR_MIPMAP_NEAREST );
      glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
		       GL_LINEAR );
      
      /* Generate The MipMapped Texture */
      gluBuild2DMipmaps( GL_TEXTURE_2D, GL_RGB, TextureImage->w,
			 TextureImage->h, GL_RGB,
			 GL_UNSIGNED_BYTE, TextureImage->pixels );
      
      if ( TextureImage )
	SDL_FreeSurface( TextureImage );
    } else
      status = FALSE;
  }

  return status;
}

/* function to reset our viewport after a window resize */
int resizeWindow( int width, int height )
{
#ifdef MAEMO
  SDL_SysWMinfo info;
  SDL_VERSION(&info.version);                                                                                                                          
  if ( SDL_GetWMInfo(&info) ) {                                                                                                                        

    /* We use the SDL GFX display (we're using the GFX window too after all) */
    Display *dpy = info.info.x11.display;
    unsigned long val = 1;
    Atom atom_zoom = XInternAtom(dpy, "_HILDON_ZOOM_KEY_ATOM", 0);
    info.info.x11.lock_func();
    Window win = info.info.x11.window;
    XUnmapWindow(dpy,win);
    XChangeProperty (dpy,win,atom_zoom,XA_INTEGER,32,PropModeReplace,(unsigned char *) &val,1);
    XMapWindow(dpy,win);
    info.info.x11.unlock_func();
  }
#endif
  
    /* Height / width ration */
    GLfloat ratio;
 
    /* Protect against a divide by zero */
    if ( height == 0 )
	height = 1;

    ratio = ( GLfloat )width / ( GLfloat )height;

    /* Setup our viewport. */
    glViewport( 0, 0, ( GLint )width, ( GLint )height );

    /* change to the projection matrix and set our viewing volume. */
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity( );

    /* Set our perspective */
    gluPerspective( 45.0f, ratio, 0.1f, 100.0f );

    /* Make sure we're chaning the model view and not the projection */
    glMatrixMode( GL_MODELVIEW );

    /* Reset The View */
    glLoadIdentity( );

    return( TRUE );
}

/* function to handle key press events */
void handleKeyPress( SDL_keysym *keysym )
{
  printf("keysym %d\n", keysym->sym);

    switch ( keysym->sym )
	{
	case SDLK_ESCAPE:
	    /* ESC key was pressed */
	    Quit( 0 );
	    break;
	case SDLK_l:
	    /* 'l' key was pressed
	     * this toggles the light
	     */
	    light = !light;
	    if ( !light )
		glDisable( GL_LIGHTING );
	    else
		glEnable( GL_LIGHTING );
	    break;
	case SDLK_i:
	case SDLK_F6:
	    /* 'i' key was pressed
	     * this zooms into the scene
	     */
	    z += 0.02f;
	    break;
	case SDLK_o:
	case SDLK_F7:
	    /* 'o' key was pressed
	     * this zooms out of the scene
	     */
	    z -= 0.02f;
	    break;
	case SDLK_RIGHT:
	    /* Right arrow key was pressed
	     * this affects the y rotation
	     */
	    yspeed += 0.01f;
	    break;
	case SDLK_LEFT:
	    /* Left arrow key was pressed
	     * this affects the y rotation
	     */
	    yspeed -= 0.01f;
	    break;
	case SDLK_F1:
	    /* 'f' key was pressed
	     * this toggles fullscreen mode
	     */
	    SDL_WM_ToggleFullScreen( surface );
	    break;
	default:
	    break;
	}

    return;
}

/* general OpenGL initialization function */
int initGL( void ) {
#ifdef MAEMO
    SDL_GLES_Init(SDL_GLES_VERSION_1_1);

    /* specify size of depth buffer (NEW) */
    SDL_GLES_SetAttribute(SDL_GLES_DEPTH_SIZE, 24);

    context = SDL_GLES_CreateContext();
    SDL_GLES_MakeCurrent(context);
#endif

    /* Load in the texture */
    if ( !LoadGLTextures( ) )
	return FALSE;

    /* Enable Texture Mapping */
    glEnable( GL_TEXTURE_2D );

    /* Enable smooth shading */
    glShadeModel( GL_SMOOTH );

    /* Set the background black */
    glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );

    /* Depth buffer setup */
#ifdef GL_VERSION_ES_CM_1_0
    glClearDepthf( 1.0f );
#else
    glClearDepth( 1.0f );
#endif

    /* Enables Depth Testing */
    glEnable( GL_DEPTH_TEST );

    /* The Type Of Depth Test To Do */
    glDepthFunc( GL_LEQUAL );

    /* Really Nice Perspective Calculations */
    glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

    /* Setup The Ambient Light */
    glLightfv( GL_LIGHT1, GL_AMBIENT, LightAmbient );

    /* Setup The Diffuse Light */
    glLightfv( GL_LIGHT1, GL_DIFFUSE, LightDiffuse );

    glLightfv(GL_LIGHT1, GL_SPECULAR, LightSpecular );

    /* Position The Light */
    glLightfv( GL_LIGHT1, GL_POSITION, LightPosition );

    /* Enable Light One */
    glEnable( GL_LIGHT1 );
    glEnable( GL_LIGHTING );

    glEnable(GL_NORMALIZE);

    /* Create A Pointer To The Quadric Object */
    quadratic = gluNewQuadric( );
    /* Create Smooth Normals */
    gluQuadricNormals( quadratic, GLU_SMOOTH );
    /* Create Texture Coords */
    gluQuadricTexture( quadratic, GL_TRUE );

    return( TRUE );
}

/* Here goes our drawing code */
int drawGLScene( ) {
    /* These are to calculate our fps */
    static GLint T0     = 0;
    static GLint Frames = 0;

    /* Clear The Screen And The Depth Buffer */
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    /* Reset the view */
    glLoadIdentity( );

    /* Translate Into/Out Of The Screen By z */
    glTranslatef( 0.0f, 0.0f, z );

    // enable color tracking
    glEnable(GL_COLOR_MATERIAL);

    if(!order) {
      glRotatef( yrot, 1.0f, 0.0f, 0.0f); /* Rotate On The Y Axis By yrot */
      glRotatef( xrot, 0.0f, 1.0f, 0.0f); /* Rotate On The X Axis By xrot */
    } else {
      glRotatef( xrot, 0.0f, 1.0f, 0.0f); /* Rotate On The X Axis By xrot */
      glRotatef( yrot, 1.0f, 0.0f, 0.0f); /* Rotate On The Y Axis By yrot */
    }

    /* Select Texture */
    glBindTexture( GL_TEXTURE_2D, texture[0] );

    glMaterialf(GL_FRONT_AND_BACK, GL_SHININESS, 96.0);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, LightSpecular);

    gluSphere( quadratic, 1.28f, 64, 64 );

    /* Enable Blending */
    glEnable( GL_BLEND );
 
    /* Select Texture */
    glBindTexture( GL_TEXTURE_2D, texture[1] );

    glBlendFunc( GL_ONE, GL_ONE );
    gluSphere( quadratic, 1.3f, 64, 64 );

    glDisable( GL_BLEND );

    /* Draw it to the screen */
#ifdef MAEMO
    SDL_GLES_SwapBuffers( );
#else
    SDL_GL_SwapBuffers( );
#endif

    /* Gather our frames per second */
    Frames++;
    {
	GLint t = SDL_GetTicks();
	if (t - T0 >= 5000) {
	    GLfloat seconds = (t - T0) / 1000.0;
	    GLfloat fps = Frames / seconds;
	    printf("%d frames in %g seconds = %g FPS\n", Frames, seconds, fps);
	    T0 = t;
	    Frames = 0;
	}
    }

    //    yrot += yspeed; /* Add yspeed To yrot */

    return( TRUE );
}

int main( int argc, char *argv[] ) {
  /* main loop variable */
  int done = FALSE;
  /* used to collect events */
  SDL_Event event;
  /* whether or not the window is active */
  int isActive = TRUE;
  
#ifndef MAEMO
  /* Flags to pass to SDL_SetVideoMode */
  int videoFlags;
  /* this holds some info about our display */
  const SDL_VideoInfo *videoInfo;
#endif
  
  /* initialize SDL */
  if ( SDL_Init( SDL_INIT_VIDEO ) < 0 ) {
    fprintf( stderr, "Video initialization failed: %s\n",
	     SDL_GetError( ) );
    Quit( 1 );
  }
  
  SDL_WM_SetCaption("Earth 3D", "Earth 3D");

#ifdef MAEMO
  surface = SDL_SetVideoMode(0, 0, 16, SDL_SWSURFACE | SDL_FULLSCREEN);
  SDL_ShowCursor(0);

  acc_init();
#else
  /* Fetch the video info */
  videoInfo = SDL_GetVideoInfo( );
  if ( !videoInfo ) {
    fprintf( stderr, "Video query failed: %s\n",
	     SDL_GetError( ) );
    Quit( 1 );
  }

  /* the flags to pass to SDL_SetVideoMode                            */
  videoFlags  = SDL_OPENGL;          /* Enable OpenGL in SDL          */
  videoFlags |= SDL_GL_DOUBLEBUFFER; /* Enable double buffering       */
  videoFlags |= SDL_HWPALETTE;       /* Store the palette in hardware */
  videoFlags |= SDL_RESIZABLE;       /* Enable window resizing        */
  
  /* This checks to see if surfaces can be stored in memory */
  if ( videoInfo->hw_available ) videoFlags |= SDL_HWSURFACE;
  else                           videoFlags |= SDL_SWSURFACE;
  
  /* This checks if hardware blits can be done */
  if ( videoInfo->blit_hw )
    videoFlags |= SDL_HWACCEL;
  
  /* Sets up OpenGL double buffering */
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
  
  /* get a SDL surface */
  surface = SDL_SetVideoMode( SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_BPP,
			      videoFlags );
  
  /* Verify there is a surface */
  if ( !surface ) {
    fprintf( stderr,  "Video mode set failed: %s\n", SDL_GetError( ) );
    Quit( 1 );
  }
#endif
  
  /* Enable key repeat */
  if ( ( SDL_EnableKeyRepeat( 100, SDL_DEFAULT_REPEAT_INTERVAL ) ) ) {
    fprintf( stderr, "Setting keyboard repeat failed: %s\n",
	     SDL_GetError( ) );
    Quit( 1 );
  }
  
  /* initialize OpenGL */
  if ( initGL( ) == FALSE ) {
    fprintf( stderr, "Could not initialize OpenGL.\n" );
    Quit( 1 );
  }
  
  /* resize the initial window */
  resizeWindow( SCREEN_WIDTH, SCREEN_HEIGHT );

  /* wait for events */
  while ( !done ) {
    /* handle the events in the queue */
    
    while ( SDL_PollEvent( &event ) ) {
      switch( event.type ) {
      case SDL_ACTIVEEVENT:
	/* Something's happend with our focus
	 * If we lost focus or we are iconified, we
	 * shouldn't draw the screen
	 */
	if ( event.active.gain == 0 )
	  isActive = FALSE;
	else
	  isActive = TRUE;
	break;			    
#ifndef MAEMO
      case SDL_VIDEORESIZE:
	/* handle resize event */
	surface = SDL_SetVideoMode( event.resize.w,
				    event.resize.h,
				    16, videoFlags );
	if ( !surface ) {
	  fprintf( stderr, "Could not get a surface after resize: %s\n", 
		   SDL_GetError( ) );
	  Quit( 1 );
	}
	resizeWindow( event.resize.w, event.resize.h );
	break;
#endif
      case SDL_KEYDOWN:
	/* handle key presses */
	handleKeyPress( &event.key.keysym );
	break;
      case SDL_MOUSEBUTTONDOWN:
      case SDL_QUIT:
	/* handle quit requests */
	done = TRUE;
	break;
      default:
	break;
      }
    }
    
#ifdef MAEMO
    pthread_mutex_lock(&acc_mutex);

    /* this is good if axis[1] or axis[2] is significantly != 0.0 */
    float x = acc_axis[0]/1000.0f;
    if(x < -1.0f) x = -1.0f;
    if(x >  1.0f) x =  1.0f;

    float y = acc_axis[1]/1000.0f;
    if(y < -1.0f) y = -1.0f;
    if(y >  1.0f) y =  1.0f;

    //    if(fabs(x) > sqrt(2.0f)) {
    //    if(fabs(x) < fabs(y)) {
    if(1) {
      xrot = -asin(x) / M_PI * 180.0f;
      yrot = atan2(acc_axis[2], acc_axis[1]) / M_PI * 180.0f - 90.0f;
      order = 0;
    } else {
      xrot = atan2(acc_axis[2], acc_axis[0]) / M_PI * 180.0f + 90.0f;
      yrot = -90.0 - acos(y) / M_PI * 180.0f;
      order = 1;
    }

    pthread_mutex_unlock(&acc_mutex);
#endif

    /* draw the scene */
    if ( isActive )
      drawGLScene( );
  }
  
  /* clean ourselves up and exit */
  Quit( 0 );
  
  /* Should never get here */
  return( 0 );
}
