/****************************************************************************
** Dooble - The Secure Internet Web Browser
**
** Copyright (c) 2008, 2009, 2010, 2011, 2012, 2013 Alexis Megas,
** Gunther van Dooble, and the Dooble Team.
** All rights reserved.
**
** License: GPL2 only:
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; version 2 of the License only.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
** or see here: http://www.gnu.org/licenses/gpl.html
**
** For the WebKit library, please see: http://webkit.org.
**
** THE CODE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY
** EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
** IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS APPLICATION, EVEN IF ADVISED
** OF THE POSSIBILITY OF SUCH DAMAGE.
**
** Please report all praise, requests, bugs, and problems to the project
** team and administrators: http://sf.net/projects/dooble.
**
** You can find us listed at our project page. New team members are welcome.
** The name of the authors should not be used to endorse or promote products
** derived from Dooble without specific prior written permission.
** If you use this code for other projects, please let us know.
**
** Web sites:
**   http://sf.net/projects/dooble
**   http://dooble.sf.net
****************************************************************************/

#include <QDir>
#include <QBuffer>
#include <QtEndian>
#include <QProcess>
#include <QSettings>
#include <QSqlQuery>
#include <QProgressBar>
#include <QSqlDatabase>
#include <QtCore/qmath.h>
#include <QDesktopWidget>
#include <QFileIconProvider>
#include <QCryptographicHash>

#include "dmisc.h"
#include "dooble.h"
#include "derrorlog.h"

/*
** OK, so you know. One of the below proxy methods (a static one)
** returns a list of proxies. How does one detect which proxy is valid?
*/

/*
** The method systemProxyForQuery() may take several
** seconds to execute on Windows systems. Therefore,
** some intelligence may need to be introduced in order to
** enhance Dooble's response times.
*/

extern "C"
{
#include <errno.h>
#include <pthread.h>

#if defined(PTHREAD_H) || defined(_PTHREAD_H) || defined(_PTHREAD_H_)
  GCRY_THREAD_OPTION_PTHREAD_IMPL;
#endif
}

#if !(defined(PTHREAD_H) || defined(_PTHREAD_H) || defined(_PTHREAD_H_))
#include <QMutex>
extern "C"
{
  int gcry_qthread_init(void)
  {
    return 0;
  }

  int gcry_qmutex_init(void **mutex)
  {
    *mutex = static_cast<void *> (new QMutex());

    if(*mutex)
      return 0;
    else
      return -1;
  }

  int gcry_qmutex_destroy(void **mutex)
  {
    delete static_cast<QMutex *> (*mutex);
    return 0;
  }

  int gcry_qmutex_lock(void **mutex)
  {
    QMutex *m = static_cast<QMutex *> (*mutex);

    if(m)
      {
	m->lock();
	return 0;
      }
    else
      return -1;
  }

  int gcry_qmutex_unlock(void **mutex)
  {
    QMutex *m = static_cast<QMutex *> (*mutex);

    if(m)
      {
	m->unlock();
	return 0;
      }
    else
      return -1;
  }
}

struct gcry_thread_cbs gcry_threads_qt =
  {
    GCRY_THREAD_OPTION_USER, gcry_qthread_init, gcry_qmutex_init,
    gcry_qmutex_destroy, gcry_qmutex_lock, gcry_qmutex_unlock,
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
  };
#endif

bool dmisc::s_passphraseWasAuthenticated = false;
dcrypt *dmisc::s_crypt = 0;
dcrypt *dmisc::s_reencodeCrypt = 0;

void dmisc::destroyCrypt(void)
{
  if(s_crypt)
    delete s_crypt;

  if(s_reencodeCrypt)
    delete s_reencodeCrypt;
}

void dmisc::initializeCrypt(void)
{
  if(!gcry_control(GCRYCTL_INITIALIZATION_FINISHED_P))
    {
#if defined(PTHREAD_H) || defined(_PTHREAD_H) || defined(_PTHREAD_H_)
      gcry_control(GCRYCTL_SET_THREAD_CBS, &gcry_threads_pthread, 0);
#else
      logError(QObject::tr("dmisc::initializeCrypt(): Using gcry_threads_qt's "
			   "address as the second parameter to "
			   "gcry_control()."));
      gcry_control(GCRYCTL_SET_THREAD_CBS, &gcry_threads_qt, 0);
#endif

      if(!gcry_check_version(GCRYPT_VERSION))
	logError(QObject::tr("dmisc::initializeCrypt(): "
			     "gcry_check_version() failure."));
      else
	{
	  /*
	  ** Disable secure memory.
	  */

	  gcry_control(GCRYCTL_ENABLE_M_GUARD);
	  gcry_control(GCRYCTL_SUSPEND_SECMEM_WARN);
	  gcry_control(GCRYCTL_INIT_SECMEM, 16384, 0);
	  gcry_control(GCRYCTL_RESUME_SECMEM_WARN);
	  gcry_control(GCRYCTL_INITIALIZATION_FINISHED, 0);
	}
    }
  else
    logError(QObject::tr("dmisc::initializeCrypt(): It appears that "
			 "the gcrypt library is already initialized."));
}

void dmisc::setCipherPassphrase(const QString &passphrase,
				const bool save,
				const QString &hashType,
				const QString &cipherType,
				const int iterationCount,
				const QByteArray &salt)
{
  if(s_crypt)
    delete s_crypt;

  s_crypt = new dcrypt(salt, cipherType, hashType, passphrase, iterationCount);

  if(!s_crypt->initialized())
    {
      delete s_crypt;
      s_crypt = 0;
      s_passphraseWasAuthenticated = false;
      return;
    }

  if(!passphrase.isEmpty())
    s_passphraseWasAuthenticated = true;

  if(save && s_passphraseWasAuthenticated)
    {
      QString hash("");

      hash = passphraseHash(passphrase, s_crypt->salt().toHex(), hashType);

      QSettings settings;

      settings.setValue("settingsWindow/cipherType", cipherType);
      settings.setValue("settingsWindow/iterationCount",
			static_cast<int> (s_crypt->iterationCount()));
      settings.setValue("settingsWindow/passphraseHash", hash);
      settings.setValue("settingsWindow/passphraseHashType", hashType);
      settings.setValue("settingsWindow/passphraseSalt",
			s_crypt->salt().toHex());
      settings.setValue("settingsWindow/saltLength",
			s_crypt->salt().length());
      dooble::s_settings["settingsWindow/cipherType"] = cipherType;
      dooble::s_settings["settingsWindow/iterationCount"] =
	static_cast<int> (s_crypt->iterationCount());
      dooble::s_settings["settingsWindow/passphraseHash"] = hash;
      dooble::s_settings["settingsWindow/passphraseHashType"] = hashType;
      dooble::s_settings["settingsWindow/passphraseSalt"] =
	s_crypt->salt().toHex();
      dooble::s_settings["settingsWindow/saltLength"] =
	s_crypt->salt().length();
    }
}

QNetworkProxy dmisc::proxyByUrl(const QUrl &url)
{
  QString str(dooble::s_settings.value("settingsWindow/browsingProxySetting",
				       "none").toString());
  QString scheme(url.scheme().toLower().trimmed());
  QNetworkProxy proxy;

  proxy.setType(QNetworkProxy::NoProxy);

#ifdef DOOBLE_VIDALIA_PLUGIN_SUPPORTED
  /*
  ** If the Vidalia plugin is loaded, we'll ignore other proxy settings and
  ** apply proxy attributes provided by Vidalia.
  ** Proxy is socks5 proxy, usually 127.0.0.1:9050. The userName and
  ** userPwd are empty by default.
  */

  if(dooble::s_settings.value("vidalia/isConnected", false).toBool())
    {
      if(!dooble::s_settings.value("vidalia/hostName",
				   "").toString().isEmpty())
	{
	  proxy.setType(QNetworkProxy::Socks5Proxy);
          proxy.setHostName(dooble::s_settings.value("vidalia/hostName",
						     "").toString());
          proxy.setPort(dooble::s_settings.value("vidalia/port", 0).toInt());
          proxy.setUser(dooble::s_settings.value("vidalia/userName", "").
			toString());
          proxy.setPassword(dooble::s_settings.value("vidalia/userPwd",
						     "").toString());
          return proxy;
	}
    }
#endif

  if(str == "system")
    {
      QNetworkProxyQuery query(url);
      QList<QNetworkProxy> list
	(QNetworkProxyFactory::systemProxyForQuery(query));

      if(!list.isEmpty())
	proxy = list.at(0);
    }
  else if(str == "manual")
    {
      if(scheme == "ftp")
	if(dooble::s_settings.value("settingsWindow/ftpBrowsingProxyEnabled",
				    false).toBool())
	  {
	    proxy.setType(QNetworkProxy::Socks5Proxy);
	    proxy.setHostName
	      (dooble::s_settings.value("settingsWindow/"
					"ftpBrowsingProxyHost",
					"").toString().trimmed());
	    proxy.setPort
	      (static_cast<quint16> (dooble::s_settings.value
				     ("settingsWindow/ftpBrowsingProxyPort",
				      1080).toInt()));
	    proxy.setUser
	      (dooble::s_settings.value("settingsWindow/"
					"ftpBrowsingProxyUser",
					"").toString());
	    proxy.setPassword
	      (dooble::s_settings.value("settingsWindow/"
					"ftpBrowsingProxyPassword",
					"").toString());
	  }

      if(scheme == "http" || scheme == "https")
	if(dooble::s_settings.value("settingsWindow/httpBrowsingProxyEnabled",
				    false).toBool())
	  {
	    QString str("");

	    str = dooble::s_settings.value
	      ("settingsWindow/httpBrowsingProxyType",
	       "Socks5").toString().trimmed().toLower();

	    if(str == "http")
	      proxy.setType(QNetworkProxy::HttpProxy);
	    else
	      proxy.setType(QNetworkProxy::Socks5Proxy);

	    proxy.setHostName
	      (dooble::s_settings.value("settingsWindow/httpBrowsingProxyHost",
					"").toString().trimmed());
	    proxy.setPort
	      (static_cast<quint16> (dooble::s_settings.value
				     ("settingsWindow/httpBrowsingProxyPort",
				      1080).toInt()));
	    proxy.setUser
	      (dooble::s_settings.value("settingsWindow/httpBrowsingProxyUser",
					"").toString());
	    proxy.setPassword(dooble::s_settings.value
			      ("settingsWindow/httpBrowsingProxyPassword",
			       "").toString());
	  }
    }

  /*
  ** Override!
  */

  if(url.host().toLower().trimmed().endsWith(".i2p") &&
     dooble::s_settings.value("settingsWindow/i2pBrowsingProxyEnabled",
			      true).toBool())
    {
      QString str("");

      str = dooble::s_settings.value("settingsWindow/i2pBrowsingProxyType",
				     "Http").toString().trimmed().
	toLower();

      if(str == "socks5")
	proxy.setType(QNetworkProxy::Socks5Proxy);
      else if(str == "http")
	proxy.setType(QNetworkProxy::HttpProxy);
      else
	proxy.setType(QNetworkProxy::HttpProxy);

      proxy.setHostName(dooble::s_settings.value
			("settingsWindow/i2pBrowsingProxyHost",
			 "127.0.0.1").toString().trimmed());
      proxy.setPort
	(static_cast<quint16> (dooble::s_settings.value
			       ("settingsWindow/i2pBrowsingProxyPort",
				4444).toInt()));
    }

  return proxy;
}

QNetworkProxy dmisc::proxyByFunctionAndUrl
(const DoobleDownloadType::DoobleDownloadTypeEnum functionType,
 const QUrl &url)
{
  QString str(dooble::s_settings.value("settingsWindow/browsingProxySetting",
				       "none").toString());
  QNetworkProxy proxy;

  proxy.setType(QNetworkProxy::NoProxy);

  switch(functionType)
    {
    case DoobleDownloadType::Ftp:
      {
	if(str == "system")
	  {
	    QNetworkProxyQuery query(url);
	    QList<QNetworkProxy> list
	      (QNetworkProxyFactory::systemProxyForQuery(query));

	    if(!list.isEmpty())
	      proxy = list.at(0);
	  }
	else if(str == "manual" &&
		dooble::s_settings.value
		("settingsWindow/ftpDownloadProxyEnabled",
		 false).toBool())
	  {
	    proxy.setType(QNetworkProxy::Socks5Proxy);
	    proxy.setHostName(dooble::s_settings.value("settingsWindow/"
						       "ftpDownloadProxyHost",
						       "").toString().
			      trimmed());
	    proxy.setPort
	      (static_cast<quint16> (dooble::s_settings.value
				     ("settingsWindow/ftpDownloadProxyPort",
				      1080).toInt()));
	    proxy.setUser(dooble::s_settings.value("settingsWindow/"
						   "ftpDownloadProxyUser",
						   "").toString());
	    proxy.setPassword
	      (dooble::s_settings.value("settingsWindow/"
					"ftpDownloadProxyPassword",
					"").toString());
	  }

	break;
      }
    case DoobleDownloadType::Http:
      {
	if(str == "system")
	  {
	    QNetworkProxyQuery query(url);
	    QList<QNetworkProxy> list
	      (QNetworkProxyFactory::systemProxyForQuery(query));

	    if(!list.isEmpty())
	      proxy = list.at(0);
	  }
	else if(str == "manual" &&
		dooble::s_settings.value
		("settingsWindow/httpDownloadProxyEnabled",
		 false).toBool())
	  {
	    QString str("");

	    str = dooble::s_settings.value
	      ("settingsWindow/httpDownloadProxyType",
	       "Socks5").toString().trimmed().
	      toLower();

	    if(str == "http")
	      proxy.setType(QNetworkProxy::HttpProxy);
	    else
	      proxy.setType(QNetworkProxy::Socks5Proxy);

	    proxy.setHostName(dooble::s_settings.value
			      ("settingsWindow/httpDownloadProxyHost",
			       "").toString().trimmed());
	    proxy.setPort
	      (static_cast<quint16> (dooble::s_settings.value
				     ("settingsWindow/httpDownloadProxyPort",
				      1080).toInt()));
	    proxy.setUser(dooble::s_settings.value
			  ("settingsWindow/httpDownloadProxyUser",
			   "").toString());
	    proxy.setPassword
	      (dooble::s_settings.value
	       ("settingsWindow/httpDownloadProxyPassword",
		"").toString());
	  }

	break;
      }
    default:
      break;
    }

  /*
  ** Override!
  */

  if(url.host().toLower().trimmed().endsWith(".i2p") &&
     dooble::s_settings.value("settingsWindow/i2pDownloadProxyEnabled",
			      true).toBool())
    {
      QString str("");

      str = dooble::s_settings.value("settingsWindow/i2pDownloadProxyType",
				     "Http").toString().trimmed().
	toLower();

      if(str == "socks5")
	proxy.setType(QNetworkProxy::Socks5Proxy);
      else if(str == "http")
	proxy.setType(QNetworkProxy::HttpProxy);
      else
	proxy.setType(QNetworkProxy::HttpProxy);

      proxy.setHostName(dooble::s_settings.value
			("settingsWindow/i2pDownloadProxyHost",
			 "127.0.0.1").toString().trimmed());
      proxy.setPort
	(static_cast<quint16> (dooble::s_settings.value
			       ("settingsWindow/i2pDownloadProxyPort",
				4444).toInt()));
    }

  return proxy;
}

void dmisc::saveIconForUrl(const QIcon &icon, const QUrl &url)
{
  if(!dooble::s_settings.value("settingsWindow/enableFaviconsDatabase",
			      false).toBool())
    return;
  else if(icon.isNull() || url.isEmpty() || !url.isValid())
    return;

  {
    QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE", "favicons");

    db.setDatabaseName(dooble::s_homePath + QDir::separator() +
		       "favicons.db");

    if(db.open())
      {
	int temporary = passphraseWasAuthenticated() ? 0 : 1;
	bool ok = true;
	QSqlQuery query(db);

	query.exec("CREATE TABLE IF NOT EXISTS favicons ("
		   "url TEXT NOT NULL, "
		   "url_hash TEXT PRIMARY KEY NOT NULL, "
		   "favicon BLOB DEFAULT NULL, "
		   "temporary INTEGER NOT NULL)");
	query.exec("CREATE INDEX IF NOT EXISTS url_hash_index ON "
		   "favicons (url_hash)");
	query.exec("PRAGMA synchronous = OFF");
	query.prepare
	  ("INSERT OR REPLACE INTO favicons "
	   "(url, url_hash, favicon, temporary) "
	   "VALUES (?, ?, ?, ?)");
	query.bindValue
	  (0,
	   encodedString(url.toEncoded(QUrl::StripTrailingSlash),
			 true, &ok).toBase64());
	query.bindValue
	  (1,
	   hashedString(url.toEncoded(QUrl::StripTrailingSlash)).toBase64());

	QBuffer buffer;
	QByteArray bytes;

	buffer.setBuffer(&bytes);
	buffer.open(QIODevice::WriteOnly);

	QDataStream out(&buffer);

	if(url.scheme().toLower().trimmed() == "ftp")
	  out << icon.pixmap(16, QIcon::Normal, QIcon::On);
	else
	  out << icon;

	if(ok)
	  query.bindValue
	    (2, encodedString(bytes, true, &ok));

	buffer.close();
	query.bindValue(3, temporary);

	if(ok)
	  query.exec();
      }

    db.close();
  }

  QSqlDatabase::removeDatabase("favicons");
}

QIcon dmisc::iconForUrl(const QUrl &url)
{
  QString scheme(url.scheme().toLower().trimmed());
  QFileIconProvider iconProvider;

  if(scheme == "ftp" || scheme == "http" || scheme == "https" ||
     scheme == "qrc")
    {
      QIcon icon;
      QPixmap pixmap;

      {
	QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE",
						    "favicons");

	db.setDatabaseName(dooble::s_homePath +
			   QDir::separator() + "favicons.db");

	if(db.open())
	  {
	    int temporary = passphraseWasAuthenticated() ? 0 : 1;
	    QSqlQuery query(db);

	    query.setForwardOnly(true);
	    query.prepare("SELECT favicon FROM favicons "
			  "INDEXED BY url_hash_index "
			  "WHERE "
			  "temporary = ? AND "
			  "url_hash = ?");
	    query.bindValue(0, temporary);
	    query.bindValue
	      (1, hashedString(url.toEncoded(QUrl::StripTrailingSlash)).
	       toBase64());

	    if(query.exec())
	      if(query.next())
		if(!query.isNull(0))
		  {
		    QBuffer buffer;
		    QByteArray bytes(query.value(0).toByteArray());

		    bytes = decodedString(bytes);
		    buffer.setBuffer(&bytes);
		    buffer.open(QIODevice::ReadOnly);

		    QDataStream in(&buffer);

		    if(scheme == "ftp")
		      in >> pixmap;
		    else
		      in >> icon;

		    buffer.close();
		  }
	  }

	db.close();
      }

      QSqlDatabase::removeDatabase("favicons");

      if(scheme == "ftp")
	icon = QIcon(pixmap);

      if(!icon.isNull())
	return icon;
    }
  else if(scheme == "file")
    {
      QFileInfo fileInfo(url.toLocalFile());

      if(fileInfo.isFile())
	return iconProvider.icon(QFileIconProvider::File);
      else if(fileInfo.isDir())
	return iconProvider.icon(QFileIconProvider::Folder);
    }

  QSettings settings
    (dooble::s_settings.value("iconSet").toString(), QSettings::IniFormat);

  return QIcon(settings.value("mainWindow/emptyIcon").toString());
}

QByteArray dmisc::decodedString(dcrypt *crypt,
				const QByteArray &byteArray,
				bool *ok)
{
  if(crypt)
    return crypt->decodedString(byteArray, ok);
  else if(ok)
    *ok = false;

  return byteArray;
}

QByteArray dmisc::decodedString(const QByteArray &byteArray,
				bool *ok)
{
  if(s_crypt)
    return s_crypt->decodedString(byteArray, ok);
  else if(ok)
    *ok = false;

  return byteArray;
}

QByteArray dmisc::encodedString(const QByteArray &byteArray,
				const bool shouldEncode,
				bool *ok)
{
  if(s_crypt && shouldEncode)
    return s_crypt->encodedString(byteArray, ok);
  else if(ok)
    *ok = false;

  return byteArray;
}

QStringList dmisc::hashTypes(void)
{
  return dcrypt::hashTypes();
}

QStringList dmisc::cipherTypes(void)
{
  return dcrypt::cipherTypes();
}

QString dmisc::passphraseHash(const QString &passphrase,
			      const QString &salt,
			      const QString &hashType)
{
  int algorithm = -1;
  QString hash("");
  QString saltedPassphrase("");

  saltedPassphrase.append(passphrase).append(salt);

  if(hashType == "sha512")
    algorithm = GCRY_MD_SHA512;
  else if(hashType == "sha384")
    algorithm = GCRY_MD_SHA384;
  else if(hashType == "sha256")
    algorithm = GCRY_MD_SHA256;
  else if(hashType == "sha224")
    algorithm = GCRY_MD_SHA224;
  else if(hashType == "tiger")
    algorithm = GCRY_MD_TIGER;

  if(algorithm != -1 && isHashTypeSupported(hashType))
    {
      unsigned int length = gcry_md_get_algo_dlen(algorithm);

      if(length > 0)
	{
	  QByteArray byteArray(length, 0);

	  gcry_md_hash_buffer
	    (algorithm,
	     static_cast<void *> (byteArray.data()),
	     static_cast<const void *> (saltedPassphrase.toUtf8().constData()),
	     static_cast<size_t> (saltedPassphrase.toUtf8().length()));
	  hash = byteArray.toHex();
	}
      else
	{
	  logError(QObject::tr("dmisc::passphraseHash(): "
			       "gcry_md_get_algo_dlen() "
			       "returned zero. Using "
			       "Qt's sha1 "
			       "implementation."));
	  hash = QCryptographicHash::hash(saltedPassphrase.toUtf8(),
					  QCryptographicHash::Sha1).
	    toHex();
	}
    }
  else
    {
      logError(QObject::tr("dmisc::passphraseHash(): Unsupported "
			   "hash type %1 (%2). Using "
			   "Qt's sha1 "
			   "implementation.").arg(hashType).
	       arg(algorithm));
      hash = QCryptographicHash::hash
	(saltedPassphrase.toUtf8(), QCryptographicHash::Sha1).toHex();
    }

  return hash;
}

bool dmisc::isHashTypeSupported(const QString &hashType)
{
  int algorithm = gcry_md_map_name(hashType.toLatin1().constData());

  if(algorithm != 0 && gcry_md_test_algo(algorithm) == 0)
    return true;
  else
    return false;
}

int dmisc::levenshteinDistance(const QString &str1,
			       const QString &str2)
{
  if(str1.isEmpty())
    return str2.length();
  else if(str2.isEmpty())
    return str1.length();

  int cost = 0;
  QChar str1_c = 0;
  QChar str2_c = 0;
  QVector<QVector<int> > matrix(str1.length() + 1,
				QVector<int> (str2.length() + 1));

  for(int i = 0; i <= str1.length(); i++)
    matrix[i][0] = i;

  for(int i = 0; i <= str2.length(); i++)
    matrix[0][i] = i;

  for(int i = 1; i <= str1.length(); i++)
    {
      str1_c = str1.at(i - 1);

      for(int j = 1; j <= str2.length(); j++)
	{
	  str2_c = str2.at(j - 1);

	  if(str1_c == str2_c)
	    cost = 0;
	  else
	    cost = 1;

	  matrix[i][j] = qMin(qMin(matrix[i - 1][j] + 1,
				   matrix[i][j - 1] + 1),
			      matrix[i - 1][j - 1] + cost);
	}
    }

  return matrix[str1.length()][str2.length()];
}

bool dmisc::passphraseWasAuthenticated(void)
{
  return s_passphraseWasAuthenticated;
}

QString dmisc::findUniqueFileName(const QString &fileName,
				  const QDir &path)
{
  QFileInfo fileInfo;

  fileInfo.setFile(path, fileName);

  if(fileInfo.exists())
    {
      int count = 1;
      QFileInfo info(fileName);

      if(info.suffix().trimmed().isEmpty())
	fileInfo.setFile(path, QString("%1(%2)").arg(fileName).arg(count));
      else
	fileInfo.setFile
	  (path,
	   QString("%1(%2).%3").arg(info.completeBaseName()).arg(count).
	   arg(info.suffix().trimmed()));

      while(fileInfo.exists())
	{
	  count += 1;

	  if(info.suffix().trimmed().isEmpty())
	    fileInfo.setFile(path, QString("%1(%2)").arg(fileName).arg(count));
	  else
	    fileInfo.setFile
	      (path, QString("%1(%2).%3").arg(info.completeBaseName()).
	       arg(count).
	       arg(info.suffix().trimmed()));
	}
    }

  return fileInfo.absoluteFilePath();
}

void dmisc::reencodeFavicons(QProgressBar *progress)
{
  if(!passphraseWasAuthenticated())
    return;

  if(progress)
    {
      progress->setMaximum(-1);
      progress->setVisible(true);
      progress->update();
    }

  {
    QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE", "favicons");

    db.setDatabaseName(dooble::s_homePath + QDir::separator() +
		       "favicons.db");

    if(db.open())
      {
	if(progress)
	  {
	    progress->setMaximum(-1);
	    progress->update();
	  }

	int temporary = -1;
	QSqlQuery query(db);

	query.setForwardOnly(true);

	if(query.exec("SELECT url, favicon FROM favicons WHERE "
		      "temporary = 0"))
	  while(query.next())
	    {
	      QUrl url
		(QUrl::fromEncoded
		 (decodedString
		  (s_reencodeCrypt, QByteArray::fromBase64
		   (query.value(0).toByteArray())),
		  QUrl::StrictMode));

	      if(isSchemeAcceptedByDooble(url.scheme()))
		{
		  QIcon icon;
		  QBuffer buffer;
		  QByteArray bytes(query.value(1).toByteArray());

		  bytes = decodedString(s_reencodeCrypt, bytes);
		  buffer.setBuffer(&bytes);
		  buffer.open(QIODevice::ReadOnly);

		  QDataStream in(&buffer);

		  in >> icon;
		  buffer.close();

		  bool ok = true;
		  QSqlQuery insertQuery(db);

		  insertQuery.prepare
		    ("INSERT OR REPLACE INTO favicons "
		     "(url, url_hash, favicon, temporary) "
		     "VALUES (?, ?, ?, ?)");
		  insertQuery.bindValue
		    (0,
		     encodedString(url.toEncoded(QUrl::StripTrailingSlash).
				   toBase64(), true, &ok));
		  insertQuery.bindValue
		    (1,
		     hashedString(url.toEncoded(QUrl::StripTrailingSlash)).
		     toBase64());
		  bytes.clear();
		  buffer.setBuffer(&bytes);
		  buffer.open(QIODevice::WriteOnly);

		  QDataStream out(&buffer);

		  out << icon;

		  if(ok)
		    insertQuery.bindValue(2, encodedString(bytes, true, &ok));

		  insertQuery.bindValue(3, temporary);

		  if(ok)
		    insertQuery.exec();

		  buffer.close();
		}
	    }

	query.exec("DELETE FROM favicons WHERE temporary <> -1");
	query.exec("VACUUM");
	query.exec("UPDATE favicons SET temporary = 0");
      }

    db.close();
  }

  QSqlDatabase::removeDatabase("favicons");

  if(progress)
    progress->setVisible(false);
}

void dmisc::clearFavicons(void)
{
  {
    QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE", "favicons");

    db.setDatabaseName(dooble::s_homePath + QDir::separator() +
		       "favicons.db");

    if(db.open())
      {
	QSqlQuery query(db);

	query.exec("PRAGMA synchronous = OFF");

	if(passphraseWasAuthenticated())
	  query.exec("DELETE FROM favicons");
	else
	  query.exec("DELETE FROM favicons WHERE temporary = 1");

	query.exec("VACUUM");
      }

    db.close();
  }

  QSqlDatabase::removeDatabase("favicons");
}

void dmisc::purgeTemporaryData(void)
{
  QStringList list;

  list << "favicons";

  for(int i = 0; i < list.size(); i++)
    {
      {
	QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE", list.at(i));

	db.setDatabaseName(dooble::s_homePath + QDir::separator() +
			   QString("%1.db").arg(list.at(i)));

	if(db.open())
	  {
	    QSqlQuery query(db);

	    query.exec("PRAGMA synchronous = OFF");
	    query.exec(QString("DELETE FROM %1 WHERE temporary = 1").
		       arg(list.at(i)));
	    query.exec("VACUUM");
	  }

	db.close();
      }

      QSqlDatabase::removeDatabase(list.at(i));
    }
}

QRect dmisc::balancedGeometry(const QRect &geometry, QWidget *widget)
{
  /*
  ** Qt may place the window underneath a desktop's panel bar.
  ** The geometry variable contains the desired geometry.
  */

  QRect rect(geometry);
  QRect available(QApplication::desktop()->availableGeometry(widget));

  if(!available.contains(rect))
    {
      rect.setX(rect.x() + available.x());
      rect.setY(rect.y() + available.y());
#if defined(Q_OS_UNIX) && !defined(Q_OS_MAC)
      /*
      ** X11 is quite difficult to predict. Let's add this simple
      ** offset.
      */

      rect.setX(rect.x() + 50);
#endif

      if(rect.width() > available.width())
	rect.setWidth(0.85 * available.width());

      if(rect.height() > available.height())
	rect.setHeight(0.85 * available.height());
    }

  return rect;
}

bool dmisc::isSchemeAcceptedByDooble(const QString &scheme)
{
  QString l_scheme(scheme.toLower().trimmed());

  if(l_scheme == "file" ||
     l_scheme == "ftp" ||
     l_scheme == "http" ||
     l_scheme == "https" ||
     l_scheme == "qrc")
    return true;
  else
    return false;
}

void dmisc::removeRestorationFiles(const QUuid &id)
{
  if(passphraseWasAuthenticated())
    {
      /*
      ** Remove all restoration files or the files that are
      ** associated with the current process.
      */

      QDir dir(dooble::s_homePath + QDir::separator() + "Histories");
      QString idStr("");
      QFileInfoList files(dir.entryInfoList(QDir::Files, QDir::Name));

      if(!id.isNull())
	{
	  idStr = id.toString();
	  idStr.remove("{");
	  idStr.remove("}");
	  idStr.remove("-");
	}

      while(!files.isEmpty())
	{
	  QFileInfo fileInfo(files.takeFirst());

	  if(!idStr.isEmpty())
	    {
	      if(fileInfo.fileName().startsWith(idStr))
		QFile::remove(fileInfo.absoluteFilePath());
	    }
	  else
	    QFile::remove(fileInfo.absoluteFilePath());
	}
    }
}

void dmisc::removeRestorationFiles(const QUuid &pid,
				   const qint64 wid)
{
  if(passphraseWasAuthenticated())
    {
      /*
      ** Remove all restoration files that are associated
      ** with the current process and window.
      */

      QDir dir(dooble::s_homePath + QDir::separator() + "Histories");
      QString idStr(pid.toString().remove("{").remove("}").remove("-") +
		    QString::number(wid).rightJustified(20, '0'));
      QFileInfoList files(dir.entryInfoList(QDir::Files, QDir::Name));

      while(!files.isEmpty())
	{
	  QFileInfo fileInfo(files.takeFirst());

	  if(fileInfo.fileName().startsWith(idStr))
	    QFile::remove(fileInfo.absoluteFilePath());
	}
    }
}

void dmisc::setActionForFileSuffix(const QString &suffix,
				   const QString &action)
{
  if(!suffix.isEmpty())
    {
      if(action.isEmpty())
	dooble::s_applicationsActions[suffix] = "prompt";
      else
	dooble::s_applicationsActions[suffix] = action;
    }
  else
    /*
    ** Damaged suffix.
    */

    return;

  {
    QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE",
						"applications");

    db.setDatabaseName(dooble::s_homePath +
		       QDir::separator() + "applications.db");

    if(db.open())
      {
	QSqlQuery query(db);
	QByteArray bytes;

	query.setForwardOnly(true);
	query.exec("CREATE TABLE IF NOT EXISTS applications ("
		   "file_suffix TEXT PRIMARY KEY NOT NULL, "
		   "action TEXT DEFAULT NULL, "
		   "icon BLOB DEFAULT NULL)");

	if(query.exec(QString("SELECT icon FROM applications WHERE "
			      "file_suffix = '%1'").
		      arg(suffix)))
	  if(query.next())
	    bytes = query.value(0).toByteArray();

	if(bytes.isEmpty())
	  {
	    QBuffer buffer;
	    QFileIconProvider iconProvider;

	    buffer.setBuffer(&bytes);
	    buffer.open(QIODevice::WriteOnly);

	    QDataStream out(&buffer);

	    out << iconProvider.icon(QFileIconProvider::File).pixmap
	      (16, QIcon::Normal, QIcon::On);
	    buffer.close();
	  }

	query.exec("PRAGMA synchronous = OFF");
	query.prepare
	  ("INSERT OR REPLACE INTO applications ("
	   "file_suffix, action, icon) "
	   "VALUES (?, ?, ?)");
	query.bindValue(0, suffix);
	query.bindValue(1, action);
	query.bindValue(2, bytes);
	query.exec();
      }

    db.close();
  }

  QSqlDatabase::removeDatabase("applications");
}

bool dmisc::canDoobleOpenLocalFile(const QUrl &url)
{
  static QStringList list;

  list << "gif"
       << "html"
       << "jpg"
       << "jpeg"
       << "png";

  QFileInfo fileInfo(url.toLocalFile());

  return list.contains(fileInfo.suffix().toLower().trimmed());
}

QString dmisc::fileNameFromUrl(const QUrl &url)
{
  QString path(url.path());
  QString fileName("");

  /*
  ** Attempt to gather the file name from the URL.
  */

  if(!path.isEmpty())
    fileName = QFileInfo(path).fileName();

  if(fileName.isEmpty())
    fileName = "dooble.download";

  return fileName;
}

void dmisc::launchApplication(const QString &program,
			      const QStringList &arguments)
{
#ifdef Q_OS_MAC
  QProcess::startDetached
    ("open", QStringList("-a") << program << "--args" << arguments);
#else
  QProcess::startDetached(program, arguments);
#endif
}

QIcon dmisc::iconForFileSuffix(const QString &suffix)
{
  QPixmap pixmap;

  {
    QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE",
						"applications");

    db.setDatabaseName(dooble::s_homePath +
		       QDir::separator() + "applications.db");

    if(db.open())
      {
	QSqlQuery query(db);

	query.setForwardOnly(true);

	if(query.exec(QString("SELECT icon FROM applications WHERE "
			      "file_suffix = '%1'").arg(suffix)))
	  if(query.next())
	    if(!query.isNull(0))
	      {
		QBuffer buffer;
		QByteArray bytes(query.value(0).toByteArray());

		buffer.setBuffer(&bytes);
		buffer.open(QIODevice::ReadOnly);

		QDataStream in(&buffer);

		in >> pixmap;
		buffer.close();
	      }
      }

    db.close();
  }

  QSqlDatabase::removeDatabase("applications");

  if(pixmap.isNull())
    {
      QFileIconProvider iconProvider;

      return iconProvider.icon(QFileIconProvider::File);
    }
  else
    return QIcon(pixmap);
}

bool dmisc::passphraseWasPrepared(void)
{
  QString str1(""), str2(""), str3("");

  if(dooble::s_settings.contains("settingsWindow/passphraseHash"))
    str1 = dooble::s_settings.value("settingsWindow/passphraseHash", "").
      toString();

  if(dooble::s_settings.contains("settingsWindow/passphraseHashType"))
    str2 = dooble::s_settings.value("settingsWindow/passphraseHashType", "").
      toString();

  if(dooble::s_settings.contains("settingsWindow/passphraseSalt"))
    str3 = dooble::s_settings.value("settingsWindow/passphraseSalt", "").
      toString();

  return !str1.isEmpty() && !str2.isEmpty() && !str3.isEmpty();
}

bool dmisc::isKDE(void)
{
  /*
  ** Trust the shell? Trust the user?
  */

  return QVariant(qgetenv("KDE_FULL_SESSION")).toBool();
}

bool dmisc::isGnome(void)
{
  /*
  ** Trust the shell? Trust the user?
  */

  QString session(qgetenv("DESKTOP_SESSION").toLower().trimmed());

  if(session == "gnome" ||
     session == "ubuntu")
    return true;
  else
    return false;
}

void dmisc::logError(const QString &error)
{
  if(dooble::s_errorLog)
    dooble::s_errorLog->logError(error);
}

QString dmisc::elidedTitleText(const QString &text)
{
  QString l_text(text);

  if(l_text.length() > dooble::MAX_NUMBER_OF_MENU_TITLE_CHARACTERS)
    l_text = l_text.mid
      (0, dooble::MAX_NUMBER_OF_MENU_TITLE_CHARACTERS - 3).trimmed() + "...";

  l_text.replace("&", "&&");
  return l_text;
}

QString dmisc::formattedSize(const qint64 size)
{
  QString str("");

  if(size >= 0)
    {
      if(size == 0)
	str = QObject::tr("0 Bytes");
      else if(size == 1)
	str = QObject::tr("1 Byte");
      else if(size < 1024)
	str = QString(QObject::tr("%1 Bytes")).arg(size);
      else if(size < 1048576)
	str = QString(QObject::tr("%1 KiB")).arg
	  (QString::number(qRound(size / 1024.0)));
      else
	str = QString(QObject::tr("%1 MiB")).arg
	  (QString::number((size * 1.0) / 1048576, 'f', 1));
    }
  else
    str = "0 Bytes";

  return str;
}

QUrl dmisc::correctedUrlPath(const QUrl &url)
{
  QUrl l_url(url);

  if(!l_url.path().isEmpty())
    {
      QString path(QDir::cleanPath(l_url.path()));

      path = path.remove("../");
      path = path.remove("/..");
      path = path.remove("/../");
      l_url.setPath(path);
    }

  return l_url;
}

QByteArray dmisc::hashedString(const QByteArray &byteArray)
{
  QByteArray hash;

  if(!s_crypt || !s_crypt->passphraseHash())
    {
      int saltLength = qMax
	(256, dooble::s_settings.value("settingsWindow/saltLength",
				       256).toInt());

      logError
	(QObject::tr("dmisc::hashedString(): s_crypt or s_crypt->passphraseHash() is "
		     "zero. Using a temporary %1-byte key.").arg(saltLength));
      hash.resize(saltLength);
      gcry_randomize(static_cast<void *> (hash.data()),
		     static_cast<size_t> (hash.length()), GCRY_STRONG_RANDOM);
    }
  else
    hash.append(s_crypt->passphraseHash(), s_crypt->passphraseHashLength());

  int algorithm = -1;
  gcry_error_t err = 0;
  gcry_md_hd_t hd;
  QByteArray hashedArray(byteArray);

  if(s_crypt)
    algorithm = s_crypt->hashAlgorithm();
  else
    {
      algorithm = GCRY_MD_SHA512;
      logError
	(QObject::tr("dmisc::hashedString(): s_crypt is zero. "
		     "Defaulting to sha512."));
    }

  if((err = gcry_md_open(&hd, algorithm,
			 GCRY_MD_FLAG_SECURE |
			 GCRY_MD_FLAG_HMAC)) != 0)
    logError(QObject::tr("dmisc::hashedString(): "
			 "gcry_md_open() failure (%1).").
	     arg(gcry_strerror(err)));
  else
    {
      if((err = gcry_md_setkey(hd,
			       static_cast<const void *> (hash.constData()),
			       static_cast<size_t> (hash.length()))) != 0)
	logError(QObject::tr("dmisc::hashedString(): "
			     "gcry_md_setkey() failure (%1).").
		 arg(gcry_strerror(err)));
      else
	{
	  gcry_md_write
	    (hd,
	     static_cast<const void *> (byteArray.constData()),
	     static_cast<size_t> (byteArray.length()));

	  unsigned char *buffer = gcry_md_read(hd, algorithm);

	  if(buffer)
	    {
	      unsigned int length = gcry_md_get_algo_dlen(algorithm);

	      if(length > 0)
		{
		  hashedArray.resize(length);
		  memcpy(static_cast<void *> (hashedArray.data()),
			 static_cast<const void *> (buffer),
			 static_cast<size_t> (hashedArray.size()));
		}
	      else
		logError(QObject::tr("dmisc::hashedString(): "
				     "gcry_md_get_algo_dlen() "
				     "returned zero."));
	    }
	  else
	    logError(QObject::tr("dmisc::hashedString(): "
				 "gcry_md_read() returned zero."));
	}
    }

  gcry_md_close(hd);
  return hashedArray;
}

qint64 dmisc::faviconsSize(void)
{
  return QFileInfo(dooble::s_homePath + QDir::separator() + "favicons.db").
    size();
}

void dmisc::prepareReencodeCrypt(void)
{
  if(s_reencodeCrypt)
    delete s_reencodeCrypt;

  s_reencodeCrypt = new dcrypt(s_crypt);
}

void dmisc::destroyReencodeCrypt(void)
{
  if(s_reencodeCrypt)
    {
      delete s_reencodeCrypt;
      s_reencodeCrypt = 0;
    }
}
