/****************************************************************************
** Dooble - The Secure Internet Web Browser
**
** Copyright (c) 2012, 2013 Alexis Megas,
** Gunther van Dooble, and the Dooble Team.
** All rights reserved.
**
** License: GPL2 only:
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; version 2 of the License only.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
** or see here: http://www.gnu.org/licenses/gpl.html
**
** For the WebKit library, please see: http://webkit.org.
**
** THE CODE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY
** EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
** IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS APPLICATION, EVEN IF ADVISED
** OF THE POSSIBILITY OF SUCH DAMAGE.
**
** Please report all praise, requests, bugs, and problems to the project
** team and administrators: http://sf.net/projects/dooble.
**
** You can find us listed at our project page. New team members are welcome.
** The name of the authors should not be used to endorse or promote products
** derived from Dooble without specific prior written permission.
** If you use this code for other projects, please let us know.
**
** Web sites:
**   http://sf.net/projects/dooble
**   http://dooble.sf.net
****************************************************************************/

#include <QMutexLocker>
#include <QObject>
#include <QtCore/qmath.h>

#include "dcrypt.h"
#include "dmisc.h"
#include "dooble.h"

dcrypt::dcrypt(const QByteArray &salt,
	       const QString &cipherType,
	       const QString &hashType,
	       const QString &passphrase,
	       const int iterationCount)
{
  m_cipherHandle = 0;
  m_key = 0;
  m_keyLength = 0;
  m_passphraseHash = 0;
  m_passphraseHashLength = 0;
  m_initialized = setCipherAlgorithm(cipherType) && setHashAlgorithm(hashType);
  setIterationCount(iterationCount);
  setSalt(salt);

  if(m_initialized)
    m_initialized = openCipherHandle();

  if(m_initialized)
    m_initialized = setCipherPassphrase(passphrase);
}

dcrypt::dcrypt(dcrypt *other)
{
  if(other)
    {
      bool ok = true;

      m_cipherAlgorithm = other->m_cipherAlgorithm;
      m_cipherHandle = 0;
      m_hashAlgorithm = other->m_hashAlgorithm;
      m_initialized = false;
      m_iterationCount = other->m_iterationCount;
      m_key = static_cast<char *>
	(gcry_calloc_secure(other->m_keyLength, sizeof(char)));
      m_keyLength = 0;

      if(m_key)
	{
	  m_keyLength = other->m_keyLength;
	  memcpy(static_cast<void *> (m_key),
		 static_cast<const void *> (other->m_key),
		 m_keyLength);
	}
      else
	{
	  ok = false;
	  dmisc::logError
	    (QObject::tr("dcrypt::dcrypt(): "
			 "gcry_calloc_secure() failure."));
	}

      m_passphraseHash = 0;
      m_passphraseHashLength = 0;

      if(ok)
	{
	  m_passphraseHash = static_cast<char *>
	    (gcry_calloc_secure(other->m_passphraseHashLength, sizeof(char)));

	  if(m_passphraseHash)
	    {
	      m_passphraseHashLength = other->m_passphraseHashLength;
	      memcpy(static_cast<void *> (m_passphraseHash),
		     static_cast<const void *> (other->m_passphraseHash),
		     m_passphraseHashLength);
	    }
	  else
	    {
	      ok = false;
	      dmisc::logError
		(QObject::tr("dcrypt::dcrypt(): "
			     "gcry_calloc_secure() failure."));
	    }
	}

      m_salt = other->m_salt;

      if(ok)
	{
	  m_initialized = openCipherHandle();

	  if(gcry_cipher_setkey(m_cipherHandle,
				static_cast<const void *> (m_key),
				m_keyLength) != 0)
	    {
	      m_initialized = false;
	      dmisc::logError
		(QObject::tr("dcrypt::dcrypt(): "
			     "gcry_cipher_setkey() failure."));
	    }
	}
    }
  else
    {
      m_cipherAlgorithm = 0;
      m_cipherHandle = 0;
      m_hashAlgorithm = 0;
      m_iterationCount = 0;
      m_key = 0;
      m_keyLength = 0;
      m_initialized = false;
      m_passphraseHash = 0;
      m_passphraseHashLength = 0;
      m_salt.clear();
    }
}

dcrypt::~dcrypt()
{
  gcry_cipher_close(m_cipherHandle);
  gcry_free(m_key);
  gcry_free(m_passphraseHash);
}

QByteArray dcrypt::decodedString(const QByteArray &byteArray,
				 bool *ok)
{
  QByteArray encodedArray(byteArray);
  QMutexLocker locker(&m_cipherMutex);

  if(!setInitializationVector(encodedArray))
    {
      if(ok)
	*ok = false;

      dmisc::logError(QObject::tr("dcrypt::decodedString(): "
				  "setInitializationVector() failure."));
      return byteArray;
    }
  else
    {
      size_t blockLength = gcry_cipher_get_algo_blklen(m_cipherAlgorithm);

      if(blockLength == 0)
	{
	  if(ok)
	    *ok = false;

	  dmisc::logError
	    (QObject::tr("dcrypt::decodedString(): "
			 "gcry_cipher_get_algo_blklen() returned "
			 "zero."));
	  return byteArray;
	}

      QByteArray decodedArray(encodedArray);

      /*
      ** Block ciphers require the length of the buffers
      ** to be multiples of the cipher's block size.
      */

      if(decodedArray.isEmpty())
	decodedArray = decodedArray.leftJustified(blockLength, 0);
      else
	decodedArray = decodedArray.leftJustified
	  (blockLength * qCeil((qreal) decodedArray.length() / (qreal)
			       blockLength), 0);

      gcry_error_t err = 0;

      if((err = gcry_cipher_decrypt(m_cipherHandle,
				    static_cast<void *> (decodedArray.data()),
				    static_cast<size_t> (decodedArray.
							 length()),
				    static_cast<const void *> (0),
				    static_cast<size_t> (0))) == 0)
	{
	  int s = 0;
	  QByteArray originalLength
	    (decodedArray.mid(decodedArray.length() - 4, 4));
	  QDataStream in(&originalLength, QIODevice::ReadOnly);

	  in >> s;

	  if(s >= 0 && s <= decodedArray.length())
	    {
	      if(ok)
		*ok = true;

	      return decodedArray.mid(0, s);
	    }
	  else
	    {
	      if(ok)
		*ok = false;

	      return decodedArray;
	    }
	}
      else
	{
	  if(ok)
	    *ok = false;

	  dmisc::logError(QObject::tr("dcrypt::decodedString(): "
				      "gcry_cipher_decrypt() failure (%1).").
			  arg(gcry_strerror(err)));
	  return byteArray;
	}
    }

  if(ok)
    *ok = false;

  return byteArray;
}

QByteArray dcrypt::encodedString(const QByteArray &byteArray,
				 bool *ok)
{
  QByteArray iv;
  QMutexLocker locker(&m_cipherMutex);

  if(!setInitializationVector(iv))
    {
      if(ok)
	*ok = false;

      dmisc::logError(QObject::tr("dcrypt::encodedString(): "
				  "setInitializationVector() failure."));
      return byteArray;
    }
  else
    {
      size_t blockLength = gcry_cipher_get_algo_blklen(m_cipherAlgorithm);

      if(blockLength == 0)
	{
	  if(ok)
	    *ok = false;

	  dmisc::logError(QObject::tr("dcrypt::encodedString(): "
				      "gcry_cipher_get_algo_blklen() returned "
				      "zero."));
	  return byteArray;
	}

      QByteArray encodedArray(byteArray);

      /*
      ** Block ciphers require the length of the buffers
      ** to be multiples of the cipher's block size.
      */

      if(encodedArray.isEmpty())
	encodedArray = encodedArray.leftJustified(blockLength, 0);
      else
	encodedArray = encodedArray.leftJustified
	  (blockLength * qCeil((qreal) encodedArray.length() / (qreal)
			       blockLength), 0);

      encodedArray.append(QByteArray(blockLength, 0));

      QByteArray originalLength;
      QDataStream out(&originalLength, QIODevice::WriteOnly);

      out << byteArray.length();
      encodedArray.remove(encodedArray.length() - 4, 4);
      encodedArray.append(originalLength);

      gcry_error_t err = 0;

      if((err = gcry_cipher_encrypt(m_cipherHandle,
				    static_cast<void *> (encodedArray.data()),
				    static_cast<size_t> (encodedArray.
							 length()),
				    static_cast<const void *> (0),
				    static_cast<size_t> (0))) == 0)
	{
	  if(ok)
	    *ok = true;

	  return iv + encodedArray;
	}
      else
	{
	  if(ok)
	    *ok = false;

	  dmisc::logError(QObject::tr("dcrypt::encodedString(): "
				      "gcry_cipher_encrypt() failure (%1).").
			  arg(gcry_strerror(err)));
	  return byteArray;
	}
    }

  if(ok)
    *ok = false;

  return byteArray;
}

QByteArray dcrypt::salt(void) const
{
  return m_salt;
}

QStringList dcrypt::cipherTypes(void)
{
  QStringList types;

  /*
  ** Block ciphers only!
  */

  types << "aes256"
	<< "camellia256"
	<< "serpent256"
	<< "twofish"
	<< "aes192"
	<< "aes128";

  for(int i = types.size() - 1; i >= 0; i--)
    {
      int algorithm = gcry_cipher_map_name(types.at(i).toLatin1().
					   constData());

      if(!(algorithm != 0 && gcry_cipher_test_algo(algorithm) == 0))
	types.removeAt(i);
    }

  return types;
}

QStringList dcrypt::hashTypes(void)
{
  QStringList types;

  types << "sha512"
	<< "sha384"
	<< "sha256"
	<< "sha224"
	<< "tiger";

  for(int i = types.size() - 1; i >= 0; i--)
    {
      int algorithm = gcry_md_map_name(types.at(i).toLatin1().constData());

      if(!(algorithm != 0 && gcry_md_test_algo(algorithm) == 0))
	types.removeAt(i);
    }

  return types;
}

bool dcrypt::initialized(void) const
{
  return m_initialized;
}

bool dcrypt::openCipherHandle(void)
{
  gcry_cipher_close(m_cipherHandle);

  gcry_error_t err = 0;

  if((err = gcry_cipher_open(&m_cipherHandle, m_cipherAlgorithm,
			     GCRY_CIPHER_MODE_CBC,
			     GCRY_CIPHER_SECURE | GCRY_CIPHER_CBC_CTS)) != 0)
    dmisc::logError(QObject::tr("dcrypt::openCipherHandle(): "
				"gcry_cipher_open() failure (%1).").
		    arg(gcry_strerror(err)));

  return err == 0;
}

bool dcrypt::setCipherPassphrase(const QString &passphrase)
{
  char *l_passphrase = 0;
  size_t passphraseLength = 0;

  if(passphrase.isEmpty())
    {
      passphraseLength = 256;
      l_passphrase = static_cast<char *>
	(gcry_calloc_secure(passphraseLength, sizeof(char)));
    }
  else
    {
      passphraseLength = passphrase.toLatin1().length();
      l_passphrase = static_cast<char *>
	(gcry_calloc_secure(passphraseLength, sizeof(char)));
    }

  if(!l_passphrase)
    {
      dmisc::logError
	(QObject::tr("dcrypt::setCipherPassphrase(): "
		     "gcry_calloc_secure() returned zero."));
      return false;
    }

  if(passphrase.isEmpty())
    gcry_randomize(static_cast<void *> (l_passphrase), passphraseLength,
		   GCRY_STRONG_RANDOM);
  else
    memcpy(static_cast<void *> (l_passphrase),
	   static_cast<const void *> (passphrase.toLatin1().constData()),
	   static_cast<size_t> (passphrase.toLatin1().length()));

  gcry_error_t err = 0;

#if DOOBLE_MINIMUM_GCRYPT_VERSION >= 0x010500
  if(m_key)
    {
      gcry_free(m_key);
      m_key = 0;
    }

  m_keyLength = gcry_cipher_get_algo_keylen(m_cipherAlgorithm);

  if(m_keyLength == 0)
    {
      err = GPG_ERR_INV_KEYLEN;
      dmisc::logError
	(QObject::tr("dcrypt::setCipherPassphrase(): "
		     "gcry_cipher_get_algo_keylen() "
		     "returned zero."));
      goto error_label;
    }
  else if((m_key = static_cast<char *> (gcry_calloc_secure(m_keyLength,
							   sizeof(char)))))
    {
      if((err = gcry_kdf_derive(static_cast<const void *> (l_passphrase),
				passphraseLength,
				GCRY_KDF_PBKDF2,
				m_hashAlgorithm,
				static_cast<const void *> (m_salt.constData()),
				static_cast<size_t> (m_salt.length()),
				m_iterationCount,
				m_keyLength,
				static_cast<void *> (m_key))) == 0)
	{
	  unsigned int length = gcry_md_get_algo_dlen(m_hashAlgorithm);

	  if(length > 0)
	    {
	      if(m_passphraseHash)
		gcry_free(m_passphraseHash);

	      QByteArray byteArray(length, 0);

	      gcry_md_hash_buffer
		(m_hashAlgorithm,
		 static_cast<void *> (byteArray.data()),
		 static_cast<const void *> (m_key),
		 m_keyLength);
	      m_passphraseHash = static_cast<char *>
		(gcry_calloc_secure(static_cast<size_t> (byteArray.
							 toHex().length()),
				    sizeof(char)));

	      if(m_passphraseHash)
		{
		  m_passphraseHashLength = byteArray.toHex().length();
		  memcpy
		    (static_cast<void *> (m_passphraseHash),
		     static_cast<const void *> (byteArray.toHex().constData()),
		     static_cast<size_t> (byteArray.toHex().length()));
		}
	      else
		{
		  err = GPG_ERR_ENOMEM;
		  dmisc::logError
		    (QObject::tr("dcrypt::setCipherPassphrase(): "
				 "m_passphraseHash is "
				 "zero."));
		  goto error_label;
		}
	    }
	  else
	    {
	      err = GPG_ERR_INV_LENGTH;
	      dmisc::logError
		(QObject::tr("dcrypt::setCipherPassphrase(): "
			     "gcry_md_get_algo_dlen() "
			     "returned zero."));
	      goto error_label;
	    }

	  if((err = gcry_cipher_setkey(m_cipherHandle,
				       static_cast<const void *> (m_key),
				       m_keyLength)) != 0)
	    {
	      dmisc::logError
		(QObject::tr("dcrypt::setCipherPassphrase(): "
			     "gcry_cipher_setkey() failure (%1).").
		 arg(gcry_strerror(err)));
	      goto error_label;
	    }
	}
      else
	{
	  dmisc::logError
	    (QObject::tr("dcrypt::setCipherPassphrase(): "
			 "gcry_kdf_derive() failure (%1).").
	     arg(gcry_strerror(err)));
	  goto error_label;
	}
    }
  else
    {
      err = GPG_ERR_ENOMEM;
      m_keyLength = 0;
      dmisc::logError
	(QObject::tr("dcrypt::setCipherPassphrase(): "
		     "gcry_calloc_secure() "
		     "returned zero."));
      goto error_label;
    }
#else
  dmisc::logError
    (QObject::tr("dcrypt::setCipherPassphrase(): "
		 "gcry_kdf_derive() is not defined. "
		 "Using the provided passphrase's hash as the key."));

  if(m_key)
    {
      gcry_free(m_key);
      m_key = 0;
    }

  m_keyLength = gcry_cipher_get_algo_keylen(m_cipherAlgorithm);

  if(m_keyLength == 0)
    {
      err = GPG_ERR_INV_KEYLEN;
      dmisc::logError
	(QObject::tr("dcrypt::setCipherPassphrase(): "
		     "gcry_cipher_get_algo_keylen() "
		     "returned zero."));
      goto error_label;
    }
  else if((m_key = static_cast<char *> (gcry_calloc_secure(m_keyLength,
							   sizeof(char)))))
    {
      /*
      ** Retain the passphrase's hash. We'll use the hash as a key.
      */

      unsigned int length = gcry_md_get_algo_dlen(m_hashAlgorithm);

      if(length > 0)
	{
	  if(m_passphraseHash)
	    gcry_free(m_passphraseHash);

	  QByteArray byteArray(length, 0);

	  gcry_md_hash_buffer(m_hashAlgorithm,
			      static_cast<void *> (byteArray.data()),
			      static_cast<const void *> (l_passphrase),
			      passphraseLength);
	  m_passphraseHash = static_cast<char *>
	    (gcry_calloc_secure(static_cast<size_t> (byteArray.
						     toHex().length()),
				sizeof(char)));

	  if(m_passphraseHash)
	    {
	      m_passphraseHashLength = byteArray.toHex().length();
	      memcpy(static_cast<void *> (m_passphraseHash),
		     static_cast<const void *> (byteArray.toHex().constData()),
		     static_cast<size_t> (byteArray.toHex().length()));
	      memcpy(static_cast<void *> (m_key),
		     static_cast<const void *> (m_passphraseHash),
		     qMin(m_keyLength, m_passphraseHashLength));

	      if((err = gcry_cipher_setkey(m_cipherHandle,
					   static_cast<const void *> (m_key),
					   m_keyLength)) != 0)
		{
		  dmisc::logError
		    (QObject::tr("dcrypt::setCipherPassphrase(): "
				 "gcry_cipher_setkey() failure (%1).").
		     arg(gcry_strerror(err)));
		  goto error_label;
		}
	    }
	  else
	    {
	      err = GPG_ERR_ENOMEM;
	      dmisc::logError
		(QObject::tr("dcrypt::setCipherPassphrase(): "
			     "m_passphraseHash is "
			     "zero."));
	      goto error_label;
	    }
	}
      else
	{
	  err = GPG_ERR_INV_LENGTH;
	  dmisc::logError
	    (QObject::tr("dcrypt::setCipherPassphrase(): "
			 "gcry_md_get_algo_dlen() returned "
			 "zero."));
	  goto error_label;
	}
    }
  else
    {
      err = GPG_ERR_ENOMEM;
      m_keyLength = 0;
      dmisc::logError
	(QObject::tr("dcrypt::setCipherPassphrase(): "
		     "gcry_calloc_secure() "
		     "returned zero."));
      goto error_label;
    }
#endif

 error_label:
  gcry_free(l_passphrase);
  return err == 0;
}

bool dcrypt::setCipherAlgorithm(const QString &cipherType)
{
  m_cipherAlgorithm = gcry_cipher_map_name(cipherType.toLatin1().constData());

  if(m_cipherAlgorithm == 0)
    m_cipherAlgorithm = gcry_cipher_map_name("aes256");

  return gcry_cipher_test_algo(m_cipherAlgorithm) == 0;
}

bool dcrypt::setHashAlgorithm(const QString &hashType)
{
  m_hashAlgorithm = gcry_md_map_name(hashType.toLatin1().constData());

  if(m_hashAlgorithm == 0)
    m_hashAlgorithm = gcry_md_map_name("sha512");

  return gcry_md_test_algo(m_hashAlgorithm) == 0;
}

bool dcrypt::setInitializationVector(QByteArray &byteArray)
{
  size_t ivLength = 0;
  gcry_error_t err = 0;

  if((ivLength = gcry_cipher_get_algo_blklen(m_cipherAlgorithm)) == 0)
    {
      err = GPG_ERR_INV_LENGTH;
      dmisc::logError
	(QObject::tr("dcrypt::setInitializationVector(): "
		     "gcry_cipher_get_algo_blklen() "
		     "returned zero."));
    }
  else
    {
      char *iv = static_cast<char *> (gcry_calloc(ivLength, sizeof(char)));

      if(iv)
	{
	  if(byteArray.isEmpty())
	    {
	      gcry_create_nonce(static_cast<void *> (iv), ivLength);
	      byteArray.append(iv, ivLength);
	    }
	  else
	    {
	      memcpy
		(static_cast<void *> (iv),
		 static_cast<const void *> (byteArray.constData()),
		 qMin(ivLength, static_cast<size_t> (byteArray.length())));
	      byteArray.remove(0, ivLength);
	    }

	  gcry_cipher_reset(m_cipherHandle);

	  if((err = gcry_cipher_setiv(m_cipherHandle,
				      static_cast<const void *> (iv),
				      ivLength)) != 0)
	    dmisc::logError
	      (QObject::tr("dcrypt::setInitializationVector(): "
			   "gcry_cipher_setiv() failure (%1).").
	       arg(gcry_strerror(err)));

	  gcry_free(iv);
	}
      else
	{
	  err = GPG_ERR_ENOMEM;
	  dmisc::logError
	    (QObject::tr("dcrypt::setInitializationVector(): "
			 "gcry_calloc() "
			 "returned zero."));
	}
    }

  return err == 0;
}

char *dcrypt::passphraseHash(void) const
{
  /*
  ** Dooble does not store the passphrase's hash in the Dooble.ini file,
  ** it stores a salted hash (hash(passphrase + salt)).
  ** m_passphraseHash is a hash of either a derived key or a hash of
  ** the passphrase.
  */

  return m_passphraseHash;
}

int dcrypt::hashAlgorithm(void) const
{
  return m_hashAlgorithm;
}

unsigned long dcrypt::iterationCount(void) const
{
  return m_iterationCount;
}

size_t dcrypt::passphraseHashLength(void) const
{
  return m_passphraseHashLength;
}

void dcrypt::setIterationCount(const int iterationCount)
{
  m_iterationCount = qMax(1000, iterationCount);
}

void dcrypt::setSalt(const QByteArray &salt)
{
  if(salt.isEmpty())
    {
      int saltLength = qMax
	(256, dooble::s_settings.value("settingsWindow/saltLength",
				       256).toInt());
      QByteArray l_salt;

      l_salt.resize(saltLength);
      gcry_randomize(static_cast<void *> (l_salt.data()),
		     static_cast<size_t> (l_salt.length()),
		     GCRY_STRONG_RANDOM);
      m_salt = l_salt;
    }
  else
    m_salt = salt;
}

char *dcrypt::key(void) const
{
  return m_key;
}

size_t dcrypt::keyLength(void) const
{
  return m_keyLength;
}
