/****************************************************************************
** Dooble - The Secure Internet Web Browser
**
** Copyright (c) 2012 Alexis Megas,
** Gunther van Dooble, and the Dooble Team.
** All rights reserved.
**
** License: GPL2 only:
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; version 2 of the License only.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
** or see here: http://www.gnu.org/licenses/gpl.html
**
** For the WebKit library, please see: http://webkit.org.
**
** THE CODE IS PROVIDED BY THE AUTHORS ''AS IS'' AND ANY
** EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
** ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
** GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
** INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
** IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
** ARISING IN ANY WAY OUT OF THE USE OF THIS APPLICATION, EVEN IF ADVISED
** OF THE POSSIBILITY OF SUCH DAMAGE.
**
** Please report all praise, requests, bugs, and problems to the project
** team and administrators: http://sf.net/projects/dooble.
**
** You can find us listed at our project page. New team members are welcome.
** The name of the authors should not be used to endorse or promote products
** derived from Dooble without specific prior written permission.
** If you use this code for other projects, please let us know.
**
** Web sites:
**   http://sf.net/projects/dooble
**   http://dooble.sf.net
****************************************************************************/

#include <QtCore>
#include <QBuffer>
#if QT_VERSION >= 0x050000
#include <QtConcurrent>
#endif

#include "dooble.h"
#include "dnetworkcache.h"

dnetworkcache::dnetworkcache(void)
{
  m_cacheSize = 0;
  QDir().mkpath(dooble::s_homePath + QDir::separator() + "Cache");
  m_dir.setPath(dooble::s_homePath + QDir::separator() + "Cache");
  m_timer.setInterval(10000);
  connect(&m_timer,
	  SIGNAL(timeout(void)),
	  this,
	  SLOT(slotTimeout(void)));
  m_timer.start();
}

dnetworkcache::~dnetworkcache()
{
  m_timer.stop();
  clearTemp();
}

void dnetworkcache::populate(void)
{
  if(dmisc::passphraseWasAuthenticated())
    clearTemp();
}

qint64 dnetworkcache::cacheSize(void) const
{
  return m_cacheSize;
}

QIODevice *dnetworkcache::data(const QUrl &url)
{
  if(!dooble::s_settings.value("settingsWindow/diskCacheEnabled",
			       true).toBool())
    return 0;
  else if(url.isEmpty() || !url.isValid())
    return 0;
  else if(dooble::s_cacheExceptionsWindow->allowed(url.host()))
    return 0;

  QFile file;
  QString hash(dmisc::hashedString(url.toEncoded(QUrl::StripTrailingSlash)).
	       toHex());
  QString path(hash.mid(0, 2));
  QByteArray bytes; 

  if(path.isEmpty())
    path = "xy";

  file.setFileName
    (m_dir.absolutePath() + QDir::separator() + path + QDir::separator() +
     hash + "_data_" + QString::number(!dmisc::passphraseWasAuthenticated()));

  if(file.open(QIODevice::ReadOnly))
    if(!(bytes = file.readAll()).isEmpty())
      bytes = dmisc::decodedString(bytes);

  file.close();

  if(bytes.isEmpty())
    return 0;

  QBuffer *buffer = new QBuffer();

  buffer->setData(bytes);
  buffer->open(QIODevice::ReadOnly);
  return buffer;
}

void dnetworkcache::insert(QIODevice *device)
{
  if(!device)
    return;
  else if(!dooble::s_settings.value("settingsWindow/diskCacheEnabled",
				    true).toBool())
    {
      device->deleteLater();
      return;
    }

  QBuffer *buffer = qobject_cast<QBuffer *> (device);

  if(!buffer)
    {
      device->deleteLater();
      return;
    }

  QUrl url(buffer->property("dooble-url").toUrl());

  if(url.isEmpty() || !url.isValid())
    {
      buffer->deleteLater();
      return;
    }
  else if(dooble::s_cacheExceptionsWindow->allowed(url.host()))
    {
      buffer->deleteLater();
      return;
    }

  insertEntry(buffer->data(), url);
  buffer->deleteLater();
}

QNetworkCacheMetaData dnetworkcache::metaData(const QUrl &url)
{
  QNetworkCacheMetaData metaData;

  if(!dooble::s_settings.value("settingsWindow/diskCacheEnabled",
			       true).toBool())
    return metaData;
  else if(url.isEmpty() || !url.isValid())
    return metaData;
  else if(dooble::s_cacheExceptionsWindow->allowed(url.host()))
    return metaData;

  QFile file;
  QString hash(dmisc::hashedString(url.toEncoded(QUrl::StripTrailingSlash)).
	       toHex());
  QString path(hash.mid(0, 2));

  if(path.isEmpty())
    path = "xy";

  file.setFileName
    (m_dir.absolutePath() + QDir::separator() + path + QDir::separator() +
     hash + "_metadata_" +
     QString::number(!dmisc::passphraseWasAuthenticated()));

  if(file.open(QIODevice::ReadOnly))
    {
      QByteArray bytes;

      if(!(bytes = file.readAll()).isEmpty())
	{
	  bytes = dmisc::decodedString(bytes);

	  QDataStream in(&bytes, QIODevice::ReadOnly);

	  in >> metaData;
	}
    }

  file.close();
  return metaData;
}

QIODevice *dnetworkcache::prepare(const QNetworkCacheMetaData &metaData)
{
  if(!dooble::s_settings.value("settingsWindow/diskCacheEnabled",
			       true).toBool())
    return 0;
  else if(!metaData.isValid() || metaData.url().isEmpty() ||
	  !metaData.url().isValid() || !metaData.saveToDisk())
    return 0;
  else if(dooble::s_cacheExceptionsWindow->allowed(metaData.url().host()))
    return 0;

  int cacheSize = 1048576 * dooble::s_settings.value
    ("settingsWindow/webDiskCacheSize", 50).toInt();
  qint64 size = 0;

  foreach(QNetworkCacheMetaData::RawHeader header, metaData.rawHeaders())
    if(header.first.toLower() == "content-length")
      {
	size = header.second.toInt();

	/*
	** Be careful not to cache large objects.
	*/

	if(size > (cacheSize * 3) / 4)
	  return 0;
      }
    else if(header.first.toLower() == "content-type" &&
	    header.second.toLower().contains("application"))
      return 0;
    else if(header.first.toLower() == "content-type" &&
	    header.second.toLower().contains("javascript"))
      return 0;

  prepareEntry(metaData);

  QBuffer *buffer = new QBuffer(this);

  buffer->setProperty("dooble-url", metaData.url());
  buffer->open(QIODevice::ReadWrite);
  return buffer;
}

bool dnetworkcache::remove(const QUrl &url)
{
  if(!dooble::s_settings.value("settingsWindow/diskCacheEnabled",
			       true).toBool())
    return false;
  else if(url.isEmpty() || !url.isValid())
    return false;

  short bit = 1;
  QString hash(dmisc::hashedString(url.toEncoded(QUrl::StripTrailingSlash)).
	       toHex());
  QString path(hash.mid(0, 2));

  if(path.isEmpty())
    path = "xy";

  bit &= QFile::remove
    (m_dir.absolutePath() + QDir::separator() + path + QDir::separator() +
     hash + "_data_" + QString::number(!dmisc::passphraseWasAuthenticated()));
  bit &= QFile::remove
    (m_dir.absolutePath() + QDir::separator() + path + QDir::separator() +
     hash + "_metadata_" +
     QString::number(!dmisc::passphraseWasAuthenticated()));

  QDir dir(m_dir);

  dir.rmdir(path);
  return bit;
}

void dnetworkcache::updateMetaData(const QNetworkCacheMetaData &metaData)
{
  if(!dooble::s_settings.value("settingsWindow/diskCacheEnabled",
			       true).toBool())
    return;

  QUrl url(metaData.url());

  if(url.isEmpty() || !url.isValid())
    return;
  else if(dooble::s_cacheExceptionsWindow->allowed(url.host()))
    return;

  prepareEntry(metaData);
}

void dnetworkcache::clear(void)
{
  QStringList list(m_dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot));

  while(!list.isEmpty())
    {
      QDir dir(m_dir);

      dir.cd(list.takeFirst());

      QStringList files(dir.entryList(QDir::Files));

      while(!files.isEmpty())
	dir.remove(files.takeFirst());

      m_dir.rmdir(dir.absolutePath());
    }
}

void dnetworkcache::clearTemp(void)
{
  QStringList list(m_dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot));

  while(!list.isEmpty())
    {
      QDir dir(m_dir);

      dir.cd(list.takeFirst());

      QStringList files(dir.entryList(QStringList("*_1"), QDir::Files));

      while(!files.isEmpty())
	dir.remove(files.takeFirst());

      m_dir.rmdir(dir.absolutePath());
    }
}

void dnetworkcache::reencode(QProgressBar *progress)
{
  if(!dmisc::passphraseWasAuthenticated())
    return;

  if(progress)
    {
      progress->setMaximum(m_cacheList.size());
      progress->setVisible(true);
      progress->update();
    }

  m_timer.stop();
  m_future.waitForFinished();
  clear();

  for(int i = 0; i < m_cacheList.size(); i++)
    {
      QPair<QByteArray, QNetworkCacheMetaData> pair
	(m_cacheList.at(i));

      prepareEntry(pair.second);
      insertEntry(pair.first, pair.second.url());

      if(progress)
	progress->setValue(i + 1);
    }

  m_cacheList.clear();

  if(progress)
    progress->setVisible(false);

  m_timer.start();
}

void dnetworkcache::slotTimeout(void)
{
  if(m_future.isFinished())
    {
      int diskCacheSize = 1048576 * dooble::s_settings.value
	("settingsWindow/webDiskCacheSize", 50).toInt();

      m_future = QtConcurrent::run
	(this, &dnetworkcache::computeCacheSizeAndCleanDirectory,
	 diskCacheSize, m_dir.absolutePath());
    }
}

void dnetworkcache::prepareCacheList(void)
{
  m_cacheList.clear();
  clearTemp();

  QStringList list(m_dir.entryList(QDir::Dirs | QDir::NoDotAndDotDot));

  while(!list.isEmpty())
    {
      QDir dir(m_dir);

      dir.cd(list.takeFirst());

      QStringList files(dir.entryList(QStringList("*_data_0"), QDir::Files));

      while(!files.isEmpty())
	{
	  QFile file;
	  QString fileName(files.takeFirst());

	  file.setFileName
	    (dir.absolutePath() + QDir::separator() + fileName);

	  QByteArray data;

	  if(file.open(QIODevice::ReadOnly))
	    if(!(data = file.readAll()).isEmpty())
	      data = dmisc::decodedString(data);

	  file.close();

	  QNetworkCacheMetaData metaData;

	  file.setFileName
	    (dir.absolutePath() + QDir::separator() +
	     fileName.replace("_data", "_metadata"));

	  if(file.open(QIODevice::ReadOnly))
	    {
	      QByteArray bytes;

	      if(!(bytes = file.readAll()).isEmpty())
		{
		  bytes = dmisc::decodedString(bytes);

		  QDataStream in(&bytes, QIODevice::ReadOnly);

		  in >> metaData;
		}
	    }

	  file.close();

	  if(!data.isEmpty() && metaData.isValid())
	    {
	      QPair<QByteArray, QNetworkCacheMetaData> pair(data, metaData);

	      m_cacheList.append(pair);
	    }
	}
    }
}

void dnetworkcache::insertEntry(const QByteArray &data,
				const QUrl &url)
{
  QFile file;
  QString hash(dmisc::hashedString(url.toEncoded(QUrl::StripTrailingSlash)).
	       toHex());
  QString path(hash.mid(0, 2));

  if(path.isEmpty())
    path = "xy";

  m_dir.mkpath(path);
  file.setFileName
    (m_dir.path() + QDir::separator() + path + QDir::separator() +
     hash + "_data_" + QString::number(!dmisc::passphraseWasAuthenticated()));

  if(file.open(QIODevice::Truncate | QIODevice::WriteOnly))
    file.write(dmisc::encodedString(data, true));

  file.flush();
  file.close();
}

void dnetworkcache::prepareEntry(const QNetworkCacheMetaData &metaData)
{
  QFile file;
  QString hash(dmisc::hashedString(metaData.url().
				   toEncoded(QUrl::StripTrailingSlash)).
	       toHex());
  QString path(hash.mid(0, 2));

  if(path.isEmpty())
    path = "xy";

  m_dir.mkpath(path);
  file.setFileName
    (m_dir.path() + QDir::separator() + path + QDir::separator() +
     hash + "_metadata_" +
     QString::number(!dmisc::passphraseWasAuthenticated()));

  if(file.open(QIODevice::Truncate | QIODevice::WriteOnly))
    {
      QByteArray bytes;
      QDataStream out(&bytes, QIODevice::WriteOnly);

      out << metaData;
      file.write(dmisc::encodedString(bytes, true));
    }

  file.flush();
  file.close();
}

void dnetworkcache::computeCacheSizeAndCleanDirectory
(const int cacheSizeDesired, const QString &path)
{
  /*
  ** Please note that this method modifies the contents of the
  ** m_cacheSize variable.
  */

  QDir dir1(path);
  qint64 size = 0;

  {
    QFileInfoList list(dir1.entryInfoList(QDir::Dirs | QDir::NoDotAndDotDot));

    while(!list.isEmpty())
      {
	QDir dir2(dir1);

	dir2.cd(list.takeFirst().fileName());

	QFileInfoList files(dir2.entryInfoList(QDir::Files));

	while(!files.isEmpty())
	  size += files.takeFirst().size();
      }
  }

  double p = static_cast<double> (size) /
    static_cast<double> (qMax(1, cacheSizeDesired));

  if(size <= 0 || p <= 0.80)
    {
      m_cacheSize = size;
      return;
    }

  dir1.setPath(path);

  {
    QStringList list(dir1.entryList(QDir::Dirs | QDir::NoDotAndDotDot));

    while(!list.isEmpty())
      {
	QDir dir2(dir1);

	dir2.cd(list.takeFirst());

	QFileInfoList files
	  (dir2.entryInfoList(QDir::Files, QDir::Reversed | QDir::Time));

	while(!files.isEmpty())
	  {
	    QFileInfo fileInfo(files.takeFirst());

	    size -= fileInfo.size();
	    dir2.remove(fileInfo.fileName());
	    fileInfo = QFileInfo
	      (fileInfo.fileName().replace("_data", "_metadata"));
	    size -= fileInfo.size();
	    dir2.remove(fileInfo.fileName());
	    dir1.rmdir(dir2.absolutePath());
	    p = static_cast<double> (size) /
	      static_cast<double> (qMax(1, cacheSizeDesired));

	    if(size <= 0 || p <= 0.80)
	      {
		m_cacheSize = size;
		return;
	      }
	  }
      }
  }

  m_cacheSize = size;
}
