"""
@todo Add an agenda view to the task list
	Tree of days, with each successive 7 days dropping the visibility of further lower priority items
@todo Add a map view
	Using new api widgets people are developing)
	Integrate GPS w/ fallback to default location
	Use locations for mapping
@todo Add a quick search (OR within a property type, and between property types) view
	Drop down for multi selecting priority
	Drop down for multi selecting tags
	Drop down for multi selecting locations
	Calendar selector for choosing due date range
@todo Remove blocking operations from UI thread
"""

import webbrowser
import datetime
import urlparse
import base64

import gobject
import gtk

import coroutines
import toolbox
import gtk_toolbox
import rtm_backend
import rtm_api


def abbreviate(text, expectedLen):
	singleLine = " ".join(text.split("\n"))
	lineLen = len(singleLine)
	if lineLen <= expectedLen:
		return singleLine

	abbrev = "..."

	leftLen = expectedLen // 2 - 1
	rightLen = max(expectedLen - leftLen - len(abbrev) + 1, 1)

	abbrevText =  singleLine[0:leftLen] + abbrev + singleLine[-rightLen:-1]
	assert len(abbrevText) <= expectedLen, "Too long: '%s'" % abbrevText
	return abbrevText


def abbreviate_url(url, domainLength, pathLength):
	urlParts = urlparse.urlparse(url)

	netloc = urlParts.netloc
	path = urlParts.path

	pathLength += max(domainLength - len(netloc), 0)
	domainLength += max(pathLength - len(path), 0)

	netloc = abbreviate(netloc, domainLength)
	path = abbreviate(path, pathLength)
	return netloc + path


def get_token(username, apiKey, secret):
	token = None
	rtm = rtm_api.RTMapi(username, apiKey, secret, token)

	authURL = rtm.getAuthURL()
	webbrowser.open(authURL)
	mb = gtk_toolbox.MessageBox2("You need to authorize DoneIt with\nRemember The Milk.\nClick OK after you authorize.")
	mb.run()

	token = rtm.getToken()
	return token


def get_credentials(credentialsDialog):
	username, password = credentialsDialog.request_credentials()
	token = get_token(username, rtm_backend.RtMilkManager.API_KEY, rtm_backend.RtMilkManager.SECRET)
	return username, password, token


def item_sort_by_priority_then_date(items):
	sortedTasks = list(items)
	sortedTasks.sort(
		key = lambda taskDetails: (
			taskDetails["priority"].get_nothrow(4),
			taskDetails["dueDate"].get_nothrow(datetime.datetime.max),
		),
	)
	return sortedTasks


def item_sort_by_date_then_priority(items):
	sortedTasks = list(items)
	sortedTasks.sort(
		key = lambda taskDetails: (
			taskDetails["dueDate"].get_nothrow(datetime.datetime.max),
			taskDetails["priority"].get_nothrow(4),
		),
	)
	return sortedTasks


def item_in_agenda(item):
	taskDate = item["dueDate"].get_nothrow(datetime.datetime.max)
	today = datetime.datetime.now()
	delta = taskDate - today
	dayDelta = abs(delta.days)

	priority = item["priority"].get_nothrow(4)
	weeksVisible = 5 - priority

	isVisible = not bool(dayDelta / (weeksVisible * 7))
	return isVisible


def item_sort_by_fuzzydate_then_priority(items):
	sortedTasks = list(items)

	def advanced_key(taskDetails):
		dueDate = taskDetails["dueDate"].get_nothrow(datetime.datetime.max)
		priority = taskDetails["priority"].get_nothrow(4)
		isNotSameYear = not toolbox.is_same_year(dueDate)
		isNotSameMonth = not toolbox.is_same_month(dueDate)
		isNotSameDay = not toolbox.is_same_day(dueDate)
		return isNotSameDay, isNotSameMonth, isNotSameYear, priority, dueDate

	sortedTasks.sort(key=advanced_key)
	return sortedTasks


class ItemListView(object):

	ID_IDX = 0
	COMPLETION_IDX = 1
	NAME_IDX = 2
	PRIORITY_IDX = 3
	DUE_IDX = 4
	FUZZY_IDX = 5
	LINK_IDX = 6
	NOTES_IDX = 7

	def __init__(self, widgetTree, errorDisplay):
		self._errorDisplay = errorDisplay
		self._manager = None
		self._projId = None
		self._showCompleted = False
		self._showIncomplete = True

		self._editDialog = gtk_toolbox.EditTaskDialog(widgetTree)
		self._notesDialog = gtk_toolbox.NotesDialog(widgetTree)

		self._itemList = gtk.ListStore(
			gobject.TYPE_STRING, # id
			gobject.TYPE_BOOLEAN, # is complete
			gobject.TYPE_STRING, # name
			gobject.TYPE_STRING, # priority
			gobject.TYPE_STRING, # due
			gobject.TYPE_STRING, # fuzzy due
			gobject.TYPE_STRING, # Link
			gobject.TYPE_STRING, # Notes
		)
		self._completionColumn = gtk.TreeViewColumn('') # Complete?
		self._completionCell = gtk.CellRendererToggle()
		self._completionCell.set_property("activatable", True)
		self._completionCell.connect("toggled", self._on_completion_change)
		self._completionColumn.pack_start(self._completionCell, False)
		self._completionColumn.set_attributes(self._completionCell, active=self.COMPLETION_IDX)
		self._priorityColumn = gtk.TreeViewColumn('') # Priority
		self._priorityCell = gtk.CellRendererText()
		self._priorityColumn.pack_start(self._priorityCell, False)
		self._priorityColumn.set_attributes(self._priorityCell, text=self.PRIORITY_IDX)
		self._nameColumn = gtk.TreeViewColumn('Name')
		self._nameCell = gtk.CellRendererText()
		self._nameColumn.pack_start(self._nameCell, True)
		self._nameColumn.set_attributes(self._nameCell, text=self.NAME_IDX)
		self._dueColumn = gtk.TreeViewColumn('Due')
		self._dueCell = gtk.CellRendererText()
		self._dueColumn.pack_start(self._nameCell, False)
		self._dueColumn.set_attributes(self._nameCell, text=self.FUZZY_IDX)
		self._linkColumn = gtk.TreeViewColumn('') # Link
		self._linkCell = gtk.CellRendererText()
		self._linkColumn.pack_start(self._nameCell, False)
		self._linkColumn.set_attributes(self._nameCell, text=self.LINK_IDX)
		self._notesColumn = gtk.TreeViewColumn('') # Notes
		self._notesCell = gtk.CellRendererText()
		self._notesColumn.pack_start(self._nameCell, False)
		self._notesColumn.set_attributes(self._nameCell, text=self.NOTES_IDX)

		self._todoBox = widgetTree.get_widget("todoBox")
		self._todoItemScroll = gtk.ScrolledWindow()
		self._todoItemScroll.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
		self._todoItemTree = gtk.TreeView()
		self._todoItemScroll.add(self._todoItemTree)
		self._onItemSelectId = 0

	def enable(self, manager, projId):
		self._manager = manager
		self._projId = projId

		self._todoBox.pack_start(self._todoItemScroll)
		self._todoItemScroll.show_all()

		self._itemList.clear()
		self._todoItemTree.append_column(self._completionColumn)
		self._todoItemTree.append_column(self._priorityColumn)
		self._todoItemTree.append_column(self._nameColumn)
		self._todoItemTree.append_column(self._dueColumn)
		self._todoItemTree.append_column(self._linkColumn)
		self._todoItemTree.append_column(self._notesColumn)
		try:
			self.reset_task_list(self._projId)
		except StandardError, e:
			self._errorDisplay.push_exception()

		self._todoItemTree.set_headers_visible(False)
		self._nameColumn.set_expand(True)

		self._onItemSelectId = self._todoItemTree.connect("row-activated", self._on_item_select)

	def disable(self):
		self._manager = None
		self._projId = None

		self._todoBox.remove(self._todoItemScroll)
		self._todoItemScroll.hide_all()
		self._todoItemTree.disconnect(self._onItemSelectId)

		self._todoItemTree.remove_column(self._completionColumn)
		self._todoItemTree.remove_column(self._priorityColumn)
		self._todoItemTree.remove_column(self._nameColumn)
		self._todoItemTree.remove_column(self._dueColumn)
		self._todoItemTree.remove_column(self._linkColumn)
		self._todoItemTree.remove_column(self._notesColumn)
		self._itemList.clear()
		self._itemList.set_model(None)

	def reset_task_list(self, projId):
		self._projId = projId
		self._itemList.clear()
		self._populate_items()

	def _populate_items(self):
		projId = self._projId
		rawTasks = self._manager.get_tasks_with_details(projId)
		filteredTasks = (
			taskDetails
			for taskDetails in rawTasks
				if self._showCompleted and taskDetails["isCompleted"] or self._showIncomplete and not taskDetails["isCompleted"]
		)
		# filteredTasks = (taskDetails for taskDetails in filteredTasks if item_in_agenda(taskDetails))
		sortedTasks = item_sort_by_priority_then_date(filteredTasks)
		# sortedTasks = item_sort_by_date_then_priority(filteredTasks)
		# sortedTasks = item_sort_by_fuzzydate_then_priority(filteredTasks)
		for taskDetails in sortedTasks:
			id = taskDetails["id"]
			isCompleted = taskDetails["isCompleted"]
			name = abbreviate(taskDetails["name"], 100)
			priority = str(taskDetails["priority"].get_nothrow(""))
			if taskDetails["dueDate"].is_good():
				dueDate = taskDetails["dueDate"].get()
				dueDescription = dueDate.strftime("%Y-%m-%d %H:%M:%S")
				fuzzyDue = toolbox.to_fuzzy_date(dueDate)
			else:
				dueDescription = ""
				fuzzyDue = ""

			linkDisplay = taskDetails["url"]
			linkDisplay = abbreviate_url(linkDisplay, 20, 10)

			notes = taskDetails["notes"]
			notesDisplay = "%d Notes" % len(notes) if notes else ""

			row = (id, isCompleted, name, priority, dueDescription, fuzzyDue, linkDisplay, notesDisplay)
			self._itemList.append(row)
		self._todoItemTree.set_model(self._itemList)

	def _on_item_select(self, treeView, path, viewColumn):
		try:
			# @todo See if there is a way to use the new gtk_toolbox.ContextHandler
			taskId = self._itemList[path[0]][self.ID_IDX]

			if viewColumn is self._priorityColumn:
				pass
			elif viewColumn is self._nameColumn:
				self._editDialog.enable(self._manager)
				try:
					self._editDialog.request_task(self._manager, taskId)
				finally:
					self._editDialog.disable()
				self.reset_task_list(self._projId)
			elif viewColumn is self._dueColumn:
				due = self._manager.get_task_details(taskId)["dueDate"]
				if due.is_good():
					calendar = gtk_toolbox.PopupCalendar(None, due.get())
					calendar.run()
			elif viewColumn is self._linkColumn:
				webbrowser.open(self._manager.get_task_details(taskId)["url"])
			elif viewColumn is self._notesColumn:
				self._notesDialog.enable()
				try:
					self._notesDialog.run(self._manager, taskId)
				finally:
					self._notesDialog.disable()
		except StandardError, e:
			self._errorDisplay.push_exception()

	def _on_completion_change(self, cell, path):
		try:
			taskId = self._itemList[path[0]][self.ID_IDX]
			self._manager.complete_task(taskId)
			self.reset_task_list(self._projId)
		except StandardError, e:
			self._errorDisplay.push_exception()


class QuickAddView(object):

	def __init__(self, widgetTree, errorDisplay, addSink):
		self._errorDisplay = errorDisplay
		self._manager = None
		self._projId = None
		self._addSink = addSink

		self._clipboard = gtk.clipboard_get()
		self._editDialog = gtk_toolbox.EditTaskDialog(widgetTree)

		self._taskNameEntry = widgetTree.get_widget("add-taskNameEntry")
		self._addTaskButton = widgetTree.get_widget("add-addTaskButton")
		self._pasteTaskNameButton = widgetTree.get_widget("add-pasteTaskNameButton")
		self._clearTaskNameButton = widgetTree.get_widget("add-clearTaskNameButton")
		self._onAddId = None
		self._onAddClickedId = None
		self._onAddReleasedId = None
		self._addToEditTimerId = None
		self._onClearId = None
		self._onPasteId = None

	def enable(self, manager, projId):
		self._manager = manager
		self._projId = projId

		self._onAddId = self._addTaskButton.connect("clicked", self._on_add)
		self._onAddClickedId = self._addTaskButton.connect("pressed", self._on_add_pressed)
		self._onAddReleasedId = self._addTaskButton.connect("released", self._on_add_released)
		self._onPasteId = self._pasteTaskNameButton.connect("clicked", self._on_paste)
		self._onClearId = self._clearTaskNameButton.connect("clicked", self._on_clear)

	def disable(self):
		self._manager = None
		self._projId = None

		self._addTaskButton.disconnect(self._onAddId)
		self._addTaskButton.disconnect(self._onAddClickedId)
		self._addTaskButton.disconnect(self._onAddReleasedId)
		self._pasteTaskNameButton.disconnect(self._onPasteId)
		self._clearTaskNameButton.disconnect(self._onClearId)

	def reset_task_list(self, projId):
		self._projId = projId
		isMeta = self._manager.get_project(self._projId)["isMeta"]
		# @todo RTM handles this by defaulting to a specific list
		self._addTaskButton.set_sensitive(not isMeta)

	def _on_add(self, *args):
		try:
			name = self._taskNameEntry.get_text()

			projId = self._projId
			taskId = self._manager.add_task(projId, name)

			self._taskNameEntry.set_text("")
			self._addSink.send((projId, taskId))
		except StandardError, e:
			self._errorDisplay.push_exception()

	def _on_add_edit(self, *args):
		try:
			name = self._taskNameEntry.get_text()

			projId = self._projId
			taskId = self._manager.add_task(projId, name)

			try:
				self._editDialog.enable(self._manager)
				try:
					self._editDialog.request_task(self._manager, taskId)
				finally:
					self._editDialog.disable()
			finally:
				self._taskNameEntry.set_text("")
				self._addSink.send((projId, taskId))
		except StandardError, e:
			self._errorDisplay.push_exception()

	def _on_add_pressed(self, widget):
		try:
			self._addToEditTimerId = gobject.timeout_add(1000, self._on_add_edit)
		except StandardError, e:
			self._errorDisplay.push_exception()

	def _on_add_released(self, widget):
		try:
			if self._addToEditTimerId is not None:
				gobject.source_remove(self._addToEditTimerId)
			self._addToEditTimerId = None
		except StandardError, e:
			self._errorDisplay.push_exception()

	def _on_paste(self, *args):
		try:
			entry = self._taskNameEntry.get_text()
			entry += self._clipboard.wait_for_text()
			self._taskNameEntry.set_text(entry)
		except StandardError, e:
			self._errorDisplay.push_exception()

	def _on_clear(self, *args):
		try:
			self._taskNameEntry.set_text("")
		except StandardError, e:
			self._errorDisplay.push_exception()


class GtkRtMilk(object):

	def __init__(self, widgetTree, errorDisplay):
		"""
		@note Thread agnostic
		"""
		self._errorDisplay = errorDisplay
		self._manager = None
		self._credentials = "", "", ""

		self._projectsList = gtk.ListStore(gobject.TYPE_STRING)
		self._projectsCombo = widgetTree.get_widget("projectsCombo")
		self._onListActivateId = 0

		self._itemView = ItemListView(widgetTree, self._errorDisplay)
		addSink = coroutines.func_sink(lambda eventData: self._itemView.reset_task_list(eventData[0]))
		self._addView = QuickAddView(widgetTree, self._errorDisplay, addSink)
		self._credentialsDialog = gtk_toolbox.LoginWindow(widgetTree)

	@staticmethod
	def name():
		return "Remember The Milk"

	def load_settings(self, config):
		"""
		@note Thread Agnostic
		"""
		blobs = (
			config.get(self.name(), "bin_blob_%i" % i)
			for i in xrange(len(self._credentials))
		)
		creds = (
			base64.b64decode(blob)
			for blob in blobs
		)
		self._credentials = tuple(creds)

	def save_settings(self, config):
		"""
		@note Thread Agnostic
		"""
		config.add_section(self.name())
		for i, value in enumerate(self._credentials):
			blob = base64.b64encode(value)
			config.set(self.name(), "bin_blob_%i" % i, blob)

	def login(self):
		"""
		@note UI Thread
		"""
		if self._manager is not None:
			return

		credentials = self._credentials
		while True:
			try:
				self._manager = rtm_backend.RtMilkManager(*credentials)
				self._credentials = credentials
				return # Login succeeded
			except rtm_api.AuthStateMachine.NoData:
				# Login failed, grab new credentials
				credentials = get_credentials(self._credentialsDialog)

	def logout(self):
		"""
		@note Thread Agnostic
		"""
		self._credentials = "", "", ""
		self._manager = None

	def enable(self):
		"""
		@note UI Thread
		"""
		self._projectsList.clear()
		self._populate_projects()

		currentProject = self._get_project()
		projId = self._manager.lookup_project(currentProject)["id"]
		self._addView.enable(self._manager, projId)
		self._itemView.enable(self._manager, projId)

		self._onListActivateId = self._projectsCombo.connect("changed", self._on_list_activate)

	def disable(self):
		"""
		@note UI Thread
		"""
		self._projectsCombo.disconnect(self._onListActivateId)

		self._addView.disable()
		self._itemView.disable()

		self._projectsList.clear()
		self._projectsCombo.set_model(None)
		self._projectsCombo.disconnect("changed", self._on_list_activate)

		self._manager = None

	def _populate_projects(self):
		for project in self._manager.get_projects():
			projectName = project["name"]
			isVisible = project["isVisible"]
			row = (projectName, )
			if isVisible:
				self._projectsList.append(row)
		self._projectsCombo.set_model(self._projectsList)
		cell = gtk.CellRendererText()
		self._projectsCombo.pack_start(cell, True)
		self._projectsCombo.add_attribute(cell, 'text', 0)
		self._projectsCombo.set_active(0)

	def _reset_task_list(self):
		projectName = self._get_project()
		projId = self._manager.lookup_project(projectName)["id"]
		self._addView.reset_task_list(projId)
		self._itemView.reset_task_list(projId)

	def _get_project(self):
		currentProjectName = self._projectsCombo.get_active_text()
		return currentProjectName

	def _on_list_activate(self, *args):
		try:
			self._reset_task_list()
		except StandardError, e:
			self._errorDisplay.push_exception()
