import sys
import StringIO
import urllib
from xml.dom import minidom
import datetime


class Optional(object):
	"""
	Taglines:
	Even you don't have to worry about knowing when to perform None checks
	When the NULL object pattern just isn't good enough

	>>> a = Optional()
	>>> a.is_good()
	False
	>>> a.get_nothrow("Blacksheep")
	'Blacksheep'
	>>> a.set("Lamb")
	>>> a.is_good()
	True
	>>> a.get_nothrow("Blacksheep"), a.get(), a()
	('Lamb', 'Lamb', 'Lamb')
	>>> a.clear()
	>>> a.is_good()
	False
	>>> a.get_nothrow("Blacksheep")
	'Blacksheep'
	>>>
	>>> b = Optional("Lamb")
	>>> a.set("Lamb")
	>>> a.is_good()
	True
	>>> a.get_nothrow("Blacksheep"), a.get(), a()
	('Lamb', 'Lamb', 'Lamb')
	>>> a.clear()
	>>> a.is_good()
	False
	>>> a.get_nothrow("Blacksheep")
	'Blacksheep'
	"""

	class NonExistent(object):
		pass

	def __init__(self, value = NonExistent):
		self._value = value

	def set(self, value):
		self._value = value

	def clear(self):
		self._value = self.NonExistent

	def is_good(self):
		return self._value is not self.NonExistent

	def get_nothrow(self, default = None):
		return self._value if self.is_good() else default

	def get(self):
		if not self.is_good():
			raise ReferenceError("Optional not initialized")
		return self._value

	def __call__(self):
		# Implemented to imitate weakref
		return self.get()

	def __str__(self):
		return str(self.get_nothrow(""))

	def __repr__(self):
		return "<Optional at %x; to %r>" % (
			id(self), self.get_nothrow("Nothing")
		)


def open_anything(source, alternative=None):
	"""URI, filename, or string --> stream

	This function lets you define parsers that take any input source
	(URL, pathname to local or network file, or actual data as a string)
	and deal with it in a uniform manner.  Returned object is guaranteed
	to have all the basic stdio read methods (read, readline, readlines).
	Just .close() the object when you're done with it.
	"""
	if hasattr(source, "read"):
		return source

	if source == '-':
		return sys.stdin

	# try to open with urllib (if source is http, ftp, or file URL)
	try:
		return urllib.urlopen(source)
	except (IOError, OSError):
		##pass
		print "ERROR with URL ("+source+")!\n"

	# try to open with native open function (if source is pathname)
	try:
		return open(source)
	except (IOError, OSError):
		##pass
		print "ERROR with file!\n"

	# treat source as string
	if alternative == None:
		print 'LAST RESORT.  String is "'+source+'"\n'
		return StringIO.StringIO(str(source))
	else:
		print 'LAST RESORT.  String is "'+alternative+'"\n'
		return StringIO.StringIO(str(alternative))


def load_xml(source, alternative=None):
	"""load XML input source, return parsed XML document

	- a URL of a remote XML file ("http://diveintopython.org/kant.xml")
	- a filename of a local XML file ("~/diveintopython/common/py/kant.xml")
	- standard input ("-")
	- the actual XML document, as a string
	"""
	sock = open_anything(source, alternative)
	try:
		xmldoc = minidom.parse(sock).documentElement
	except (IOError, OSError):
		print "ERROR with data"
		sock.close()
		sock = open_anything('<response method="getProjects"><project projName="ERROR!"/></response>')
		xmldoc = minidom.parse(sock).documentElement
	sock.close()
	return xmldoc


def is_same_year(targetDate, todaysDate = datetime.datetime.today()):
	return targetDate.year == todaysDate.year


def is_same_month(targetDate, todaysDate = datetime.datetime.today()):
	return targetDate.month == todaysDate.month


def is_same_day(targetDate, todaysDate = datetime.datetime.today()):
	return targetDate.day == todaysDate.day


def to_fuzzy_date(targetDate, todaysDate = datetime.datetime.today()):
	"""
	Conert a date/time/datetime object to a fuzzy date

	>>> todaysDate = datetime.date(2009, 4, 16)
	>>> to_fuzzy_date(datetime.date(1, 4, 6), todaysDate)
	'Forever ago'
	>>> to_fuzzy_date(datetime.date(2008, 4, 13), todaysDate)
	'Last year'
	>>> to_fuzzy_date(datetime.date(2009, 4, 13), todaysDate)
	'Last Monday'
	>>> to_fuzzy_date(datetime.date(2009, 4, 20), todaysDate)
	'This Monday'
	>>> to_fuzzy_date(datetime.date(2010, 4, 13), todaysDate)
	'Next year'
	>>> to_fuzzy_date(datetime.date(2012, 12, 12), todaysDate)
	'Forever from now'
	"""
	delta = targetDate - todaysDate
	days = abs(delta.days)
	noDifference = datetime.timedelta()
	isFuture = noDifference < delta
	directionBy1 = "Next" if isFuture else "Last"
	directionByN = "Later" if isFuture else "Earlier"

	yearDelta = abs(targetDate.year - todaysDate.year)
	if 1 < yearDelta:
		directionByInf = "from now" if isFuture else "ago"
		return "Forever %s" % directionByInf
	elif 1 == yearDelta:
		return "%s year" % directionBy1

	monthDelta = abs(targetDate.month - todaysDate.month)
	if 1 < monthDelta:
		return "%s this year" % directionByN
	elif 1 == monthDelta:
		return "%s month" % directionBy1

	dayDelta = abs(targetDate.day - todaysDate.day)
	if 14 < dayDelta:
		return "%s this month" % directionByN
	elif 1 < dayDelta:
		directionInWeek = "This" if isFuture else "Last"
		days = ["Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday"]
		return "%s %s" % (directionInWeek, days[targetDate.weekday()])
	elif 1 == dayDelta:
		return "%s day" % directionBy1
	else:
		return "Today"
