/*
  DiskUsage - shows the disk usage under a number of operating systems.
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cmath>
#include <iomanip>
#include <iostream>
#include <sstream>

#include <Lum/Base/L10N.h>
#include <Lum/Base/Object.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/DataStream.h>
#include <Lum/Model/Integer.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/OS/Color.h>
#include <Lum/OS/Main.h>

#include <Lum/Label.h>
#include <Lum/PercentPie.h>
#include <Lum/Table.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>

#include "config.h"

#include "DiskInfo.h"
#include "Map.h"

static Lum::Def::AppInfo info;

static std::wstring DoubleToWString(double value, double unit)
{
  std::stringstream buffer;

  buffer.imbue(std::locale(""));
  buffer.setf(std::ios::fixed);
  buffer << std::setprecision(1);

  if (ceil(value*unit)>=1024*1024*1024*0.5) {
    buffer << (value*unit)/(1024*1024*1024) << " TB";
  }
  else if (ceil(value*unit)>=1024*1024*0.5) {
    buffer << (value*unit)/(1024*1024) << " GB";
  }
  else if (ceil(value*unit)>=1024*0.5) {
    buffer << (value*unit)/1024 << " MB";
  }
  else {
    buffer << value*unit << " KB";
  }

  return Lum::Base::StringToWString(buffer.str());
}

static Lum::TextValue* GetTextValueLeft(const Lum::Model::StringRef& value)
{
  Lum::TextValue   *text;
  Lum::OS::FontRef font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1));

  text=Lum::TextValue::Create(value,Lum::TextValue::left,true,false);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,25);
  // TODO: font

  return text;
}

static Lum::TextValue* GetTextValueRight(const Lum::Model::StringRef& value)
{
  Lum::TextValue   *text;
  Lum::OS::FontRef font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1));

  text=Lum::TextValue::Create(value,Lum::TextValue::right,true,false);
  text->SetMinWidth(Lum::Base::Size::stdCharWidth,25);
  // TODO: font

  return text;
}

class DiskInfoThread : public Lum::OS::Thread
{
public:
  Lum::Model::ActionRef dataAvailable;
  std::list<DiskInfo>   diskInfos;

public:
  DiskInfoThread()
   : dataAvailable(new Lum::Model::Action())
  {
    // no code
  }

  ~DiskInfoThread()
  {
    Join();
  }

  void Run()
  {
    diskInfos.clear();
    GetDisks(diskInfos);

    Lum::OS::display->QueueActionForAsyncNotification(dataAvailable);
  };
};

class Usage : public Lum::Control
{
private:
  Lum::Model::DoubleRef model;
  Lum::OS::FontRef      font;
  Lum::OS::ColorRef     red;
  Lum::OS::ColorRef     green;

public:
  Usage()
  : font(Lum::OS::display->GetFont()),
    red(1,0,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor)),
    green(0,1,0,Lum::OS::display->GetColor(Lum::OS::Display::tableTextColor))
  {
    // no code
  }

  bool SetModel(Lum::Base::Model* model)
  {
    this->model=dynamic_cast<Lum::Model::Double*>(model);

    Control::SetModel(this->model);

    return this->model.Valid();
  }

  void CalcSize()
  {
    minWidth=2*Lum::OS::display->GetSpaceHorizontal(Lum::OS::Display::spaceObjectBorder)+
             3*font->StringWidth(L"m",Lum::OS::Font::normal);
    minHeight=font->pixelHeight+2*Lum::OS::display->GetSpaceVertical(Lum::OS::Display::spaceObjectBorder);

    width=minWidth;
    height=minHeight;

    Control::CalcSize();
  }

  void Draw(int x, int y, size_t w, size_t h)
  {
    Control::Draw(x,y,w,h);

    if (!OIntersect(x,y,w,h)) {
      return;
    }

    /* --- */

    Lum::OS::DrawInfo *draw=GetDrawInfo();

    if (!model.Valid() || model->IsNull()) {
      return;
    }

    double percent=model->Get();

    if (percent>=0.9) {
      draw->PushForeground(red);
    }
    else {
      draw->PushForeground(green);
    }

    draw->FillRectangle(this->x,this->y+1,
                        (size_t)(width*percent),height-2);

    draw->PopForeground();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==this->model) {
      Redraw();
    }
    Control::Resync(model,msg);
  }
};

typedef Lum::Model::StdTable<DiskInfo>      DiskInfoModel;
typedef Lum::Base::Reference<DiskInfoModel> DiskInfoModelRef;

class DiskInfoDataProvider : public DiskInfoModel::DataProvider
{
private:
  Usage                 *percent;
  Lum::Model::DoubleRef data;

public:
  DiskInfoDataProvider()
  : percent(new Usage()),
    data(new Lum::Model::Double())
  {
    percent->SetFlex(true,true);
    percent->SetMinWidth(Lum::Base::Size::stdCharWidth,15);
    percent->SetModel(data);
  }

  virtual ~DiskInfoDataProvider()
  {
    delete percent;
  }

  std::wstring GetString(const DiskInfoModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 1:
      return iter->dir;
    default:
      return L"";
    }
  }

  Lum::Object* GetObject(const DiskInfoModel::Iterator& iter, size_t column) const
  {
    switch (column) {
    case 2:
      data->Set((iter->blocks-iter->avail)/iter->blocks);
      return percent;
    default:
      return NULL;
    }
  }
};

class MainWindow : public Lum::Dialog
{
private:
  Lum::Model::ActionRef              about;

  Lum::Model::ActionRef              showMap;

  Lum::Model::IntRef                 refreshTime;
  Lum::Model::ActionRef              timer;

  Lum::Table                         *table;
  Lum::Model::SingleLineSelectionRef selection;
  DiskInfoModelRef                   tableModel;
  Lum::Model::StringRef              device;
  Lum::Model::StringRef              dir;
  Lum::Model::StringRef              used;
  Lum::Model::StringRef              free;
  Lum::Model::StringRef              total;
  Lum::Model::DoubleDataStreamRef    data;

  std::list<DiskInfo>                list;

  DiskInfoThread                     diskInfoThread;
  ScanTask                           scanTask;

public:
  MainWindow()
  : about(new Lum::Model::Action()),
    showMap(new Lum::Model::Action()),
    refreshTime(new Lum::Model::Int(15)),
    timer(new Lum::Model::Action()),
    selection(new Lum::Model::SingleLineSelection),
    tableModel(new DiskInfoModel(new DiskInfoDataProvider())),
    device(new Lum::Model::String()),
    dir(new Lum::Model::String()),
    used(new Lum::Model::String()),
    free(new Lum::Model::String()),
    total(new Lum::Model::String()),
    data(new Lum::Model::DoubleDataStream())
  {
    data->SetNotificationMode(Lum::Model::DoubleDataStream::notifyExplicit);
    data->SetChannels(3);

    scanTask.SetParent(this);
    scanTask.SetCaption(L"Scanning directory...");

    Observe(about);
    Observe(showMap);
    Observe(refreshTime);
    Observe(timer);
    Observe(selection);
    Observe(GetOpenedAction());

    Observe(diskInfoThread.dataAvailable);

    Observe(scanTask.GetAbortedAction());
    Observe(scanTask.GetFinishedAction());
  }

  void PreInit()
  {
    Lum::Panel            *panel,*panel2;
    Lum::PercentPie       *percent;
    Lum::Model::HeaderRef headerModel;

    panel=Lum::VPanel::Create(true,true);
    panel->SetWidth(Lum::Base::Size::workHRel,20);
    panel->SetHeight(Lum::Base::Size::workVRel,40);

    panel2=Lum::HPanel::Create(true,false);
    panel2->Add(Lum::Label::Create()
                ->AddLabel(L"Device:",GetTextValueLeft(device))
                ->AddLabel(L"Directory:",GetTextValueLeft(dir))
                ->AddLabel(L"Size:",GetTextValueRight(total))
                ->AddLabel(L"Used:",GetTextValueRight(used))
                ->AddLabel(L"Free:",GetTextValueRight(free)));
    panel2->AddSpace(true);

    percent=new Lum::PercentPie();
    percent->SetModel(data);
    panel2->Add(percent);

    panel->Add(panel2);
    panel->AddSpace();

    headerModel=new Lum::Model::HeaderImpl();
    headerModel->AddColumn(L"Dir",Lum::Base::Size::stdCharWidth,25);
    headerModel->AddColumn(L"Size",Lum::Base::Size::stdCharWidth,15,true);

    table=new Lum::Table();
    table->SetFlex(true,true);
    table->SetModel(tableModel);
    table->SetHeaderModel(headerModel);
    table->SetSelection(selection);
    table->SetDoubleClickAction(showMap);
    table->SetShowHeader(true);
    table->GetTableView()->SetAutoFitColumns(true);
    panel->Add(table);

    SetMain(panel);

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupHelp()
        //->AddMenuItemAction(_ld(menuHelpHelp),NULL)
        ->ActionAbout(about)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void UpdateDisplay()
  {
    std::list<DiskInfo>::const_iterator iter;

    iter=list.begin();

    // First try to update existing entries in the current table
    while (iter!=list.end()) {
      size_t i;

      for (i=1; i<=tableModel->GetRows(); i++) {
        if (iter->dir==tableModel->GetEntry(i).dir) {
          break;
        }
      }

      if (i<=tableModel->GetRows()) {
        tableModel->GetEntry(i)=*iter;
        tableModel->RedrawRow(i);
      }
      else {
        tableModel->Append(*iter);
      }

      ++iter;
    }

    // Now see, if entries in the table have been removed and must
    // be deleted
    size_t i=1;

    while(i<=tableModel->GetRows()) {
      std::wstring                        directory=tableModel->GetEntry(i).dir;
      std::list<DiskInfo>::const_iterator iter;

      iter=list.begin();
      while (iter!=list.end()) {
        if (iter->dir==directory) {
          break;
        }

        ++iter;
      }

      if (iter==list.end()) {
        tableModel->Delete(i);
      }
      else {
        i++;
      }
    }
  }

  void ShowDetail()
  {
    std::list<DiskInfo>::const_iterator iter;

    if (selection->HasSelection()) {
      iter=list.begin();
      while (iter!=list.end() && iter->dir!=tableModel->GetEntry(selection->GetLine()).dir) {
        ++iter;
      }
    }
    else {
      iter=list.end();
    }

    if (iter!=list.end()) {
      device->Set(iter->name);
      dir->Set(iter->dir);

      total->Set(DoubleToWString(iter->blocks,iter->unit));
      used->Set(DoubleToWString(iter->blocks-iter->free,iter->unit));
      free->Set(DoubleToWString(iter->avail,iter->unit)+L" (root: "+DoubleToWString(iter->free,iter->unit)+L")");

      data->Enable();
      data->Set(0,(iter->blocks-iter->free)/iter->blocks); // used
      data->Set(1,iter->avail/iter->blocks); // free without reserved
      data->Set(2,(iter->free-iter->avail)/iter->blocks); // reserved
      data->Notify();
    }
    else {
      device->Set(L"");

      used->Set(L"");
      free->Set(L"");
      total->Set(L"");

      data->Disable();
    }
  }

  void InitiateScan()
  {
    if (!selection->HasSelection()) {
      return;
    }

    std::list<DiskInfo>::const_iterator iter=list.begin();

    while (iter!=list.end() && iter->dir!=tableModel->GetEntry(selection->GetLine()).dir) {
      ++iter;
    }

    if (iter==list.end()) {
      return;
    }

    Lum::Base::Path path;

    path.SetNativeDir(iter->dir);

    scanTask.SetSize(iter->blocks*iter->unit);
    scanTask.SetDir(path);
    scanTask.Start();
  }

  void ShowMap()
  {
    DirEntry  *top;
    MapWindow *window;

    top=scanTask.top;
    scanTask.top=NULL;

    window=new MapWindow(top,top->size/*scanTask.size*/);
    window->SetParent(this);

    if (window->Open()) {
      window->EventLoop();
      window->Close();
    }

    delete window;
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() && GetOpenedAction()->IsFinished()) {
      diskInfoThread.Start();
    }
    else if (model==diskInfoThread.dataAvailable &&
             diskInfoThread.dataAvailable->IsFinished()) {
      list=diskInfoThread.diskInfos;
      UpdateDisplay();

      if (tableModel->GetRows()>0 &&
          !selection->HasSelection()) {
        selection->SelectLine(1);
      }

      Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
    }
    else if (model==timer && timer->IsFinished()) {
      diskInfoThread.Start();
    }
    else if (model==refreshTime && IsOpen()) {
      /*
      Lum::OS::display->RemoveTimer(timer);
      if (refreshTime->Get()>0) {
        Lum::OS::display->AddTimer(refreshTime->Get(),0,timer);
      }*/
    }
    else if (model==selection) {
      ShowDetail();
    }
    else if (model==showMap && showMap->IsFinished()) {
      InitiateScan();
    }
    else if (model==scanTask.GetFinishedAction() &&
             scanTask.GetFinishedAction()->IsFinished()) {
      ShowMap();
    }
    else if (model==about && about->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }

};

class Main : public Lum::OS::MainDialog<MainWindow>
{
public:
  bool Prepare()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"Show usage of all available disks"));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2004, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::OS::MainDialog<MainWindow>::Prepare();
  }
};

LUM_MAIN(Main,L"DiskUsage")
