#include <QtGui>
#include <QLocale>
#include <QGraphicsView>
#include <QGraphicsScene>
#include <QFontMetrics>
#include <QGraphicsProxyWidget>
#if  defined(Q_WS_MAEMO_5)
#include <QtDBus/QtDBus>
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#endif

#include "digitalclockwdg.h"
#include "settingsdialog.h"
#include "dclocksettings.h"
#include "lcdclock.h"
#include "fontclock.h"

DigitalClockWdg::DigitalClockWdg(QGraphicsScene *scene)
    : QGraphicsView(scene), m_scene(scene)
{

    setWindowTitle(tr("Digital Clock"));
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

//    QPalette qp = palette();
//    QColor qc = qp.color(QPalette::Window);
//    qc.setAlpha(100);
    m_scene->setForegroundBrush(Qt::NoBrush);

    setAttribute(Qt::WA_TranslucentBackground);
    viewport()->setAutoFillBackground(false);

    clk1 = new QWidget(0);
    m_timer = new QTimer(this);
    connect(m_timer, SIGNAL(timeout()), this, SLOT(showTime()));

    m_gLine = m_scene->addLine(0,0,0,0);
    m_gLine->setPos(-size().width(),-size().height());
    m_gCity = m_scene->addText("");
    m_gCity->setPos(-size().width(),-size().height());

    setViewportUpdateMode(QGraphicsView::BoundingRectViewportUpdate);
    setCacheMode(QGraphicsView::CacheNone);
    setRenderHints(QPainter::Antialiasing | QPainter::SmoothPixmapTransform);

    m_gProxy = m_scene->addWidget(clk1);
//    m_x = x();
//    m_y = y();
}

void DigitalClockWdg::setId(QString &aAppletId)
{
    m_aplletId = aAppletId;
    loadSettings();
//showSettingsDialog();    //!!
//    initClock();
}

int DigitalClockWdg::drawCityName()
{
    // calc font + line
    // get pen color
    QFont fnt = font();
    QPalette qp = palette();
    QPen pen(qp.color(QPalette::Text));
//    pen.color() = qp.color(QPalette::Button);
    fnt.setPointSize(m_citySize);
    QFontMetrics fm(fnt);
    int textSize = fm.height();
    m_wCity = fm.width(m_cityName);
    int y = m_sc_y + textSize + 2;
    m_scene->removeItem(m_gLine);
    m_scene->removeItem(m_gCity);
    m_gLine = m_scene->addLine(0,0,size().width(),0, pen);
    m_gLine->setPos(m_sc_x,y);
    m_gCity = m_scene->addText(m_cityName,fnt);
    m_gCity->setPos(m_sc_x + (size().width()-m_wCity)/2,m_sc_y);
    return textSize + 4;
}

void DigitalClockWdg::initClock()
{
    delete clk1;
//    m_scene->removeItem(m_gProxy);
    QTime tm = calcTime();
    QString tStr = tm.toString(m_timeFormat);
    QSize asize;
    if (!m_showCity)
    {
        asize = size();
        m_gLine->setVisible(false);
        m_gCity->setVisible(false);
    }
    else
    {
        m_gLine->setVisible(true);
        m_gCity->setVisible(true);
        asize = size();
        m_hCity = drawCityName();
        asize.setHeight(asize.height()-m_hCity);
        m_sc_y = m_sc_y + m_hCity; // move Clock down
        // todo calc size for city
    }
    if (m_lcd)
    {
        clk1 = new LcdClock(m_digitColor, tStr, asize);
    }
    else
    {
        clk1 = new FontClock(m_digitColor, m_font, tStr, asize);
    }
    connect(this,SIGNAL(updateClock(QString)),clk1,SLOT(showTime(QString)));
//    connect(this,SIGNAL(rotateClock()),clk1,SLOT(rotate()));
    m_gProxy = m_scene->addWidget(clk1);
//    m_gProxy->resize(asize);
    m_gProxy->setPos(m_sc_x,m_sc_y);
}

QTime DigitalClockWdg::calcTime()
{
    if (m_tzOffset == -50000)
        return QTime::currentTime();
    else
    {
        QDateTime dt = QDateTime::currentDateTime();
        QDateTime udt = dt.toUTC();
        dt = udt.addSecs(m_tzOffset);
        return dt.time();
    }
}

void DigitalClockWdg::showTime()
{
    //TODO update clock
    QTime tt = calcTime();

//    QTime time = QTime::currentTime();
    QString text = tt.toString(m_timeFormat);
    if (m_blinking1)
    {
        int lngt = text.length();
        if ((tt.second() % 2) == 0)
        {
            text[lngt-m_blinking1] = ' ';
            if (m_blinking2)
                text[lngt-m_blinking2] = ' ';
        }
    }
    emit updateClock(text);
}

void DigitalClockWdg::showSettingsDialog()
    {
    SettingsDialog dialog(m_aplletId, 0);
    if (dialog.exec() == QDialog::Accepted) {

        dialog.dumpSettings();
        loadSettings();
    }
}

void DigitalClockWdg::loadSettings()
{
#if  defined(Q_WS_MAEMO_5)
    m_digitColor = dclocksettings::getInstance().getColor(m_aplletId);
    bool showSeconds = dclocksettings::getInstance().isShowSeconds(m_aplletId);
    int timeFormatIdx = dclocksettings::getInstance().getTimeFormat(m_aplletId);
    m_size = dclocksettings::getInstance().getSize(m_aplletId);
    bool blinking = dclocksettings::getInstance().isBlinking(m_aplletId);
    m_openclock = dclocksettings::getInstance().isOpenClockEnabled(m_aplletId);
    m_showCity = dclocksettings::getInstance().isShowCityEnabled(m_aplletId);
    m_corner = dclocksettings::getInstance().getCornerNumber(m_aplletId);
#else
    m_digitColor = QColor(Qt::blue);
    bool showSeconds = true;
    int timeFormatIdx = 0;
    m_size = QSize(300,300);
    bool blinking = false;
#endif
    m_blinking1 = 0;
    m_blinking2 = 0;

//    QString timeFormat;
    if (showSeconds)
        switch (timeFormatIdx)
        {
            case 1:     // 12 H (AM/PM)
                m_timeFormat = QString(" h:mm:ss ap");
                if (blinking)
                {
                    m_blinking1 = 6;
                    m_blinking2 = 9;
                }
                break;
            case 2:     // mmilitary
                m_timeFormat = QString("HHmmss");
                break;
            default:    // 24 H
                m_timeFormat = QString("HH:mm:ss");
                if (blinking)
                {
                    m_blinking1 = 3;
                    m_blinking2 = 6;
                }
            }
    else
        switch (timeFormatIdx)
        {
            case 1:     // 12 H (AM/PM)
                m_timeFormat = QString(" h:mm ap");
                if (blinking)
                {
                    m_blinking1 = 6;
                }
                break;
            case 2:     // mmilitary
                m_timeFormat = QString("HHmm");
                break;
            default:    // 24 H
                m_timeFormat = QString("HH:mm");
                if (blinking)
                {
                    m_blinking1 = 3;
                }

        }

    bool rotatable;
#if  defined(Q_WS_MAEMO_5)
    m_lcd = dclocksettings::getInstance().isLcd(m_aplletId);
    m_font = dclocksettings::getInstance().getFont(m_aplletId);
    m_tzOffset = dclocksettings::getInstance().getTZoffset(m_aplletId);
    int aTr = 255 - dclocksettings::getInstance().getTransparency(m_aplletId);
    rotatable = dclocksettings::getInstance().isRotateEnabled(m_aplletId);
#else
    m_lcd = false;
    m_font = QFont("Helvetica",30);
    m_tzOffset = 14400;
    float aTr = float(50);
    rotatable = false;
#endif
//    m_transparency = aTr/100;

    QPalette qp = palette();
    QColor qc = qp.color(QPalette::Window);
    qc.setAlpha(aTr);
    m_scene->setBackgroundBrush(qc);

    m_cityName = dclocksettings::getInstance().getTZcity(m_aplletId);
    m_citySize = dclocksettings::getInstance().getCitySize(m_aplletId);

    m_timer->stop();
    m_timer->start(1000);
    m_sc_x = -m_size.width()/2;
    m_sc_y = -m_size.height()/2;
    m_scene->setSceneRect(m_sc_x, m_sc_y,m_size.width(),m_size.height());
    resize(m_size.width(),m_size.height());
    initClock();
    enableRotationMonitor(rotatable);    // !! to test
}

void DigitalClockWdg::homeScreenChanged(bool isOnCurrentHomescreen)
{
    if (isOnCurrentHomescreen)
    {
        if (!m_timer->isActive())
//            m_timer->start(m_timer_tick);
            m_timer->start(1000);
        showTime();
    }
    else
        m_timer->stop();
}


void DigitalClockWdg::mouseReleaseEvent ( QMouseEvent * e )
{
    if ((m_mouseClick) && (e->pos() == m_lastPoint) && (m_openclock) )
    {
#if  defined(Q_WS_MAEMO_5)
        showClock();
#else
        orientationChanged(QString("a"));
#endif
    }
}

void DigitalClockWdg::mousePressEvent ( QMouseEvent * e )
{
    m_lastPoint = e->pos();
    m_mouseClick = true;
}

void DigitalClockWdg::showClock()
{
#if  defined(Q_WS_MAEMO_5)
    QDBusConnection bus = QDBusConnection::sessionBus();
    QDBusInterface dbus_iface("com.nokia.HildonDesktop.AppMgr", "/com/nokia/HildonDesktop/AppMgr",
                          "com.nokia.HildonDesktop.AppMgr", bus);
    dbus_iface.call("LaunchApplication","worldclock");
#endif
}

void DigitalClockWdg::enableRotationMonitor(bool aRotatable)
{
#if  defined(Q_WS_MAEMO_5)
    m_rotate = false;
    if (aRotatable)
    {
        QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
            MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ));
        QDBusConnection::systemBus().connect("", MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
            MCE_DEVICE_ORIENTATION_SIG, this, SLOT(orientationChanged(QString)));
    }
    else
    {
        QDBusConnection::systemBus().call(QDBusMessage::createMethodCall(MCE_SERVICE, MCE_REQUEST_PATH,
            MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ));
        QDBusConnection::systemBus().disconnect("", MCE_SIGNAL_PATH, MCE_SIGNAL_IF,
            MCE_DEVICE_ORIENTATION_SIG, this, SLOT(orientationChanged(QString)));
        m_rotate = false;
    }
#else
    m_rotate = true;
#endif
}

void DigitalClockWdg::orientationChanged(QString aMessage)
{
#if  defined(Q_WS_MAEMO_5)
    int wd = size().width();
    int hg = size().height();
    int x = pos().x();
    int y = pos().y();
    if ((aMessage == QLatin1String(MCE_ORIENTATION_PORTRAIT)) && !m_rotate)
    {
        setTransform(QTransform()
               .translate(size().width() / 2, size().height() / 2)
               .rotate(270, Qt::ZAxis)
               .translate(-size().width() / 2, -size().height() / 2));
        m_rotate = true;
        switch (m_corner)
        {
        case 1:
            break;
        case 2:
            x = x + wd - hg;
            break;
        case 3:
            x = x + wd - hg;
            y = y - wd + hg;
            break;
        case 4:
            y = y - wd + hg;
            break;
        }
        move(x,y);
        resize(QSize(hg,wd));
        update();
    }
    else if (m_rotate)
    {
        setTransform(QTransform());
        m_rotate = false;
        switch (m_corner)
        {
        case 1:
            break;
        case 2:
            x = x + wd - hg;
            break;
        case 3:
            x = x + wd - hg;
            y = y - wd + hg;
            break;
        case 4:
            y = y - wd + hg;
            break;
        }
        move(x,y);
        resize(QSize(hg,wd));
        update();
    }
#else
    Q_UNUSED(aMessage);
    if (m_rotate)
    {
        m_rotate = false;
        setTransform(QTransform()
               .translate(size().width() / 2, size().height() / 2)
               .rotate(270, Qt::ZAxis)
               .translate(-size().width() / 2, -size().height() / 2));
    }
    else
    {
        m_rotate = true;
        setTransform(QTransform());
    }
#endif
}
