#include "mainwindow.h"
#include "ui_mainwindow.h"

/**
 * Constructor. Connects signals and slots for the user interface.
 *
 * @fn MainWindow
 * @param QWidget * parent - the parent widget.
 */
MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    qDebug() << "Starting Dicer";
    previousX = 0;
    previousY = 0;
    previousZ = 0;
    accelerometerPoller = 0;
    // Initialise the settings.
    settings = new QSettings("WillemLiu", "dicer");
    // We always start in landscape mode.
    landscape = true;
    if(settings->contains("Landscape"))
    {
        landscape = settings->value("Landscape").toBool();
    }
    setAttribute(Qt::WA_Maemo5PortraitOrientation, !landscape);

    ui->setupUi(this);
    connect(&dice, SIGNAL(signalDieAdded(const Die*)), this, SLOT(slotDieAdded(const Die*)));
    connect(&dice, SIGNAL(signalDieCleared()), this, SLOT(slotDiceCleared()));

    ui->actionShake->setChecked(settings->value("Accelerometer", false).toBool());
    on_actionShake_triggered();
}

/**
 * Destructor.
 *
 * @fn ~MainWindow
 */
MainWindow::~MainWindow()
{
    delete ui;
}

/**
 * Create and start the accelerometer thread and connect SLOTS.
 *
 * @fn startAccelerometer
 */
void MainWindow::startAccelerometer()
{
    if(accelerometerPoller == 0)
    {
        // Create AccelerometerPoller thread for polling the sensors.
        accelerometerPoller = new AccelerometerPoller(10);
        accelerometerPoller->start();
        connect(accelerometerPoller, SIGNAL(signalSensorReading(QAccelerometerReading*)), this, SLOT(slotSensorReading(QAccelerometerReading*)), Qt::QueuedConnection);
    }
}

/**
 * Stop and delete the accelerometer thread and disconnect used SLOTS.
 *
 * @fn stopAccelerometer
 */
void MainWindow::stopAccelerometer()
{
    if(accelerometerPoller != 0)
    {
        // Disconnect specified SIGNAL from all SLOTS.
        disconnect(accelerometerPoller, SIGNAL(signalSensorReading(QAccelerometerReading*)), 0, 0);
        accelerometerPoller->stop();
        while(accelerometerPoller->isRunning());
        accelerometerPoller->deleteLater();
        accelerometerPoller = 0;
    }
}

/**
 * Read sensor data.
 *
 * @fn slotSensorReading
 *
 * @param QAccelerometerReading * - The readings from the sensor.
 */
void MainWindow::slotSensorReading(QAccelerometerReading * reading)
{
    // later
    qreal x = reading->x();
    qreal y = reading->y();
    qreal z = reading->z();
    if((x-previousX) > 8 || (x-previousX) < -8 )
    {
        qDebug() << "Sensor x:" << x << "y:"<< y << "z:" << z;
        on_throwDicePushButton_clicked();
    }
    previousX = x;
    previousY = y;
    previousZ = z;
}

/**
 * Slot for receiving the signal fired from the Dice class.
 * The Dice class will fire the signal when a Die has been added.
 *
 * @fn slotDieAdded
 * @param Die const* die - the pointer to the added Die.
 */
void MainWindow::slotDieAdded(Die const* die)
{
    QVariant eyes(die->getEyes());
    qDebug() << "Die added:" << eyes.toString();
    QString diceText = ui->diceLineEdit->text();
    if(diceText.length() > 0)
    {
        diceText.append(" + ");
    }
    diceText.append(eyes.toString());
    ui->diceLineEdit->setText(diceText);
}

/**
 * Slot for receiving the signal fired from the Dice class.
 * The Dice class will fire the signal when all the Dice are
 * cleared.
 *
 * @fn slotDiceCleared
 */
void MainWindow::slotDiceCleared()
{
    qDebug() << "Dice cleared";
    ui->diceLineEdit->clear();
}

/**
 * Is called when the About menu item is triggered.
 *
 * @fn on_actionRotate_triggered
 */
void MainWindow::on_actionAbout_triggered()
{
    qDebug() << "About";
    QString aboutText;
    aboutText.append("Dicer (c) 2010\n\n");
    aboutText.append("Created by Willem Liu.\n");
    aboutText.append("Created with QtCreator.\n");
    QMessageBox::about(this, "Dicer", aboutText);
}

/**
 * Slot for receiving the signal from the Add Die button.
 *
 * @fn on_addDiePushButton_clicked
 */
void MainWindow::on_addDiePushButton_clicked()
{
    int eyes = ui->spinBox->value();
    qDebug() << "Add die of" << eyes;
    dice.slotAddDie(eyes);
}

/**
 * Slot for receiving the signal from the Clear Dice button.
 *
 * @fn on_clearDicePushButton_clicked
 */
void MainWindow::on_clearDicePushButton_clicked()
{
    qDebug() << "Clear dice";
    dice.slotClearDie();
}

/**
 * Slot for receiving the signal from the Throw Dice button. It will retrieve the
 * random result for each Die.
 *
 * @fn on_throwDicePushButton_clicked
 */
void MainWindow::on_throwDicePushButton_clicked()
{
    qDebug() << "Throw dice";
    QVector<Die* > diceVector = dice.getDice();
    int total = 0;
    QString outcomeText("");
    ui->diceOutcomeLineEdit->clear();
    foreach(Die * die, diceVector)
    {
        QVariant outcome(die->getOutcome());
        qDebug() << "Outcome:" << outcome;
        total += outcome.toInt();
        if(outcomeText.length() > 0)
        {
            outcomeText.append(" + ");
        }
        outcomeText.append(outcome.toString());
    }
    outcomeText.append("\nTotal: ");
    QVariant totalVariant(total);
    outcomeText.append(totalVariant.toString());
    ui->diceOutcomeLineEdit->setText(outcomeText);
}

/**
 * Is called when the Rotate menu item is triggered.
 *
 * @fn slotActionRotate
 * @param QAction* action - the action.
 */
void MainWindow::on_actionRotate_triggered()
{
    qDebug() << "Rotate";
    if(landscape)
    {
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        settings->setValue("Landscape", false);
        landscape = false;
    }
    else
    {
        setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
        settings->setValue("Landscape", true);
        landscape = true;
    }
}

void MainWindow::on_actionShake_triggered()
{
    bool accelerometer = ui->actionShake->isChecked();
    qDebug() << "Shake" << accelerometer;
    settings->setValue("Accelerometer", accelerometer);
    if(accelerometer)
    {
        startAccelerometer();
    }
    else
    {
        stopAccelerometer();
    }
}
