{$INCLUDE valkyrie.inc}
unit vluastate;
interface

uses typinfo, Variants, Classes, SysUtils, vrltools, vluatools, vutil;

type
  ELuaStateException = class(EException);


type
TLuaReferencedObject = interface['vluastate.luareferencedobject']
    function GetLuaIndex   : Integer;
    function GetID         : AnsiString;
    function GetProtoTable : AnsiString;
end;

const GLOBALSINDEX  = -10002;


type

{ TLuaState }

TLuaState = object
    constructor Init( State : Pointer );
    constructor ObjectInit( State : Pointer; out RefObject : TObject );
    constructor Create;
    procedure Reset;
    function StackSize : LongInt;
    procedure Error( const Message : AnsiString );
    procedure PopRaise( PopAmount : Integer; const Message : AnsiString );
    function ToString( Index : Integer ) : AnsiString; overload;
    function ToInteger( Index : Integer ) : Integer; overload;
    function ToFloat( Index : Integer ) : Single; overload;
    function ToBoolean( Index : Integer ) : Boolean; overload;
    function ToChar( Index : Integer ) : Char;
    function ToFlags( Index : Integer ) : TFlags;
    function ToVariant( Index : Integer ) : Variant;
    function ToObject( Index : Integer ) : TObject;
    function ToObjectOrNil( Index : Integer ) : TObject;
    function ToStringArray( Index : Integer ) : TAnsiStringArray;
    function ToCoord( Index : Integer ) : TCoord2D;
    function ToArea( Index : Integer ) : TArea;

    function ToString( Index : Integer; const DValue : AnsiString ) : AnsiString; overload;
    function ToInteger( Index : Integer; DValue : Integer ) : Integer; overload;
    function ToFloat( Index : Integer;   DValue : Single  ) : Single; overload;
    function ToBoolean( Index : Integer; DValue : Boolean ) : Boolean; overload;

    function IsNil( Index : Integer ) : Boolean;
    function IsNumber( Index : Integer ) : Boolean;
    function IsBoolean( Index : Integer ) : Boolean;
    function IsString( Index : Integer ) : Boolean;
    function IsTable( Index : Integer ) : Boolean;
    function IsObject( Index : Integer ) : Boolean;
    function IsCoord( Index : Integer ) : Boolean;
    function IsArea( Index : Integer ) : Boolean;

    function GetField( Index : Integer; const Key : Variant ) : Variant;
    function GetField( Index : Integer; const Key, DValue : Variant ) : Variant;
    procedure SetField( Index : Integer; const Key, Value : Variant );
    function RawGetField( Index : Integer; const Key : Variant ) : Variant;
    function RawGetField( Index : Integer; const Key, DValue : Variant ) : Variant;
    procedure RawSetField( Index : Integer; const Key, Value : Variant );

    procedure Push( Value : Single ); overload;
    procedure Push( const Value : AnsiString ); overload;
    procedure Push( Value : Boolean ); overload;
    procedure Push( Value : LongInt ); overload;
    procedure Push( Value : TLuaReferencedObject ); overload;
    procedure Push(const Args: array of const); overload;
    procedure PushCoord( Value : TCoord2D );
    procedure PushArea( Value : TArea );
    procedure PushNil;
    procedure PushVariant( Value : Variant );
    procedure PushUserdata( Value : Pointer );
    procedure PushReference( Value : Integer );
    procedure PushNewLuaObject( const Name : AnsiString; const ConstructorParams : array of const );
    procedure RegisterEnumValues( EnumTypeInfo : PTypeInfo; UpperCase : Boolean = True; Index : Integer = GLOBALSINDEX );
    function  RegisterNewObject( Obj : TObject; const Prototype : AnsiString ) : Integer;
    procedure SetPrototypeTable(Obj: TLuaReferencedObject);
    procedure UnRegisterObject( Obj : TLuaReferencedObject );
    procedure RegisterNewSubObject( Ref : Integer; Obj : TObject; const Field, Prototype : AnsiString );
    function  RunHook( Obj : TLuaReferencedObject; HookName : AnsiString; const Params : array of const ) : Variant;
    function  CallFunction( Name : AnsiString; const Params : array of const; idx : Integer = GLOBALSINDEX ) : Variant;
    destructor Done;
    procedure RegisterSubTable( Ref : Integer; const Name : AnsiString );
    procedure SubTableToStream( Obj : TLuaReferencedObject; const Name : AnsiString; OSt : TStream );
    procedure SubTableFromStream( Obj : TLuaReferencedObject; const Name : AnsiString; ISt : TStream );

protected
    procedure PushPrototypeTable( Obj : TLuaReferencedObject );
    procedure CloneTableWithPtr( const CloneName : AnsiString; Ptr : Pointer );
  protected
    FState      : Pointer;
    FOwner      : Boolean;
    FStartStack : Integer;
  public
    property Stack[ Index : Integer ] : Variant read ToVariant;
  end;

implementation

uses lua;

{ TLuaState }

constructor TLuaState.Init(State: Pointer);
begin
  FState      := State;
  FOwner      := False;
  FStartStack := lua_gettop( FState );
end;

constructor TLuaState.ObjectInit(State: Pointer; out RefObject: TObject);
begin
  FState      := State;
  FOwner      := False;
  RefObject   := ToObject(1);
  FStartStack := lua_gettop( FState );
end;

constructor TLuaState.Create;
begin
  FState := lua_open();
end;

procedure TLuaState.Reset;
begin
  lua_settop( FState, FStartStack );
end;

function TLuaState.StackSize: LongInt;
begin
  Exit( lua_gettop( FState ) );
end;

procedure TLuaState.Error(const Message: AnsiString);
begin
  luaL_error( FState, PChar(Message));
end;

procedure TLuaState.PopRaise(PopAmount: Integer; const Message: AnsiString);
begin
  lua_pop( FState, PopAmount );
  raise ELuaStateException.Create(Message);
end;

function TLuaState.ToString(Index: Integer): AnsiString;
begin
  Exit( lua_tostring( FState, Index ) );
end;

function TLuaState.ToInteger(Index: Integer): Integer;
begin
  Exit( lua_tointeger( FState, Index ) );
end;

function TLuaState.ToFloat(Index: Integer): Single;
begin
  Exit( lua_tonumber( FState, Index ) );
end;

function TLuaState.ToBoolean(Index: Integer): Boolean;
begin
  Exit( lua_toboolean( FState, Index ) );
end;

function TLuaState.ToVariant( Index : Integer ): Variant;
begin
  Exit( lua_tovariant( FState, Index ) );
end;

function TLuaState.ToChar( Index: Integer ): Char;
begin
  if (lua_type( FState, Index ) <> LUA_TSTRING) or
     (lua_objlen( FState, Index ) <> 1) then
    Error( 'Char expected as parameter '+IntToStr(Index)+'!' );
  Exit( lua_tostring( FState, Index )[1] );
end;

function TLuaState.ToFlags( Index: Integer ): TFlags;
begin
  ToFlags := [];
  if lua_istable( FState, Index ) then
  begin
    // table is in the stack at index 't'
    lua_pushnil( FState );  // first key */
    while (lua_next( FState, Index ) <> 0) do
    begin
       // uses 'key' (at index -2) and 'value' (at index -1) */
       Include( ToFlags, lua_tointeger( FState ,-1 ) );
       // removes 'value'; keeps 'key' for next iteration */
       lua_pop( FState, 1 );
    end;
  end;
end;

function TLuaState.ToObject(Index: Integer): TObject;
begin
  if not lua_istable( FState ,Index ) then Error( 'Object expected as parameter '+IntToStr(Index)+'!');
  lua_getfield( FState, Index, '__ptr' );
  if not lua_isuserdata( FState, -1 ) then Error( 'Object at parameter '+IntToStr(Index)+' has no __ptr!');
  ToObject := TObject( lua_touserdata( FState, -1 ) );
  lua_pop( FState, 1 );
end;

function TLuaState.ToObjectOrNil(Index: Integer): TObject;
begin
  if not lua_istable( FState ,Index ) then Exit( nil );
  lua_getfield( FState, Index, '__ptr' );
  if not lua_isuserdata( FState, -1 ) then Error( 'Object at parameter '+IntToStr(Index)+' has no __ptr!' );
  ToObjectOrNil := TObject( lua_touserdata( FState, -1 ) );
  lua_pop( FState, 1 );
end;

function TLuaState.ToStringArray(Index: Integer): TAnsiStringArray;
var arr : TAnsiStringArray;
    cnt : Word;
begin
  Index := lua_absindex( FState, Index );
  lua_pushnil( FState );

  cnt := 0;
  while lua_next( FState, Index ) <> 0 do
  begin
    SetLength(arr, cnt+1);
    arr[cnt] := lua_tostring( FState, -1 );
    lua_pop( FState , 1 );
    inc(cnt);
  end;

  Exit(arr);
end;

function TLuaState.ToCoord(Index: Integer): TCoord2D;
begin
  Exit( vlua_tocoord( FState, lua_absindex( FState, Index ) ) );
end;

function TLuaState.ToArea(Index: Integer): TArea;
begin
  Exit( vlua_toarea( FState, lua_absindex( FState, Index ) ) );
end;

function TLuaState.ToString(Index: Integer; const DValue: AnsiString
  ): AnsiString;
begin
  if lua_type( FState, Index ) = LUA_TSTRING
     then Exit( lua_tostring( FState, Index ) )
     else Exit( DValue );
end;

function TLuaState.ToInteger(Index: Integer; DValue: Integer): Integer;
begin
  if lua_type( FState, Index ) = LUA_TNUMBER
     then Exit( lua_tointeger( FState, Index ) )
     else Exit( DValue );
end;

function TLuaState.ToFloat(Index: Integer; DValue: Single): Single;
begin
  if lua_type( FState, Index ) = LUA_TNUMBER
     then Exit( lua_tonumber( FState, Index ) )
     else Exit( DValue );
end;

function TLuaState.ToBoolean(Index: Integer; DValue: Boolean): Boolean;
begin
  if lua_type( FState, Index ) = LUA_TBOOLEAN
     then Exit( lua_toboolean( FState, Index ) )
     else Exit( DValue );
end;

function TLuaState.IsNil(Index: Integer): Boolean;
begin
  Exit( lua_isnil( FState, Index ) or lua_isnone( FState, Index ) );
end;

function TLuaState.IsNumber(Index: Integer): Boolean;
begin
  Exit( lua_type( FState, Index ) = LUA_TNUMBER );
end;

function TLuaState.IsBoolean(Index: Integer): Boolean;
begin
  Exit( lua_type( FState, Index ) = LUA_TBOOLEAN );
end;

function TLuaState.IsString(Index: Integer): Boolean;
begin
  Exit( lua_type( FState, Index ) = LUA_TSTRING );
end;

function TLuaState.IsTable(Index: Integer): Boolean;
begin
  Exit( lua_type( FState, Index ) = LUA_TTABLE );
end;

function TLuaState.IsObject(Index: Integer): Boolean;
begin
  if lua_type( FState, Index ) <> LUA_TTABLE then Exit(False);
  lua_getfield( FState, Index, '__ptr' );
  IsObject := lua_isuserdata( FState, -1 );
  lua_pop( FState, 1 );
end;

function TLuaState.IsCoord(Index: Integer): Boolean;
begin
  Exit( vlua_iscoord( FState, lua_absindex( FState, Index ) ) );
end;

function TLuaState.IsArea(Index: Integer): Boolean;
begin
  Exit( vlua_isarea( FState, lua_absindex( FState, Index ) ) );
end;

function TLuaState.GetField(Index: Integer; const Key: Variant): Variant;
begin
  Index := lua_absindex( FState, Index );
  lua_pushvariant( FState, Key );
  lua_gettable( FState, Index );
  if lua_isnil( FState, -1 ) then PopRaise( 1, 'TLuaState.GetField - key '+Key+' not present in table!' );
  GetField := lua_tovariant( FState, -1 );
  lua_pop( FState, 1 );
end;

function TLuaState.GetField(Index: Integer; const Key, DValue: Variant
  ): Variant;
begin
  Index := lua_absindex( FState, Index );
  lua_pushvariant( FState, Key );
  lua_gettable( FState, Index );
  if lua_isnil( FState, -1 )
     then GetField := DValue
     else GetField := lua_tovariant( FState, -1 );
  lua_pop( FState, 1 );
end;

procedure TLuaState.SetField(Index: Integer; const Key, Value: Variant);
begin
  Index := lua_absindex( FState, Index );
  lua_pushvariant( FState, Key );
  lua_pushvariant( FState, Value );
  lua_settable( FState, Index );
end;

function TLuaState.RawGetField(Index: Integer; const Key: Variant): Variant;
begin
  Index := lua_absindex( FState, Index );
  lua_pushvariant( FState, Key );
  lua_rawget( FState, Index );
  if lua_isnil( FState, -1 ) then PopRaise( 1, 'TLuaState.GetField - key '+Key+' not present in table!' );
  RawGetField := lua_tovariant( FState, -1 );
  lua_pop( FState, 1 );
end;

function TLuaState.RawGetField(Index: Integer; const Key, DValue: Variant
  ): Variant;
begin
  Index := lua_absindex( FState, Index );
  lua_pushvariant( FState, Key );
  lua_rawget( FState, Index );
  if lua_isnil( FState, -1 )
     then RawGetField := DValue
     else RawGetField := lua_tovariant( FState, -1 );
  lua_pop( FState, 1 );
end;

procedure TLuaState.RawSetField(Index: Integer; const Key, Value: Variant);
begin
  Index := lua_absindex( FState, Index );
  lua_pushvariant( FState, Key );
  lua_pushvariant( FState, Value );
  lua_rawset( FState, Index );
end;

procedure TLuaState.Push(Value: Single);
begin
  lua_pushnumber( FState, Value );
end;

procedure TLuaState.Push(const Value: AnsiString);
begin
  lua_pushstring( FState, Value );
end;

procedure TLuaState.Push(Value: Boolean);
begin
  lua_pushboolean( FState, Value );
end;

procedure TLuaState.Push(Value: LongInt);
begin
  lua_pushinteger( FState, Value );
end;

procedure TLuaState.Push(Value: TLuaReferencedObject);
begin
  if Value = nil
     then lua_pushnil( FState )
     else lua_rawgeti( FState, LUA_REGISTRYINDEX, Value.GetLuaIndex );
end;

procedure TLuaState.PushNil;
begin
  lua_pushnil( FState );
end;

procedure TLuaState.PushVariant(Value: Variant);
begin
  lua_pushvariant( FState, Value );
end;

procedure TLuaState.PushUserdata(Value: Pointer);
begin
  lua_pushlightuserdata( FState, Value );
end;

procedure TLuaState.PushReference(Value: Integer);
begin
  lua_rawgeti( FState, LUA_REGISTRYINDEX, Value );
end;

procedure TLuaState.PushNewLuaObject(const Name: AnsiString;
  const ConstructorParams: array of const);
begin
  lua_getglobal( FState, Name );
  lua_pushstring( FState, 'new' );
  lua_rawget( FState, -2 );
  lua_insert( FState, -2 ); // swap table,function
  lua_pop( FState, 1 ); // pop table
  Push( ConstructorParams );
  if lua_pcall( FState, High( ConstructorParams ) + 1, 1, 0 ) <> 0 then
    PopRaise( 1, 'Lua constructor error : '+lua_tostring( FState, -1) );
  // leave the object on stack
end;

procedure TLuaState.RegisterEnumValues(EnumTypeInfo: PTypeInfo; UpperCase : Boolean; Index : Integer );
var Count, i : DWord;
    Name     : AnsiString;
    Value    : Integer;
begin
  Count := GetEnumNameCount( EnumTypeInfo );
  if Count > 0 then
  for i := 0 to Count-1 do
  begin
    Name  := GetEnumName( EnumTypeInfo, i );
    Value := GetEnumValue( EnumTypeInfo, Name );
    if UpperCase then Name := UpCase(Name);
    lua_pushstring( FState, Name );
    lua_pushinteger( FState, Value );
    lua_rawset( FState, Index );
  end;
end;

procedure TLuaState.Push(const Args: array of const);
var NArgs, i : LongInt;
begin
  NArgs := High(Args);
  if NArgs >= 0 then
  for i:=0 to NArgs do
  case Args[i].vtype of
    vtBoolean    : lua_pushboolean(FState, Args[i].VBoolean);
    vtChar       : lua_pushstring (FState, Args[i].VChar);
    vtInteger    : lua_pushinteger(FState, Args[i].VInteger);
    vtString     : lua_pushstring (FState, Args[i].VString^ );
    vtExtended   : lua_pushnumber (FState, Args[i].vextended^ );
    vtAnsiString : lua_pushstring (FState, Args[i].VAnsiString );
    vtVariant    : lua_pushvariant(FState, args[i].VVariant^ );
    vtInt64      : lua_pushnumber (FState, Args[i].VInt64^ );
    vtQWord      : lua_pushnumber (FState, Args[i].VQWord^ );
    vtObject     : begin
      lua_rawgeti(FState, LUA_REGISTRYINDEX, (Args[i].VObject as TLuaReferencedObject).GetLuaIndex );
      if not lua_istable( FState, -1 ) then PopRaise( i+2, 'Lua reference parameter #'+IntToStr(i+1)+' not found!');
    end;
  end;
end;

procedure TLuaState.PushCoord(Value: TCoord2D);
begin
  vlua_pushcoord( FState, Value );
end;

procedure TLuaState.PushArea(Value: TArea);
begin
  vlua_pusharea( FState, Value );
end;


procedure TLuaState.RegisterNewSubObject(Ref : Integer; Obj: TObject;
  const Field, Prototype: AnsiString);
begin
  lua_rawgeti( FState, LUA_REGISTRYINDEX, Ref);
  lua_pushstring( FState, Field );
  CloneTableWithPtr( Prototype, Obj );
  lua_rawset( FState, -3 );
  lua_pop( FState, 1 );
end;

function TLuaState.RunHook(Obj: TLuaReferencedObject; HookName: AnsiString;
  const Params: array of const): Variant;
var Count : DWord;
begin
  PushPrototypeTable( Obj );

  lua_pushstring( FState, HookName );
  lua_rawget( FState, -2 );
  if not lua_isfunction( FState, -1) then PopRaise( 3, Obj.GetProtoTable+'['+Obj.GetID+'].'+HookName+' not found!');

  lua_rawgeti(FState, LUA_REGISTRYINDEX, Obj.GetLuaIndex);
  if not lua_istable( FState, -1 ) then PopRaise( 4, 'Object not found!');

  Count := High( Params ) + 2;
  Push( Params );

  if lua_pcall( FState, Count, 1, 0 ) <> 0 then PopRaise( 3, 'Lua error : '+lua_tostring( FState, -1 ) );

  RunHook := lua_tovariant( FState, -1 );
  lua_pop( FState, 2 );
end;

function TLuaState.CallFunction( Name : AnsiString;
    const Params : array of const; idx : Integer = GLOBALSINDEX ) : Variant;
var Index : Integer;
begin
  Index := lua_absindex( FState, idx );
  lua_pushstring( FState, Name );
  lua_rawget( FState, Index );
  if not lua_isfunction( FState, -1) then PopRaise( 1, Name+' not found!');

  Push( Params );

  if lua_pcall( FState, High( Params ) + 1, 1, 0 ) <> 0 then PopRaise( 1, 'Lua error : '+lua_tostring( FState, -1) );
  CallFunction := lua_tovariant( FState, -1 );
  lua_pop( FState, 1 );
end;

function TLuaState.RegisterNewObject(Obj: TObject;
  const Prototype: AnsiString): Integer;
begin
  CloneTableWithPtr( Prototype, Obj );
  Exit( luaL_ref( FState, LUA_REGISTRYINDEX ) );
end;

procedure TLuaState.RegisterSubTable(Ref : Integer; const Name: AnsiString);
begin
  lua_rawgeti( FState, LUA_REGISTRYINDEX, Ref);
  lua_pushstring( FState, Name );
  lua_newtable( FState );
  lua_rawset( FState, -3 );
  lua_pop( FState, 1 );
end;

procedure TLuaState.SubTableToStream(Obj: TLuaReferencedObject;
  const Name: AnsiString; OSt: TStream);
begin
  lua_rawgeti( FState, LUA_REGISTRYINDEX, Obj.GetLuaIndex );
  lua_getfield( FState, -1, PChar(Name) );
  if not lua_istable( FState, -1 ) then PopRaise( 1, Name+' is not a valid table!');
  lua_tabletostream( FState, -1, Ost );
  lua_pop( FState, 2 );
end;

procedure TLuaState.SubTableFromStream(Obj: TLuaReferencedObject;
  const Name: AnsiString; ISt: TStream);
begin
  lua_rawgeti( FState, LUA_REGISTRYINDEX, Obj.GetLuaIndex );
  lua_getfield( FState, -1, PChar(Name) );
  if not lua_istable( FState, -1 ) then PopRaise( 1, Name+' is not a valid table!');
  lua_tablefromstream( FState, -1, Ist );
  lua_pop( FState, 2 );
end;

procedure TLuaState.SetPrototypeTable(Obj: TLuaReferencedObject);
begin
  Push( Obj );
  lua_pushstring( FState, 'proto' );
  PushPrototypeTable( Obj );
  lua_rawset( FState, -3 );
  lua_pop( FState, 1 );
end;

procedure TLuaState.UnRegisterObject(Obj: TLuaReferencedObject);
begin
  lua_rawgeti( FState, LUA_REGISTRYINDEX, Obj.GetLuaIndex );
  lua_pushstring( FState, '__ptr' );
  lua_pushboolean( FState, False );
  lua_rawset( FState, -3 );
  lua_pop( FState, 1 );
  luaL_unref( FState, LUA_REGISTRYINDEX, Obj.GetLuaIndex );
end;

destructor TLuaState.Done;
begin
  if FOwner then lua_close( FState );
end;

procedure TLuaState.PushPrototypeTable(Obj: TLuaReferencedObject);
begin
  lua_getglobal( FState, Obj.GetProtoTable );
  if not lua_istable( FState, -1 ) then PopRaise( 1, Obj.GetProtoTable+' is not a valid table!');
  lua_pushstring( FState, Obj.GetID );
  lua_gettable( FState, -2 );
  if not lua_istable( FState, -1 ) then PopRaise( 2, Obj.GetProtoTable+'['+Obj.GetID+'] is not a valid table!');
  lua_insert( FState, -2 ); // swap
  lua_pop( FState, 1 ); // free base table
end;

procedure TLuaState.CloneTableWithPtr(const CloneName: AnsiString;
  Ptr : Pointer);
begin
  lua_getglobal( FState, CloneName );
  lua_shallowcopy( FState, -1 );
  lua_pushlightuserdata( FState, Ptr );
  lua_setfield( FState, -2, '__ptr' );

  lua_getmetatable( FState, -2 );
  lua_setmetatable( FState, -2 );

  lua_insert( FState, -2 );
  lua_pop( FState, 1 );
end;

end.

