{$INCLUDE valkyrie.inc}
// @abstract(Debugging unit for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(May 7, 2004)
// @lastmod(Jan 14, 2006)
//
// Core debuging engine for Valkyrie. Used by almost all
// files in the Valkyrie library. Implements Logging, Errors,
// and Warnings. All debug data is written to log.txt.
//
// vdebug also defines the @link(Debug) boolean constant. If set to
// True a lot of additional Log informations are generated.
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>

unit vdebug;
interface

type TLogLevel = ( LCRITICAL, LERROR, LWARNING, LINFO, LDEBUG, LTRACE );

const
// The global Debug
    Debug : Boolean = True;
// The global LogLevel
    LogLevel : TLogLevel = LINFO;

// Calls a critical error, and halts execution.
procedure CritError(const CritErrorString : Ansistring);
// Issues a warning.
procedure Warning  (const WarningString   : Ansistring);
// Logs a string into the logfile.
procedure Log      (const LogString       : Ansistring);
// Calls a critical error, and halts execution. Parameters handled via vutil.VFormat.
procedure CritError(const CritErrorString : Ansistring; const param : array of Const);
// Issues a warning. Parameters handled via vutil.VFormat.
procedure Warning  (const WarningString   : Ansistring; const param : array of Const);
// Logs a string into the logfile. Parameters handled via vutil.VFormat.
procedure Log      (const LogString       : Ansistring; const param : array of Const);


// Logs a string into the logfile.
procedure Log      (Level : TLogLevel; const LogString       : Ansistring);
// Logs a string into the logfile. Parameters handled via vutil.VFormat.
procedure Log      (Level : TLogLevel; const LogString       : Ansistring; const param : array of Const);


implementation
uses vutil, vconsole;

var LogFile : Text;

//const LogName : array[TLogLevel] of string = (
//  'CRITICAL', 'ERROR', 'WARNING', 'INFO', 'DEBUG', 'TRACE'
//);


procedure CritError(const CritErrorString : Ansistring);
begin
  Writeln(LogFile,'Critical Error: ',CritErrorString);
  Writeln('Critical Error: '+CritErrorString);
  Flush(LogFile);
  Readln;
  Halt(0);
end;

procedure Warning  (const WarningString   : Ansistring);
begin
  Writeln(LogFile,'Warning: ',WarningString);
  if Console <> nil then
  begin
    Console.Print('Warning: '+WarningString);
    Console.Call;
  end
  else
  begin
    Writeln('Warning: '+WarningString);
    Readln;
  end;
  Flush(LogFile);
end;

procedure Log      (const LogString       : Ansistring);
begin
  if LogLevel < LINFO then Exit;
  Log( LINFO, LogString );
end;

procedure CritError(const CritErrorString: Ansistring; const param: array of const);
begin
  CritError(VFormat(CritErrorString,param));
end;

procedure Warning(const WarningString: Ansistring; const param: array of const);
begin
  Warning(VFormat(WarningString,param));
end;

procedure Log(const LogString: Ansistring; const param: array of const);
begin
  if LogLevel < LINFO then Exit;
  Log( LINFO, VFormat(LogString,param));
end;

procedure Log(Level : TLogLevel; const LogString       : Ansistring);
begin
  if Level > LogLevel then Exit;
  if Console <> nil then Console.Print(LogString);
  Writeln(LogFile,LogString);
  Flush(LogFile);
end;

procedure Log(Level : TLogLevel; const LogString: Ansistring; const param: array of const);
begin
  if Level > LogLevel then Exit;
  Log(LogLevel,VFormat(LogString,param));
end;


initialization

Assign(LogFile,'log.txt');
Rewrite(LogFile);
Writeln(LogFile,'--- Logging start : '+TimeStamp+' ---');

finalization
Writeln(LogFile,'--- Logging end   : '+TimeStamp+' ---');
Close(LogFile);

end.

{ $Log
  -- 14.I.2006 Added AnsiString support
  -- Created 2.06.2003
}
