{$MODE OBJFPC}
{ $DEFINE SOUND}

unit rlui;

interface

uses SysUtils, vlua, vsystem, vui, vtextui, vrltools, rlglobal, rlthing, vini,
     vutil, vinput, rlnpc, rlplayer, vtextut, vds, rlitem,
     rlconfig{$IFDEF SOUND},rlsound {$ENDIF};

type TWindow   = TTextUIWindow;
     TTWindow  = TTextUIWindow;
     TContent  = TTextUIContent;
     TMap      = TTextUIMap;
     TSeparator= TTextUISeparator;
     TMessages = TTextUIMessages;
     TCmdSet   = set of byte;

type TMenuCallback = procedure (choice: byte) of object;

type TMenu = class(TByteTextUIMenu)
    FScroll : byte;
    constructor Create(newParent: TUIElement; TerminationKeys: TKeyFilter; cback: TMenuCallback; initial: byte = 1);
    procedure Add(name : Ansistring; active : boolean = True; color : byte = 0);
    procedure Draw; override;
end;

type TUIStatusArea = class(TUIArea)
  s1, s2, s3 : string;
  procedure Draw; override;
end;

type TPlotWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newDimensions: TRect; newTitle: AnsiString; nText: AnsiString);
  procedure Draw; override;
  procedure Run;
  destructor Destroy;
  private
  Text: TStringArray;
  Count: Word;
end;

type TInventoryWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newTitle: AnsiString = 'Inventory');
  procedure Draw; Override;
  procedure Run;
  destructor Destroy; override;
private
  S : TSeparator;
  Chooser : TMenu;
  InvMode : (Inventory,Equipment);
  procedure EqCallback(Choice : byte);
  procedure InvCallback(Choice : byte);
end;

type TJournalWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newTitle: AnsiString = 'Journal');
  procedure Draw; Override;
  procedure Run;
  destructor Destroy; override;
  private
  Chooser : TMenu;
end;

type TShopWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newTitle: AnsiString = ''; cback: TTextMenuCallback = nil);
  procedure Draw; Override;
  Procedure Add(Content : string; Color: byte = LightGray);
  function Run: Word;
  destructor Destroy; override;
private
  Chooser : TMenu;
  Count : byte;
end;

type TCharWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newTitle: AnsiString = 'Character');
  procedure Draw; Override;
  procedure Run;
  destructor Destroy; override;
end;

type TSpellWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newTitle: AnsiString = 'Spells');
  procedure Draw; override;
  procedure Run;
  destructor Destroy; override;
end;

type TSkillWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newTitle : AnsiString = 'Choose Skill');
  function Run : word;
  procedure Prepare;
  destructor destroy; override;
  private
  Chooser : TMenu;
end;

type TQSWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newTitle : AnsiString = 'Quick slots');
  function Run : word;
  procedure Prepare;
  destructor destroy; override;
  private
  Chooser : TMenu;
end;


type TItemWindow = class(TWindow)
  constructor Create(newParent: TUIElement; newItem: TItem ;newTitle : AnsiString = '');
  procedure Draw; override;
  procedure Run;
private
  Item: TItem;
end;

type

{ TMainUIArea }

TMainUIArea = class(TUIArea)
  procedure Draw; override;
end;


{TGameUI}

type TGameUI = class(TSystem, IMap)
  public
    {$IFDEF SOUND}Sound    : TRLSound;{$ENDIF}
    Messages : TUIMessages;
    VUI      : TUIElement;
    LeftPanel: TWindow;
    RightPanel:TWindow;
    Player   : TPlayer;
    Map      : TMap;
    Config     : TDiabloConfig;
    constructor Create(); override;
    function getASCII(x,y : Longint) : Word;
    function getColor(x,y : LongInt) : DWord;
    function getTile(x,y : LongInt; layer : byte = 0) : DWord;
    procedure ClassSelectCallback(Choice: byte);
    procedure SelectKlass();
    destructor Destroy(); override;
    procedure Outro();
    procedure Intro();
    function MainMenu: byte;
    procedure MainMenuCallback(Choice: byte);
    procedure Draw();
    procedure ShowMortem();
    procedure ReadConfig();
    procedure Prepare();
    // Marks the given tile with specified glyph
    procedure MarkCell( c : TCoord2D; atr : byte; chr: char );
    //Places the cursor over the specified cell
    procedure MoveCursor( c : TCoord2D );
    //focuses onto the specified cell
    procedure Focus( c : TCoord2D );
    //wipe all marks
    procedure MarkClear();
    //adds a new message
    procedure Msg(MsgText : AnsiString);
    //creates a screenshot
    procedure ScreenShot(BBCode: boolean = false);
    //waits for Enter key
    procedure PressEnter();
    //front-end for TInput.GetCommand
    function GetCommand(valid : TCmdSet = []) : byte;
    //front-end for TInput.GetKey
    function GetKey(valid : TKeyFilter = []) : byte;
    //delays for time given in ms
    procedure Delay(ms : word);
    //Put a message onto status bar
    procedure UpdateStatus(c: TCoord2D);
    procedure UpdateStatus(STarget : TThing);
    //reviews last messages
    procedure ShowRecent;
    //shows game manual
    procedure ShowManual;
    //Plot text window
    Procedure PlotText(const Text: AnsiString);
    //Draw explosion
    procedure Explosion( c : TCoord2D; Color, Range : Byte; DrawDelay: Word = 50);
    //Resizes the map matching opened panels
    procedure AdjustMap;
    //Sound procedures wrapping
    procedure PlayMusic(sID: Ansistring);
    procedure PlaySound(sID: Ansistring; x : Integer = 0; y : Integer = 0);
    procedure HaltSound();
    procedure LoadSound(INIFile: TINI);
    procedure Mute();
    procedure Unmute();
    procedure SetMusicVolume(Volume : Byte);
    procedure SetSoundVolume(Volume : Byte);
    procedure SetListenerPosition(x : Integer; y : Integer);
    class procedure RegisterLuaAPI( Lua : TLua );
  private
    tempstr : array[1..6] of string[20];
    CallbackArea1 : TUIArea;
    CallbackArea2 : TUIArea;
    StatusArea : TUIStatusArea;
    MapArea    : TUIArea;
    MsgArea    : TUIArea;
    AnimCount  : DWord;
end;

{ TTalkWindow }

TTalkWindow = class(TWindow)
  constructor Create( const nIntro : string );
  procedure Add( const Option : string; active : boolean = true );
  function Run : Byte;
  destructor Destroy; override;
private
  Chooser : TMenu;
end;

function InputCommandParser(const Token : string) : byte;
function CommandDirection(Command: byte): TDirection;

var UI: TGameUI = nil;

implementation

uses voutput, rlshopinv, rllua, rlgame, vsystems, vtinput, vtoutput,
     rlcell, vmath, rllevel, vluaconfig;

function CommandDirection(Command : byte) : TDirection;
begin
  case Command of
    COMMAND_WALKWEST  : CommandDirection.Create(4);
    COMMAND_WALKEAST  : CommandDirection.Create(6);
    COMMAND_WALKNORTH : CommandDirection.Create(8);
    COMMAND_WALKSOUTH : CommandDirection.Create(2);
    COMMAND_WALKNW    : CommandDirection.Create(7);
    COMMAND_WALKNE    : CommandDirection.Create(9);
    COMMAND_WALKSW    : CommandDirection.Create(1);
    COMMAND_WALKSE    : CommandDirection.Create(3);
    COMMAND_WAIT      : CommandDirection.Create(5);
    else CommandDirection.Create(0);
  end;
end;

function InputCommandParser(const Token : string) : byte;
begin
  if Token = 'WALKNORTH'   then Exit(COMMAND_WALKNORTH);
  if Token = 'WALKSOUTH'   then Exit(COMMAND_WALKSOUTH);
  if Token = 'WALKEAST'    then Exit(COMMAND_WALKEAST);
  if Token = 'WALKWEST'    then Exit(COMMAND_WALKWEST);
  if Token = 'WALKNE'      then Exit(COMMAND_WALKNE);
  if Token = 'WALKSE'      then Exit(COMMAND_WALKSE);
  if Token = 'WALKNW'      then Exit(COMMAND_WALKNW);
  if Token = 'WALKSW'      then Exit(COMMAND_WALKSW);
  if Token = 'WAIT'        then Exit(COMMAND_WAIT);
  if Token = 'PICKUP'      then Exit(COMMAND_PICKUP);
  if Token = 'DROP'        then Exit(COMMAND_DROP);
  if Token = 'INVENTORY'   then Exit(COMMAND_INVENTORY);
  if Token = 'EQUIPMENT'   then Exit(COMMAND_EQUIPMENT);
  if Token = 'ACT'         then Exit(COMMAND_ACT);
  if Token = 'LOOK'        then Exit(COMMAND_LOOK);
  if Token = 'CAST'        then Exit(COMMAND_CAST);
  if Token = 'FIRE'        then Exit(COMMAND_FIRE);
  if Token = 'USE'         then Exit(COMMAND_USE);
  if Token = 'PLAYERINFO'  then Exit(COMMAND_PLAYERINFO);
  if Token = 'SAVE'        then Exit(COMMAND_SAVE);
  if Token = 'RUNMODE'     then Exit(COMMAND_RUNMODE);
  if Token = 'STAIRSDOWN'  then Exit(COMMAND_STAIRSDOWN);
  if Token = 'STAIRSUP'    then Exit(COMMAND_STAIRSUP);
  if Token = 'JOURNAL'     then Exit(COMMAND_JOURNAL);
  if Token = 'QUICKSLOT'   then Exit(COMMAND_QUICKSLOT);
  if Token = 'SPELLBOOK'   then Exit(COMMAND_SPELLBOOK);
  if Token = 'QUICKSKILL'  then Exit(COMMAND_SPELLS);
  if Token = 'SAVE'        then Exit(COMMAND_SAVE);
  if Token = 'SOUNDVOLUP'  then Exit(COMMAND_SOUNDVOLUP);
  if Token = 'SOUNDVOLDN'  then Exit(COMMAND_SOUNDVOLDN);
  if Token = 'MUSICVOLUP'  then Exit(COMMAND_MUSICVOLUP);
  if Token = 'MUSICVOLDN'  then Exit(COMMAND_MUSICVOLDN);
  Exit(0);
end;

{ TMenu }

constructor TMenu.Create(newParent: TUIElement; TerminationKeys: TKeyFilter; cback: TMenuCallback; initial: byte = 1);
var AD: TRect;
begin
  inherited Create(newParent, TerminationKeys, cback, false, initial);
  FScroll:=0;
  if ParentIsArea then
  begin
    AD:=TUIArea(newParent).getAbsoluteDimensions;
    Dimensions.X2:=AD.W;
    Dimensions.Y2:=AD.H;
  end;
end;

procedure TMenu.Add(name: Ansistring; active: boolean; color : byte);
begin
  inherited Add(name,Count+1,active,color);
  Dec(Dimensions.Y2);
end;

procedure TMenu.Draw;
var AD: TRect;
    cn: DWORD;
    clr:byte;
    text:string;
begin
  AD:=getAbsoluteDimensions;
  Output.ClearRect(AD.x1,AD.y1,AD.x2,AD.y2);
  if ad.h > count then
    FScroll:=0
  else
  begin
    FScroll:=max(0, choice - AD.h div 2 );
    if FScroll+ad.h > count then FScroll:=count - ad.h;
  end;
  for cn:=FScroll+1 to min(ad.h+FScroll, Count) do
    with Fields[cn] do
    begin
      clr:=Style^.MainColor;
      text:=name;
      if ( not Active ) then
        if cn = choice then clr:=Style^.WrongColor
                       else clr:=Style^.InactiveColor
        else if cn = choice then clr:=Style^.BoldColor
                       else clr:=Color;
        Output.DrawString(AD.x1, AD.y1 + cn - 1 - FScroll, clr, text)
    end;
end;

{ TGameUI }

constructor TGameUI.Create();
begin
  inherited Create();
  {$IFDEF SOUND}Systems.Add(Sound, TRLSound.Create);{$ENDIF}
  Systems.Add(Output, TTextmodeOutput.Create);
  Systems.Add(Input, TTextModeInput.Create);
  VUI := TMainUIArea.Create(nil, NewRectXY(1,1,80,25));
  Output.UI := VUI;
  AnimCount := 0;
end;

procedure TGameUI.Draw;
begin
  Output.Clear;
  Map.setXY(Player.Position.X, Player.Position.Y);
end;

procedure TGameUI.MarkCell( c : TCoord2D; atr : byte; chr: char );
begin
  Map.Mark(c.x, c.y, chr, atr)
end;

procedure TGameUI.MoveCursor( c : TCoord2D );
begin
  Output.MoveCursor(Map.ScreenX(C.X), Map.ScreenY(C.Y));
end;

procedure TGameUI.Focus(c :  TCoord2D);
begin
  Map.setXY(c.X, c.Y);
end;

procedure TGameUI.MarkClear;
begin
  Map.Update;
end;

procedure TGameUI.Delay(ms : word);
begin
  Sleep(ms);
end;

procedure TGameUI.ShowRecent;
var View  : TTextViewer;
    Arr   : TStringArray;
    Count : DWord;
begin
  Arr  := TStringArray.Create;
  for Count := Min(2000-1,Messages.Size) downto 0 do
     if Messages.Get(Count) <> '' then
       Arr.Push(Messages.Get(Count));
  Output.UI := nil;
  View := TTextViewer.Create('Past messages viewer');
  View.SetArray( Arr );
  View.Recent;
  View.Run;
  Output.UI := VUI;
  FreeAndNil(Arr);
  FreeAndNil(View);
end;

procedure TGameUI.ShowManual;
begin
  Output.UI := nil;
  if not FileExists('manual.txt') then Exit;
  with TTextViewer.Create('@lDiabloRL Manual (manual.txt)') do
  begin
    Load('manual.txt');
    Run;
    Free;
  end;
  Output.UI := VUI;
end;


procedure TUIStatusArea.Draw;
var cnt : byte;
var dc : byte;

  function DurColor(Dur,MaxDur : Word) : Byte;
  begin
    if Dur >= (MaxDur div 4) then DurColor := LightGray
    else if Dur <= Max(MaxDur div 6,1) then DurColor := LightRed else
      DurColor := Brown;
  end;

  procedure DrawOrb(Value, ValueMax : Word; const color,x : Byte);
  var percent : Integer;
  begin
    if ValueMax = 0 then percent:=0 else
      percent := (Value * 100) div ValueMax;
    case percent of
      0..75   : Output.DrawString(x+1,22,color,'....');
      76..84  : Output.DrawString(x+1,22,color,'----');
      85..93  : Output.DrawString(x+1,22,color,'====');
      94..255 : Output.DrawString(x+1,22,color,'####');
    end;
    case percent of
      0..50   : Output.DrawString(x,23,color,'......');
      51..58  : Output.DrawString(x,23,color,'------');
      59..66  : Output.DrawString(x,23,color,'======');
      67..255 : Output.DrawString(x,23,color,'######');
    end;
    case percent of
      0..25   : Output.DrawString(x,24,color,'......');
      26..33  : Output.DrawString(x,24,color,'------');
      34..42  : Output.DrawString(x,24,color,'======');
      43..255 : Output.DrawString(x,24,color,'######');
    end;
    case percent of
      0..25   : Output.DrawString(x,24,color,'......');
      26..33  : Output.DrawString(x,24,color,'------');
      34..42  : Output.DrawString(x,24,color,'======');
      43..255 : Output.DrawString(x,24,color,'######');
    end;
    case percent of
      0..6    : Output.DrawString(x+1,25,color,'....');
      7..12   : Output.DrawString(x+1,25,color,'----');
      13..19  : Output.DrawString(x+1,25,color,'====');
      20..255 : Output.DrawString(x+1,25,color,'####');
    end;
  end;

begin
  Output.DrawString(1,23,DarkGray,Padded('', 80, '-'));
  if UI.Player = nil then exit;
  with UI.Player do
  begin
    DrawOrb(max(HP,0),getLife,Red,4);
    if nfManaShield in flags then
      DrawOrb(MP,getMana,LightBlue,72)
    else
      DrawOrb(MP,getMana,Blue,72);
    Output.ClearRect(10,24,70,25);
    Output.LeftDrawString(70,23,LightGray,' '+Game.Level.Name+' ');
    if LevelUp then
    begin
      Output.DrawString(5,20,Red,'[@y+@r]');
      Output.DrawString(3,21,White,'Level Up!');
    end;
    if Eq[slotHead] <> nil then
      with Eq[slotHead] do begin
        dc := DurColor(Dur,DurMax);
        if dc <> LightGray then
          Output.DrawChar(79,15,dc,#127);
      end;
    if Eq[slotTorso] <> nil then
      with Eq[slotTorso] do begin
        dc := DurColor(Dur,DurMax);
        if dc <> LightGray then
          Output.DrawString(79,16,dc,#219);
      end;
    if Eq[slotRHand] <> nil then
      with Eq[slotRHand] do begin
        dc := DurColor(Dur,DurMax);
        if dc <> LightGray then
        begin
          Output.DrawChar(78,15,dc,#179);
          Output.DrawChar(78,16,dc,#197);
        end;
      end;
    if Eq[slotLHand] <> nil then
      with Eq[slotLHand] do begin
        dc := DurColor(Dur,DurMax);
        if dc <> LightGray then
          if Eq[SlotLHand].IType = TYPE_WEAPON then
          begin
            Output.DrawChar(78,15,dc,#179);
            Output.DrawChar(78,16,dc,#197);
          end
          else
          begin
            Output.DrawChar(80,15,dc,'_');
            Output.DrawChar(80,16,dc,'U');
          end;
        end;
      Output.DrawString(11,23,DarkGray,' [');
      Output.DrawString(12+ITEMS_QS+1,23,DarkGray,'] ');
      for cnt := 1 to ITEMS_QS do
        if QuickSlots[cnt] = nil then Output.DrawChar(12+cnt,23,LightGray,'.')
                             else Output.DrawChar(12+cnt,23,QuickSlots[cnt].Color,QuickSlots[cnt].Pic);
      if Spell > 0 then
        if Spell = 254 then
        begin
          if UI.Player.Eq[SlotRHand] = nil then
            Spell := 0
          else if UI.Player.Eq[SlotRHand].Charges = 0 then
            Spell := 0
          else
            Output.DrawString(12,24,DarkGray,UI.Player.Eq[SlotRHand].getName(SpellName))
        end
        else
        with TLuaTable.Create(Game.Lua,'spells',Spell) do
        try
          Output.DrawString(12,24,DarkGray,getString('name'));
        finally
          Free;
        end;
  end;
  if S3 <> '' then
    Output.CenterDrawString(40,23,DarkGray, S3);
  Output.CenterDrawString(40,24,LightGray, S1);
  Output.CenterDrawString(40,25,LightGray, S2);
end;

procedure TGameUI.UpdateStatus(c: TCoord2D);
begin
  with StatusArea do
  begin
    s1 := ''; s2 := ''; s3 := '';
    if not Game.Level.isVisible(c) then exit;
    if not(Game.Level.CellCheck(Player.Target,[cfNoMonsters])) then
      UpdateStatus(Game.Level.Map[c.x,c.y].NPC)
    else if not(Game.Level.CellCheck(c,[cfNoItems])) then
      UpdateStatus(Game.Level.Map[c.x,c.y].Item)
    else
      s1 := Cells[Game.Level.Map[c.x,c.y].Cell].name;
  end;
end;

procedure TGameUI.UpdateStatus(STarget : TThing);
begin
  with StatusArea do
  begin
    s1 := ''; s2 := ''; s3 := '';
    if STarget = nil then Exit;
    if (STarget is TNPC) then
      if (nfInvisible in STarget.Flags) then
        s1 := Cells[Game.Level.Map[STarget.Position.x,STarget.Position.y].Cell].name
      else
      with STarget as TNPC do
      begin
        s1 := name;
        if isPlayer then exit;
        if AI in AIMonster+[AIGolem] then
        begin
          s3 := '[@r'+Padded('',((hp*10) div hpmax), '#')+'@d'+Padded('',10-((hp*10) div hpmax),'.')+']';
          if Player.KillData[Name] > 0 then
            s2 := 'Total kills : ' + inttostr(Player.KillData[Name]);
          if Player.KillData[Name] > 15 then
          if [nfFireImmune,nfFireResist,nfLightningImmune,nfLightningResist,nfMagicImmune,nfMagicResist]*flags<>[] then
          begin
            s2:=s2+'   (@R';
            if nfFireImmune in flags then s2:=s2+'I' else
            if nfFireResist in flags then s2:=s2+'R' else s2:=s2+'-';
            s2:=s2+'@y';
            if nfLightningImmune in flags then s2:=s2+'I' else
            if nfLightningResist in flags then s2:=s2+'R' else s2:=s2+'-';
            s2:=s2+'@B';
            if nfMagicImmune in flags then s2:=s2+'I' else
            if nfMagicResist in flags then s2:=s2+'R' else s2:=s2+'-';
            s2:=s2+'@>)';
          end
          else s2:=s2+'   (@R-@y-@B-@>)';
          if Player.KillData[Name] > 30 then
            s2:=s2+'   HP: '+inttostr(Game.Lua.IndexedTable['npcs',tid,'hpmin'])+'-'+
                           inttostr(Game.Lua.IndexedTable['npcs',tid,'hpmax']);
        end;
      end;

    if STarget is TItem then
    with STarget as TItem do
    begin
      if GetName(Status1) = '' then
        s1 := Output.ColorToVCode(Color)+GetName(PlainName)
      else
      begin
        s3 := '['+Output.ColorToVCode(Color)+GetName(PlainName)+'@>]';
        s1 := Output.ColorToVCode(Color)+GetName(Status1);
      end;
      s2 := Output.ColorToVCode(Color)+GetName(Status2);
    end;
  end;
end;


function TGameUI.getASCII(x,y : Longint) : Word;

var Coord : TCoord2D;
const black = ord(' ') + LightGray shl 8;

  function Color(Value : Byte) : Byte;
  begin
    if Value <= 16 then Exit(Value);
    case Value of
      ColorPortal    : case AnimCount mod 4 of 0 : Exit(Blue); 1,3 : Exit(LightBlue); 2 : Exit(White); end;
      ColorRedPortal : case AnimCount mod 4 of 0 : Exit(Red); 1,3 : Exit(LightRed); 2 : Exit(White); end;
      ColorFire      : case AnimCount mod 4 of 0 : Exit(Red); 1,3 : Exit(LightRed); 2 : Exit(Yellow); end;
    else Exit(Red)
    end;
  end;

begin
  Coord.Create(x,y);
  with Game.Level do
  begin
    if not ProperCoord(Coord) then Exit(Black);
    if not (isVisible(Coord)) then
      if (nfInfravision in Player.Flags) and (Map[x,y].NPC <> nil) then
        if not(nfInvisible in Map[x,y].NPC.Flags) then
          exit(ord(Map[x,y].NPC.Pic) + LightRed shl 8);
    if not IsExplored(Coord) then Exit(Black);
    if isVisible(Coord) then
    begin
      if (Map[x,y].NPC <> nil) then
        if not(nfInvisible in Map[x,y].NPC.Flags) then
          if nfStatue in Map[x,y].NPC.Flags then
            exit(ord(Map[x,y].NPC.Pic) + DarkGray shl 8)
          else
            exit(Map[x,y].NPC.Picture);
      if (Map[x,y].Item <> nil) then
        exit(Map[x,y].Item.Picture);
      exit(ord(Cells[Map[x,y].Cell].Pic) + Color(Cells[Map[x,y].Cell].Color) shl 8);
    end
    else
      exit(ord(Cells[Map[x,y].Cell].Pic) + DarkGray shl 8);
  end;
end;

function TGameUI.getColor(x,y : LongInt) : DWord;
begin

end;

function TGameUI.getTile(x,y : LongInt; layer : byte = 0) : DWord;
begin

end;

procedure TGameUI.Msg(MsgText : AnsiString);
begin
  Messages.Add(MsgText);
end;

procedure TGameUI.Explosion( c : TCoord2D; Color, Range : byte; DrawDelay: Word = 50);
var ax, ay, d, c1, c2, c3, Step : byte;
    vis : boolean;
begin
  case Color of
    Blue    : begin c1 := Blue;     c2 := LightBlue;  c3 := White;  end;
    Magenta : begin c1 := Magenta;  c2 := Red;        c3 := Blue;   end;
    Green   : begin c1 := Green;    c2 := LightGreen; c3 := White;  end;
    LightRed: begin c1 := LightRed; c2 := Yellow;     c3 := White;  end;
      else    begin c1 := Red;      c2 := LightRed;   c3 := Yellow; end;
  end;
  for Step := 1 to Range + 3 do
  begin
    Vis := false;
    Draw();
    //Display the explosion
    for ax := Max( 1, c.x-Range ) to Min(MapSizeX, c.x + Range) do
      for ay := Max( 1, c.y-Range ) to Min(MapSizeY, c.y + Range) do
      begin
        if Game.Level.isVisible(NewCoord2D(ax, ay)) then Vis := true else continue;
        d := Distance(ax, ay, c.x, c.y);
        if d > Range then Continue;
        if d = Step   then UI.MarkCell(NewCoord2D(ax, ay), c1, '*');
        if d = Step-1 then UI.MarkCell(NewCoord2D(ax, ay), c2, '*');
        if d = Step-2 then UI.MarkCell(NewCoord2D(ax, ay), c3, '*');
        if d = Step-3 then UI.MarkCell(NewCoord2D(ax, ay), c1, '*');
      end;
    if Vis then
    begin
      Output.Update;
      Delay(DrawDelay);
      MarkClear();
    end;
    Output.Update;
  end;
end;


procedure TGameUI.Prepare;
begin
  Output.Clear;
  MapArea := TUIArea.Create(VUI,NewRectXY(1,3,80,22));
  MsgArea := TUIArea.Create(VUI,NewRectXY(2,1,79,2));
  Messages := TMessages.Create(MsgArea, 1000, [MSG_CONTINUE, MSG_MORE], '@b[more]');
  Map := TMap.Create(MapArea, self, [UIMAP_CENTERING, UIMAP_MARKING]);
  StatusArea := TUIStatusArea.Create(VUI,NewRectXY(1,23,80,25));
  LeftPanel := nil;
  RightPanel := nil;
  UI.Player := Game.Player;
end;

procedure TGameUI.ClassSelectCallback(Choice: byte);
begin
  PlaySound('sound/sfx/items/titlemov.wav');
  CallbackArea1.DestroyChildren;
  CallbackArea2.DestroyChildren;
  with TLuaTable.Create( Game.Lua, 'klasses', Choice)  do
  try
    TContent.Create(CallbackArea1, 1, '@l Level     : ' + IntToStr(GetNumber('level')));
    TContent.Create(CallbackArea1, 2, '');
    TContent.Create(CallbackArea1, 3, '@l Strength  : ' + IntToStr(GetNumber('str')));
    TContent.Create(CallbackArea1, 4, '@l Magic     : ' + IntToStr(GetNumber('mag')));
    TContent.Create(CallbackArea1, 5, '@l Dexterity : ' + IntToStr(GetNumber('dex')));
    TContent.Create(CallbackArea1, 6, '@l Vitality  : ' + IntToStr(GetNumber('vit')));
    TContent.Create(CallbackArea2, Getstring('desc'));
  finally
    Free;
  end;
end;

procedure TGameUI.SelectKlass;
var
  i, Count: byte;
  ClassSelect: TMenu;
  ClassSelectWnd: TUIWindow;
  ClassSelectWnd2: TUIWindow;
  Separator: TUISeparator;
begin
  Count := 0;
  Output.ClearRect(1, 15, 80, 25);
  ClassSelectWnd := TWindow.Create(VUI,NewRectXY(1, 15, 34, 24), 'Choose class');
  Separator := TUISeparator.Create(ClassSelectWnd, false, 12, 2, 1);
  ClassSelect := TMenu.Create(Separator.Left, [VKEY_ENTER], @UI.ClassSelectCallback);
  CallbackArea1 := Separator.Right;
  ClassSelectWnd2 := TWindow.Create(VUI,NewRectXY(35, 15, 80, 24), 'Description');
  ClassSelectWnd2.HBorder := 1;
  ClassSelectWnd2.VBorder := 1;
  CallbackArea2 := ClassSelectWnd2;
  Count := 3;
  for i := 1 to Count do
  begin
    ClassSelect.Add(Game.Lua.IndexedTable['klasses',i,'name']);
  end;
  ClassSelect.Run;
  PlaySound('sound/sfx/items/titlslct.wav');
  Game.Player := TPlayer.Create(Game.Lua.IndexedTable['klasses',ClassSelect.Return,'id']);
  FreeAndNil(ClassSelectWnd);
  FreeAndNil(ClassSelectWnd2);
end;

procedure TGameUI.MainMenuCallback(Choice: byte);
begin
  PlaySound('sound/sfx/items/titlemov.wav');
end;

function TGameUI.MainMenu: byte;
var
  Count: byte;
  Menu: TMenu;
  MenuWnd: TUIWindow;
begin
  Count := 0;
//  Output.ClearRect(1, 15, 80, 25);
  MenuWnd := TWindow.Create(VUI,NewRectXY(30, 17, 50, 24), '');
  MenuWnd.VBorder := 2;
  MenuWnd.HBorder := 5;
  Menu := TMenu.Create(MenuWnd, [VKEY_ENTER], @UI.MainMenuCallback);
  Menu.Add(' New Game');
  Menu.Add(' Load Game', FileExists('save'));
  Menu.Add('Show Manual');
  Menu.Add(' Quit Game');
  Menu.Run;
  Count := Menu.Return;
  PlaySound('sound/sfx/items/titlslct.wav');
  FreeAndNil(Menu);
  FreeAndNil(MenuWnd);
  exit(Count);
end;

function TGameUI.GetCommand(valid : TCmdSet = []) : byte;
begin
  Inc(AnimCount);
  repeat
    if valid = [] then
      GetCommand := Input.GetCommand
    else
      GetCommand := Input.GetCommand(valid+[COMMAND_SOUNDVOLUP, COMMAND_SOUNDVOLDN,
                                            COMMAND_MUSICVOLUP, COMMAND_MUSICVOLDN]);
   {$IFDEF SOUND}
   case GetCommand of
     COMMAND_SOUNDVOLUP : Sound.setSoundVolume(min(Sound.mySoundVolume div 10 * 10 + 10, 100));
     COMMAND_SOUNDVOLDN : Sound.setSoundVolume(max(Sound.mySoundVolume div 10 * 10 - 10, 0));
     COMMAND_MUSICVOLUP : Sound.setMusicVolume(min(Sound.myMusicVolume div 10 * 10 + 10, 100));
     COMMAND_MUSICVOLDN : Sound.setMusicVolume(max(Sound.myMusicVolume div 10 * 10 - 10, 0));
   end;
    {$ENDIF}
  until not(GetCommand in [COMMAND_SOUNDVOLUP, COMMAND_SOUNDVOLDN,
                           COMMAND_MUSICVOLUP, COMMAND_MUSICVOLDN]);
  Messages.Update;
  Messages.Draw;
end;

function TGameUI.GetKey(valid : TKeyFilter = []) : byte;
var Key:byte;
begin
  repeat
    if valid = [] then
      GetKey := Input.GetKey
    else
      GetKey := Input.GetKey(valid);
    if not(GetKey in Valid) then
    begin
      Key:=GetKey;
      Key:=Input.VCodeToCommand(GetKey);
      if Valid<>[] then continue;
    end else break;
  until (GetKey in Valid)or(valid=[]);
end;


procedure TGameUI.PressEnter;
begin
  GetKey([VKEY_ENTER]);
  PlaySound('sound/sfx/items/titlslct.wav');
end;

destructor TGameUI.Destroy();
begin
  //FreeAndNil(Config);
  inherited Destroy;
end;

procedure TGameUI.Intro;
begin
  PlayMusic('sound/music/dintro.wav');
  Output.Clear;
  Output.DrawString(20,3,Red,   '  ####                           ####   ');
  Output.DrawString(20,4,Red,   '  #####  #    ##    ###    #    ######  ');
  Output.DrawString(20,5,Red,   '  ## ##  #   #  #   #  #   #    ##  ##  ');
  Output.DrawString(20,6,Red,   '  ## ##  #   #  #   ###    #    ##  ##  ');
  Output.DrawString(20,7,Red,   '  #@y# #@r#  #   @y#@r##@y#   #  #   #    @y#@r#  #@y#  ');
  Output.DrawString(20,8,Yellow,'  ####   #   #  #   ###    #### ######  ');
  Output.DrawString(20,9,Yellow,'  ###           #                ####   ');
  Output.DrawString(20,10,Brown,'           R O G U E L I K E            ');
  Output.CenterDrawString(38,11,Brown,VERSION);

  Output.DrawString(20,13,Brown,'         by Kornel Kisielewicz          ');
  Output.DrawString(20,14,Brown,' Chris Johnson and Mel''nikova Anastasia');


  Output.DrawString(13,16,Brown,'This  is  the  beta  version,  pre-release  of  Diablo');
  Output.DrawString(13,17,Brown,'Roguelike.  Many  more  things are planned for it  but');
  Output.DrawString(13,18,Brown,'whether we will  continue this project, it all depends');
  Output.DrawString(13,19,Brown,'on you.  If you  like  the  idea,   please drop by the');
  Output.DrawString(13,20,Brown,'ChaosForge forums ( http://forum.chaosforge.org )  and');
  Output.DrawString(13,21,Brown,'leave a  positive  comment  at the  DiabloRL  board to');
  Output.DrawString(13,22,Brown,'encourage further development!                        ');
  Output.DrawString(13,24,Brown,'Press <@LEnter@>> to continue...');

  Output.Update;
  PressEnter;
end;

procedure TGameUI.PlotText(const Text: AnsiString);
var Window: TPlotWindow;
begin
  Window := TPlotWindow.Create(VUI, NewRectXY(8,4,72,21), '', Text);

  Window.Show;
  Window.Draw;
  Window.Run;
  HaltSound();
  FreeAndNil(Window);
end;

procedure TGameUI.ShowMortem;
begin
  Output.UI := nil;
  if not FileExists('mortem.txt') then Exit;
  with TTextViewer.Create('@lPostMortem (mortem.txt)') do
  begin
    Load('mortem.txt');
    Run;
    Free;
  end;
end;

procedure TGameUI.AdjustMap;
begin
  if not ((LeftPanel = nil) xor (RightPanel = nil)) then
  begin
    Map.Dimensions.x1 := 1;
    Map.Dimensions.x2 := 80;
    Focus(Player.Position);
    Map.Update;
    exit;
  end;
  if LeftPanel = nil then
  begin
    Map.Dimensions.x1 := 1;
    Map.Dimensions.x2 := 40;
  end
  else
  begin
    Map.Dimensions.x1 := 41;
    Map.Dimensions.x2 := 80;
  end;
  Focus(Player.Position);
  Map.Update;
end;

procedure TGameUI.ReadConfig;
var INIFile : TINI;
begin
  {INIFile := TINI.Load('diablorl.ini');
  INIFile.SetSection('KeyBindings');
  Input.Load(INIFile,@InputCommandParser);}
  Config := TDiabloConfig.Create('config.lua');
  //Input.RegisterCommand(VKEY_ESCAPE, COMMAND_ESCAPE);
  Input.RegisterCommand(VKEY_BACKSPACE, COMMAND_CWIN);
  {LoadSound(INIFile);
  FreeAndNil(INIFile);}

  if GodMode then
  begin
    Input.RegisterCommand(VKEY_F2,COMMAND_GOD1);
    Input.RegisterCommand(VKEY_F3,COMMAND_GOD2);
    Input.RegisterCommand(VKEY_F4,COMMAND_GOD3);
    Input.RegisterCommand(VKEY_F11,COMMAND_GOD4);
    Input.RegisterCommand(VKEY_F12,COMMAND_GOD5);
  end;
end;

procedure TGameUI.ScreenShot(BBCode: boolean = false);
var key: word;
begin
  Key:=1;
  While FileExists('DiabloRL'+inttostr(Key)+'.txt') do inc(Key);
  if BBCode then
    Output.ScreenShot('DiabloRL'+inttostr(Key)+'.txt', 1)
  else
    Output.ScreenShot('DiabloRL'+inttostr(Key)+'.txt', 0);
  if not FileExists('DiabloRL'+inttostr(Key)+'.txt') then exit;
  if BBCode then
    Msg('BB code screenshot "'+'DiabloRL'+inttostr(Key)+'.txt" created.')
  else
    Msg('Screenshot "'+'DiabloRL'+inttostr(Key)+'.txt" created.');
end;

procedure TGameUI.Outro;
begin
  Output.UI := nil;
  Output.Clear;
  Output.DrawString(3,3 ,LightGray,'Thank you for playing Diablo Roguelike!');
  Output.DrawString(3,4 ,LightGray,'This is just a beta, keep your eyes open for the full release!');
  Output.DrawString(3,6 ,White,'Features planned for DiabloRL:');
  Output.DrawString(3,7 ,LightGray,' -- the full range of Diablo items, monsters, uniques, prefixes and spells');
  Output.DrawString(3,8 ,LightGray,' -- catacombs, caves and hell -- each with a unique level generator');
  Output.DrawString(3,9 ,LightGray,' -- all the original Diablo quests with the original texts');
  Output.DrawString(3,10,LightGray,' -- all the hidden quest locations and uniques');
  Output.DrawString(3,11,LightGray,' -- additional quests by Blizzard that didn''t appear in Diablo');
  Output.DrawString(3,12,LightGray,' -- full spell and missile weapon system');
  Output.DrawString(3,13,LightGray,' -- six classes -- Warrior, Mage, Rogue, Monk, Bard and Barbarian');
  Output.DrawString(3,14,LightGray,' -- maybe additional Hellfire content -- spells, items, quests, uniques.');
  Output.DrawString(3,15,LightGray,' -- Programmer''s edition -- how the author himself see''s the world of Diablo');
  Output.DrawString(3,16,LightGray,' -- proper shops, and special rooms in dungeons');
  Output.DrawString(3,19,LightGray,'Well, at least that would be if DiabloRL would be continued. It all depends');
  Output.DrawString(3,20,LightGray,'on @Lyou@>! If you want this project continued then drop me a note at');
  Output.DrawString(3,21,LightGray,'@Ladmin(at)chaosforge.org@>... Comments, suggestions, death threats all welcome.');

  Output.DrawString(3,23,LightGray,'Again, thank you for your time spent playing DiabloRL.');
  Output.DrawString(3,24,LightGray,'Press @<Enter@> to quit...');

  Output.Update;
  PressEnter;
  Output.Clear;
  Output.Update;
end;

procedure TGameUI.PlayMusic(sID: Ansistring);
begin
  {$IFDEF SOUND} Sound.PlayMusic(sID); {$ENDIF}
end;

procedure TGameUI.PlaySound(sID: Ansistring; x : Integer = 0; y : Integer = 0);
begin
  {$IFDEF SOUND} Sound.PlaySound(sID, x, y); {$ENDIF}
end;

procedure TGameUI.HaltSound();
begin
  {$IFDEF SOUND} Sound.HaltSound; {$ENDIF}
end;

procedure TGameUI.LoadSound(INIFile: TINI);
begin
  {$IFDEF SOUND} Sound.Load(INIFile); {$ENDIF}
end;

procedure TGameUI.Mute();
begin
  {$IFDEF SOUND} Sound.Mute; {$ENDIF}
end;

procedure TGameUI.Unmute();
begin
  {$IFDEF SOUND} Sound.UnMute; {$ENDIF}
end;

procedure TGameUI.SetMusicVolume(Volume : Byte);
begin
  {$IFDEF SOUND} Sound.SetMusicVolume(Volume); {$ENDIF}
end;

procedure TGameUI.SetSoundVolume(Volume : Byte);
begin
  {$IFDEF SOUND} Sound.SetSoundVolume(Volume); {$ENDIF}
end;

procedure TGameUI.SetListenerPosition(x : Integer; y : Integer);
begin
  {$IFDEF SOUND} Sound.SetListenerPosition(x,y); {$ENDIF}
end;

{ TTalkWindow }

constructor TTalkWindow.Create( const nIntro : string );
var S: TSeparator;
begin
  inherited Create(UI.VUI,NewRectXY(41,3,80,22), '');
  s := TSeparator.Create(self, true, 3, 1);
  TContent.Create(s.Top, 1, Padded('', (Dimensions.W - 6 - length(nIntro)) div 2)+nIntro);
  TContent.Create(s.Bottom, 2, '     What would you like to?');
  Chooser := TMenu.Create(s.Bottom, [VKEY_ENTER,VKEY_ESCAPE], nil);
  inc(Chooser.Dimensions.Y1, 3);
end;

procedure TTalkWindow.Add(const Option: string; active : boolean = true);
begin
  Chooser.Add(Padded('', (Dimensions.W - 6 - length(Option)) div 2)+Option, active);
end;

function TTalkWindow.Run: Byte;
begin
  Show;
  Draw;
  if (Chooser.Run = VKEY_ENTER) then Exit( Chooser.Return ) else Exit (0);
end;

destructor TTalkWindow.Destroy;
begin
  FreeAndNil( Chooser );
  inherited Destroy;
end;

{ TPlotWindow }

constructor TPlotWindow.Create(newParent: TUIElement; newDimensions: TRect; newTitle: AnsiString; nText: AnsiString);
var Size: Word;
    tmpstr: ansistring;
begin
  inherited Create(newParent, newDimensions, newTitle);
  Count:=1;
  Text:=TStringArray.Create;
  while Count<length(nText) do
  begin
    tmpstr:=copy(nText, Count, dimensions.w-hborder*2);
    Size:=pos('@/', tmpstr);
    if Size>0 then
    begin
      delete(tmpstr, Size, 80);
      Count+=2;
    end
    else
    begin
      Size:=Length(tmpstr);
      while not(tmpstr[Size] in [' ',',','.',';','?','!','-']) do
      begin
        delete(tmpstr,Size,1);
        dec(Size);
      end;
    end;
    if (Length(tmpstr) > 0) then begin
       Count+=Length(tmpstr);
       if tmpstr[1]=' ' then delete(tmpstr,1,1);
    end;
    Text.Push(tmpstr);
  end;
  Count:=1;
end;

procedure TPlotWindow.Draw;
var Size: Word;
begin
  output.clearrect(dimensions.x1,dimensions.y1,dimensions.x2, dimensions.y2);
  inherited Draw;
  Output.DrawString(2,1,LightGray,'Press <@<Enter@>> or <@<Escape@>> to skip.');
  for Size := 0 to Count-1 do
  begin
    if (Size<=Text.Size)and(Count-Size<=dimensions.h-vborder*2) then
      Output.drawString(dimensions.x1+HBorder, dimensions.y2-vborder-Count+Size+1, Yellow, text[Size]);
  end;
end;

procedure TPlotWindow.Run;
var i: Word;
begin
  for i:=1 to dimensions.h-vborder*2+Text.Size+1 do
  begin
    Count:=i;
    Draw;
    if Input.KeyPressed then
      if UI.GetKey in [VKEY_ENTER, VKEY_ESCAPE] then exit;
    UI.Delay(1400);
  end;
  UI.GetKey([VKEY_ESCAPE,VKEY_ENTER]);
end;

destructor TPlotWindow.Destroy;
begin
  FreeAndNil(Text);
end;

{ TInventoryWindow }

procedure TInventoryWindow.InvCallback(Choice : byte);
begin
  UI.PlaySound('sound/sfx/items/invgrab.wav');
  with Game.Player do
    UI.UpdateStatus(Inv[Choice]);
end;

procedure TInventoryWindow.EqCallback(Choice : byte);
begin
  UI.PlaySound('sound/sfx/items/invgrab.wav');
  with Game.Player do
    UI.UpdateStatus(Eq[Choice]);
end;

constructor TInventoryWindow.Create(newParent: TUIElement; newTitle: AnsiString = 'Inventory');
begin
  inherited Create(newParent, NewRectXY(41,3,80,22), newTitle);
  VBorder := 1;
  HBorder := 1;
  S := TSeparator.Create(self, true, 8, 1);
  UI.Player.InvSort;
  InvMode := Inventory;
end;

procedure TInventoryWindow.Draw;
var Count: word;
begin
  S.Top.DestroyChildren;
  S.Bottom.DestroyChildren;
  if not Visible then exit;
  Output.DrawString(2,1,DarkGray,'Inventory mode: [@lTAB@>] switch, [@lENTER@>] equip/use, [@ld@>] drop, [@lESC@>] exit mode.');
  with UI.Player do
  begin
    if InvMode <> Equipment then
    for Count := 1 to ITEMS_EQ do
      if Eq[Count] = nil then TContent.Create(S.Top,Count,'@d'+SlotName(Count)+' :@> ---')
                         else TContent.Create(S.Top,Count,'@d'+SlotName(Count)+' :@> '+DPad(Eq[Count].GetName(PlainName),31));
    Count := 1;
    if invSize <> 0 then
    repeat
      if Inv[Count] <> nil then TContent.Create(S.Bottom,Count,Inv[Count].GetName(InvName));
      Inc(Count);
      if Count > 10 then Break;
    until False
    else TContent.Create(S.Bottom,' < @lEMPTY@> > ');
  end;
  inherited Draw;
  Output.LeftDrawString(Dimensions.X2-1,Dimensions.Y2,DarkGray,'[@<@1@>/@2]',[UI.Player.InvVolume,MaxVolume]);
end;

procedure TInventoryWindow.Run;
var Choice, cmd, slot : byte;
    Item : TItem;
begin
  repeat
    Draw;
    //empty inventory
    with UI.Player do
    case InvMode of
      Inventory :
        if InvSize = 0 then
        begin
          choice :=  UI.GetKey([VKEY_ENTER,VKEY_ESCAPE,VKEY_TAB,VKEY_BACKSPACE,
                     Input.CommandToVCode(COMMAND_JOURNAL),
                     Input.CommandToVCode(COMMAND_INVENTORY),
                     Input.CommandToVCode(COMMAND_PLAYERINFO)]);
          if Input.VCodeToCommand(Choice) in [COMMAND_CWIN, COMMAND_INVENTORY, COMMAND_JOURNAL, COMMAND_PLAYERINFO] then
            Choice := Input.VCodeToCommand(Choice);
          case choice of
            VKEY_TAB : InvMode := Equipment;
            COMMAND_PLAYERINFO:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_PLAYERINFO));
                  exit;
                end;
            COMMAND_JOURNAL:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_JOURNAL));
                  exit;
                end;
            VKEY_BACKSPACE:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_CWIN));
                  exit;
                end;
            else break;
          end;
        end
        else
        begin
          Chooser := TMenu.Create(self, [VKEY_ENTER,VKEY_ESCAPE,VKEY_TAB,VKEY_BACKSPACE,ord('d'),
                                        Input.CommandToVCode(COMMAND_QUICKSLOT),
                                        Input.CommandToVCode(COMMAND_JOURNAL),
                                        Input.CommandToVCode(COMMAND_INVENTORY),
                                        Input.CommandToVCode(COMMAND_PLAYERINFO){,

                                        <<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>>>>
                                                    FIX     NEEDED
                                        <<<<<<<<<<<<<<<<<<>>>>>>>>>>>>>>>>>>>

                                        Input.CommandToVCode(COMMAND_QUICKSLOT1),
                                        Input.CommandToVCode(COMMAND_QUICKSLOT2),
                                        Input.CommandToVCode(COMMAND_QUICKSLOT3),
                                        Input.CommandToVCode(COMMAND_QUICKSLOT4),
                                        Input.CommandToVCode(COMMAND_QUICKSLOT5),
                                        Input.CommandToVCode(COMMAND_QUICKSLOT6),
                                        Input.CommandToVCode(COMMAND_QUICKSLOT7),
                                        Input.CommandToVCode(COMMAND_QUICKSLOT8)}], @InvCallback);
          Chooser.Dimensions.Y1:=9;
          Choice := 1;
          repeat
            Chooser.Add(Inv[choice].GetName(InvName), true, Inv[choice].InvColor);
            Inc(Choice);
            if Choice > ITEMS_INV then Break;
          until Inv[Choice] = nil;
          cmd := Chooser.Run;
          Choice := Chooser.Return;
          FreeAndNil(Chooser);
          if Input.VCodeToCommand(cmd) in {COMMAND_QUICKSLOTS+}[COMMAND_JOURNAL,
            COMMAND_QUICKSLOT, COMMAND_INVENTORY, COMMAND_PLAYERINFO] then cmd := Input.VCodeToCommand(cmd);
          case cmd of
            VKEY_TAB           : InvMode := Equipment;
            VKEY_ESCAPE,
            COMMAND_INVENTORY  : Break;
            COMMAND_QUICKSLOT  : if Inv[Choice].IType in [TYPE_POTION, TYPE_SCROLL] then
                                   if ItemCheck(Inv[Choice]) then
                                     if GetFreeSlot <> 0 then
                                     begin
                                       UI.Msg('You put '+Inv[Choice].GetName(TheName)+' behind your belt.');
                                       UI.PlaySound(Inv[Choice].Sound1);
                                       QuickSlots[GetFreeSlot] := Inv[Choice];
                                       Dec(SpeedCount,SpdMov);
                                       Inv[Choice] := nil;
                                       Break;
                                     end
                                     else
                                     begin
                                       UI.Msg('You have no free quickslots!');
                                       continue;
                                     end
                                   else
                                   begin
                                     UI.Msg('@<"I can''t use this yet"@>');
                                     UI.Player.PlaySound('13');
                                     continue;
                                   end
                                 else
                                 begin
                                   UI.Msg('Not a quickslot item!');
                                   continue;
                                 end;
            Ord('d')           : begin
                                   if Inv[Choice].Amount > 5000 then
                                   begin
                                     Item:=TItem.Create('gold');
                                     Item.Amount:=5000;
                                     dec(Inv[Choice].Amount, 5000);
                                     dec(Inv[Choice].Volume);
                                   end
                                   else
                                   begin
                                     Item:=Inv[Choice];
                                     Inv[Choice] := nil;
                                   end;
                                   UI.Msg('You drop '+Item.GetName(TheName)+'.');
                                   Dec(SpeedCount,SpdMov);
                                   TLevel(Parent).DropItem(Item,Position);
                                   Break;
                                 end;
            COMMAND_PLAYERINFO:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_PLAYERINFO));
                  exit;
                end;
            COMMAND_JOURNAL:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_JOURNAL));
                  exit;
                end;
            VKEY_BACKSPACE:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_CWIN));
                  exit;
                end;
{           COMMAND_QUICKSLOT1 ..
            COMMAND_QUICKSLOT8    :if Inv[Choice].IType in [TYPE_POTION, TYPE_SCROLL] then
                                   begin
                                     if ItemCheck(Inv[Choice]) then
                                       if QuickSlots[cmd-COMMAND_QUICKSLOT1+1] = nil then
                                       begin
                                         UI.Msg('You put '+Inv[Choice].GetName(TheName)+' behind your belt.');
                                         UI.PlaySound(Inv[Choice].Sound1);
                                         QuickSlots[cmd-COMMAND_QUICKSLOT1+1] := Inv[Choice];
                                         Dec(SpeedCount,SpdMov);
                                         Inv[Choice] := nil;
                                         Break;
                                       end
                                       else
                                       begin
                                         Item := QuickSlots[cmd-COMMAND_QUICKSLOT1+1];
                                         UI.Msg('You swap '+Inv[Choice].GetName(TheName)+' with '+Item.GetName(TheName)+'.');
                                         QuickSlots[cmd-COMMAND_QUICKSLOT1+1] := Inv[Choice];
                                         Dec(SpeedCount,SpdMov);
                                         Inv[Choice] := Item;
                                         Break;
                                       end
                                     else
                                     begin
                                       UI.Msg('@<"I can''t use this yet"@>');
                                       UI.Player.PlaySound('13');
                                       continue;
                                     end
                                   end
                                   else
                                   begin
                                     UI.Msg('Not a quickslot item!');
                                     continue;
                                   end;}
            else
            begin
              slot := ItemToSlot(Inv[Choice]);
              case slot of
                1..ITEMS_EQ : if Eq[slot] = nil then
                                  begin
                                    if slot in [SlotRHand, SlotLHand] then
                                      UI.Msg('You wield '+Inv[Choice].GetName(TheName)+'.')
                                    else
                                      UI.Msg('You wear '+Inv[Choice].GetName(TheName)+'.');
                                    UI.PlaySound(Inv[Choice].Sound1);
                                    Dec(SpeedCount,SpdMov);
                                    Eq[slot] := Inv[Choice];
                                    if (Eq[Slot].Spell > 0) then Spell := 254;
                                    Inv[Choice] := nil;
                                    Game.Lua.TableExecute('achievment', 'OnEquip', [Slot]);
                                    Break;
                                  end
                                else
                                  if Eq[slot].Volume <= MaxVolume-InvVolume+Inv[Choice].Volume then
                                    begin
                                      Item := Inv[Choice];
                                      Inv[Choice] := nil;
                                      UI.Msg('You take off '+Eq[slot].GetName(TheName)+'.');
                                      Inv[InvFreeSlot] := Eq[slot];
                                      if slot in [SlotRHand, SlotLHand] then
                                        UI.Msg('You wield '+Item.GetName(TheName)+'.')
                                      else
                                        UI.Msg('You wear '+Item.GetName(TheName)+'.');
                                      UI.PlaySound(Item.Sound1);
                                      Eq[slot] := Item;
                                      if Eq[Slot].Spell > 0 then Spell := 254;
                                      Dec(SpeedCount,2*SpdMov);
                                      Game.Lua.TableExecute('achievment', 'OnEquip', [Slot]);
                                      Break;
                                    end;
                SlotPotion, SlotBook,
                SlotScroll    : begin
                                  UseItem(Inv[Choice]);
                                  Choice := 0;
                                  Break;
                                end;
                SlotFailTown  : begin
                                  UI.Msg('@<"I can''t use this here"@>');
                                  UI.Player.PlaySound('27');
                                end;
                SlotFailReq   : begin
                                  UI.Msg('@<"I can''t use this yet"@>');
                                  UI.Player.PlaySound('13');
                                end;
                SlotFail2H    : begin
                                  if ifTwoHanded in Inv[Choice].flags then
                                    if ((Eq[SlotRHand]=nil) and(InvVolume-Inv[Choice].Volume+Eq[SlotLHand].Volume <= MaxVolume))
                                      or((Eq[SlotRHand]<>nil)and(InvVolume-Inv[Choice].Volume+Eq[SlotRHand].Volume+Eq[SlotLHand].Volume <= MaxVolume)) then
                                    begin
                                      if Eq[SlotRHand]<>nil then
                                      begin
                                        UI.Msg('You take off '+Eq[SlotRHand].GetName(TheName)+'.');
                                        Inv[InvFreeSlot] := Eq[SlotRHand];
                                      end;
                                      Eq[SlotRHand] := Inv[Choice];
                                      if Eq[SlotLHand]<>nil then
                                        UI.Msg('You take off '+Eq[SlotLHand].GetName(TheName)+'.');
                                      Inv[Choice]:=Eq[SlotLHand];
                                      Eq[SlotLHand]:=nil;
                                      UI.Msg('You wield '+Eq[SlotRHand].GetName(TheName)+'.');
                                      UI.PlaySound(Eq[SlotRHand].Sound1);
                                      Game.Lua.TableExecute('achievment', 'OnEquip', [SlotRHand]);
                                      Dec(SpeedCount,SpdMov);
                                      Break;
                                    end
                                    else
                                      UI.Msg('What?')
                                  else
                                    if (InvVolume-Inv[Choice].Volume+Eq[SlotRHand].Volume <= MaxVolume) then
                                    begin
                                      UI.Msg('You take off '+Eq[SlotRHand].GetName(TheName)+'.');
                                      Eq[SlotLHand] := Inv[Choice];
                                      Inv[Choice]   := Eq[SlotRHand];
                                      Eq[SlotRHand]:=nil;
                                      UI.Msg('You wield '+Eq[SlotLHand].GetName(TheName)+'.');
                                      UI.PlaySound(Eq[SlotLHand].Sound1);
                                      Dec(SpeedCount,SpdMov);
                                      Break;
                                    end
                                    else
                                      UI.Msg('What?')
                                end;
                else UI.Msg('What?');
              end; //case Slot
            end; //case cmd
          end;
        end;

      Equipment :
        begin
          Output.DrawString(2,1,DarkGray,'Equipment mode: [@lTAB@>] switch, [@lENTER@>] unequip, [@ld@>] drop, [@lESC@>] exit mode.  ');
          Chooser := TMenu.Create(self, [VKEY_ENTER,VKEY_ESCAPE,VKEY_TAB,VKEY_BACKSPACE, ord('d'), ord('i')], @EqCallback);
          Chooser.Dimensions.Y2:=7;
          for Choice := 1 to ITEMS_EQ do
            if Eq[Choice] = nil then Chooser.Add('@d'+SlotName(Choice)+' :@> ---', false)
                                else Chooser.Add('@d'+SlotName(Choice)+' :@> '+DPad(Eq[Choice].GetName(PlainName),31), true, Eq[Choice].Color);
          cmd := Chooser.Run;
          Choice := Chooser.Return;
          FreeAndNil(Chooser);
          if Input.VCodeToCommand(cmd) in [COMMAND_INVENTORY, COMMAND_JOURNAL, COMMAND_PLAYERINFO] then
            cmd := Input.VCodeToCommand(cmd);
          case cmd of
            VKEY_ESCAPE,
            COMMAND_INVENTORY : Break;
            VKEY_TAB    : InvMode := Inventory;
            Ord('d')    : if Eq[Choice] <> nil then
                          begin
                            UI.Msg('You drop '+Eq[Choice].GetName(TheName)+'.');
                            TLevel(Parent).DropItem(Eq[Choice],Position);
                            Dec(SpeedCount,SpdMov);
                            Eq[Choice] := nil;
                            Break;
                          end;
            COMMAND_PLAYERINFO:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_PLAYERINFO));
                  exit;
                end;
            COMMAND_JOURNAL:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_JOURNAL));
                  exit;
                end;
            VKEY_BACKSPACE:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_CWIN));
                  exit;
                end;
            else
             if Eq[Choice] <> nil then
             begin
               if Eq[Choice].Volume <= MaxVolume-InvVolume then
               begin
                 UI.Msg('You take off '+Eq[Choice].GetName(TheName)+'.');
                 Inv[InvFreeSlot] := Eq[Choice];
                 Dec(SpeedCount,SpdMov);
                 Eq[Choice] := nil;
                 Break;
               end;
             end;
          end;
        end;
      else exit;
    end;// case InvMode
  until False;
  Input.QueueKey(Input.CommandToVCode(COMMAND_INVENTORY));
end;

destructor TInventoryWindow.Destroy;
begin
  inherited Destroy;
end;

{ TCharWindow }

constructor TCharWindow.Create(newParent: TUIElement; newTitle: AnsiString = 'Character');
begin
  Inherited Create(newParent, NEWRectXY(1,3,40,22), newTitle);
end;

procedure TCharWindow.Draw;

  function ModColor(Modifier: shortint): byte;
  begin
    case modifier of
      -1: exit(LightRed);
       0: exit(LightGray);
       1: exit(LightBlue);
    end;
  end;

  function DualString(Stat,MaxStat : Integer) : string;
    begin
      if Stat = MaxStat then
      case length(IntToStr(Stat)) of
        1  : Exit('  @l'+IntToStr(Stat)+'@>|@l'+IntToStr(MaxStat)+'@>');
        2  : Exit(' @l'+IntToStr(Stat)+'@>|@l'+IntToStr(MaxStat)+'@>');
        else Exit('@l'+IntToStr(Stat)+'@>|@l'+IntToStr(MaxStat)+'@>');
      end;
      if Stat < MaxStat then
      case length(IntToStr(Stat)) of
        1  : Exit('  @l'+IntToStr(Stat)+'@>|@B'+IntToStr(MaxStat)+'@>');
        2  : Exit(' @l'+IntToStr(Stat)+'@>|@B'+IntToStr(MaxStat)+'@>');
        else Exit('@l'+IntToStr(Stat)+'@>|@B'+IntToStr(MaxStat)+'@>');
      end;
      if Stat > MaxStat then
      case length(IntToStr(Stat)) of
        1  : Exit('  @l'+IntToStr(Stat)+'@>|@R'+IntToStr(MaxStat)+'@>');
        2  : Exit(' @l'+IntToStr(Stat)+'@>|@R'+IntToStr(MaxStat)+'@>');
        else Exit('@l'+IntToStr(Stat)+'@>|@R'+IntToStr(MaxStat)+'@>');
      end;
    end;

begin
  if not Visible then exit;
  inherited Draw;
  with UI.Player do
  begin
    if LevelUp then Output.DrawString(2,2,DarkGray,'LevelUp - press [@<s@>] Strength, [@<m@>] Magic, [@<d@>] Dexterity. [@<v@>] for Vitality.');
    Output.DrawString(3, 5,DarkGray,'Name                Class    ');
    Output.DrawString(3, 6,DarkGray,'Level               Exp      ');
    Output.DrawString(3, 7,DarkGray,'                    NextLev  ');

    Output.DrawString(3, 9,DarkGray,'Strength            Gold     ');
    Output.DrawString(3,10,DarkGray,'Magic               Armor    ');
    Output.DrawString(3,11,DarkGray,'Dexterity           ToHit    ');
    Output.DrawString(3,12,DarkGray,'Vitality            Damage   ');
    Output.DrawString(3,13,DarkGray,'Points             ');
    Output.DrawString(3,14,DarkGray,'                    ResMagic     @l5%@>');
    Output.DrawString(3,15,DarkGray,'Life                ResFire      @l5%@>');
    Output.DrawString(3,16,DarkGray,'Mana                ResLight     @l5%@>');
    Output.DrawString(3,17,DarkGray,'                    ');

    Output.LeftDrawString(18, 5,LightGray,Name);
    Output.LeftDrawString(18, 6,LightGray,IntToStr(Level));

    Output.LeftDrawString(37, 5,LightGray,UpCase(KlassName));
    Output.LeftDrawString(37, 6,LightGray,IntToStr(Exp));
    if Level<50 then
      Output.LeftDrawString(37, 7,LightGray,IntToStr(ExpTable[Level+1]))
    else
      Output.LeftDrawString(37, 7,LightGray,'-----');

    Output.LeftDrawString(37, 9,LightGray,IntToStr(getGold));
    Output.LeftDrawString(37,10,LightGray,IntToStr(getAC));

    Output.LeftDrawString(37,11,ModColor(sgn(GetItemStatBonus(PROP_TOHIT))),IntToStr(getToHit+GetItemStatBonus(PROP_TOHIT))+'%');
    Output.LeftDrawString(37,12,ModColor(sgn(GetItemStatBonus(PROP_DMGMOD))),IntToStr(getDmgMin)+'-'+IntToStr(getDmgMax));

    if (Points=0) or (Str>=Game.Lua.IndexedTable['klasses',Klass,'strmax']) then
      Output.DrawString(13, 9,DarkGray,DualString(Str, getStr))
    else
      Output.LeftDrawString(19, 9,DarkGray,'@l'+inttostr(Str)+'@>|@r[@ys@r]');
    if (Points=0) or (Mag>=Game.Lua.IndexedTable['klasses',Klass,'magmax']) then
      Output.DrawString(13,10,DarkGray,DualString(Mag, getMag))
    else
      Output.LeftDrawString(19,10,DarkGray,'@l'+inttostr(Mag)+'@>|@r[@ym@r]');
    if (Points=0) or (Dex>=Game.Lua.IndexedTable['klasses',Klass,'dexmax']) then
      Output.DrawString(13,11,DarkGray,DualString(Dex, getDex))
    else
      Output.LeftDrawString(19,11,DarkGray,'@l'+inttostr(Dex)+'@>|@r[@yd@r]');
    if (Points=0) or (Vit>=Game.Lua.IndexedTable['klasses',Klass,'vitmax']) then
      Output.DrawString(13,12,DarkGray,DualString(Vit, getVit))
    else
      Output.LeftDrawString(19,12,DarkGray,'@l'+inttostr(Vit)+'@>|@r[@yv@r]');
    Output.DrawString(14,13,LightGray,IntToStr(Points));
    Output.DrawString(13,15,DarkGray,DualString(getLife,HP));
    Output.DrawString(13,16,DarkGray,DualString(getMana,MP));
  end;
end;

procedure TCharWindow.Run;
var Key : byte;
    tmp : Word;
    Keys: TKeyFilter;
begin
  with UI.Player do
  begin
    LevelUp := False;
    if Points > 0 then
    begin
      repeat
        Output.DrawString(2,2,DarkGray,'LevelUp - press [@<s@>] Strength, [@<m@>] Magic, [@<d@>] Dexterity. [@<v@>] for Vitality.');
        Keys := [VKEY_ESCAPE,
          Input.CommandToVCode(COMMAND_JOURNAL),
          Input.CommandToVCode(COMMAND_PLAYERINFO),
          Input.CommandToVCode(COMMAND_INVENTORY)];
        if Str < Game.Lua.IndexedTable['klasses',klass,'strmax'] then include(keys,ord('s'));
        if Mag < Game.Lua.IndexedTable['klasses',klass,'magmax'] then include(keys,ord('m'));
        if Dex < Game.Lua.IndexedTable['klasses',klass,'dexmax'] then include(keys,ord('d'));
        if Vit < Game.Lua.IndexedTable['klasses',klass,'vitmax'] then include(keys,ord('v'));
        Key := UI.GetKey(Keys);

        if Input.VCodeToCommand(Key) in [COMMAND_JOURNAL, COMMAND_PLAYERINFO, COMMAND_INVENTORY] then
          Key := Input.VCodeToCommand(Key);
        case Key of

          Ord('s') : Inc(Str);
          Ord('m') : Inc(Mag);
          Ord('d') : Inc(Dex);
          Ord('v') : Inc(Vit);

          COMMAND_JOURNAL :
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_JOURNAL));
                  exit;
                end;

          COMMAND_INVENTORY:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_INVENTORY));
                  exit;
                end;

          else  begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_PLAYERINFO));
                  break;
                end;
        end;
        Dec(Points);
        Draw;
      until (Points = 0);
    end
  end;
end;

destructor TCharWindow.Destroy;
begin
  inherited Destroy;
end;

{ TJournalWindow  }

constructor TJournalWindow.Create(newParent: TUIElement; newTitle: AnsiString = 'Journal');
begin
  Inherited Create(newParent, NEWRectXY(1,3,40,22), newTitle);
end;


procedure TJournalWindow.Draw;
var Size, Count, i : byte;
begin
  inherited Draw;
  Output.DrawString(2,1,DarkGray,' @<Enter@> to view, @<Escape@> or @<'+Input.VCodeToString(Input.CommandToVCode(COMMAND_JOURNAL))+'@> to exit.');
  Size := 0;

  if chooser = nil then
  begin
    Count := Game.Lua.Table['quests','__counter'];;

    for i:= 1 to Count do
     if Game.Player.Quests[i] > 0 then
      with TLuaTable.Create( Game.Lua, 'quests', i ) do
      try
        if Game.Player.Quests[i] < getNumber('completed') then
        begin
          inc(Size);
          Output.DrawString(4,4+Size,DarkGray, getString('name'));
        end;
      finally
        Free;
      end;
    if Size > 0 then
      Output.DrawString(4, 5+Size, DarkGray, 'Close')
    else
    begin
      Output.CenterDrawString(19,5, LightGray, 'Currently, You are not involved in');
      Output.CenterDrawString(19,6, LightGray, 'any quests.');
    end;
  end;
end;

procedure TJournalWindow.Run;
var Choice  : Byte;
    Size    : Byte;
    i       : Byte;
    temp    : array[1..MaxQuests] of byte; //There are 20 quests in both Original and Expansion
begin
  Size := 0;
  Choice := Game.Lua.Table['quests','__counter'];;

  for i:= 1 to Choice do
   if Game.Player.Quests[i] > 0 then
    with TLuaTable.Create( Game.Lua, 'quests', i ) do
    try
      if Game.Player.Quests[i] < getNumber('completed') then
      begin
        inc(Size);
        temp[size]:=i;
      end;
    finally
      Free;
    end;
  if Size = 0 then exit;

  Chooser := TMenu.Create(self, [VKEY_ESCAPE, VKEY_ENTER, VKEY_BACKSPACE,
    Input.CommandToVCode(COMMAND_JOURNAL),
    Input.CommandToVCode(COMMAND_PLAYERINFO),
    Input.CommandToVCode(COMMAND_INVENTORY)], nil);
  //load quests
  for Choice := 1 to Size do
    with TLuaTable.Create( Game.Lua, 'quests', Temp[Choice] ) do
    try
      Chooser.Add(GetString('name'));
    finally
      Free;
    end;
    Chooser.Add('Close');
    repeat
      Choice := Chooser.Run;
      if Input.CommandToVCode(Choice) in [COMMAND_JOURNAL, COMMAND_PLAYERINFO, COMMAND_INVENTORY] then
        Choice := Input.VCodeToCommand(Choice);
      case Choice of
      VKEY_ENTER :
                begin
                  Choice := Chooser.Return;
                  if Choice > Size then break;
                  Game.Lua.TableExecute('quests',Temp[Choice],'OnJournal');
                end;

      VKEY_BACKSPACE :
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_CWIN));
                  exit;
                end;

      COMMAND_INVENTORY:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_INVENTORY));
                  exit;
                end;

      COMMAND_PLAYERINFO:
                begin
                  Input.QueueKey(Input.CommandToVCode(COMMAND_PLAYERINFO));
                  exit;
                end;
      else
        break;
      end;
    until false;
    UI.Player.doJournal;
end;

destructor TJournalWindow.Destroy;
begin
  FreeAndNil(Chooser);
  inherited Destroy;
end;

{ TSpellWindow }

constructor TSpellWindow.Create(newParent: TUIElement; newTitle: AnsiString);
begin
  inherited Create(newParent, NewRectXY(41,3,80,22), newTitle);
end;

procedure TSpellWindow.Draw;
var Count  : Byte;
    Name   : Ansistring;
    Effect : Byte;
    SType  : Byte;
    Page   : Byte;
    Slvl   : Byte;
    Cost   : DWord;
    DMin   : DWord;
    DMax   : DWord;
    Size   : Byte;
    State  : TRLLuaState;
begin
  Size := 0;
  inherited Draw;
  if UI.Player.Skill > 0 then
  begin
    Name := Game.Lua.IndexedTable['spells',UI.Player.Skill,'name'];
    Output.DrawString(43,4,LightGray,Padded(Name,30));
    inc(Size);
  end;
  for Count := 1 to MaxSpells do
    if Game.Lua.Defined('spells',Count) then
    with TLuaTable.Create(Game.Lua,'spells',Count) do
    try
      Slvl   := UI.Player.Spells[Count];
      Page   := getNumber('page');
      if Page <> 0 then // ignore unlisted
      begin
        inc(Size);
        if slvl <> 0 then
        begin
          State.Init( LuaState.NativeState );
          Name   := getString('name');
          Effect := getNumber('effect');
          SType  := getNumber('type');
          Cost   := max(State.CallFunction('cost',[20, UI.Player],-1),(State.CallFunction('cost',[slvl, UI.Player],-1)*UI.Player.SpellCost)div 100);
          DMin   := State.CallFunction('dmin',[slvl, UI.Player],-1);
          DMax   := State.CallFunction('dmax',[slvl, UI.Player],-1);
          if (dmax>0) then
            Output.DrawString(43,3+Size,LightGray,'@1 (L:@<@2@>) @<@3-@4@> Cost: @<@5',
              [Padded(Name,13), slvl+UI.Player.getItemStatBonus(PROP_SPELLLEVEL), dmin, dmax, cost])
          else
            Output.DrawString(43,3+Size,LightGray,'@1 (L:@<@2@>) Cost: @<@3',
              [Padded(Name,13), slvl+UI.Player.getItemStatBonus(PROP_SPELLLEVEL), cost]);
        end;
      end;
    finally
      Free;
    end;
end;

procedure TSpellWindow.Run;
begin
  Exit;
end;

destructor TSpellWindow.Destroy;
begin
  inherited Destroy;
end;

{ TSkillWindow }

constructor TSkillWindow.Create( newParent: TUIElement; newTitle : AnsiString = 'Choose Skill' );
begin
  inherited Create(newParent, NewRectXY(46,13,72,22), newTitle);
  VBorder := 1;
  HBorder := 2;
  Chooser := nil;
  Prepare;
end;

procedure TSkillWindow.Prepare;

var Count : byte;

  function getQuick(ID: byte): string;
  var i : byte;
  begin
    for i:= 1 to MaxQuickSkills do
      if UI.Player.QuickSkills[i] = ID then
        exit('('{+Input.CommandToString(COMMAND_QUICKSKILL1+i-1)}+')');
    exit('');
  end;

begin
  Chooser := TMenu.Create(self, [VKEY_ENTER,VKEY_ESCAPE,VKEY_BACKSPACE,
                               {Input.CommandToVCode(COMMAND_QUICKSKILL1),
                                Input.CommandToVCode(COMMAND_QUICKSKILL2),
                                Input.CommandToVCode(COMMAND_QUICKSKILL3),
                                Input.CommandToVCode(COMMAND_QUICKSKILL4),}
                                Input.CommandToVCode(COMMAND_SPELLS)], nil);

  TByteTextUIMenu(Chooser).Add('  none', 0);
  if UI.Player.Skill > 0 then
    TByteTextUIMenu(Chooser).Add(Padded('@@ '+Game.Lua.IndexedTable['spells',UI.Player.Skill,'name'],20)+getQuick(UI.Player.Skill),UI.Player.Skill );
  for Count := 1 to MaxSpells do
    if UI.Player.Spells[Count] <> 0 then
    if Game.Lua.Defined('spells',Count) then
    with TLuaTable.Create(Game.Lua,'spells',Count) do
    try
      TByteTextUIMenu(Chooser).Add(Padded('@@ '+getString('name')+' lvl '+inttostr((UI.Player.Spells[Count]+UI.Player.getItemStatBonus(PROP_SPELLLEVEL))),20)+getQuick(Count), Count);
    finally
      Free;
    end;
  if UI.Player.Eq[SlotRHand]<>nil then
    if UI.Player.Eq[SlotRHand].Spell<>0 then
      if UI.Player.Eq[SlotRHand].Charges > 0 then
      begin
        Count:=20+length(UI.Player.Eq[SlotRHand].getName(SpellName))-length(Output.Strip(UI.Player.Eq[SlotRHand].getName(SpellName)));
        TByteTextUIMenu(Chooser).Add(Padded('/ '+UI.Player.Eq[SlotRHand].getName(SpellName),Count)+getQuick(254), 254);
      end;
  Chooser.Dimensions.y2:=min(Chooser.Size, 8);
end;

function TSkillWindow.Run: Word;
var Choice: Byte;
begin
  Show;
  Draw;
  Choice := Chooser.Run;
{  if Input.VCodeToCommand(Choice) in [COMMAND_QUICKSKILL1..COMMAND_QUICKSKILL1+MaxQuickSkills-1] then
  begin
    Choice := Input.VCodeToCommand(Choice);
    UI.Player.QuickSkills[Choice - COMMAND_QUICKSKILL1 + 1] := Chooser.Return;
    FreeAndNil(Chooser);
    Prepare;
    exit(Self.Run);
  end;}
  case Choice of
    VKEY_ENTER: Exit( Chooser.Return );
    VKEY_BACKSPACE: Input.QueueKey(Input.CommandToVCode(COMMAND_CWIN));
  else Exit ($ff);
  end;
end;

destructor TSkillWindow.Destroy;
begin
  FreeAndNil( Chooser );
  inherited Destroy;
end;

{ TShopWindow  }

constructor TShopWindow.Create(newParent: TUIElement; newTitle: AnsiString = ''; cback: TTextMenuCallback = nil);
begin
  Inherited Create(newParent, NewRectXY(8,4,72,21), newTitle);
  VBorder := 1;
  HBorder := 2;
  Count   := 1;
  Chooser := TMenu.Create(self, [VKEY_ENTER,VKEY_ESCAPE, Input.CommandToVCode(COMMAND_SPELLS)], cback);
end;

procedure TShopWindow.Add(Content : string; Color: byte = LightGray);
begin
  TByteTextUIMenu(Chooser).Add(Content, Count, true, Color);
  inc(Count);
end;

procedure TShopWindow.Draw;
begin
  inherited Draw;
//  Output.DrawString(2,1,DarkGray,hint);
end;

function TShopWindow.Run: word;
begin
  TByteTextUIMenu(Chooser).Add(Padded('', 26)+'Close', 0);
  Dec(Chooser.Dimensions.Y2, Count);
  Show;
  Draw;
  if (Chooser.Run = VKEY_ENTER) then exit(Chooser.Return) else Exit(0);
end;

destructor TShopWindow.Destroy;
begin
  FreeAndNil(Chooser);
  inherited Destroy;
end;

{ TItemWindow }

constructor TItemWindow.Create(newParent: TUIElement; newItem: TItem ;newTitle : AnsiString = '');
begin
  inherited Create(newParent, NewRectXY(5, 9, 75, 15), newTitle);
  Item:=newItem;
end;

procedure TItemWindow.Draw;
begin
  inherited Draw;
  Output.CenterDrawString(40, 11, Item.Color, Item.GetName(PlainName));
  if Item.GetName(Status1) = '' then
    Output.CenterDrawString(40, 12, Item.Color, Item.GetName(Status2))
  else
  begin
    Output.CenterDrawString(40, 12, Item.Color, Item.GetName(Status1));
    Output.CenterDrawString(40, 13, Item.Color, Item.GetName(Status2));
  end;
end;

procedure TItemWindow.Run;
begin
  Show;
  Draw;
  UI.GetKey;
end;

{ TQSWindow }

constructor TQSWindow.Create( newParent: TUIElement; newTitle : AnsiString = 'Quick slots' );
begin
  inherited Create(newParent, NewRectXY(2,3,40,13), newTitle);
  VBorder := 1;
  HBorder := 2;
  Chooser := nil;
  Prepare;
end;

procedure TQSWindow.Prepare;

var Count : byte;

begin
  Chooser := TMenu.Create(self, [VKEY_ENTER,VKEY_ESCAPE,VKEY_BACKSPACE, ord('d'),
                                Input.CommandToVCode(COMMAND_INVENTORY),
                               {Input.CommandToVCode(COMMAND_QUICKSLOT1),
                                Input.CommandToVCode(COMMAND_QUICKSLOT2),
                                Input.CommandToVCode(COMMAND_QUICKSLOT3),
                                Input.CommandToVCode(COMMAND_QUICKSLOT4),
                                Input.CommandToVCode(COMMAND_QUICKSLOT5),
                                Input.CommandToVCode(COMMAND_QUICKSLOT6),
                                Input.CommandToVCode(COMMAND_QUICKSLOT7),
                                Input.CommandToVCode(COMMAND_QUICKSLOT8),}
                                Input.CommandToVCode(COMMAND_QUICKSLOT)], @UI.Player.QsCallback);
{  for Count := 1 to ITEMS_QS do
    if UI.Player.QuickSlots[Count] <> nil then
      TByteTextUIMenu(Chooser).Add(Input.CommandToString(COMMAND_QUICKSLOT1+Count-1)+' '+UI.Player.QuickSlots[Count].GetName(PlainName), Count)
    else
      TByteTextUIMenu(Chooser).Add(Input.CommandToString(COMMAND_QUICKSLOT1+Count-1)+' (empty)', Count, false);}
  TByteTextUIMenu(Chooser).Add('  close', 0);
  Chooser.Dimensions.y2:=9;
end;

function TQSWindow.Run: Word;
var Choice: Byte;
    Slot : Byte;
    Item  : TItem;
begin
  Show;
  Draw;
  Choice := Chooser.Run;
  Slot  := Chooser.Return;
  FreeAndNil(Chooser);
  {
  if (Slot<>0)and(Input.VCodeToCommand(Choice) in [COMMAND_QUICKSLOT1..COMMAND_QUICKSLOT1+ITEMS_QS-1]) then
  begin
    Choice := Input.VCodeToCommand(Choice);
    Item := UI.Player.QuickSlots[Slot];
    UI.Player.QuickSlots[Slot] :=  UI.Player.QuickSlots[Choice - COMMAND_QUICKSLOT1 + 1];
    UI.Player.QuickSlots[Choice - COMMAND_QUICKSLOT1 + 1] := Item;
    Prepare;
    exit(Self.Run);
  end;
  }
  if Choice = Input.CommandToVCode(COMMAND_INVENTORY) then Choice:=ord('i');
  case Choice of
    VKEY_ENTER: Exit( Slot );
    VKEY_BACKSPACE: Input.QueueKey(Input.CommandToVCode(COMMAND_CWIN));
    ord('i'): if Slot > 0 then
              with UI.Player do
              begin
                if InvFreeSlot = 0 then
                begin
                  UI.Msg('You have no space in your backpack.');
                  Prepare;
                  exit(Self.Run)
                end;
                UI.Msg('You remove '+QuickSlots[Slot].GetName(TheName)+' from your belt.');
                UI.PlaySound(QuickSlots[Slot].Sound1);
                Inv[InvFreeSlot] := QuickSlots[Slot];
                Dec(SpeedCount,SpdMov);
                QuickSlots[Slot] := nil;
              end;
    ord('d'): if Slot > 0 then
              with UI.Player do
              begin
                UI.Msg('You drop '+QuickSlots[Slot].GetName(TheName)+'.');
                TLevel(Parent).DropItem(QuickSlots[Slot],Position);
                QuickSlots[Slot] := nil;
                Dec(SpeedCount,SpdMov);
              end;
  end;
  FreeAndNil(Chooser);
  exit(0);
end;

destructor TQSWindow.Destroy;
begin
  FreeAndNil( Chooser );
  inherited Destroy;
end;

{ TMainUIArea }

procedure TMainUIArea.Draw;
begin
  if Input.RegisterCalled then Exit;
  inherited Draw;
end;


function lua_ui_get_key(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
    KeyFilter: TKeyFilter = [];
    Count:byte;
begin
  State.Init(L);
  Count:=State.StackSize;
  while Count>0 do
  begin
    include(KeyFilter,State.ToInteger(Count));
    dec(Count);
  end;
  State.Push( UI.GetKey(KeyFilter) );
  Result := 1;
end;

function lua_ui_msg(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
begin
  State.Init(L);
  if State.StackSize = 0 then Exit(0);
  UI.Msg(State.ToString(1));
  UI.Draw;
  Result := 0;
end;

function lua_ui_delay(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
begin
  State.Init(L);
  UI.Delay(State.ToInteger(1));
  Result := 0;
end;

function lua_ui_plot_talk(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
begin
  State.Init(L);
  UI.PlotText(State.ToString(1));
  Result := 0;
end;



function lua_ui_talk_run(L: Plua_State): Integer; cdecl;
var State   : TRLLuaState;
    Window  : TTalkWindow = nil;
    Count   : byte;
    Choice  : byte;
    Value   : AnsiString;
begin
  State.Init(L);
  Count := State.StackSize;
  if Count < 2 then Exit(0);

  FreeAndNil(UI.LeftPanel);

  if UI.RightPanel is TTalkWindow
    then Window:=TTalkWindow(UI.RightPanel)
    else FreeAndNil(UI.RightPanel);
  UI.RightPanel := TTalkWindow.Create( State.ToString(1) );
  UI.AdjustMap;

  for Choice := 2 to Count do
  begin
    Value := State.ToString(Choice);
    TTalkWindow(UI.RightPanel).Add(Value, (Length(Value) > 0) and (Value[1] <> '@'));
  end;
  Output.Update;
  UI.Draw;
  Choice := TTalkWindow(UI.RightPanel).Run;

  FreeAndNil(UI.RightPanel);
  UI.RightPanel := Window;
  UI.AdjustMap;
  State.Push( Choice );
  Result := 1;
end;

function lua_ui_shop_run(L: Plua_State): Integer; cdecl;
var State   : TRLLuaState;
    Window  : TShopWindow;
    Shop    : TShopInventory;
    Count   : byte;
    Choice  : byte;
    Source  : Byte;
    Desc    : AnsiString;
begin
  State.Init(L);
  Count  := State.StackSize;
  if Count < 1 then Exit(0);
  Source := State.ToInteger(1);
  Desc   := State.ToString(2);
  Shop   := Game.Shops[Source];
  Shop.Resort;
  Output.DrawString(2, 1, DarkGray,' @<Enter@> to buy, @<Escape@> to exit.                                                ');
  Window := TShopWindow.Create(UI.VUI, Desc, @Shop.Callback);
  for Choice := 1 to Shop.getCount do
     Window.Add(Shop.Items[choice].GetName(BuyName),Shop.Items[choice].InvColor);
  Choice := Window.Run;
  FreeAndNil( Window );
  State.Push(Choice);
  Result := 1;
end;

// ----------------------------- SOUND FUNCTIONS ---------------------- //

function lua_ui_play_music(L: Plua_State): Integer; cdecl;
var State   : TRLLuaState;
begin
  State.Init(L);
  if State.StackSize = 1 then UI.PlayMusic(State.ToString(1));
  Result := 0;
end;

function lua_ui_play_sound(L: Plua_State): Integer; cdecl;
var State   : TRLLuaState;
    x : Integer = 0;
    y : Integer = 0;
    nargs : Integer;
begin
  State.Init(L);
  nargs := State.StackSize;
  if nargs >= 3 then y := State.ToInteger(3);
  if nargs >= 2 then x := State.ToInteger(2);
  if nargs >= 1 then UI.PlaySound(State.ToString(1), x, y);
  Result := 0;
end;



class procedure TGameUI.RegisterLuaAPI(Lua: TLua);
begin
  Lua.SetTableFunction('ui', 'msg',             @lua_ui_msg);
  Lua.SetTableFunction('ui', 'get_key',         @lua_ui_get_key);
  Lua.SetTableFunction('ui', 'delay',           @lua_ui_delay);
  Lua.SetTableFunction('ui', 'talk_run',        @lua_ui_talk_run);
  Lua.SetTableFunction('ui', 'shop_run',        @lua_ui_shop_run);
  Lua.SetTableFunction('ui', 'plot_talk',       @lua_ui_plot_talk);

  Lua.SetTableFunction('ui', 'play_music',          @lua_ui_play_music);
  Lua.SetTableFunction('ui', 'play_sound',          @lua_ui_play_sound);
end;



initialization

with VDefaultWindowStyle do
begin
  BackColor  := Black;
  MainColor  := LightGray;
  BoldColor  := White;
  TitleColor := LightGray;
  FrameColor := DarkGray;
  WrongColor := Red;
  Frame      := 'ĳĳڿ';
end;

end.
