{$MODE OBJFPC}
// @abstract(Thing Object for RL Core)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(January 17, 2005)
// @lastmod(January 17, 2005)
//
// This unit holds the TThing class. A TThing holds common data and
// methods for TItem and TNPC.

unit rlthing;
interface
uses classes, vlua, vrltools, rlgobj;

// Generic Thing class -- ancestor to both TItem and TNPC.
type

{ TThing }

TThing = class(TGameObject)
       Position : TCoord2D;
       Pic      : Char;
       Color    : Byte;
       Level    : Word;
       DmgMin   : Byte;
       DmgMax   : Byte;
       // Overriden constructor -- sets x,y to (1,1).
       constructor Create(const thingID :string); override;
       // We would love to name this one just Move, but Move is a
       // method of TNode. All manipulation on x,y should be done by
       // displace, cause it also displaces the Thing on the map!
       procedure Displace( new : TCoord2D );
       // TThing time flow. Currently just passes time flow to TGameObject.
       procedure TimeFlow(time : LongInt); override;
       // Overriden destructor, does nothing (yet)
       destructor Destroy; override;
       // returns DmgMin
       function getDmgMin : Word; virtual;
       // returns DmgMax
       function getDmgMax : Word; virtual;
       // Returns true if the thing is in the vision field of the player.
       function Visible : boolean;
       // Returns a property value
       function getProperty( PropertyID : Byte ) : Variant; override;
       // Sets a property value
       procedure setProperty( PropertyID : Byte; Value : Variant ); override;
       // returns Picture (color + char)
       function Picture : Word;

       // Stream constructor
       constructor CreateFromStream( ISt : TStream ); override;
       // Stream writer
       procedure ToStream( OSt : TStream ); override;

       // register lua functions
       class procedure RegisterLuaAPI( Lua : TLua );

     end;

implementation
uses rllua, rlnpc, rlitem, rllevel, rlglobal, voutput;

constructor TThing.Create(const thingID :string);
begin
  Pic := '?';
  Color := LightGray;
  // With TNode descendants ALWAYS call inherited Init and Done.
  inherited Create(thingID);
  // To be on the safe side we set x,y to 1, 'cause (0,0) is not
  // a valid map coordinate in this game.
  Position.Create( 1, 1 );
end;

// TThing time flow. Currently just passes time flow to TGameObject.
procedure TThing.TimeFlow(time : LongInt);
begin
  inherited TimeFlow(time);
end;


procedure TThing.Displace( new : TCoord2D );
begin
  // Note that TThing's parent MUST be TLevel, or else we
  // can't access Parent in the way we do!
  with Parent as TLevel do
  begin
    // We have to determinate wether we are a NPC or Item to
    // access the proper mapfields.
    if Self is TNPC then
    begin
      if Map[Position.x,Position.y].NPC = Self then Map[Position.x,Position.y].NPC := nil;
      Map[new.x,new.y].NPC := TNPC(Self);
    end else
    if Self is TItem then
    begin
      if Map[Position.x,Position.y].Item = Self then Map[Position.x,Position.y].Item := nil;
      Map[new.x,new.y].Item := TItem(Self);
    end
    else CritError('We have a TThing that is neither a TNPC nor a TItem!');
  end;
  // Now we can update the coordinates.
  Position := new;
end;

function TThing.Visible : boolean;
begin
  Exit(lfLighted in TLevel(Parent).Map[Position.x,Position.y].Light);
end;

// returns DmgMin
function TThing.getDmgMin : Word;
begin
  Exit(DmgMin);
end;

// returns DmgMax
function TThing.getDmgMax : Word;
begin
  Exit(DmgMax);
end;

function TThing.Picture : Word;
begin
  exit(Color shl 8 + ord(Pic));
end;

destructor TThing.Destroy;
begin
  // With TNode descendants ALWAYS call inherited Init and Done.
  inherited Destroy;
end;

function TThing.getProperty(PropertyID: Byte): Variant;
begin
  case PropertyID of
    PROP_X      : Exit( Position.X );
    PROP_Y      : Exit( Position.Y );
    PROP_PIC    : Exit( Pic );
    PROP_COLOR  : Exit( Color );
    PROP_LEVEL  : Exit( Level );

    PROP_DMGMIN : Exit( DmgMin );
    PROP_DMGMAX : Exit( DmgMax );

    PROP_VISIBLE: Exit( Visible );

    else Exit( inherited getProperty(PropertyID) )
  end;
end;

procedure TThing.setProperty(PropertyID: Byte; Value: Variant);
begin
  case PropertyID of
    // PROP_X      : Position.X := Value; // READ-ONLY
    // PROP_Y      : Position.Y := Value; // READ-ONLY
    PROP_PIC    : Pic := Value;
    PROP_COLOR  : Color := Value;
    PROP_LEVEL  : Level := Value;

    PROP_DMGMIN : DmgMin := Value;
    PROP_DMGMAX : DmgMax := Value;
    else inherited setProperty(PropertyID, Value);
  end;
end;

constructor TThing.CreateFromStream(ISt: TStream);
begin
  inherited CreateFromStream(ISt);
  ISt.Read(Position,SizeOf(TCoord2D));
  Pic   := Chr(ISt.ReadByte);
  Color := ISt.ReadWord;
  Level := ISt.ReadWord;
  DmgMin:= ISt.ReadByte;
  DmgMax:= ISt.ReadByte;
end;

procedure TThing.ToStream(OSt: TStream);
begin
  inherited ToStream(OSt);
  OSt.Write(Position,SizeOf(TCoord2D));
  OSt.WriteByte(Ord(Pic));
  OSt.WriteWord(Color);
  OSt.WriteWord(Level);
  OSt.WriteByte(DmgMin);
  OSt.WriteByte(DmgMax);
end;

function lua_thing_get_name(L: Plua_State) : Integer; cdecl;
var State : TRLLuaState;
    th    : TThing;
begin
  State.ObjectInit(L,th);
  if State.StackSize = 0 then
      State.Push( th.GetName( PlainName ) )
  else
      State.Push( th.GetName( TNameOutputType(State.ToInteger(2)) ) );
  Result := 1;
end;

function lua_thing_get_position(L: Plua_State) : Integer; cdecl;
var State : TRLLuaState;
    th    : TThing;
begin
  State.ObjectInit(L,th);
  State.PushCoord( th.Position );
  Result := 1;
end;

function lua_thing_move(L: Plua_State) : Integer; cdecl;
var State : TRLLuaState;
    th    : TThing;
begin
  State.ObjectInit(L,th);
  th.Displace( State.ToCoord( 2 ) );
  Result := 0;
end;

function lua_thing_is_visible(L: Plua_State) : Integer; cdecl;
var State : TRLLuaState;
    th    : TThing;
begin
  State.ObjectInit(L,th);
  State.Push( th.Visible );
  Result := 1;
end;

class procedure TThing.RegisterLuaAPI(Lua: TLua);
begin
  Lua.SetTableFunction('thing','get_name',     @lua_thing_get_name);
  Lua.SetTableFunction('thing','move',         @lua_thing_move);
  Lua.SetTableFunction('thing','get_position', @lua_thing_get_position);
  Lua.SetTableFunction('thing','is_visible',   @lua_thing_is_visible);
end;

end.
