// @abstract(Sound Unit for RL Core)
// @author(Brett Foster)
// @created(June 27, 2010)
//
// This unit holds the game's sound class : TRLSound.

unit rlsound;
interface
uses vini, vmath, vsdlsound, vsound, SDL_Mixer;

// Maximum distance of pan and fade effects
const MAXDISTANCE = 15;

type

{ TRLSound }

TRLSound = class(TSDLSound)
       mySoundVolume : Byte;
       myMusicVolume : Byte;
       myMusic   : Ansistring;
       xListener : Integer;
       yListener : Integer;
       constructor Create; override;
       procedure PlayMusic(sID: Ansistring) reintroduce;
       procedure PlaySound(sID: Ansistring; x : Integer = 0; y : Integer = 0) reintroduce;
       procedure HaltSound();
       procedure Load(INIFile: TINI);
       procedure Mute();
       procedure Unmute();
       procedure SetMusicVolume(Volume : Byte);
       procedure SetSoundVolume(Volume : Byte);
       procedure SetListenerPosition(x : Integer; y : Integer);
     end;

implementation

constructor TRLSound.Create;
begin
  inherited Create;
  mySoundVolume := 0;
  myMusicVolume := 0;
  Mute();
end;

procedure TRLSound.SetMusicVolume(Volume : Byte);
var resume: boolean;
begin
     resume := ( myMusicVolume = 0 );
     myMusicVolume := Volume;
     inherited SetMusicVolume(Volume);
     if resume then PlayMusic(myMusic);
end;

procedure TRLSound.SetSoundVolume(Volume : Byte);
begin
     mySoundVolume := Volume;
     inherited SetSoundVolume(Volume);
end;

procedure TRLSound.Mute();
begin
     inherited SetSoundVolume(0);
     inherited SetMusicVolume(0);
end;

procedure TRLSound.Unmute();
begin
     inherited SetSoundVolume(mySoundVolume);
     inherited SetMusicVolume(myMusicVolume);
end;

procedure TRLSound.Load(INIFile: TINI);
begin
     if (INIFile.TrySetSection('Sound')) then
     begin
          if (INIFile.Defined('MusicVolume')) then SetMusicVolume(INIFile.getNumber('MusicVolume'));
          if (INIFile.Defined('SoundVolume')) then SetSoundVolume(INIFile.getNumber('SoundVolume'));
     end;
end;

procedure TRLSound.PlayMusic(sID: Ansistring);
var
    mID: Word;
    pThis: TSound;
begin
     if (sID = '') then exit;
     MyMusic := sID;
     if (myMusicVolume = 0) then exit;
     if (not MusicExists(sID)) then
        try
           RegisterMusic(sID, sID);
        except
           Log('Music not found: ' + sID);
        end;
     mID := MusicID(sID);
     if (MusicArray[mID] <> nil) then
     begin
          if (MusicPlaying <> mID) then
          begin
               // pThis is used to avoid a compiler bug
               // We just want to call the base class function TSound.PlayMusic
               pThis := self;
               pThis.PlayMusic(mID);
          end;
     end;
end;

procedure TRLSound.PlaySound(sID: Ansistring; x : Integer; y : Integer);
var
    mID: Word;
    Delta : Integer;
    Volume : Byte = 128;
    Pan : Integer = -1;
begin
     if (sID = '') then exit;
     if (GetSoundVolume() = 0) then exit;
     if (not SampleExists(sID)) then
        try
           RegisterSample(sID, sID);
        except
           Log('Sample not found: ' + sID);
        end;
     mID := SampleID(sID);
     if (SampleArray[mID] = nil) then begin
        // vsound doesn't increment SampleMax on exception
        // But we want it to retain a nil sample in this case
        // so that RegisterSample will only be called once
        if (mID > SampleMax) then SampleMax := mID;
     end
     else begin
          if (x > 0) then begin
             // Attenuate based on simple distance to listener
             // Maximum volume is 100% (128) at range 0 or 1
             // Minimum volume is 25% (32) at range > MAXDISTANCE
             Delta := Distance(x,y,xListener,yListener);
             Volume := 32 + Round(((MAXDISTANCE - Min(Delta - 1,MAXDISTANCE)) * 96) / MAXDISTANCE);
             // Pan based on delta x
             // Minimum pan is none at range 0 or 1
             // Maximum pan is full at maximum range
             Delta := x - xListener;
             if (Abs(Delta) > 1) then begin
               // Calculated value in the range -MAXDISTANCE to MAXDISTANCE
               Pan := Min(Abs(Delta) - 1,MAXDISTANCE) * Sgn(Delta);
               // Input to PlaySample is either -1 (no pan) or
               // a range from 0 (full left) to 255 (full right)
               Pan := Round(((Pan + MAXDISTANCE) * 255) / (2 * MAXDISTANCE));
             end;
          end;
          PlaySample(mID,Volume,Pan);
     end;
end;

// TSDLSound doesn't offer any way to stop a sample
// It's needed so users can opt out of the long gossip wavs
procedure TRLSound.HaltSound();
begin
     Mix_HaltChannel(-1);
end;

procedure TRLSound.SetListenerPosition(x : Integer; y : Integer);
begin
     xListener := x;
     yListener := y;
end;

end.
