{$MODE OBJFPC}
// @abstract(Player for RL Core)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(January 17, 2005)
// @lastmod(January 17, 2009)
//

unit rlplayer;
interface
uses classes, vlua, vutil, rlglobal, rlnpc, rlitem, vmath, vrltools;

// TKillCountArray - keep track of kill counts
// Required because base class TIntAssocArray doesn't provide any way to iterate
type TKillCountArray = class(TIntAssocArray)
     procedure WriteAll(var VF : Text; ShowCount : boolean);

end;

type TInventory = array[1..ITEMS_INV] of TItem;
type TEquipment = array[1..ITEMS_EQ] of TItem;

const
  //Target mode for choosetarget - Look mode
  TM_LOOK = 1;
  //Target mode for choosetarget - Fire mode
  TM_FIRE = 2;

type

{ TPlayer }

TPlayer = class(TNPC)
       klass   : byte;
       Inv     : TInventory;
       Eq      : TEquipment;
       Exp     : DWord;
       LevelUp : Boolean;
       Points  : Byte;
       PortalLevel: Byte;
       PortalCount: Byte;

       HPMod   : ShortInt; //non-item HP bonus/penalty
       MP      : Word;
       MPMax   : Word;
       MPMod   : ShortInt; //non-item MP bonus/penalty
       spdblk  : byte;
       Str     : Word;
       Dex     : Word;
       Vit     : Word;
       Mag     : Word;
       //Spellcasting speed
       SpdMag : Word;
       Skill  : byte;
       // Current ready spell. 0 if none, 254 if staff in hands;
       Spell      : Byte;
       SpellCost  : Byte;
       Spells     : array[1..MaxSpells] of Byte;
       Quests     : array[1..MaxQuests] of Byte;
       QuickSlots : array[1..ITEMS_QS] of TItem;
       QuickSkills: array[1..MaxQuickSkills] of Byte;
       //Selected Enemy
       Querry  : TNPC;
       TargetDir : TDirection;
       // Maximum depth reached
       MaxDepth  : Byte;
       // Kill count
       KillCount : DWord;
       // Exact Kill counts:
       KillData  : TKillCountArray;
       KillDataUnique  : TKillCountArray;

       // Last shopping time
       LastShop  : DWord;

       // Overriden constructor
       constructor Create(const thingID :string); override;
       // Death procedure -- does all cleanup.
       procedure Die(Attacker: TNPC = nil); override;
       procedure Displace( new : TCoord2D );
       // Command Act
       procedure doAct;
       // Command Inventory
       procedure doInv;
       // QuickSlot view
       procedure QsCallback(Choice: byte);
       procedure doQuickSlot;
       //find the first free quickslot
       function  getFreeSlot : byte;
       // Command Pickup
       procedure doPickUp;
       // review unfinished quests
       procedure doJournal;
       // Spell menu
       procedure doSpellBook;
       procedure doQuickSkill;
       // perform firing
       procedure doFire;
       procedure FireArrow;
       // Cast spell
       // if SpellLevel <> 0 then spell is cast like from item
       function CastSpell( SpellID : Byte; SpellLevel : Byte = 0 ) : Boolean;
       // Player action
       procedure Action; override;
       // Player Attack monster
       function Attack(NPC : TNPC; Ranged: boolean = false; SpellID : Byte = 0) : Boolean; override;
       // Block Attack
       function Block(npcAttacker : TNPC) : Boolean; override;
       // Block chance
       function  getBlock : Word;
       // Damage base
       function  getDmgBase : Word;
       // returns DmgMin {with Items and charbonus}
       function  getDmgMin : Word; override;
       // returns DmgMax {with Items and charbonus}
       function  getDmgMax : Word; override;
       // returns AC {with Items}
       function  getAC : Word; override;
       // returns ToHit{with Items}
       function  getToHit : Word; override;
       // returns ToHit {with clvl and class bonus}
       function  getToHitMelee : Word;
       // returns ToHit {with clvl and class bonus}
       function  getToHitMagic : Word;
       // returns ToHit {with clvl and class bonus}
       function  getToHitRange(Distance : Word) : Word;
       //returns attack speed with equipped weapon
       function getAtkSpeed : Word;
       // returns HPMax
       function  getLife : Word; override;
       function  getMana : Word;
       function  getStr : Word;
       function  getMag : Word;
       function  getDex : Word;
       function  getVit : Word;
       function  getGold : DWord;
       function  getItemHasFlags(aFlags: TFlags):boolean;
       function  getItemStatBonus(Stat : Byte) : Integer;
       // Takes care of the items.
       destructor Destroy; override;
       // Adds given amount of gold. Returns number of pieces that
       // didn't fit.
       function  addGold(Amount : DWord) : DWord;
       // Removes the given amount of gold if available, returns false if not
       function  removeGold(Amount : DWord) : Boolean;
       // Adds given amount of experience.
       procedure addExp(Amount : DWord);
       procedure durabilityCheck(Slot : Byte);
       procedure WriteMemorial;
       function  SlotName(slot : byte) : string;
       function  ItemToSlot(Item : TItem) : byte;
       procedure InvSort;
       function  InvSize : Byte;
       function  InvVolume : Word;
       function  InvFreeSlot : Byte;
       function  InvGoldSlot : Byte;
       function  KlassName : string;
       function  KlassPronoun : string;
       function  KlassSalutation : string;
       procedure doCharInfo;
       // moves the item from anywhere to inventory
       function AddItem(aItem: TItem): byte;
       // returns slot in which item exist or 0
       function FindItemInv(item: string):byte;
       function FindItemEq(item: string):byte;
       function FindItemQs(item: string):byte;
       // checks wether item can be weared
       function  ItemCheck(item : TItem) : Boolean;
       // Scrolls and Potions only -- uses the item and disposes of it.
       // Returns wether successful. If successful var argument set to nil.
       function  UseItem(var item : TItem) : Boolean;
       //Targeting
       function  ChooseTarget( TargetMode : Byte = TM_LOOK; FireCmd : byte = COMMAND_LOOK ) : boolean;
       //Targeting
       function  ChooseDirection : boolean;
              // Returns a property value
       function getProperty( PropertyID : Byte ) : Variant; override;
       // Sets a property value
       procedure setProperty( PropertyID : Byte; Value : Variant ); override;
       procedure PlaySound(sID : string);
       //quick item slot handling
       procedure useQuickSlot( slot: byte );
       //quick skill slot handling
       procedure useQuickSkill( slot: byte );
       //Initialize before construction
       procedure Init;
       // Stream constructor
       constructor CreateFromStream( ISt : TStream ); override;
       // Stream writer
       procedure ToStream( OSt : TStream ); override;

       // register lua functions
       class procedure RegisterLuaAPI(Lua: TLua);
     end;

const SlotNone    = 0;
      SlotFail2H  = 100;
      SlotFailReq = 101;
      SlotPotion  = 102;
      SlotScroll  = 103;
      SlotBook    = 104;
      SlotFailTown= 105;

implementation
uses sysutils, voutput, rlgobj, rllevel, rlcell, rlgame, vdebug, rllua, variants,
rlui, vinput;

var
      VF : Text;

function TPlayer.ChooseTarget( TargetMode : Byte = TM_LOOK; FireCmd : byte = COMMAND_LOOK ) : boolean;
var Key : Byte;
    Dir : TDirection;
begin
    Target := Position;

    if TargetMode <> TM_LOOK then
    begin
      if (Querry = nil) or (not Querry.Visible) or (Querry.AI in [AINPC, AIGolem]) then
        Querry := ClosestTarget(AIMonster,min(max(100+getItemStatBonus(PROP_LIGHTMOD), 40)div 10, 15)-1);
      if Querry <> nil then
        Target := Querry.Position;
    end;

    UI.MoveCursor(Target);
    Output.ShowCursor;
  repeat
    Output.Clear;

    UI.MoveCursor(Target);

    UI.UpdateStatus(Target);

    if TargetMode in [TM_FIRE] then
      if TLevel(Parent).isExplored(Target) then
        UI.MarkCell(Target, Red, 'X');
    Key := UI.GetCommand(COMMANDS_MOVE+[COMMAND_ESCAPE,FireCmd]);
    if Key in [COMMAND_ESCAPE, COMMAND_LOOK] then
    begin
      Target.x := 0;
      Output.HideCursor;
      UI.MarkClear;
      Exit(False);
    end;

    if (Key in COMMANDS_MOVE) then
    begin
      Dir := CommandDirection(Key);
      if TLevel(Parent).ProperCoord(Target+Dir) and ((TargetMode in [TM_LOOK]) or
        (TLevel(Parent).isVisible(Target+Dir) and (TargetMode in [TM_FIRE])))then
          Target += Dir;
    end;
    UI.MarkClear;
    if TargetMode = TM_LOOK then
      UI.Focus(Target);
  until Key in [FireCmd];

  if Position = Target then
  begin
    UI.Msg('Suicide? Too easy for you...');
    Output.HideCursor;
    exit(false);
  end;
  Output.HideCursor;
  Exit(True);
end;

function TPlayer.ChooseDirection: boolean;
var Key : Byte;
begin
  UI.Msg('Choose direction...');
  Key := UI.GetCommand(COMMANDS_MOVE+[COMMAND_ESCAPE]);
  if Key = COMMAND_ESCAPE
    then TargetDir.code := 0
    else TargetDir := CommandDirection(Key);
  Exit( TargetDir.isProper );
end;

//find the first free quickslot
function TPlayer.GetFreeSlot : byte;
var Slot: byte;
begin
  for Slot:=1 to ITEMS_QS do
    if QuickSlots[Slot] = nil then exit(Slot);
  exit(0);
end;

function TPlayer.SlotName(slot : byte) : string;
begin
  case Slot of
    SlotHead   : Exit('Head');
    SlotAmulet : Exit('Neck');
    SlotTorso  : Exit('Body');
    SlotRHand  : Exit('Wpn ');
    SlotLHand  : Exit('Shld');
    SlotRRing  : Exit('RRng');
    SlotLRing  : Exit('LRng');
  end;
end;

function TPlayer.ItemToSlot(Item : TItem) : byte;
begin
  if Item = nil then Exit(SlotNone);
  if not ItemCheck(Item) then Exit(SlotFailReq);

  if ifTwoHanded in Item.Flags then
    if (Eq[SlotLHand] = nil) then
      Exit(SlotRHand)
    else Exit(SlotFail2H);

  if Item.IType = TYPE_WEAPON then Exit(SlotRHand);
  if Item.IType = TYPE_BOW then Exit(SlotRHand);

  if Item.IType = TYPE_SHIELD then
    if Eq[SlotRHand] = nil then Exit(SlotLHand)
    else if (ifTwoHanded in Eq[SlotRHand].Flags) then
      Exit(SlotFail2H)
    else Exit(SlotLHand);

  if Item.IType = TYPE_HELM   then Exit(SlotHead);
  if Item.IType = TYPE_ARMOR  then Exit(SlotTorso);
  if Item.IType = TYPE_AMULET then Exit(SlotAmulet);

  if Item.IType = TYPE_RING then
    if Eq[SlotRRing] = nil then Exit(SlotRRing) else
      if Eq[SlotLRing] = nil then Exit(SlotLRing) else
         Exit(SlotRRing);

  if ifNotInTown in Item.Flags then
    if Game.LevelNumber = 0 then Exit(SlotFailTown);
  if Item.IType = TYPE_SCROLL then Exit(SlotScroll);
  if Item.IType = TYPE_POTION then Exit(SlotPotion);
  if Item.IType = TYPE_BOOK   then Exit(SlotBook);
  Exit(SlotNone);
end;

function TPlayer.InvSize : Byte;
var Count : byte;
    Size  : byte;
begin
  Size := 0;
  for Count := 1 to ITEMS_INV do
    if Inv[Count] <> nil then Inc(size);
  Exit(Size);
end;

function TPlayer.InvVolume : Word;
var Count  : byte;
    Volume : Word;
begin
  Volume := 0;
  for Count := 1 to ITEMS_INV do
    if Inv[Count] <> nil then Inc(Volume,Inv[Count].Volume);
  Exit(Volume);
end;

function TPlayer.InvFreeSlot : Byte;
var Count : byte;
begin
  for Count := 1 to ITEMS_INV do
    if Inv[Count] = nil then Exit(Count);
  Exit(0);
end;

function TPlayer.InvGoldSlot : Byte;
var Count : byte;
begin
  for Count := 1 to ITEMS_INV do
    if Inv[Count] <> nil then
      if Inv[Count].IType = TYPE_GOLD then Exit(Count);
  Exit(0);
end;

function TPlayer.KlassName: string;
begin
  case Klass of
    KlassWarrior  : Exit('Warrior');
    KlassRogue    : Exit('Rogue');
    KlassSorceror : Exit('Sorceror');
  end;
end;

function TPlayer.KlassPronoun: string;
begin
  if Klass in [KlassRogue] then Exit('She') else Exit('He');
end;

function TPlayer.KlassSalutation: string;
begin
  if Klass in [KlassRogue] then Exit('lass') else Exit('lad');
end;

function TPlayer.getGold : DWord;
var slot : Byte;
begin
  slot := InvGoldSlot;
  if slot = 0 then Exit(0) else Exit(Inv[slot].Amount);
end;


function TPlayer.getVit : Word;
begin
  Exit(Vit+getItemStatBonus(PROP_VIT));
end;

function TPlayer.getStr : Word;
begin
  Exit(Str+getItemStatBonus(PROP_STR));
end;

function TPlayer.getMag : Word;
begin
  Exit(Mag+getItemStatBonus(PROP_MAG));
end;

function TPlayer.getDex : Word;
begin
  Exit(Dex+getItemStatBonus(PROP_DEX));
end;

function TPlayer.getAC : Word;
begin
  case Klass of
    KlassWarrior,
    KlassRogue, KlassSorceror: Exit(max(0,(getDex div 5+getItemStatBonus(PROP_AC)*100+getItemStatBonus(PROP_ACMODPERCENT))div 100));
  end;
end;

// Block chance
function TPlayer.getBlock : Word;
begin
  case Klass of
    KlassWarrior  : Exit(getDex+30);
    KlassRogue    : Exit(getDex+20);
    KlassSorceror : Exit(getDex+10);
  end;
end;


function TPlayer.getToHit : Word;
begin
  Exit(50 + getDex div 2);
end;

function TPlayer.getAtkSpeed : Word;
begin
  case Klass of
    KlassWarrior : if Eq[SlotRHand]=nil then
                     if Eq[SlotLHand]=nil then
                       Exit(45)        //Bare hands
                     else Exit(45)      //Shield
                   else begin
                     if (ifAxe in Eq[SlotRHand].flags)   then exit(50);
                     if (TYPE_STAFF = Eq[SlotRHand].IType) then exit(55);
                     if (TYPE_BOW = Eq[SlotRHand].IType)then exit(55);
                     exit(45);
                   end;
    KlassRogue   : if Eq[SlotRHand]=nil then
                     if Eq[SlotLHand]=nil then
                       Exit(50)         //Bare hands
                     else Exit(50)      //Shield
                   else begin
                     if (ifAxe in Eq[SlotRHand].flags)   then exit(65);
                     if (TYPE_STAFF = Eq[SlotRHand].IType) then exit(55);
                     if (TYPE_BOW = Eq[SlotRHand].IType)then exit(35);
                     Exit(50);
                   end;
    KlassSorceror: if Eq[SlotRHand]=nil then
                     if Eq[SlotLHand]=nil then
                       Exit(45)     //Bare hands
                     else Exit(60)  //Shield
                   else begin
                     if (ifAxe in Eq[SlotRHand].flags)   then exit(80);
                     if (TYPE_BOW = Eq[SlotRHand].IType)then exit(80);
                     Exit(60);
                   end;
  end;
end;

function TPlayer.getToHitMelee : Word;
begin
  case Klass of
    KlassWarrior  : Exit(getToHit+Level+20+GetItemStatBonus(PROP_TOHIT));
    KlassRogue    : Exit(getToHit+Level+GetItemStatBonus(PROP_TOHIT));
    KlassSorceror : Exit(getToHit+Level+GetItemStatBonus(PROP_TOHIT));
  end;
end;

function TPlayer.getToHitMagic : Word;
begin
  case Klass of
    KlassWarrior  : Exit(50+getMag+GetItemStatBonus(PROP_TOHIT));
    KlassRogue    : Exit(50+getMag+GetItemStatBonus(PROP_TOHIT));
    KlassSorceror : Exit(50+getMag+GetItemStatBonus(PROP_TOHIT)+20);
  end;
end;

function TPlayer.getToHitRange(Distance : Word) : Word;
begin
  case Klass of
    KlassWarrior  : Exit(Max(50+getDex+Level-((Distance*Distance) div 2)+10,0)+GetItemStatBonus(PROP_TOHIT));
    KlassRogue    : Exit(Max(50+getDex+Level-((Distance*Distance) div 2)+20,0)+GetItemStatBonus(PROP_TOHIT));
    KlassSorceror : Exit(Max(50+getDex+Level-((Distance*Distance) div 2)   ,0)+GetItemStatBonus(PROP_TOHIT));
  end;
end;

// returns HPMax
function TPlayer.getLife : Word;
begin
  getLife:= HPMax;
  case Klass of
    KlassWarrior  : HPMax           := 2*Vit
                                     + 2*getItemStatBonus(PROP_VIT)
                                     + getItemStatBonus(PROP_HPMOD)
                                     + 2*Level + 18;
    KlassRogue    : HPMax           := 1*Vit
                                     + Round(1.5*getItemStatBonus(PROP_VIT))
                                     + getItemStatBonus(PROP_HPMOD)
                                     + 2*Level+23;
    KlassSorceror : HPMax           := 1*Vit
                                     + getItemStatBonus(PROP_VIT)
                                     + getItemStatBonus(PROP_HPMOD)
                                     + Level+9;
  end;
  HPMax += HPMod;
  if hp > 0 then
    hp:=max(hp+HPMax-getLife,1);
  Exit(HPMax);
end;

function TPlayer.getMana : Word;
begin
  getMana:=MPMax;
  case Klass of
    KlassWarrior  : MPMax           := 1*Mag
                                     + getItemStatBonus(PROP_MAG)
                                     + getItemStatBonus(PROP_MPMOD)
                                     + Level - 1;
    KlassRogue    : MPMax           := 1*Mag
                                     + Round(1.5*getItemStatBonus(PROP_MAG))
                                     + getItemStatBonus(PROP_MPMOD)
                                     + 2*Level + 5;
    KlassSorceror : MPMax           := 2*Mag
                                     + 2*getItemStatBonus(PROP_MAG)
                                     + getItemStatBonus(PROP_MPMOD)
                                     + 2*Level - 2;
  end;
  MPMax += MPMod;
  mp:=max(mp+MPMax-getMana,1);
  Exit(MPMax);
end;

// returns Dmg
function TPlayer.getDmgBase : Word;
begin
  case Klass of
     KlassWarrior   : if (Eq[SlotRHand]<>nil) and (TYPE_BOW = Eq[SlotRHand].IType) then
       exit(GetStr*level div 200) else exit(GetStr*level div 100);
     KlassSorceror  : if (Eq[SlotRHand]<>nil) and (TYPE_BOW = Eq[SlotRHand].IType) then
       exit(GetStr*level div 200) else exit(GetStr*level div 100);
     KlassRogue     : exit((GetStr+GetDex)*level div 200);
  end;
end;

// returns DmgMin
function TPlayer.getDmgMin : Word;
begin
  Exit((getDmgBase+max(getItemStatBonus(PROP_DMGMIN),1)+getItemStatBonus(PROP_DMGMOD)));
end;

// returns DmgMax
function TPlayer.getDmgMax : Word;
begin
  if (Eq[SlotRHand] = nil) and (Eq[SlotLHand] <> nil) and (Eq[SlotLHand].IType = TYPE_SHIELD) then
    Exit((getDmgBase+3+getItemStatBonus(PROP_DMGMOD)));
  Exit((getDmgBase+max(getItemStatBonus(PROP_DMGMAX),1)+getItemStatBonus(PROP_DMGMOD)));
end;

function TPlayer.getItemHasFlags(aFlags: TFlags):boolean;
var Count : byte;
    Item  : TItem;
begin
  for Count := 1 to ITEMS_EQ do
    if Eq[Count] <> nil then
      if not(ifUnknown in Eq[Count].flags) then
      begin
        Item:=Eq[Count];
        Eq[Count]:=nil;
        if (aFlags * Item.Flags = aFlags) and ItemCheck(Item) then
        begin
          Eq[Count]:=Item;
          exit(true);
        end;
        Eq[Count]:=Item;
      end;
  exit(false);
end;


function TPlayer.getItemStatBonus(Stat : Byte) : Integer;
var Count : byte;
    Bonus : Integer;
    Bonus_: Integer;
    Item  : TItem;
begin
  Bonus := 0;
  Bonus_:=0;
  case Stat of
  PROP_LIFESTEAL: begin
                   for Count := 1 to ITEMS_EQ do
                     if Eq[Count] <> nil then
                     if not(ifUnknown in Eq[Count].flags) then
                     begin
                       Item:=Eq[Count];
                       Eq[Count]:=nil;
                       if (Item.LifeSteal <> 0) and ItemCheck(Item) then
                         Bonus := max(Item.LifeSteal, Bonus);
                       if (ifVampiric in Item.flags) then
                         Bonus_ := randrange(0,125);
                       Eq[Count]:=Item;
                     end;
                   Exit(Bonus+Bonus_);
                 end;

  PROP_DMGMOD,
  PROP_DMGMIN,
  PROP_DMGMAX: begin
                   for Count := 1 to ITEMS_EQ do
                     if Eq[Count] <> nil then
                     if (stat <> PROP_DMGMOD) or not(ifUnknown in Eq[Count].flags) then
                     begin
                       Item:=Eq[Count];
                       Eq[Count]:=nil;
                       if ((Item.GetProperty(Stat) <> 0) or (Item.DmgModPercent <> 0)) and ItemCheck(Item) then
                       begin
                         Bonus  += Item.GetProperty(Stat);;
                         Bonus_ += Item.DmgModPercent;
                       end;
                       Eq[Count]:=Item;
                     end;

                   if Bonus_>0 then
                     Bonus+=max(1, Bonus_*Bonus div 100)
                   else if Bonus_<0 then
                     Bonus-=max(1,(-Bonus_*Bonus) div 100);
                   Exit(Bonus);
                 end;

  PROP_MANASTEAL,
  PROP_SPDATK,
  PROP_SPDHIT,
  PROP_SPDBLK:   begin
                   for Count := 1 to ITEMS_EQ do
                     if Eq[Count] <> nil then
                     if not(ifUnknown in Eq[Count].flags) then
                     begin
                       Item:=Eq[Count];
                       Eq[Count]:=nil;
                       Bonus_ := Item.GetProperty(Stat);
                       if (Bonus_ <> 0) and ItemCheck(Item) then
                         Bonus := max(Bonus_, Bonus);
                       Eq[Count]:=Item;
                     end;
                   Exit(Bonus);
                 end;

  else begin
         for Count := 1 to ITEMS_EQ do
         if Eq[Count] <> nil then
         if not(ifUnknown in Eq[Count].flags) then
         begin
           Item:=Eq[Count];
           Eq[Count]:=nil;
           Bonus_ := Item.GetProperty(Stat);
           if (Bonus_ <> 0) and ItemCheck(Item) then
             Bonus := Bonus + Bonus_;
           Eq[Count]:=Item;
         end;
         Exit(Bonus);
       end;
  end;
end;

procedure TPlayer.InvSort;
var cn   : byte;
    ci   : byte;
    function Greater(Item1,Item2 : TItem) : Boolean;
    begin
      if Item1 = nil then Exit(True);
      if Item2 = nil then Exit(False);
      if Item1.Color > Item2.Color then Exit(True);
      if Item1.Color < Item2.Color then Exit(False);
      if Ord(Item1.Pic) > Ord(Item2.Pic) then Exit(False);
      if Ord(Item1.Pic) < Ord(Item2.Pic) then Exit(True);
      Exit(False);
    end;
    procedure Swap(var Item1,Item2 : TItem);
    var temp : TItem;
    begin
      temp := Item1;
      Item1 := Item2;
      Item2 := temp;
    end;
begin
  for cn := 1 to ITEMS_INV-2 do
    for ci := 1 to ITEMS_INV-cn do
      if Greater(Inv[ci],Inv[ci+1]) then Swap(Inv[ci],Inv[ci+1]);
end;

procedure TPlayer.Die(Attacker: TNPC = nil);
begin
  PlaySound('71');
  UI.Msg('You die... Press <@<Enter@>>...');
  UI.Draw;
  UI.PressEnter;
  Querry:=Attacker;
  GameEnd := True;
end;

function TPlayer.ItemCheck(item : TItem) : Boolean;
begin
  if Item = nil then Exit(False);
  if (Item.StrReq > Str)or(Item.DexReq > Dex) then
  begin
    if Item.StrReq > getStr then Exit(False);
    if Item.DexReq > getDex then Exit(False);
  end;
  if Item.IType = TYPE_BOOK then
  begin
    if Item.BookMagReq > getMag then Exit(False);
  end
  else
    if Item.MagReq > getMag then Exit(False);
  Exit(True);
end;

function TPlayer.UseItem(var item : TItem) : Boolean;
var State : TRLLuaState;
begin

  if (ifNotInTown in item.Flags) and (Game.LevelNumber = 0) then
  begin
    UI.Msg('Not in town!');
    Exit(False);
  end;
       if Item.IType = TYPE_POTION then UI.Msg('You drink '+item.GetName(TheName)+'.')
  else if Item.IType in [TYPE_SCROLL,TYPE_BOOK] then UI.Msg('You read '+item.GetName(TheName)+'.')
  else Exit(False);
       if Item.IType in [TYPE_SCROLL,TYPE_POTION] then UI.PlaySound(Item.Sound1)
  else UI.PlaySound('sound/sfx/items/readbook.wav');

  Game.Lua.TableExecute('achievment', 'OnUse', [Item.tID]);
  if Item.Spell <> 0 then
    if Item.IType = TYPE_BOOK then
    begin
      with TLuaTable.Create( Game.Lua, 'spells', Item.Spell ) do
      try
        if Spells[Item.Spell] < MaxSpellLvl then
        begin
          State.Init( LuaState.NativeState );
          Inc( Spells[Item.Spell] );
          MP := Min( getMana, MP + State.CallFunction('cost',[1,Self],-1) );
        end;
      finally
        Free;
      end;
      Dec(SpeedCount,SpdMov);
    end
    else
    begin
      if not CastSpell( Item.Spell, Max(1,Spells[Item.Spell]) ) then Exit(False);
    end
  else
  begin
    Item.RunHook( Hook_OnUse, [Self] );
    Dec(SpeedCount,SpdMov);
  end;
  FreeAndNil(item);
  Exit(True);
end;


// Adds given amount of experience.
procedure TPlayer.addExp(Amount : DWord);
begin
  if Level = 50 then exit;
  Exp := Exp + Amount;
  if Exp >= ExpTable[Level+1] then
  begin
    UI.PlaySound('sound/sfx/misc/levelup.wav');
    Inc(Level);
    if Level < 50 then Inc(Points,5);
    UI.Msg('You advance to @<level '+IntToStr(Level)+'@>!');
    HP := getLife;
    MP := getMana;
    if (UI.LeftPanel<>nil) and (UI.LeftPanel is TCHarWindow) then
      TCharWindow(UI.LeftPanel).Run else LevelUp := true;
  end;
end;

// Player Attack monster
function TPlayer.Attack(NPC : TNPC; Ranged : boolean = false; SpellID : Byte = 0) : Boolean;
var HitChance  : Integer;
    Damage     : Integer;
    aRanged    : boolean;
    DmgType    : byte;
    State      : TRLLuaState;
begin
  if NPC = nil then Exit;
  if NPC = self then Exit;
  aRanged:=Ranged;
  if not aRanged then
  begin
    if Eq[SlotRhand]<>nil then
      aRanged:=(TYPE_BOW = Eq[SlotRhand].IType);
    Dec(SpeedCount,GetAtkSpeed-getItemStatBonus(PROP_SPDATK));
  end;

  if NPC.Visible then
    Querry := NPC;


  if not ( nfStatue in NPC.flags) then
  begin
    if aRanged then
      if SpellID = 0
        then HitChance := getToHitRange(Distance(Position, NPC.Position))
        else HitChance := getToHitMagic
    else begin
      HitChance := getToHitMelee;
      if (Random(2) = 0) then UI.PlaySound('sound/sfx/misc/swing.wav')
      else UI.PlaySound('sound/sfx/misc/swing2.wav');
    end;
    HitChance := HitChance-NPC.getAC;

    if HitChance < 5  then HitChance := 5;
    if HitChance > 95 then HitChance := 95;
    if Random(100) >= HitChance then
    begin
      if (NPC.Visible) then
        UI.Msg('You miss '+NPC.GetName(TheName)+'.');
      Exit(false);
    end;
  end;

  DmgType := DAMAGE_GENERAL;

  if SpellID = 0 then
  begin
    if Eq[SlotRHand] <> nil then
    begin
      Damage := getDmgBase+{+%}+getItemStatBonus(PROP_DMGMOD)+RandRange(Eq[SlotRHand].DmgMin,Eq[SlotRHand].DmgMax);
      if (nfUndead in NPC.flags) and (ifSlayUndead in flags) then damage:=Damage * 3 div 2;
      if (nfUndead in NPC.flags) and (ifWeakUndead in flags) then damage:=Damage div 2;
      if (nfUndead in NPC.flags) and (ifBaneUndead in flags) then damage:=Damage * 3;
      if (nfAnimal in NPC.flags) and (ifSlayAnimal in flags) then damage:=Damage * 3 div 2;
      if (nfAnimal in NPC.flags) and (ifWeakAnimal in flags) then damage:=Damage div 2;
      if (nfAnimal in NPC.flags) and (ifBaneAnimal in flags) then damage:=Damage * 3;
      if (nfDemon in NPC.flags) and (ifSlayDemon in flags) then damage:=Damage * 3 div 2;
      if (nfDemon in NPC.flags) and (ifWeakDemon in flags) then damage:=Damage  div 2;
      if (nfDemon in NPC.flags) and (ifBaneDemon in flags) then damage:=Damage * 3;
      Damage:=max(Damage,1);
    end
    else
      Damage := getDmgBase+1{+%}+getItemStatBonus(PROP_DMGMOD);
  end
  else // spell
  begin
    with TLuaTable.Create(Game.Lua,'spells',SpellID) do
    try
      State.Init( LuaState.NativeState );
      Damage := RandRange(
        State.CallFunction('dmin',[Spells[SpellID],Self],-1),
        State.CallFunction('dmax',[Spells[SpellID],Self],-1)
      );

      DmgType:=getNumber('type');

    finally
      Free;
    end;
  end;
  // Critical hit
  if (not aRanged) and (Klass = KlassWarrior) and (Random(100) < Level)
    then begin Damage := 2*Damage;
               UI.Msg('You criticaly hit '+NPC.GetName(TheName)+'.');
               end
    else UI.Msg('You hit '+NPC.GetName(TheName)+'.');

  // Warning -- NPC may be invalid afterwards.
  if getItemHasFlags([ifKnockback]) then
    NPC.Knockback(NewDirection(Position, NPC.Position));

  NPC.ApplyDamage(Damage, DmgType,self);

  if ((not Ranged) and (SpellID=0)) then begin
    // Lifesteal and Manasteal
    LifeSteal := GetItemStatBonus(PROP_LIFESTEAL);
    if LifeSteal>0 then
      HP+=max((Damage*LifeSteal+500)div 1000,1)
    else if LifeSteal<0 then
      ApplyDamage(max(-(Damage*LifeSteal-500)div 1000,1));

    LifeSteal := GetItemStatBonus(PROP_MANASTEAL);
    if LifeSteal>0 then
      MP:=min(MPMax,MP+max((Damage*LifeSteal+500)div 1000,1));

    DurabilityCheck(SlotRHand);
  end;

  // Attack hit
  Exit(true);
end;

function TPlayer.Block(npcAttacker : TNPC) : Boolean;
var BlockChance : Word;
begin
    // If npcAttacker = nil this function returns the chance of blocking a trap
    // Rules for blocking: Jarulf 2.2.1
    // Note: "You can only block while standing still or while doing a melee attack."
    // TODO: Should we deny a fleeing player any chance to block?
    if (Recovery = 0) then begin
      if ((Eq[SlotLhand] <> nil) and (Eq[SlotLhand].IType = TYPE_SHIELD)) then begin
        BlockChance := getBlock;
        if (npcAttacker <> nil) then
           BlockChance += 2*(Level-npcAttacker.Level);
        if (Random(100) < BlockChance) then begin
          UI.PlaySound('sound/sfx/items/invsword.wav');
          if (npcAttacker = nil) then
              UI.Msg('You block the trap.')
          else
              UI.Msg('You block the blow of '+npcAttacker.GetName(TheName)+'.');
          Recovery := 10;
          durabilityCheck(SlotLHand);
          Exit(true);
        end;
      end;
    end;
    // Attack was not blocked
    Exit(false);
end;

procedure TPlayer.durabilityCheck(Slot : Byte);
begin
  // Jarulf 3.7.1 - Losing durability
  // Special case for head/torso
  if Slot = slotTorso then begin
    if Eq[slotTorso] = nil then
      Slot := slotHead
    else if Eq[slotHead] <> nil then
      if Random(3)=0 then
        Slot := slotHead;
  end;
  if Eq[slot] = nil then Exit;
  if ifIndest in Eq[slot].flags then exit;
  case slot of
    slotRHand : begin
                  if TYPE_BOW = Eq[Slot].IType then begin
                    if Random(40)=0 then Dec(Eq[slot].Dur);    //Bow
                  end else if Random(30)=0 then Dec(Eq[slot].Dur);
                  // If two weapons are wielded check both
                  if (Eq[slotLHand]<>nil)and(Eq[slotLHand].IType in [ TYPE_WEAPON, TYPE_STAFF ])
                    and (Random(30)=0) then Dec(Eq[slotLHand].Dur);
                end;
    slotTorso, slotHead : if Random(4)<>0 then Dec(Eq[slot].Dur);
    slotLHand : if Random(10)=0 then Dec(Eq[slot].Dur);
  end;
  if Eq[slot].Dur = 0 then
  begin
    UI.Msg('Your '+Eq[slot].GetName(PlainName)+' breaks!');
    case Eq[slot].IType of
         TYPE_HELM: UI.PlaySound('sound/sfx/items/hlmtfkd.wav');
         TYPE_ARMOR: UI.PlaySound('sound/sfx/items/armrfkd.wav');
         TYPE_SHIELD: UI.PlaySound('sound/sfx/items/shielfkd.wav');
    else
         UI.PlaySound('sound/sfx/items/Swrdfkd.wav');
    end;
    FreeAndNil(Eq[slot]);
  end;
end;

procedure TPlayer.Init;
begin
  with TLuaTable.Create( Game.Lua, 'klasses', tid ) do
  try
    klass := GetNumber('klassid');
    spdblk := GetNumber('spdblk');
    spdmag := GetNumber('spdmag');
    if Defined('spellcost') then SpellCost:=getNumber('spellcost') else SpellCost:=100;
    if Defined('skill') then Skill:=Game.Lua.IndexedTable['spells',getString('skill'),'nid'] else Skill := 0;
  finally
    Free;
  end;

  TargetDir.code := 0;
  LevelUp := (Points > 0);

  Game.Lua.RegisterPlayer(Self);

  KillData   := TKillCountArray.Create(true);
  KillDataUnique := TKillCountArray.Create(true);
end;

constructor TPlayer.Create(const thingID :string);
var Count : byte;
begin
  // Set other atributes and find given thingID
  inherited Create(thingID);
  Spell := 0;
  Skill := 0;
  Points := 0;
  Init;

  for Count := 1 to ITEMS_INV do Inv[Count] := nil;
  for Count := 1 to ITEMS_EQ  do Eq [Count] := nil;
  for Count := 1 to MaxSpells  do Spells[Count] := 0;
  for Count := 1 to MaxQuests  do Quests[Count] := 0;
  for Count := 1 to ITEMS_QS do QuickSlots[Count] := nil;
  for Count := 1 to MaxQuickSkills do QuickSkills[Count] := 0;

  with TLuaTable.Create( Game.Lua, 'klasses', tid ) do
  try
    Str := GetNumber('str');
    Mag := GetNumber('mag');
    Dex := GetNumber('dex');
    Vit := GetNumber('vit');

    Name := GetString('dname');
    if isFunction( 'OnCreate' ) then Count := 1 else Count := 0;
  finally
    Free;
  end;

  HP   := getLife;
  MP   := getMana;
  Spell:= 0;

  PortalLevel:= 0;
  PortalCount:= 0;
  MaxDepth   := 0;
  KillCount  := 0;
  LastShop   := 0;

  if Count = 0 then exit;
  Game.Lua.TableExecute('klasses',thingID,'OnCreate');
end;

procedure TPlayer.doAct;
var Dir : TDirection;
begin
  UI.Msg('Choose direction...');
  UI.Draw;
  Dir := CommandDirection( UI.GetCommand );
  if Dir.isProper then
  begin
    if (ActivateCell(Position + Dir)) then
      Dec(SpeedCount,SpdMov);
  end;
end;

//show unfinished quests
procedure TPlayer.doJournal;
begin
  if UI.LeftPanel <> nil then
    if UI.LeftPanel is TJournalWindow then
    begin
      FreeAndNil(UI.LeftPanel);
      UI.AdjustMap;
      exit;
    end;
  FreeAndNil(UI.LeftPanel);
  UI.LeftPanel := TJournalWindow.Create(UI.VUI);
  UI.AdjustMap;
  TJournalWindow(UI.LeftPanel).Run;
end;

//show unfinished quests
procedure TPlayer.doSpellBook;
begin
  if UI.RightPanel <> nil then
    if UI.RightPanel is TSpellWindow then
    begin
      FreeAndNil(UI.RightPanel);
      UI.AdjustMap;
      exit;
    end;
  FreeAndNil(UI.RightPanel);
  UI.RightPanel := TSpellWindow.Create(UI.VUI);
  UI.AdjustMap;
  TSpellWindow(UI.RightPanel).Run;
end;

procedure TPlayer.doQuickSkill;
var Window: TSkillWindow;
    Count : word;
begin
  Window := TSkillWindow.Create(UI.VUI);
  Count :=Window.Run;
  if Count<>$ff then Spell := Count;
  FreeAndNil(Window);
end;

procedure TPlayer.FireArrow;
begin
    UI.PlaySound('sound/sfx/misc/bfire.wav');
    SendMissile(Target, mt_Arrow);
    Dec(SpeedCount,GetAtkSpeed-getItemStatBonus(PROP_SPDATK));
    DurabilityCheck(SlotRHand);
end;

procedure TPlayer.doFire;
begin
  if Game.LevelNumber = 0 then
  begin
    UI.Msg('@<"That won''t work here"@>');
    PlaySound('41');
    Exit;
  end;

  if (Eq[SlotRhand] = nil) or ( not (TYPE_BOW = Eq[SlotRhand].IType) ) or ( not ( ItemCheck(Eq[SlotRHand]) ) ) then
  begin
    UI.Msg('You have no ranged weapon.');
    Exit;
  end;

  if ChooseTarget(TM_FIRE, COMMAND_FIRE) then FireArrow();
end;

function TPlayer.CastSpell(SpellID: Byte; SpellLevel : Byte = 0): Boolean;
var Cost       : DWord;
    slvl       : DWord;
    TargetType : DWord;
    TownSafe   : Boolean;
    fromStaff  : boolean;
    State      : TRLLuaState;
begin
  if SpellID = 0 then
  begin
    UI.Msg('@<"I don''t have a spell ready"@>');
    PlaySound('34');
    Exit(False);
  end;

  if SpellID = 254 then
  begin
    FromStaff := true;
    SpellID := Eq[SlotRHand].Spell;
    SpellLevel := max(1, Spells[SpellID]);
  end
  else
    FromStaff := false;

  if SpellLevel = 0 then
    slvl := Spells[SpellID]
  else
    slvl := SpellLevel;
  slvl+=getItemStatBonus(PROP_SPELLLEVEL);

  with TLuaTable.Create(Game.Lua,'spells',SpellID) do
  try
    if FromStaff then
      if ifUnique in eq[SlotRHand].flags then
        Cost := 0
      else
        Cost := getNumber( 'magic' )
    else
    begin
      State.Init( LuaState.NativeState );
      Cost := max(State.CallFunction('cost',[20, Self],-1),(State.CallFunction('cost',[slvl, Self],-1)*SpellCost)div 100);
    end;
    TargetType:= getNumber( 'target' );
    TownSafe  := getBoolean( 'townsafe' );
  finally
    Free;
  end;
  if (Game.LevelNumber = 0) and (not TownSafe) then
  begin
    UI.Msg('@<"I can''t cast that here"@>');
    PlaySound('27');
    Exit(False);
  end;
  if (SpellLevel = 0) and (Cost > MP) then
  begin
    UI.Msg('@<"Not enough mana"@>');
    PlaySound('35');
    Exit(False);
  end else if FromStaff and (Cost > getMag) then
  begin
    UI.Msg('@<"I can''t cast that yet"@>');
    PlaySound('28');
    Exit(False);
  end;

  case TargetType of
    SPELL_DIRECTION : if not ChooseDirection then Exit(False);
    SPELL_TARGET    : if not ChooseTarget( TM_FIRE, COMMAND_CAST ) then Exit(False);
    else Target:=Position;
  end;
  if SpellLevel = 0 then
    MP -= Cost
  else if FromStaff then
    dec(Eq[SlotRHand].Charges);

  doSpell( SpellID, slvl );
  Dec(SpeedCount,SpdMag);
  Exit(True);
end;


// Command Inventory
procedure TPlayer.doInv;
begin
  if UI.RightPanel <> nil then
  begin
    if UI.RightPanel is TInventoryWindow then
    begin
      FreeAndNil(UI.RightPanel);
      UI.AdjustMap;
      exit;
    end;
  FreeAndNil(UI.RightPanel);
  end;
  UI.RightPanel := TInventoryWindow.Create(UI.VUI);
  UI.AdjustMap;
  TInventoryWindow(UI.RightPanel).Run;
end;

procedure TPlayer.QsCallback(Choice : Byte);
begin
  if not(Choice in [1..ITEMS_QS]) then exit;
  UI.UpdateStatus(QuickSlots[Choice]);
end;

procedure TPlayer.doQuickSlot;
var Window: TQsWindow;
    Choice : word;

begin
  Window := TQsWindow.Create(UI.VUI);
  Choice :=Window.Run;
  FreeAndNil(Window);
  if Choice=0 then exit;
  case ItemToSlot(QuickSlots[Choice]) of
    SlotPotion,
    SlotScroll    : begin
                      UseItem(QuickSlots[Choice]);
                      exit;
                    end;
    SlotFailTown  : begin
                      UI.Msg('@<"I can''t cast that here"@>');
                      PlaySound('27');
                    end;
    else UI.Msg('What?');
  end;
  doQuickSlot;
end;

function TPlayer.addGold(Amount : DWord) : DWord;
var slot : byte;
    rest : DWord;
begin
  slot := InvGoldSlot;
  if slot = 0 then
  begin
    if InvVolume >= MaxVolume then Exit(Amount);
    slot := InvFreeSlot;
    if slot = 0 then Exit(Amount);
    Inv[slot] := TItem.Create('gold');
    Inv[slot].Amount := Amount;
    exit(0);
  end;
  rest := GoldVolume-(Inv[slot].Amount mod GoldVolume);
  if rest = GoldVolume then rest := 0;
  if Amount <= rest then
  begin
    Inc(Inv[slot].Amount,Amount);
    Exit(0);
  end;
  Inc(Inv[slot].Amount,Rest);
  Dec(Amount,Rest);
  while Amount > 0 do
  begin
    if InvVolume >= MaxVolume then Exit(Amount);
    if Amount <= GoldVolume then
    begin
      Inc(Inv[slot].Amount,Amount);
      Inc(Inv[slot].Volume);
      Exit(0);
    end;
    Inc(Inv[slot].Amount,GoldVolume);
    Inc(Inv[slot].Volume);
    Dec(Amount,GoldVolume);
  end;
  Exit(0);
end;

function TPlayer.removeGold(Amount: DWord): Boolean;
var Gold : Byte;
begin
  if getGold < Amount then Exit(False);
  Gold := InvGoldSlot;
  Dec(Inv[Gold].Amount, Amount);
  if Inv[Gold].Amount = 0 then
      FreeAndNil(Inv[Gold])
  else
      Inv[Gold].Volume := ((Inv[Gold].Amount-1) div GoldVolume) + 1;
  Exit(True);
end;

//returns 0 if the item was placed into backpack
//1 if gold, 3 if quickslot, 255 if noroom, 254 if nil
function TPlayer.AddItem(aItem: TItem): byte;
var Gold : Word;
begin
  //check if the item is exist
  if aItem = nil then exit(254);
  //lift the item from the ground
  if (Parent <> nil) then
    if TLevel(Parent).Map[aItem.Position.x, aItem.Position.y].Item = aItem then
      TLevel(Parent).Map[aItem.Position.x, aItem.Position.y].Item := nil;
  aItem.Detach;
  //if it's gold
  if aItem.IType = TYPE_GOLD then
    if InvGoldSlot = 0 then
    begin
      if InvFreeSlot > 0 then
      begin
        UI.Msg('You got '+aItem.GetName(TheName)+'.');
        Dec(SpeedCount,SpdMov);
        Inv[InvFreeSlot] := aItem;
        Exit(1);
      end
      else
        Result := 255;
    end
    else
    begin
      gold := AddGold(aItem.Amount);
      if Gold = 0 then
      begin
        UI.Msg('You got '+aItem.GetName(TheName)+'.');
        Dec(SpeedCount,SpdMov);
        FreeAndNil(aItem);
        Exit(1);
      end;
      Dec(SpeedCount,SpdMov);
      aItem.Amount := gold;
      UI.Msg('You ran out of space in your backpack.');
      Result := 1;
    end
  //or it can be placed into quickslot
  else if (GetFreeSlot>0) and (aItem.IType in [TYPE_POTION, TYPE_SCROLL])and ItemCheck(aItem) then
  begin
    Dec(SpeedCount,SpdMov);
    QuickSlots[GetFreeSlot] := aItem;
    exit(2);
  end
//put it into backpack
  else if (InvSize <> ITEMS_INV) and (InvVolume+aItem.Volume <= MaxVolume) then
    begin
      Inv[InvFreeSlot] := aItem;
      Dec(SpeedCount,SpdMov);
      Exit(0);
    end
  else Result := 255;
  TLevel(Parent).DropItem(aItem, Position);
end;

function TPlayer.FindItemInv(item: string): byte;
var slot : byte;
begin
  for slot:=1 to ITEMS_INV do
    if Inv[slot]<>nil then
      if Inv[slot].tid = item then exit(slot);
  exit(0);
end;

function TPlayer.FindItemEq(item: string): byte;
var slot : byte;
begin
  for slot:=1 to ITEMS_EQ do
    if Eq[slot]<>nil then
      if Eq[slot].tid = item then exit(slot);
  exit(0);
end;

function TPlayer.FindItemQs(item: string): byte;
var slot : byte;
begin
  for slot:=1 to ITEMS_QS do
    if QuickSlots[slot]<>nil then
      if QuickSlots[slot].tid = item then exit(slot);
  exit(0);
end;


procedure TPlayer.doPickUp;
var nItem: TItem;
var nSound: Byte;
begin
  nItem := TLevel(Parent).Map[Position.x,Position.y].Item;
  case AddItem(nItem) of
  0:    begin
          nItem.RunHook( Hook_OnPickUp, [] );
          UI.Msg('You put '+nItem.GetName(TheName)+' into your backpack.');
          Game.Lua.TableExecute('achievment', 'OnPickUp', [nItem.ID]);
        end;
  2:    begin
          UI.Msg('You found '+nItem.GetName(AName)+'.');
          Game.Lua.TableExecute('achievment', 'OnPickUp', [nItem.ID]);
        end;
  254:  UI.Msg('But there is nothing here!');
  255:  begin
          nSound := 14 + Random(3);
          case nSound of
          14: UI.Msg('@<"I can''t carry any more"@>');
          15: UI.Msg('@<"I have no room!"@>');
          16: UI.Msg('@<"Where would I put this?"@>');
          end;
          PlaySound(IntToStr(nSound));
        end;
  end;
end;

procedure TPlayer.doCharInfo;
begin
  if UI.LeftPanel <> nil then
    if UI.LeftPanel is TCharWindow then
    begin
      FreeAndNil(UI.LeftPanel);
      UI.AdjustMap;
      exit;
    end;
  FreeAndNil(UI.LeftPanel);
  UI.LeftPanel := TCharWindow.Create(UI.VUI);
  UI.AdjustMap;
  TCharWindow(UI.LeftPanel).Run;
end;

procedure TPlayer.Action;
var Key : Byte;
    Cell: byte;
    Spec : Variant;

   procedure TryMove( dir : TDirection );
   var NPC : TNPC;
   begin
     with TLevel(Parent) do
     begin
       NPC := Map[Position.x+dir.x,Position.y+dir.y].NPC;
       if NPC <> nil then
       begin
         if NPC.AI = AINPC then
         begin
           NPC.RunHook( Hook_OnTalk, [] );
         end else if NPC.AI = AIGolem then begin
           Map[NPC.Position.x, NPC.Position.y].NPC := nil;
           Displace(Position+Dir);
           Dec(SpeedCount,SpdMov);
           NPC.Displace(Position+Dir.Reversed);
         end else if NPC.AI = AIGuardian then exit
         else if ((Eq[SlotRhand] <> nil) and (TYPE_BOW = Eq[SlotRhand].IType)) then begin
           Target.x := Position.x+dir.x;
           Target.y := Position.y+dir.y;
           FireArrow();
         end else begin
           Dec(SpeedCount,getAtkSpeed-getItemStatBonus(PROP_SPDATK));
           Attack(NPC);
         end;
         Exit;
       end;
       if Cells[Map[Position.x+dir.x,Position.y+dir.y].Cell].Pass then
       begin
         Displace(Position+Dir); Dec(SpeedCount,SpdMov);
         UI.PlaySound('sound/sfx/misc/walk2.wav');
         Exit;
       end;
       if Cells[Map[Position.x+dir.x,Position.y+dir.y].Cell].ActTo <> 0 then
       begin
         ActivateCell(Position+Dir);
         Dec(SpeedCount,SpdMov);
       end;
     end;
   end;

begin
  if HP>GetLife then HP:=GetLife;
  // Again we assume that the parent is a TLevel
  with TLevel(Parent) do
  begin
    LightClear;
    Key:=min(max(100+getItemStatBonus(PROP_LIGHTMOD), 40)div 10, 15);
    LightTrace(Position,Key-1);
    UI.Draw;
    ActivateCell(Position);

    // Portal possibility
    if LevelChange then
    begin
      Dec(SpeedCount,50);
      Exit;
    end;

    if Querry <> nil then
      if (not Querry.Visible)or(nfInvisible in Querry.Flags) then
        Querry := nil;

    if Querry <> nil then
      UI.UpdateStatus(Querry)
    else if Map[Position.x,Position.y].Item <> nil then
    begin
      UI.Msg(Map[Position.x,Position.y].Item.GetName(CAName)+' is lying here.');
      UI.UpdateStatus(Map[Position.x,Position.y].Item)
    end
    else
      UI.UpdateStatus(nil);

  end;

  UI.Draw;
  Key := UI.GetCommand;

  if Key = COMMAND_INVALID then
  begin
   Spec := UI.Config.RunKey( Input.LastKey );
   if VarIsOrdinal(Spec) and (not VarIsType( Spec, varBoolean) ) then
     Key := Spec;
  end;

  Cell := TLevel(Parent).Map[Position.x,Position.y].Cell;
  if CommandDirection( Key ).isProper then
    TryMove( CommandDirection( Key ) )
  else
  case Key of
    COMMAND_SAVE       : begin
                            GameEnd := True;
                            Game.Save;
                            Dec(SpeedCount,50);
                         end;
    COMMAND_ACT        : doAct;
    COMMAND_INVENTORY  : doInv;
    COMMAND_JOURNAL    : doJournal;
    COMMAND_SPELLBOOK  : doSpellBook;
    COMMAND_SPELLS     : doQuickSkill;
    COMMAND_CAST       : CastSpell( Spell );
    COMMAND_LOOK       : ChooseTarget();
    COMMAND_FIRE       : doFire;
    COMMAND_WAIT       : Dec(SpeedCount,SpdMov);
    COMMAND_GOD1       : begin Hp := HpMax; Mp := MpMax; Dec(SpeedCount,10); end;
    COMMAND_GOD2       : begin LevelChange := True; if Game.LevelNumber < 16 then Game.StairNumber := CELL_STAIR_UP; Inc(Game.LevelNumber); Dec(SpeedCount,50); end;
    COMMAND_GOD3       : begin addGold(5000); addExp(2000); end;
    COMMAND_GOD4       : begin {do something} end;
    COMMAND_GOD5       : begin
                           if nfInfravision in flags then
                             exclude(flags, nfInfravision)
                           else
                             include(flags, nfInfravision);
                         end;
    COMMAND_PLAYERINFO : doCharInfo;
    COMMAND_PICKUP     : doPickUp;
    COMMAND_STAIRSDOWN : begin
                           if Cell = CELL_STAIR_DOWN then
                             begin
                               LevelChange := True;
                               if Game.LevelNumber < MaxLevels then
                               begin
                                 Game.StairNumber := CELL_STAIR_UP;
                                 Inc(Game.LevelNumber);
                               end
                               else
                               begin
                                 Game.StairNumber := CELL_STAIR_UP_SPECIAL;
                                 Game.LevelNumber := Game.Lua.IndexedTable['levels',Game.LevelNumber,'special'];
                               end;
                             end
                           else if Cell = CELL_STAIR_DOWN_SPECIAL then
                             begin
                               LevelChange := True;
                               Game.StairNumber := CELL_STAIR_UP;
                               Game.LevelNumber := Game.Lua.IndexedTable['levels',Game.LevelNumber,'special'];
                             end
                           else if Cell = CELL_STAIR_DOWN_CATACOMBS then
                             begin
                               LevelChange := True;
                               Game.StairNumber := CELL_STAIR_UP_CATACOMBS;
                               Game.LevelNumber := 5;
                             end
                           else exit;
                           Dec(SpeedCount,50);
                         end;
    COMMAND_STAIRSUP   : begin
                           if Cell = CELL_STAIR_UP then
                             begin
                               LevelChange := True;
                               If Game.LevelNumber > MaxLevels then
                               begin
                                 Game.StairNumber := CELL_STAIR_DOWN_SPECIAL;
                                 Game.LevelNumber := Game.Lua.IndexedTable['levels',Game.LevelNumber,'special'];
                               end
                               else
                               begin
                                 Game.StairNumber := CELL_STAIR_DOWN;
                                 Dec(Game.LevelNumber);
                               end;
                           end
                           else if Cell =  CELL_STAIR_UP_SPECIAL then
                             begin
                               LevelChange := True;
                               Game.StairNumber := CELL_STAIR_DOWN;
                               Game.LevelNumber := Game.Lua.IndexedTable['levels',Game.LevelNumber,'special'];
                             end
                           else if Cell = CELL_STAIR_UP_CATACOMBS then
                             begin
                               LevelChange := True;
                               Game.StairNumber := CELL_STAIR_DOWN_CATACOMBS;
                               Game.LevelNumber := 0;
                             end
                           else exit;
                           Dec(SpeedCount, 50);
                         end;
    COMMAND_QUICKSLOT  : doQuickSlot;
    COMMAND_CWIN,
    COMMAND_ESCAPE     : begin
                           if UI.RightPanel <> nil then
                           begin
                           FreeAndNil(UI.RightPanel);
                           UI.AdjustMap;
                           end;
                           if UI.LeftPanel <> nil then
                           begin
                           FreeAndNil(UI.LeftPanel);
                           UI.AdjustMap;
                           end;
                         end;
    else UI.Msg('Unknown command. Read the help file!');
  end;
end;

procedure TPlayer.WriteMemorial;
var
    Count,Size : Word;
    Score : DWord;
    List  : Variant;

begin
  Score := (Level * Level * 1000) + Exp + (TLevel(Parent).Level * 500) + getLife * 20;
  Size := Game.Lua.Table['quests','__counter'];
  for Count := 1 to Size do
    if Quests[Count]>0 then
      with TLuaTable.Create( Game.Lua, 'quests', Count ) do
      try
        if (getBoolean('enabled'))and(Game.Player.Quests[Count] >= getNumber('completed'))and
           (Game.Player.Quests[Count] < getNumber('failed')) then
          Score+=getNumber('score')
        else  Game.Player.Quests[Count]:=0;
      finally
        Free;
      end;

  Assign(VF,'mortem.txt');
  Rewrite(VF);
  Writeln(VF,'--------------------------------------------------------------');
  Writeln(VF,'  DiabloRL '+VERSION+' roguelike postmortem character dump');
  Writeln(VF,'--------------------------------------------------------------');
  Writeln(VF,'');
  if Game.LevelNumber = 9 then
    Writeln(VF,'  '+Name+', level '+IntToStr(Level)+' '+LowerCase(KlassName)+', entered the Caves,')
  else
    if Querry <> nil then
      Writeln(VF,'  '+Name+', level '+IntToStr(Level)+' '+LowerCase(KlassName)+', killed by '+Querry.GetName(AName)+' in '+TLevel(Parent).Name+',')
    else
      Writeln(VF,'  '+Name+', level '+IntToStr(Level)+' '+LowerCase(KlassName)+', died in '+TLevel(Parent).Name+',');
  Writeln(VF,'  after '+IntToStr(Game.TurnCount div 4)+' turns.');
  Writeln(VF,'  '+KlassPronoun+' scored '+IntToStr(Score)+' points, killing '+IntToStr(KillCount)+' hellspawn.');
  Writeln(VF,'');
  Writeln(VF,'  '+KlassPronoun+' advanced to level '+IntToStr(Level)+' gaining '+IntToStr(Exp)+' experience.');
  Writeln(VF,'  '+KlassPronoun+' amassed '+IntToStr(getGold)+' gold coins.');
  for Count := 1 to Size do
    if Quests[Count]>0 then
        Writeln(VF,'  '+KlassPronoun+' '+Game.Lua.IndexedTable['quests',Count,'message']);
  Size := Game.Lua.Table['achievments','__counter'];
  if Game.LevelNumber = 9 then
    for Count := 1 to Size do
      if Game.Lua.IndexedTable['achievments',Count,'active'] then
        Writeln(VF,'  '+KlassPronoun+' '+Game.Lua.IndexedTable['achievments',Count,'message']);
  Writeln(VF,'');
  Writeln(VF,'-- Statistics ------------------------------------------------');
  Writeln(VF,'');
  Writeln(VF,'  Strength   '+IntToStr(getStr)+'/'+IntToStr(Str));
  Writeln(VF,'  Magic      '+IntToStr(getMag)+'/'+IntToStr(Mag));
  Writeln(VF,'  Dexterity  '+IntToStr(getDex)+'/'+IntToStr(Dex));
  Writeln(VF,'  Vitality   '+IntToStr(getVit)+'/'+IntToStr(Vit));
  Writeln(VF,'');
  Writeln(VF,'  Life '+IntToStr(getLife)+'/'+IntToStr(HP)+'  Mana '+IntToStr(getMana)+'/'+IntToStr(MP));
  Writeln(VF,'  Armor '+IntToStr(getAC)+'  ToHit '+IntToStr(getToHit));
  Writeln(VF,'');
  Writeln(VF,'-- Spells ----------------------------------------------------');
  for Count := 1 to MaxSpells do
    if Spells[Count] > 0 then
      Writeln(VF, '  '+Game.Lua.IndexedTable['spells',Count,'name']+' level ' + inttostr(Spells[Count]));
  Writeln(VF,'');
  Writeln(VF,'-- Equipment -------------------------------------------------');
  Writeln(VF,'');
  for Count := 1 to ITEMS_EQ do
    if Eq[Count] = nil then Writeln(VF,Output.Strip('  [ '+SlotName(Count)+' ]  nothing'))
                       else
                       begin
                         Eq[Count].Identify;
                         Writeln(VF,Output.Strip('  [ '+SlotName(Count)+' ] '+Eq[Count].GetName(PlainName)));
                       end;
  Writeln(VF,'');
  Writeln(VF,'-- Quickslots ------------------------------------------------');
  Writeln(VF,'');
  for Count := 1 to ITEMS_QS do
    if QuickSlots[Count] <> nil then Writeln(VF,Output.Strip('  [ Slot '+IntToStr(Count)+' ] '+QuickSlots[Count].GetName(PlainName)));
  Writeln(VF,'');
  Writeln(VF,'-- Inventory -------------------------------------------------');
  Writeln(VF,'');
  Count := 1;
  if invSize <> 0 then
  repeat
    if Inv[Count] <> nil then
    begin
      Inv[Count].Identify;
      Writeln(VF,Output.Strip('  '+Inv[Count].GetName(PlainName)));
    end;
    Inc(Count);
  until Count >= ITEMS_INV
  else Writeln(VF,' < EMPTY > ');
  Writeln(VF,'');
  Writeln(VF,'-- Kills -----------------------------------------------------');
  Writeln(VF,'');

  KillData.WriteAll(VF,true);
  KillDataUnique.WriteAll(VF,false);
  Writeln(VF,'');
  Writeln(VF,'-- Messages --------------------------------------------------');
  Writeln(VF,'');
  for Count := 20 downto 0 do
    if UI.Messages.Get(Count)<>'' then
      Writeln(VF,'  '+Output.Strip(UI.Messages.Get(Count)));
  Writeln(VF,'');
  Writeln(VF,'--------------------------------------------------------------');
  Close(VF);
end;

// Takes care of the items.
destructor TPlayer.Destroy;
var Count : byte;
begin
  for Count := 1 to ITEMS_QS do FreeAndNil(QuickSlots[Count]);
  for Count := 1 to ITEMS_INV do FreeAndNil(Inv[Count]);
  for Count := 1 to ITEMS_EQ do FreeAndNil(Eq [Count]);
  FreeAndNil(KillData);
  FreeAndNil(KillDataUnique);
  inherited Destroy;
end;

function TPlayer.getProperty( PropertyID : Byte ) : Variant;
begin
  case PropertyID of
    // TODO : should these be as a property?
    PROP_AC     : Exit( getAC );
    PROP_DMGMIN : Exit( getDmgMin );
    PROP_DMGMAX : Exit( getDmgMax );
    PROP_TOHIT  : Exit( getToHit );

    PROP_STR    : Exit( Str );
    PROP_MAG    : Exit( Mag );
    PROP_DEX    : Exit( Dex );
    PROP_VIT    : Exit( Vit );
    PROP_LIFE,
    PROP_HPMAX  : Exit( getLife );
    PROP_MANA   : Exit( getMana );

    PROP_KLASS  : Exit( Klass );
    PROP_EXP    : Exit( Exp );
    PROP_MP     : Exit( MP );
    PROP_POINTS : Exit( Points );
    PROP_HPMOD  : Exit( HPMod );
    PROP_MPMOD  : Exit( MPMod );
    PROP_MAXDEPTH : Exit( MaxDepth );
    PROP_PORTALLEVEL : Exit( PortalLevel );
    PROP_GOLD   : Exit( getGold );
    PROP_VOLUME : Exit( InvVolume );
    else Exit( inherited getProperty(PropertyID) )
  end;
end;

procedure TPlayer.setProperty(PropertyID: Byte; Value: Variant);
begin
  case PropertyID of
    // TODO : these should report error!
    // PROP_AC     : AC := Value;
    // PROP_DMGMIN : DmgMin := Value;
    // PROP_DMGMAX : DmgMax := Value;
    // PROP_TOHIT  : ToHit := Value;

    PROP_STR    : Str := Value;
    PROP_MAG    : Mag := Value;
    PROP_DEX    : Dex := Value;
    PROP_VIT    : Vit := Value;
    //PROP_LIFE,  //No effect Use PROP_HPMOD and PROP_MPMOD
    //PROP_HPMAX  : Stats[StatLife] := Value;
    //PROP_MANA   : Stats[StatMana] := Value;

    // PROP_KLASS  : Klass := Value; // READ-ONLY
    PROP_EXP    : AddExp( Value - Exp );
    PROP_MP     : MP := Value;
    PROP_POINTS : Points := Value;
    PROP_HPMOD  : HPMod := Value;
    PROP_MPMOD  : MPMod := Value;

    else inherited setProperty(PropertyID, Value);
  end;
end;

procedure TPlayer.PlaySound(sID : string);
begin
     // TODO: Special case for wario/warior
     UI.PlaySound(Sound + sID + '.wav');
end;

procedure TPlayer.Displace( new : TCoord2D );
begin
     inherited Displace(new);
     UI.SetListenerPosition(Position.x,Position.y);
end;

procedure TKillCountArray.WriteAll(var VF : Text; ShowCount : boolean);
var iPtr   : PAssocArrayEntry;
    iCount : DWord;
begin
  for iCount := 0 to 95 do
    begin
      iPtr := FEntries[iCount];
      while iPtr <> nil do
      begin
        if (ShowCount) then
          WriteLn(VF, '  '+padded(iPtr^.Key,25)+IntToStr(iPtr^.Value))
        else
          WriteLn(VF, '  '+iPtr^.Key);
        iPtr := iPtr^.Next;
      end;
    end;
end;

constructor TPlayer.CreateFromStream(ISt: TStream);
var Count: byte;
    tsob : set of byte; //temporary set of byte
begin
  inherited CreateFromStream(ISt);

  MP  := ISt.ReadWord;
  MPMax  := ISt.ReadWord;
  MPMod  := ISt.ReadByte;
  Str := ISt.ReadWord;
  Mag := ISt.ReadWord;
  Dex := ISt.ReadWord;
  Vit := ISt.ReadWord;
  Exp := ISt.ReadDWord;
  Points := ISt.ReadByte;

  Spell         := ISt.ReadByte;
  for Count := 1 to MaxSpells do
    Spells[Count] := ISt.ReadByte;
  for Count := 1 to MaxQuickSkills do
    QuickSkills[Count] := ISt.ReadByte;

  ISt.Read(tsob, sizeof(tsob));
  for Count := 1 to Game.Lua.Table['quests','__counter']  do
    begin
      if Count in tsob then
        Game.Lua.IndexedTable['quests', Count, 'enabled'] := true
      else
        Game.Lua.IndexedTable['quests', Count, 'enabled'] := false;
      Quests[Count] := ISt.ReadByte;
    end;

  ISt.Read(tsob, sizeof(tsob));
  for Count := 1 to Game.Lua.Table['quests','__counter']  do
    if Count in tsob then
    begin
      Game.Lua.IndexedTable['achievments', Count, 'active'] := true;
      Game.Lua.IndexedTable['achievments', Count, 'value'] := ISt.ReadWord;
    end
    else
      Game.Lua.IndexedTable['achievments', Count, 'active'] := false;

      ISt.Read(tsob, sizeof(tsob));
  for Count := 1 to ITEMS_EQ  do
    if Count in tsob then
      Eq[Count] := TItem.CreateFromStream(ISt)
    else
      Eq[Count] := nil;

  ISt.Read(tsob, sizeof(tsob));
  for Count := 1 to ITEMS_INV  do
    if Count in tsob then
      Inv[Count] := TItem.CreateFromStream(ISt)
    else
      Inv[Count] := nil;

  ISt.Read(tsob, sizeof(tsob));
  for Count := 1 to ITEMS_QS  do
    if Count in tsob then
      QuickSlots[Count] := TItem.CreateFromStream(ISt)
    else
      QuickSlots[Count] := nil;

  KillCount     := ISt.ReadDWord;
  MaxDepth      := ISt.ReadByte;
  PortalLevel   := ISt.ReadByte;
  PortalCount   := ISt.ReadByte;
  LastShop      := ISt.ReadDword;

  Init;

  KillData.ReadFromStream(ISt);
  KillDataUnique.ReadFromStream(ISt);
end;

procedure TPlayer.ToStream(OSt: TStream);
var Count: byte;
    tsob : set of byte; //temporary set of byte
begin
  inherited ToStream(OSt);

  Ost.WriteWord(MP);
  Ost.WriteWord(MPMax);
  Ost.WriteByte(MPMod);
  OSt.WriteWord(Str);
  OSt.WriteWord(Mag);
  OSt.WriteWord(Dex);
  OSt.WriteWord(Vit);
  OSt.WriteDWord(Exp);
  OSt.WriteByte(Points);

  OSt.WriteByte(Spell);
  for Count := 1 to MaxSpells do
    OSt.WriteByte(Spells[Count]);
  for Count := 1 to MaxQuickSkills do
    OSt.WriteByte(QuickSkills[Count]);

  tsob := [];
  for Count := 1 to Game.Lua.Table['quests','__counter'] do
    if Game.Lua.IndexedTable['quests', Count, 'enabled'] then
      include(tsob, Count);
  OSt.Write(tsob, sizeof(tsob));
  for Count := 1 to Game.Lua.Table['quests','__counter'] do
      OSt.WriteByte(Quests[Count]);
  tsob := [];
  for Count := 1 to Game.Lua.Table['achievments','__counter'] do
    if Game.Lua.IndexedTable['achievments', Count, 'active'] then
      include(tsob, Count);
  OSt.Write(tsob, sizeof(tsob));
  for Count := 1 to Game.Lua.Table['achievments','__counter'] do
    if (Count in tsob) then
      Ost.WriteWord(Game.Lua.IndexedTable['achievments', Count, 'value']);

  tsob := [];
  for Count := 1 to ITEMS_EQ do
    if Eq[Count]<>nil then
      include(tsob, Count);
  OSt.Write(tsob, sizeof(tsob));
  for Count := 1 to ITEMS_EQ do
    if Count in tsob then
      Eq[Count].ToStream(OSt);

  tsob := [];
  for Count := 1 to ITEMS_INV do
    if Inv[Count]<>nil then
      include(tsob, Count);
  OSt.Write(tsob, sizeof(tsob));
  for Count := 1 to ITEMS_INV do
    if Count in tsob then
      Inv[Count].ToStream(OSt);

  tsob := [];
  for Count := 1 to ITEMS_QS do
    if QuickSlots[Count]<>nil then
      include(tsob, Count);
  OSt.Write(tsob, sizeof(tsob));
  for Count := 1 to ITEMS_QS do
    if Count in tsob then
      QuickSlots[Count].ToStream(OSt);

  OSt.WriteDWord(KillCount);
  OSt.WriteByte(MaxDepth);
  OSt.WriteByte(PortalLevel);
  OSt.WriteByte(PortalCount);
  OSt.WriteDWord(LastShop);

  KillData.WriteToStream(OSt);
  KillDataUnique.WriteToStream(OSt);
end;

function lua_player_spell_get(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    player : TPlayer;
begin
  State.ObjectInit(L,player);
  State.Push(player.Spells[State.ToInteger(2)]);
  Result := 1;
end;

function lua_player_spell_set(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    player : TPlayer;
begin
  State.ObjectInit(L,player);
  player.Spells[State.ToInteger(2)] := State.ToInteger(3);
  Result := 0;
end;

function lua_player_add_gold(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    player : TPlayer;
begin
  State.ObjectInit(L,player);
  State.Push( player.addGold( State.ToInteger(2) ) );
  Result := 1;
end;

function lua_player_remove_gold(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    player : TPlayer;
begin
  State.ObjectInit(L,player);
  State.Push( player.removeGold( State.ToInteger(2) ) );
  Result := 1;
end;

function lua_player_quest_set(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    player : TPlayer;
begin
  State.ObjectInit(L,player);
  player.Quests[State.ToInteger(2)] := State.ToInteger(3);
  Result := 0;
end;

function lua_player_quest_get(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    player : TPlayer;
begin
  State.ObjectInit(L,player);
  State.Push( player.Quests[ State.ToInteger(2) ] );
  Result := 1;
end;

function lua_player_summon_portal (L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    player : TPlayer;
begin
  State.ObjectInit(L,player);
  player.PortalCount := State.ToInteger(2);
  Result := 0;
end;

function lua_player_play_sound(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    player : TPlayer;
begin
  State.ObjectInit(L,player);
  player.PlaySound(State.ToString(2));
  Result := 0;
end;

function lua_player_add_item(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    Player : TPlayer;
    Item   : TItem;
begin
  State.ObjectInit(L,player);
  if State.IsTable(2) then
    Item := State.ToObject(2) as TItem
  else
    Item := TItem.Create(State.ToString(2));
  State.Push( Player.AddItem(Item) < 254 );
  Result := 1;
end;


function lua_player_item_get(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    Player : TPlayer;
    Slot   : Word;
begin
  State.ObjectInit(L,Player);

  if not State.IsNumber(2) then
  begin
    Slot := Player.FindItemInv(State.ToString(2));
    if Slot = 0 then
    begin
      Slot := Player.FindItemEq(State.ToString(2)) + ITEMS_INV;
      if Slot = ITEMS_INV then
      begin
        Slot := Player.FindItemQs(State.ToString(2)) + ITEMS_INV + ITEMS_EQ;
        if Slot = ITEMS_INV+ITEMS_EQ then Slot := 0;
      end;
    end;
  end
  else
    Slot := State.ToInteger(2);
  if (Slot = 0) or (Slot > ITEMS_ALL) then State.Push( nil )
  else if SLOT <= ITEMS_INV then State.Push( Player.Inv[Slot] )
  else if SLOT <= ITEMS_EQ + ITEMS_INV then State.Push( Player.Eq[Slot-ITEMS_INV] )
  else State.Push( Player.QuickSlots[Slot-ITEMS_INV-ITEMS_EQ] );
  Result := 1;
end;

function lua_player_item_set(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    Player : TPlayer;
    Item   : TItem;
    ID     : AnsiString;
    Slot   : Word;
begin
  State.ObjectInit(L,Player);
  if not State.IsNumber(2) then
  begin
    Slot := Player.FindItemInv(State.ToString(2));
    if Slot = 0 then
    begin
      Slot := Player.FindItemEq(State.ToString(2)) + ITEMS_INV;
      if Slot = ITEMS_INV then
      begin
        Slot := Player.FindItemQs(State.ToString(2)) + ITEMS_INV + ITEMS_EQ;
        if Slot = ITEMS_INV+ITEMS_EQ then exit(0);
      end;
    end;
  end
  else
    Slot := State.ToInteger(2);

  if Slot = 0 then exit(0);
  Item := nil;

  if State.IsString(3) then
  begin
    ID := State.ToString(3);
    if ID <> '' then Item := TItem.Create( State.ToString(3) )
  end
  else if State.IsObject(3) then Item := State.ToObject(3) as TItem;

  if Slot <= ITEMS_INV then
  begin
    FreeAndNil(Player.Inv[slot]);
    if Item <> nil then Player.inv[slot] := Item;
  end
  else if Slot <= ITEMS_INV+ITEMS_EQ then
  begin
    Slot-=ITEMS_INV;
    FreeAndNil(Player.Eq[slot]);
    if Item <> nil then Player.Eq[slot] := Item;
  end
  else if Slot <= ITEMS_ALL then
  begin
    Slot-=ITEMS_INV+ITEMS_EQ;
    FreeAndNil(Player.QuickSlots[slot]);
    if Item <> nil then Player.QuickSlots[slot] := Item;
  end;
  Result := 0;
end;

procedure TPlayer.useQuickSlot( slot: byte );
begin
  if not (slot in [1..8]) then exit;
  if QuickSlots[slot] <> nil then
    UseItem(QuickSlots[slot])
  else
    with TLevel(Parent).Map[Position.x,Position.y] do
      if (Item <> nil) and (Item.IType in [TYPE_POTION, TYPE_SCROLL]) then
      begin
        UI.Msg('You found '+Item.GetName(AName)+'.');
        Dec(SpeedCount,SpdMov);
        Item.Detach;
        QuickSlots[slot] := Item;
        Item := nil;
      end
      else
        UI.Msg('No quick item in slot '+IntToStr(slot)+'!');
end;

procedure TPlayer.useQuickSkill( slot: byte );
begin
  if not (slot in [1..4]) then exit;
  if QuickSkills[slot] > 0 then
    Spell:=QuickSkills[slot];
end;

class procedure TPlayer.RegisterLuaAPI(Lua: TLua);
begin
  Lua.SetTableFunction('player','spell_get',    @lua_player_spell_get);
  Lua.SetTableFunction('player','spell_set',    @lua_player_spell_set);
  Lua.SetTableFunction('player','add_gold',     @lua_player_add_gold);
  Lua.SetTableFunction('player','remove_gold',  @lua_player_remove_gold);
  Lua.SetTableFunction('player','quest_get',    @lua_player_quest_get);
  Lua.SetTableFunction('player','quest_set',    @lua_player_quest_set);
  Lua.SetTableFunction('player','summon_portal',@lua_player_summon_portal);
  Lua.SetTableFunction('player','play_sound',   @lua_player_play_sound);
  Lua.SetTableFunction('player','add_item',     @lua_player_add_item);
  Lua.SetTableFunction('player','item_get',     @lua_player_item_get);
  Lua.SetTableFunction('player','item_set',     @lua_player_item_set);
end;

end.
