{$MODE OBJFPC}
// @abstract(NPC for RL Core)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(January 17, 2005)
// @lastmod(January 17, 2009)
//

unit rlnpc;
interface
uses classes, sysutils, vlua, vmath, vrltools, rlthing, rlglobal;

type

{ TNPC }

TNPC = class(TThing)
       SpeedCount : Integer;
       AI         : Byte;
       Intf       : Byte; //Intelligence factor
       Corpse     : Byte;
       Target     : TCoord2D;
       AC         : Word;
       ToHit      : Word;
       HP         : Integer;
       HPMax      : Word;
       ExpValue   : Word;
       SpdAtk     : Byte;
       SpdMov     : Byte;
       SpdHit     : Byte;
       Recovery   : Word;
       RegenTick  : Word;
       Special    : byte;  //something needed for AI;
       Special2   : byte;  //something needed for AI;
       LifeSteal  : ShortInt;
       Activated  : Boolean;
       Sound      : String;
       // Overriden constructor -- sets x,y to (1,1).
       constructor Create(const thingID :string); override;
       procedure Action; virtual;
       // Opening doors/Closing doors/Opening Sacrophagus/etc...
       function ActivateCell(c : TCoord2D) : Boolean;
       // Attack NPC
       function Attack(NPC : TNPC; Ranged: Boolean = false; SpellID : Byte = 0) : boolean; virtual;
       // Spell
       procedure doSpell( SpellID, SpellLevel : DWord);
       // Block Attack
       function Block(npcAttacker : TNPC) : Boolean; virtual;
       // TimeFlow. See GameObject.
       procedure TimeFlow(time : LongInt); override;
       // returns AC
       function getAC : Word; virtual;
       // returns ToHit
       function getToHit : Word; virtual;
       // returns HPMax
       function getLife : Word; virtual;
       // Returns NPC's name
       function GetName(outputType : TNameOutputType) : string; override;
       // Damage application.
       procedure applyDamage(Damage : DWord; DamageType: byte = DAMAGE_GENERAL; Attacker: TNPC = nil);
       // Death procedure -- does all cleanup.
       procedure Die(Attacker: TNPC = nil); virtual;
       // Displays the message only if the given NPC is a Player.
       procedure PlayerMsg(const str : string);
       // Checks wether this given NPC is the player
       function IsPlayer : boolean;
       // Returns amount of experience gained by killer of killerLevel
       function getExpValue(killerLevel : Byte) : DWord;
       // Seeks given position; dst is for distance.
       //positive value = come closer, negatice = farther, zero = walk circle;
       procedure Seek( c : TCoord2D; dst : shortint = 1 );
       // Sends a missile
       procedure SendMissile( tg : TCoord2D; mtype : Byte; mid : DWord = 0; explosion : byte = 0);
       //Seek for target
       function ClosestTarget(AISet: TAITypeSet; Radius : Byte):TNPC;
       // Returns a property value
       function getProperty( PropertyID : Byte ) : Variant; override;
       // Sets a property value
       procedure setProperty( PropertyID : Byte; Value : Variant ); override;
       // Plays a sound
       procedure PlaySound(sID : char);
       // knockback
       procedure Knockback(dir: TDirection);
       // Check charge availability
       function CanCharge( c: TCoord2D ) : boolean;
       // Stream constructor
       constructor CreateFromStream( ISt : TStream ); override;
       // Stream writer
       procedure ToStream( OSt : TStream ); override;

       // register lua functions
       class procedure RegisterLuaAPI(Lua: TLua);

     end;

implementation
uses vvision,
     vutil, voutput,
     rllua, rlgobj, rlgame,rlcell,rllevel, rlplayer,
     rlui;

type TNavPoint = class
  public
    x : integer;
    y : integer;
    _next : TNavPoint;
    _prev : TNavPoint;
    step : word;
    dist : word;

    Constructor Create(const nx,ny: Integer; nstep : word = 0; ndist : word = 0);
    procedure Move(dest: TNavPoint);
    procedure Add(new_child: TNavPoint);
    function  Includes(Coord: TCoord2D; newstep: word) : boolean;
    Destructor Destroy;
  end;

Constructor TNavPoint.Create(const nx,ny: Integer; nstep : word = 0; ndist : word = 0);
begin
  x := nx;
  y := ny;
  step := nstep;
  dist := ndist;
  _next := nil;
  _prev := nil;
end;

Destructor TNavPoint.Destroy;
begin
  if _next<>nil then _next.Free;
end;

procedure TNavPoint.Move(dest: TNavPoint);
begin
  if _prev<>nil then _prev._next := _next;
  if _next<>nil then _next._prev := _prev;
  _prev := dest;
  _next := dest._next;
  dest._next := self;
end;

procedure TNavPoint.Add(new_child: TNavPoint);
begin
  if new_child.step+new_child.dist < step+dist then
    self.Move(new_child)
  else if _next <> nil then
    _next.Add(new_child)
  else
    new_child.Move(self);
end;

function TNavPoint.Includes(Coord: TCoord2D; newstep: word) : boolean;
begin
  if (Coord.x = x) and (Coord.y = y) then
  begin
    if newstep < step then step := newstep;
    exit(true);
  end
  else if (_next<>nil) then exit(_next.Includes(Coord, newstep))
  else exit(false);
end;

constructor TNPC.Create(const thingID :string);
var table : string;
    LTable : TLuaTable;
begin
  // Set other atributes and find given thingID
  inherited Create(thingID);

  if not Game.Lua.Defined('npcs',tid) then
    if Game.Lua.Defined('klasses',tid) then
      table := 'klasses'
    else
      Game.Lua.Error('NPC "'+tid+'" not found!')
  else
    table := 'npcs';

  LTable := TLuaTable.Create( Game.Lua, table, tid );
  with LTable do
  try
    Name    := GetString('name');
    Flags   := GetFlags('flags');

    Pic     := GetString('pic')[1];
    Color   := GetNumber('color');
    Level   := GetNumber('level');

    Corpse  := CellID[GetString('corpse')];
    AI      := GetNumber('ai');
    Intf    := GetNumber('intf');
    HPMax   := RandRange(GetNumber('hpmin'),GetNumber('hpmax'));
    AC      := GetNumber('ac');
    DmgMin  := GetNumber('dmgmin');
    DmgMax  := GetNumber('dmgmax');
    ExpValue:= GetNumber('expvalue');
    if (nfUnique in Flags) then ExpValue := ExpValue * 2;
    SpdAtk  := GetNumber('spdatk');
    SpdMov  := GetNumber('spdmov');
    SpdHit  := GetNumber('spdhit');
    ToHit   := GetNumber('tohit');
    Sound   := GetString('sound');

    ReadHooks( LTable );
  finally
    Free;
  end;

  HP := HPMax;
  SpeedCount := 90-Random(40);
  Activated  := False;
  Recovery := 0;
  Special := 0;
  Special2 := 0;
end;

//Find closest target matching AI
function TNPC.ClosestTarget(AISet: TAITypeSet; Radius : Byte):TNPC;
var scx, scy, t, ran : Word;
begin
  ran:=1000;
  for scx := Max(1,Position.x-Radius-1) to Min(MapSizeX,Position.x+Radius+1) do
    for scy := Max(1,Position.y-Radius-1) to Min(MapSizeY,Position.y+Radius+1) do
      if not (TLevel(Parent).CellCheck(NewCoord2D(scx,scy), [cfNoMonsters])) then
        if (TLevel(Parent).Map[scx,scy].NPC.visible)and not(nfInvisible in TLevel(Parent).Map[scx,scy].NPC.Flags) then
          if (TLevel(Parent).Map[scx,scy].NPC.AI in AISet) then
          begin
            t := Distance(NewCoord2D(scx,scy),Position);
            if t <= ran then
            begin
              ran:=t;
              Target.x := scx;
              Target.y := scy;
            end;
          end;
  if ran = 1000 then
    exit(nil)
  else
    exit(TLevel(Parent).Map[Target.x,Target.y].NPC);
end;

// Returns NPC's name
function TNPC.GetName(outputType : TNameOutputType) : string;
begin
  if not Visible then
    if OutputType in [CAName, CTheName] then exit('Someone') else exit('someone');
  if (nfUnique in flags) then
    if OutputType in [CAName, CTheName] then exit(Capitalized(Name)) else exit(Name);
  exit(inherited getname(outputType));
end;

function TNPC.getAC : Word;
begin
  Exit(AC);
end;

function TNPC.getToHit : Word;
begin
  if nfCharge in flags then exit(500) else Exit(ToHit);
end;

function TNPC.getLife : Word;
begin
  Exit(HPMax);
end;


function TNPC.IsPlayer : boolean;
begin
  Exit(AI = AIPlayer);
end;

// Attack Player
function TNPC.Attack(NPC : TNPC; Ranged: boolean = false; SpellID : Byte = 0) : Boolean;
var HitChance  : Integer;
    Damage     : Integer;
    DmgType    : byte;
begin
  if NPC = nil then Exit;
  Dec(SpeedCount,SpdAtk);

  if not(nfStatue in NPC.flags) then
  begin
    PlaySound('a');
    HitChance := 30+getToHit+2*Level;
    HitChance := HitChance-(NPC.getAC+2*NPC.Level);

    if HitChance < 15  then HitChance := 15;
    case TLevel(Parent).Level of
      14 : if HitChance < 20  then HitChance := 20;
      15 : if HitChance < 25  then HitChance := 25;
      16 : if HitChance < 30  then HitChance := 30;
    end;

    if Random(100) >= HitChance then
    begin
      if NPC.isPlayer then
        UI.Msg(GetName(CTheName)+' misses you.')
      else
        UI.Msg(GetName(CTheName)+' misses '+NPC.GetName(TheName)+'.');
      Exit(false);
    end;

    if (NPC.Block(self)) then begin
       // Attack hit, but was blocked
       Exit(true);
    end;
  end;

  if NPC.isPlayer then
    UI.Msg(GetName(CTheName)+' hits you.')
  else
    UI.Msg(GetName(CTheName)+' hits '+NPC.GetName(TheName)+'.');

  Damage := RandRange(DmgMin,DmgMax);

  DmgType := DAMAGE_GENERAL;
  if spellID > 0 then
  begin
    DmgType:=Game.Lua.IndexedTable['spells',SpellID,'type'];
  end;

  if NPC.isPlayer then
  begin
    Damage+=TPlayer(NPC).GetItemStatBonus(PROP_DMGTAKEN);
    if Damage<1 then Damage:=1;
    TPlayer(NPC).durabilityCheck(SlotTorso);
  end;
  // Warning -- NPC may be invalid afterwards.

  RunHook( Hook_OnAttack, [NPC] );
  NPC.ApplyDamage(Damage, dmgtype, self);
  if ((not Ranged) and (SpellID=0)) then begin
     // LifeSteal for melee attacks only
     if LifeSteal>0 then
        HP+=max((Damage*LifeSteal+500)div 1000,1);
  end;

  // Attacker hit
  Exit(true);
end;

function TNPC.Block(npcAttacker : TNPC) : Boolean;
begin
     npcAttacker := npcAttacker; // Suppress unused argument hint
     // Monsters can never block attacks against them. Jarulf 2.2.1
     Exit(false);
end;

function TNPC.ActivateCell( c : TCoord2D ) : Boolean;
var CellID : Byte;
begin
  with TLevel(Parent) do
  begin
    cellID := Map[c.x,c.y].Cell;
    if (c<>position)and(Cells[CellID].pass) then
    if (Map[c.x,c.y].NPC <> nil) or (Map[c.x,c.y].Item <> nil) then
    begin
      PlayerMsg('There''s something blocking the way.');
      Exit(false);
    end;
    //focus Lua on the cell activator and execute script
    with TLuaTable.Create( Game.Lua, 'cells', Cells[CellID].id ) do
    try
      if (Cells[CellID].ActTo <> 0)and(c<>position) then
      begin
        Map[c.x,c.y].Cell := Cells[cellID].ActTo;
        if isFunction('OnAct') then
          Execute('OnAct', [c.x, c.y]);
      end
      else if (c = position) then
      begin
        if isFunction('OnStep') then
          Execute('OnStep', [c.x, c.y]);
      end else begin
        PlayerMsg('There''s nothing to act upon there.');
        Exit(false);
      end;
    finally
      Free;
    end;
  end;
  Exit(true);
end;


procedure TNPC.applyDamage(Damage : DWord; DamageType: byte = DAMAGE_GENERAL; Attacker: TNPC = nil);
begin
  if HP <= 0 then exit;  //fix player's double death
  if AI = AINPC then exit;
  if isPlayer and (Attacker = nil) and (TPlayer(self).getItemHasFlags([ifTrapResist])) then
    Damage := max(Damage div 2, 1);
  if ((DamageType = DAMAGE_HOLY) and (not (nfUndead in flags))) or
     ((DamageType = DAMAGE_FIRE) and (nfFireImmune in flags)) or
     ((DamageType = DAMAGE_LIGHTNING) and (nfLightningImmune in flags)) or
     ((DamageType = DAMAGE_MAGIC) and (nfMagicImmune in flags)) then
  begin
    if (Visible) and (not isPlayer) then UI.Msg(getName(CTheName)+' is unaffected.');
    exit;
  end;

  if ((DamageType = DAMAGE_FIRE) and (nfFireResist in flags)) or
     ((DamageType = DAMAGE_LIGHTNING) and (nfLightningResist in flags)) or
     ((DamageType = DAMAGE_MAGIC) and (nfMagicResist in flags)) then
       Damage:=max(Damage*3 div 4,1);

  if nfManaShield in flags then
  begin
    Damage:=(Damage*77)div 100;
    if self is TPlayer then
      with self as TPlayer do
      begin
         if mp > damage then
         begin
           mp-=damage;
           damage:=0;
         end
         else
         begin
           damage-=mp;
           mp:=0;
           exclude(flags, nfManaShield);
         end;
         dec(hp, Damage);
         if (SpdHit > 0) and (hp > Integer(Damage)) and (Damage>=Level) then
           Recovery := SpdHit;
           SpeedCount := min(SpeedCount, 100 - (SpdMov + SpdAtk)div 4);
      end;
  end
  else
  begin
    Dec(HP,Damage);
    if Damage > DWord(Level)+3 then
    begin
      Recovery := SpdHit;
      SpeedCount := min(SpeedCount, 100 - (SpdMov + SpdAtk)div 4);
    end;
  end;
  // warning, no access afterwards
  if HP <= 0 then Die(Attacker)
  else begin
    if (isPlayer) then
      with self as TPlayer do
      begin
           PlaySound('69'); // TODO: alternate 69b?
      end
    else
        PlaySound('h');
    if nfStatue in Flags then
    begin
      exclude(Flags, nfStatue);
      Recovery := 20;
    end else RunHook(Hook_OnSpot,[Attacker]);
  end;
end;


function TNPC.getExpValue(killerLevel : Byte) : DWord;
var value : LongInt;
begin
  Value := Round(ExpValue*(1.0+0.1*(Level-killerLevel)));
  if Value < 0 then Value := 0;
  Exit(Value);
end;

procedure TNPC.TimeFlow(time : LongInt);
begin
  if GameEnd then Exit;
  // Do your action
  if not (isPlayer()or (nfNoHeal in flags)) then
  begin
    RegenTick += time;
    if (HP < HPMax) then
    begin
      if (RegenTick*Level >= 640) then
      begin
        inc(HP);
        dec(RegenTick, 640 div Level)
      end;
    end
    else
      RegenTick := 0;
  end;
  Inc(SpeedCount,Max(0,time-Recovery));
  Recovery := Max(0,Recovery-Time);
  if Recovery = 0 then exclude( flags, nfStatue );
  while (SpeedCount >= 100) and (GameEnd = False) do Action;
  // Pass time to siblings and children.
  inherited TimeFlow(time);
end;

procedure TNPC.Die(Attacker: TNPC = nil);
begin
  if ai in AIMonster then
  begin
    Inc(Game.Player.KillCount);
    if (nfUnique in Flags) then
      Game.Player.KillDataUnique[Name] := 1
    else
      Game.Player.KillData[Name] := Game.Player.KillData[Name]+1;
  end;
  //ensure that player will not target unexisting monster
  if self = game.player.querry then
    game.player.querry := nil;
  if Visible then UI.Msg(GetName(CTheName)+' dies.');
  PlaySound('d');
  RunHook( Hook_OnDrop, [ Attacker ] );
  if AI in AIMonster then
    UI.Msg('You gain '+IntToStr(getExpValue(Game.Player.Level))+' experience.');
  Game.Player.AddExp(getExpValue(Game.Player.Level));
  TLevel(Parent).Map[Position.x,Position.y].NPC := nil;
  if Corpse <> 0 then
    if Cells[TLevel(Parent).Map[Position.x,Position.y].cell].canch then
      TLevel(Parent).Map[Position.x,Position.y].cell  := Corpse;
  RunHook( Hook_OnDie, [Attacker] );
  if AI in AIMonster then
    Game.Level.OnMonsterDie(Position);

  Move(Game.Graveyard);
end;

procedure TNPC.PlayerMsg(const str : string);
begin
  if IsPlayer then UI.Msg(str);
end;

procedure TNPC.KnockBack(dir : TDirection);
begin
  if TLevel(Parent).CellCheck(Position+Dir,[cfNoMonsters, cfNoObstacles]) then
    Displace(Position+Dir);
  Recovery := SpdHit;
end;

function TNPC.CanCharge( c : TCoord2D ): boolean;
var Ray : TBresenhamRay;
begin
  CanCharge := false;
  Ray.Init(Position,c);
  repeat
    Ray.Next;
    if Ray.GetC = c then
      CanCharge := true;
  until not TLevel(Parent).CellCheck(Ray.GetC, [cfNoMonsters, cfNoObstacles]);
  Target:=Ray.GetC;
end;

procedure TNPC.Seek( c : TCoord2D; dst : shortint = 1);

var rx,ry       : Word;
    mx,my : ShortInt;
    MoveResult  : byte;
    Dir         : TDirection;
  function TryMove(mvx,mvy: word) : Byte;
  begin with TLevel(Parent) do begin
    if Map[mvx,mvy].NPC <> nil then Exit(100);
    if not Cells[Map[mvx,mvy].Cell].Pass then Exit(100);
    if Cells[Map[mvx,mvy].Cell].Special <> 0 then Exit(100); // what is this for??
    rx := mvx; ry := mvy;
    Exit(0);
  end; end;

  function GetCloser: boolean;
  var WhiteList, BlackList, Curr : TNavPoint;
      done : boolean;
      cx, cy : ShortInt;
      cf : TFlags;
  begin
    WhiteList := TNavPoint.Create(0,0,0,0);
    BlackList := TNavPoint.Create(0,0,0,0);
    WhiteList.Add(TNavPoint.Create(c.x,c.y,1,0));
    done := false;
    repeat
      Curr := WhiteList._next;
      //log(name+' '+inttostr(Curr.step)+','+inttostr(Curr.dist)+' : '+inttostr(curr.x)+';'+inttostr(curr.y)+' -> '+inttostr(position.x)+';'+inttostr(position.y));
      if (Curr.step <= 2)or(Curr.dist < 2) then cf := [cfNoMonsters, cfNoObstacles] else cf := [cfNoObstacles];
      if (Curr.x = Position.x) and (Curr.y = Position.y) then
        done := true;
      if Curr.step <= 20 then
      for cy := Curr.y-1 to Curr.y+1 do
        for cx := Curr.x-1 to Curr.x-1 do
          if ((cx = Curr.x) and (cy = Curr.y)) or (cx > MapSizeX) or (cy > MapSizeY) then
            continue
          else
            if ((cx = Position.x)and(cy = Position.y)) then
            begin
              mx := Curr.x-cx;
              my := Curr.y-cy;
              done := true;
            end
            else if (TLevel(Parent).CellCheck(NewCoord2D(cx, cy),cf)) then
              if not (WhiteList.Includes(NewCoord2D(cx, cy),Curr.step+1)
                  or BlackList.Includes(NewCoord2D(cx, cy),Curr.step+1)) then
                WhiteList.Add(TNavPoint.Create(cx, cy, Curr.step + 1, Distance(NewCoord2D(cx, cy),Position)));
      Curr.Move(BlackList);
    until (WhiteList._next = nil) or (Done);
    BlackList.Free;
    WhiteList.Free;
    exit( Done );
  end;

begin
  MoveResult := 0;
  Dir.CreateSmooth(Position, c);
  if not((dst = 1)and(GetCloser)) then
  begin
    mx := Sgn(dst*Dir.X);
    my := Sgn(dst*Dir.Y);
  end;

  if ((mx<>0)or(my<>0)) then
    MoveResult := TryMove(Position.x+mx,Position.y+my)
  else
    MoveResult := 100;

  if nfCharge in flags then
    if MoveResult = 0 then
    begin
      Dec(SpeedCount,SpdMov div 4);
      Displace(NewCoord2D(Position.x+mx,Position.y+my));
      Exit;
    end
    else
    begin
      if TLevel(Parent).Map[Position.x+mx,Position.y+my].NPC<>nil then
        if not(TLevel(Parent).Map[Position.x+mx,Position.y+my].NPC.AI in AIMonster+[AINPC]) then
        begin
          Attack(TLevel(Parent).Map[Position.x+mx,Position.y+my].NPC);
          exit;
        end;
      Recovery := spdhit;
      dec(speedcount, spdmov div 4);
      exclude(flags, nfCharge);
      exit;
    end;

  if MoveResult = 100 then
    if mx = 0 then
      MoveResult := TryMove(Position.x+(Random(2)*2-1),Position.y+my)
    else
      MoveResult := TryMove(Position.x+mx,Position.y);
  if MoveResult = 100 then
    if my = 0 then
      MoveResult := TryMove(Position.x+mx,Position.y+(Random(2)*2-1))
    else
      MoveResult := TryMove(Position.x,Position.y+my);

  if MoveResult = 0 then
  begin
    Dec(SpeedCount,SpdMov);
    Displace(NewCoord2D(rx,ry));
    Exit;
  end;
end;

procedure TNPC.SendMissile( tg : TCoord2D; mtype : Byte; mid : DWord; explosion : byte = 0);
const HITDELAY  = 50;
var Ray       : TBresenhamRay;
    State     : TRLLuaState;
    Old       : TCoord2D;
    DrawDelay : Word;
    Pict      : Char;
    Col       : Byte;
    Dmin, DMax: Word;
    bHit      : Boolean;
    dmgtype   : byte;
begin
  dmgtype := DAMAGE_GENERAL;
  if not isPlayer then Dec(SpeedCount, SpdAtk);

  DrawDelay := 10;
  case mtype of
    mt_Arrow :
      begin
        DrawDelay := 20;
        Pict      := '-';
        Col       := Brown;
      end;
    mt_Spell :
      begin
        with TLuaTable.Create(Game.Lua,'spells',mid) do
        try
          DrawDelay := 20;
          Pict      := getString('picture')[1];
          Col       := getNumber('color');
          dmgtype   := getNumber('type');
          if isPlayer then
            with self as TPlayer do
            begin
              State.Init( LuaState.NativeState );
              DMin := State.CallFunction('dmin',[spells[mid],Self],-1);
              DMax := State.CallFunction('dmax',[spells[mid],Self],-1);
            end
          else
          begin
            DMin   := dmgmin;
            DMax   := dmgmax;
          end
        finally
          Free;
        end;
      end;
  end;

  Ray.Init(Position,tg);
  repeat
    Old := Ray.GetC;
    Ray.Next;
    //hit obstacle
    if not TLevel(Parent).CellCheck(Ray.GetC, [cfNoBlock]) then begin
      // Missile hits non-passable feature
      if (mtype = mt_Arrow) then
         UI.PlaySound('sound/sfx/misc/arrowall.wav',Ray.GetC.x,Ray.GetC.y);
      if explosion > 0 then
      begin
        include(flags, nfUnAffected);
        TLevel(Parent).Explosion(Old, Col, explosion, DMin, DMax, 50, dmgtype, self);
        exclude(flags, nfUnAffected);
      end
      else if TLevel(Parent).isVisible(Ray.GetC) then
      begin
        UI.MarkCell(Ray.GetC, LightGray, '*');
        Output.Update;
        UI.Delay(HITDELAY);
        UI.MarkClear;
      end;
      Break;
    end;
   //hit monster
   if not TLevel(Parent).CellCheck(Ray.GetC, [cfNoMonsters]) then
    begin
        bHit := false;
        case mtype of
            mt_Arrow  : if isPlayer or (tg = ray.GetC)
                         then bHit := Attack(TLevel(Parent).Map[Ray.GetX,Ray.GetY].NPC, true);
            mt_Spell  : if isPlayer or (tg = ray.GetC) then
                          if explosion > 0 then
                          begin
                            bHit := true;
                            include(flags, nfUnAffected);
                            TLevel(Parent).Explosion(Ray.GetC, Col, explosion, DMin, DMax, 50, dmgtype, self);
                            exclude(flags, nfUnAffected);
                            break;
                          end
                          else
                            bHit := Attack(TLevel(Parent).Map[Ray.GetX,Ray.GetY].NPC, true, mid);
        end;
        if bHit and (explosion = 0) then
        begin
          if TLevel(Parent).isVisible(Ray.GetC) then
            UI.MarkCell(Ray.GetC, Red, '*');
          Output.Update;
          UI.Delay(HITDELAY);
          UI.MarkClear;
          Break;
        end;
    end;

    if TLevel(Parent).isVisible(Ray.GetC) then // Draw when visible
    begin
      if Pict = '-' then
        UI.MarkCell(Ray.GetC, Col, NewDirection(Old, Ray.GetC).Picture)
      else
        UI.MarkCell(Ray.GetC, Col, Pict);
      Output.Update;
      UI.Delay(DRAWDELAY);
      UI.MarkClear;
    end;
  until False;
end;

procedure TNPC.doSpell(SpellID, SpellLevel: DWord);
var Effect      : DWord;
    Col         : Byte;
    Dmin, DMax  : Word;
    SpellTarget : TThing;
    State       : TRLLuaState;
begin
  with TLuaTable.Create(Game.Lua,'spells',SpellID) do
  try
    if Visible then UI.Msg(GetName(CTheName)+' casts '+getString('name'));
    if Defined('sound1') then UI.PlaySound('sound/sfx/misc/' + getString('sound1') + '.wav');
    if Defined('sound2') then UI.PlaySound('sound/sfx/misc/' + getString('sound2') + '.wav');
    if Defined('spdmag') then dec(SpeedCount, getNumber('spdmag')) else dec(SpeedCount, spdatk);
    Effect := getNumber( 'effect' );
    if Effect = SPELL_SCRIPT then
    begin
      SpellTarget := nil;
      if getNumber( 'target' ) = SPELL_TARGET then
        if TLevel(Parent).Map[Target.x, Target.y].NPC<>nil then
          SpellTarget := TLevel(Parent).Map[Target.x, Target.y].NPC
        else if TLevel(Parent).Map[Target.x, Target.y].Item<>nil then
          SpellTarget := TLevel(Parent).Map[Target.x, Target.y].Item;
      Game.Lua.RunHook('spells',SpellID,'script', [SpellLevel, Self, SpellTarget] );
    end;
  finally
    Free;
  end;
  case Effect of
    SPELL_BOLT  : SendMissile( Target, mt_Spell, SpellID ); // spell level is not passed -- will need a fix for cases like wands
    SPELL_BALL  : SendMissile( Target, mt_Spell, SpellID, 1);
    SPELL_BLAST : begin
                    with TLuaTable.Create(Game.Lua,'spells',SpellID) do
                    try
                      Col  := getNumber('color');
                      if isPlayer then
                      begin
                        State.Init( LuaState.NativeState );
                        DMin := State.CallFunction('dmin',[spelllevel,Self],-1);
                        DMax := State.CallFunction('dmax',[spelllevel,Self],-1);
                      end
                      else if isFunction('dminmlvl') then
                      begin
                        State.Init( LuaState.NativeState );
                        DMin := State.CallFunction('dminmlvl',[Level],-1);
                        DMax := State.CallFunction('dmaxmlvl',[Level],-1);
                      end
                      else
                      begin
                        DMin := DmgMin;
                        DMax := DmgMax;
                      end;
                      include(flags, nfUnAffected);
                      TLevel(Parent).Explosion(Target, Col, 1, DMin, DMax, 50, getNumber('type'), self);
                      exclude(flags, nfUnAffected);
                    finally
                      Free;
                    end;
                  end;
  end;
end;

procedure TNPC.Action;
begin
  //OnSpot script
  if Visible then
  begin
    if (not Activated) then begin
       RunHook(Hook_OnSpot,[]);
       Activated := True;
    end;
  end else begin
      // Cause OnSpot to be called again when player moves back into view
      if (AI = AINPC) then
         Activated := False;
  end;

  //AI
  RunHook(Hook_OnAct,[]);

  //wait 0.05 seconds
  if SpeedCount >= 100 then Dec(SpeedCount,5);
end;

function TNPC.getProperty(PropertyID: Byte): Variant;
begin
  case PropertyID of
    // TODO : these 4 could be moved to thing
    PROP_AC     : Exit( AC );
    PROP_TOHIT  : Exit( ToHit );

    PROP_SCOUNT : Exit( SpeedCount );
    PROP_AI     : Exit( AI );
    PROP_CORPSE : Exit( Corpse );
    PROP_HP     : Exit( Hp );
    PROP_HPMAX  : Exit( HpMax );
    PROP_EXPVAL : Exit( ExpValue );
    PROP_SPDATK : Exit( SpdAtk );
    PROP_SPDMOV : Exit( SpdMov );
    PROP_SPDHIT : Exit( SpdHit );
    PROP_RECOV  : Exit( Recovery );
    PROP_ACTIVE : Exit( Activated );
    PROP_AISTATE: Exit( Special );
    PROP_AISTATE2:Exit( Special2 );
    PROP_INTF   : Exit( Intf );
    PROP_LIFESTEAL: Exit(LifeSteal);
    PROP_TARGETX: Exit(target.x);
    PROP_TARGETY: Exit(target.y);
    else Exit( inherited getProperty(PropertyID) )
  end;
end;

procedure TNPC.setProperty(PropertyID: Byte; Value: Variant);
begin
  case PropertyID of
    PROP_AC     : AC := Value;
    PROP_TOHIT  : ToHit := Value;

    PROP_SCOUNT : SpeedCount := Value;
    PROP_AI     : AI := Value;
    PROP_CORPSE : Corpse := Value;
    PROP_HP     : Hp := Value;
    PROP_HPMAX  : HpMax := Value;
    PROP_EXPVAL : ExpValue := Value;
    PROP_SPDATK : SpdAtk := Value;
    PROP_SPDMOV : SpdMov := Value;
    PROP_SPDHIT : SpdHit := Value;
    PROP_RECOV  : Recovery := Value;
    PROP_AISTATE: Special := Value;
    PROP_AISTATE2:Special2 := Value;
    PROP_LIFESTEAL: LifeSteal := Value;
    PROP_INTF   : Intf := Value;
    PROP_TARGETX: target.x := Value;
    PROP_TARGETY: target.y := Value;
    else inherited setProperty(PropertyID, Value);
  end;
end;

// PlaySound
// It is assumed that the Sound property is set to the base name
// sID is one of:
//     a - attack
//     h - hit
//     d - die
//     s - shout (war cry)
procedure TNPC.PlaySound(sID: char);
begin
     if (Sound = '') then exit;
     // There are two versions of each sound, pick one randomly
     UI.PlaySound(Sound + sID + IntToStr(Random(2) + 1) + '.wav',Position.x,Position.y);
end;

constructor TNPC.CreateFromStream(ISt: TStream);
var table : string;
    LTable : TLuaTable;
begin
  inherited CreateFromStream(ISt);
  ISt.Read(Target,SizeOf(TCoord2D));
  AI        := ISt.ReadByte;
  Special   := ISt.ReadByte;
  Special2  := ISt.ReadByte;
  HP        := ISt.ReadWord;
  HPMax     := ISt.ReadWord;
  Recovery  := ISt.ReadWord;
  RegenTick := ISt.ReadWord;

  if not Game.Lua.Defined('npcs',tid) then
    if Game.Lua.Defined('klasses',tid) then
      table := 'klasses'
    else
      Game.Lua.Error('NPC "'+tid+'" not found!')
  else
    table := 'npcs';

  LTable := TLuaTable.Create( Game.Lua, table, tid );
  with LTable do
  try
    Corpse  := CellID[GetString('corpse')];
    Intf    := GetNumber('intf');
    AC      := GetNumber('ac');
    DmgMin  := GetNumber('dmgmin');
    DmgMax  := GetNumber('dmgmax');
    ExpValue:= GetNumber('expvalue');
    if (nfUnique in Flags) then ExpValue := ExpValue * 2;
    SpdAtk  := GetNumber('spdatk');
    SpdMov  := GetNumber('spdmov');
    SpdHit  := GetNumber('spdhit');
    ToHit   := GetNumber('tohit');
    Sound   := GetString('sound');

    ReadHooks(LTable);
  finally
    Free;
  end;
  Log('...Finished!');
end;

procedure TNPC.ToStream(OSt: TStream);
begin
  inherited ToStream(OSt);
  OSt.Write(Target,SizeOf(TCoord2D));
  OSt.WriteByte(AI);
  OSt.WriteByte(Special);
  OSt.WriteByte(Special2);
  OSt.WriteWord(HP);
  OSt.WriteWord(HPMax);
  OSt.WriteWord(Recovery);
  OSt.WriteWord(RegenTick);
end;

function lua_npc_is_active(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
begin
  State.ObjectInit(L,npc);
  State.Push( npc.Activated );
  Result := 1;
end;

function lua_npc_die(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
begin
  State.ObjectInit(L,npc);
  npc.die;
  Result := 0;
end;

function lua_npc_can_charge(L: Plua_State): Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
begin
  State.ObjectInit(L,npc);
  State.Push( npc.CanCharge( State.ToCoord(2) ) );
  Result := 1;
end;

function lua_npc_get_target(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
begin
  State.ObjectInit(L,npc);
  State.PushCoord( npc.Target );
  Result := 1;
end;


function lua_npc_target_closest(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
    Target  : TNPC;
begin
  State.ObjectInit(L,npc);
  if npc.AI in AIMonster then
    Target := npc.ClosestTarget([AIPlayer,AIGolem], State.toInteger(2,15) )
  else
    Target := npc.ClosestTarget(AIMonster, State.toInteger(2,15) );
  if Target <> nil then npc.Target := Target.Position;
  State.Push( Target );
  Result := 1;
end;

function lua_npc_attack(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
    c       : TCoord2D;
begin
  State.ObjectInit(L,npc);
  c := State.ToCoord(2);
  if Game.Level.Map[c.x,c.y].NPC = nil then Exit(0);
  npc.Attack(Game.Level.Map[c.x,c.y].NPC);
  Result := 0;
end;

function lua_npc_seek(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
begin
  State.ObjectInit(L,npc);
  npc.Seek( State.ToCoord(2), State.ToInteger(3,1) );
  Result := 0;
end;

function lua_npc_send_missile(L: Plua_State) : Integer; cdecl;
var State : TRLLuaState;
    npc   : TNPC;
begin
  State.ObjectInit(L,npc);
  if State.StackSize < 4 then Exit(0);
  npc.SendMissile(State.ToCoord(3),State.ToInteger(2),State.toInteger(5));
  Result := 0;
end;

function lua_npc_cast_spell(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
begin
  State.ObjectInit(L,npc);
  npc.doSpell(State.ToInteger(2),State.ToInteger(3));
  Result := 0;
end;

function lua_npc_phasing(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
    Count: byte;
    aoe : TArea;
    pos : TCoord2D;
begin
  State.ObjectInit(L,npc);
  aoe := NewArea(npc.Position,4);
  Game.Lua.Level.Area.Area.Clamp(aoe);
  Count := 10;
  repeat
    pos := aoe.RandomEdgeCoord;
    dec(Count);
  until (Count = 0) or (Game.Lua.Level.CellCheck(pos,[cfNoMonsters, cfNoObstacles]));
  if Count = 0 then
  begin
    Pos := npc.position;
    if not Game.Lua.Level.Area.FindNearSpace(pos, 6, [cfNoMonsters, cfNoObstacles]) then Exit(0);
  end;
  npc.Displace(pos);
  dec(npc.SpeedCount, 1);
  Result := 0;
end;

function lua_npc_knockback(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    npc     : TNPC;
begin
  State.ObjectInit(L,npc);
  npc.Knockback( NewDirection( State.ToCoord(2), npc.Position) );
  Result := 0;
end;

class procedure TNPC.RegisterLuaAPI(Lua: TLua);
begin
  Lua.SetTableFunction('npc','is_active',     @lua_npc_is_active);
  Lua.SetTableFunction('npc','die',           @lua_npc_die);
  Lua.SetTableFunction('npc','can_charge',    @lua_npc_can_charge);
  Lua.SetTableFunction('npc','get_target',    @lua_npc_get_target);
  Lua.SetTableFunction('npc','target_closest',@lua_npc_target_closest);
  Lua.SetTableFunction('npc','attack',        @lua_npc_attack);
  Lua.SetTableFunction('npc','seek',          @lua_npc_seek);
  Lua.SetTableFunction('npc','send_missile',  @lua_npc_send_missile);
  Lua.SetTableFunction('npc','cast_spell',    @lua_npc_cast_spell);
  Lua.SetTableFunction('npc','phasing',       @lua_npc_phasing);
  Lua.SetTableFunction('npc','knockback',     @lua_npc_knockback);
end;

end.
