{$MODE OBJFPC}
// @abstract(Game Level for RL Core)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(January 17, 2005)
// @lastmod(May 25, 2010)
//
// This unit hold's level class : TLevel.

unit rllevel;
interface
uses classes, vmaparea, vutil, vlua, vrltools, rlgobj, rlglobal, rlnpc, rlitem;

// Defines a single map cell. The pointers herin are only informational.
// Disposing of the NPCs and Items is done by disposing TLevel's children.
// Control of the cell pointers is done by the objects themselves (see
// TThing.Displace.
type TCell  = record
       Cell   : Byte;
       Light  : TLightFlags;
       NPC    : TNPC;
       Item   : TItem;
     end;

// wrapper for access to TMapArea functionality
type TLevel = class;

     { TLevelArea }

     TLevelArea = class(TMapArea)
        constructor Create( theLevel : TLevel );
        function GetCell( const aWhere : TCoord2D ) : Byte; override;
        procedure PutCell( const aWhere : TCoord2D; const aWhat : Byte ); override;
        function isEmpty( const coord : TCoord2D; EmptyFlags : TFlags32 = []) : Boolean; override;
        function FindNearSpace(var Where : TCoord2D; Range : Byte; EmptyFlags : TFlags32) : boolean;
        function FindEmptySquare : TCoord2D;

     private
        Level : TLevel;
     end;


// The TLevel class. Holds an array of TCells (see above). TLevel is the
// parent to all the TThings (Items, NPCs, and also the Player (who of
// course is also a NPC ;-) ).
TLevel = class(TGameObject)
       // The map array that holds TCells.
       Map  : array[1..MapSizeX,1..MapSizeY] of TCell;
       Level  : Byte;
       special: string;
       music  : string;
       OnCreate: boolean;//maybe we should add 'tick' and 'leave' scripts
       OnEnter : boolean;//and make them flags. but not for now.
       OnKillAll: boolean;
       Area : TLevelArea;
       MonsterCount : Integer;
       // Overriden constructor.
       constructor Create(const thingID : string); override;
       // Disposes the children (TThings) and clears the map.
       procedure Clear;
       // Clears the lighted and blocked bits from the lightmap.
       procedure LightClear;
       // Traces a light at (lx,ly) with the radius ldist.
       procedure LightTrace( c : TCoord2D; ldist : Byte);
       // Line of Sight checking between points
       function  isLoS ( c1, c2 : TCoord2D ): boolean;
       // Levels time flow procedure. See TGameObject.
       procedure TimeFlow(time : LongInt); override;
       // Properly adds monster to a given square.
       procedure AddMonster(monster : string; c : TCoord2D );
       // Properly adds monster to a given square.
       procedure AddMonster(npc : TNPC; c : TCoord2D );
       // Properly adds item to a given square.
       procedure AddItem(item : TItem; c : TCoord2D );
       // Adds an item to the nearest free square.
       procedure DropItem(item : TItem; c : TCoord2D );
       // Adds an item to the nearest free square.
       procedure DropMonster(npc : TNPC; c : TCoord2D );
       // Places a portal. Will check if portal exists and remove it.
       procedure PlacePortal(portalID : Byte; c : TCoord2D );
       // Find first given thing
       function Find(const ThingID: string) : TGameObject;
       // noMonsters,noItems,noObstacles
       function CellCheck( c : TCoord2D; const cFlags : TFlags) : boolean;
       // checks wether given cell is visible
       function isVisible( c : TCoord2D ) : boolean;
       // checks wether given cell is explored
       function isExplored( c : TCoord2D ) : boolean;
       // Overriden destructor.
       destructor Destroy; override;
       //Reveal the whole level
       procedure Reveal;
       //Remove corpse
       procedure RemoveCorpse( c : TCoord2D );
       //Calculates and draws explosion
       procedure Explosion( c : TCoord2D; Color, Range:byte; DmgMin, DmgMax : word; DrawDelay: Word = 50; DamageType: byte = DAMAGE_GENERAL; Attacker: TNPC = nil);
       //Checks coords to be proper
       function  ProperCoord( c : TCoord2D ) : boolean;
       // Call event in all nearby NPCs
       procedure BroadcastEvent(origin : TCoord2D; radius : Word; event : String);
       // Do stuff when a monster dies
       procedure OnMonsterDie(c : TCoord2D);
       // register lua functions
       class procedure RegisterLuaAPI( Lua : TLua );

       private
       // Cell ID to byte;
       function IDToByte( const CID : AnsiString ) : Byte;

       //Initialization before construction
       procedure Init;
       // Stream constructor
       constructor CreateFromStream( ISt : TStream ); override;
       // Stream writer
       procedure ToStream( OSt : TStream ); override;
     end;

implementation

uses sysutils,vnode, vdungen,vmath,rlcell,rlgen, rlgame, vvision, rllua, rlui, rlthing;

procedure TLevel.Init;
begin
  OnEnter  := false;
  OnKillAll := false;
  with TLuaTable.Create( Game.Lua, 'levels', tID ) do
  try
    level   := GetNumber('level');
    special := GetString('special');
    music   := GetString('music');
    if isFunction( 'OnEnter' )  then OnEnter := true;
    if isFunction( 'OnKillAll' )  then OnKillAll := true;
  finally
    Free;
  end;
end;

constructor TLevel.Create(const thingID : string);
var ltype : string;
//    fname : string;
    gtype : string;
    TR    : TPrintableCharToByte;
    LMap  : AnsiString;
    x     : Byte;
    npc_count : word;
    ShiftX : Integer;
    ShiftY : Integer;
begin
  // With TNode descendants ALWAYS call inherited Init and Done.
  inherited Create(thingID);
  UI.Mute();
  Init;
  Log('Created.');
  LMap := '';
  MonsterCount := 0;

  OnCreate := false;

  ShiftX := 0;
  ShiftY := 0;

  Area := TLevelArea.Create( Self );

  if not Game.Lua.Defined('levels',thingID) then CritError('Level "'+thingID+'" not found!');
  with TLuaTable.Create( Game.Lua, 'levels', thingID ) do
  try
    name    := GetString('name');
    ltype   := GetString('type');
    gtype   := GetString('generator');
    npc_count := GetNumber('npc_count');

    if isFunction( 'OnCreate' ) then OnCreate := true;

    if defined('map_key') then
    begin
      TR    := getCharTranslation('map_key',@IDToByte);
      for x := Low(TPrintableCharToByte) to High(TPrintableCharToByte) do
        if TR[x] = 0 then TR[x] := StandardTranslation[x];
    end;
    if defined('map') then
      LMap  := getString('map');
  finally
    Free;
  end;

  Clear;
  Game.Lua.Level := self;

  with TGenerator.Create( Self ) do
  try
    if ltype = 'map' then
      FromString( LMap, TR )
    else
      GenerateLevel( ltype, gtype, LMap, TR );
      ShiftX := MapCoord.x - 1;
      ShiftY := MapCoord.y - 1;
  finally
    Free;
  end;

  if (level < MaxLevels) then begin
    try
      Area.FindCell([CELL_STAIR_DOWN]);
    except
      Area.PutCell(Area.FindEmptySquare,CELL_STAIR_DOWN);
    end;
  end;
  try
    PlayerC := Area.FindCell([CELL_STAIR_UP]);
  except
    PlayerC := Area.FindEmptySquare;
    Area.PutCell(PlayerC,CELL_STAIR_UP);
  end;

  if npc_count > 0 then
    Game.Lua.TableExecute('level', 'drop_npcs', [ thingID, ShiftX, ShiftY ]);

  if OnCreate then
    Game.Lua.TableExecute('levels','level'+IntToStr(Game.LevelNumber),'OnCreate');

  UI.Unmute();
end;

procedure TLevel.OnMonsterDie(c : TCoord2D);
begin
     Game.Level.BroadcastEvent(c,5,'death_nerby');
     Dec(MonsterCount);
     if (MonsterCount = 0) and OnKillAll then Game.Lua.TableExecute('levels',tid,'OnKillAll' );
end;

function  TLevel.isLoS ( c1, c2 : TCoord2D ): boolean;
var cnt: byte;
    Ray: TBresenhamRay;
begin
  if ( c1 = c2 ) then exit(true);
  Ray.Init(c1, c2);
  cnt := 0;
  repeat
    inc(cnt);
    Ray.Next;
    if Ray.Done then exit(true);
    if not ((Ray.GetX in [1..MapSizeX]) or (Ray.GetY in [1..MapSizeY])) then Exit(false);
    if Cells[Map[Ray.GetX, Ray.GetY].Cell].bvis then exit(false);
  until cnt = 15;
  exit( false );
end;

procedure TLevel.Explosion( c : TCoord2D; Color, Range : byte; DmgMin, DmgMax : word; DrawDelay: Word = 50; DamageType: byte = DAMAGE_GENERAL; Attacker: TNPC = nil);
var ax, ay, step : byte;
begin
  //Clear nfAffected flag

    for ax := Max( 1, c.x-Range ) to Min(MapSizeX, c.x + Range) do
      for ay := Max( 1, c.y-Range ) to Min(MapSizeY, c.y + Range) do
        if ( Map[ax, ay].NPC <> nil ) then
          if (Attacker<>nil) and (Attacker.AI in AIMonster) and (Map[ax, ay].NPC.AI in AIMonster) then
              Include(Map[ax, ay].NPC.Flags, nfAffected)
          else
              Exclude(Map[ax, ay].NPC.Flags, nfAffected);
  //Draw the explosion
  UI.Explosion( c, Color, Range, DrawDelay );

  //Deal damage to creatures
  for Step:=1 to range do
  for ax := Max( 1, c.x-Range ) to Min(MapSizeX, c.x + Range) do
    for ay := Max( 1, c.y-Range ) to Min(MapSizeY, c.y + Range) do
      if Map[ax, ay].NPC <> nil then
        if ( Distance( ax, ay, c.x, c.y ) <= Step ) and
           isLoS( c, NewCoord2D( ax, ay ) ) then
          if ( [nfAffected, nfUnAffected] * Map[ax, ay].NPC.Flags <> []) then
            continue
          else
            begin
              Include(Map[ax, ay].NPC.Flags, nfAffected);
              Map[ax, ay].NPC.ApplyDamage(randrange(DmgMin, DmgMax), DamageType, Attacker);
            end;
  //Clear nfAffected flag
    for ax := Max( 1, c.x-Range ) to Min(MapSizeX, c.x + Range) do
      for ay := Max( 1, c.y-Range ) to Min(MapSizeY, c.y + Range) do
        if ( Map[ax, ay].NPC <> nil ) then Exclude(Map[ax, ay].NPC.Flags, nfAffected);
end;

procedure TLevel.RemoveCorpse( c : TCoord2D );
begin
  if Map[c.x, c.y].Cell in [ CELL_CORPSE, CELL_BONES ] then Map[c.x, c.y].Cell := CELL_FLOOR;
  if Map[c.x, c.y].Cell = CELL_CORPSE_BLOOD then Map[c.x, c.y].Cell := CELL_FLOOR_BLOOD;
end;

function TLevel.IDToByte(const CID: AnsiString): Byte;
begin
  Exit( CellID[ CID ] );
end;

// Clears the lighted and blocked bits from the lightmap.
procedure Tlevel.LightClear;
var tx, ty : Integer;
begin
  for tx := 1 to MapSizeX do
    for ty := 1 to MapSizeY do
    begin
      Exclude( Map[tx,ty].Light, lfBlocked );
      Exclude( Map[tx,ty].Light, lfLighted );
    end;
end;

// Special thanks for this procedure goes to Isaac Kuo. This beamcasting
// algorithm is a ported to FreePascal modified version of his algorithm
// posted on http://www.roguelikedevelopment.org
//   lx,ly -- the coordinates of the lightsource
//   ldist -- the maximum range of the lightsource
procedure TLevel.LightTrace(c : TCoord2D; ldist : Byte);
var tx, ty : Integer;
    mini, maxi, cor, u, v : Integer;
    quad, slope : Byte;
const quads : array[1..4] of array[1..2] of ShortInt =
      ((1,1),(-1,-1),(-1,+1),(+1,-1));
      RayNumber          = 32; // It's effective to keep this number a power of 2.
      RayWidthCorrection = 10; // Must be smaller then RayNumber/2 ;-)
begin
  for tx := 1 to MapSizeX do
    for ty := 1 to MapSizeY do
      if (Cells[Map[tx,ty].Cell].bvis) or (Distance(c.x,c.y,tx,ty) > ldist) then
        Include(Map[tx,ty].Light,lfBlocked);
  // Set 0,0 to be visible even if the player is
  // standing on something opaque
  Map[c.x,c.y].Light += [lflighted,lfvisited];

  // Check the orthogonal directions
  tx := c.x; repeat Inc(tx); Map[tx,c.y].Light += [lflighted,lfvisited]; until lfBlocked in Map[tx,c.y].Light;
  tx := c.x; repeat Dec(tx); Map[tx,c.y].Light += [lflighted,lfvisited]; until lfBlocked in Map[tx,c.y].Light;
  ty := c.y; repeat Inc(ty); Map[c.x,ty].Light += [lflighted,lfvisited]; until lfBlocked in Map[c.x,ty].Light;
  ty := c.y; repeat Dec(ty); Map[c.x,ty].Light += [lflighted,lfvisited]; until lfBlocked in Map[c.x,ty].Light;

  // Loop through the quadrants
  for quad := 1 to 4 do
  // Now loop on the diagonal directions
  for slope := 1 to RayNumber-1 do
  begin
    // initialize the v coordinate and set the beam size
    // to maximum--mini and maxi store the beam\'s current
    // top and bottom positions.
    // As long as mini<maxi, the beam has some width.
    // When mini=maxi, the beam is a thin line.
    // When mini>maxi, the beam has been blocked.

    v := slope; u := 0;
    mini := RayWidthCorrection;
    maxi := RayNumber-RayWidthCorrection;
    repeat Inc(u);
      ty:= v div RayNumber; tx:= u - ty;  //Do the transform
      cor:= RayNumber-(v mod RayNumber);         //calculate the position of block corner within beam
      if mini < cor then begin //beam is low enough to hit (x,y) block
        Map[quads[quad][1]*tx+c.x,quads[quad][2]*ty+c.y].Light += [lflighted,lfvisited];
        if lfBlocked in Map[quads[quad][1]*tx+c.x,quads[quad][2]*ty+c.y].Light then mini := cor; //beam was partially blocked
      end;
      if maxi > cor then begin //beam is high enough to hit (x-1,y+1) block
        Map[c.x+quads[quad][1]*(tx-1),c.y+quads[quad][2]*(ty+1)].Light += [lflighted,lfvisited];
        if lfBlocked in Map[c.x+quads[quad][1]*(tx-1),c.y+quads[quad][2]*(ty+1)].Light then maxi := cor; //beam was partially blocked
      end;
      v := v + slope;  //increment the beam\'s v coordinate
    until mini > maxi;
  end;
end;

Procedure TLevel.Reveal;
var dx, dy: Word;
begin
  for dx := 1 to MapSizeX do
    for dy := 1 to MapSizeY do
      Map[dx, dy].Light += [lfVisited];
end;

procedure TLevel.PlacePortal(portalID : Byte; c : TCoord2D );
var Coord : TCoord2D;
begin
  Area.Transmute([portalID],CELL_FLOOR);
  try
    Coord := Area.Drop(c,[cfNoMonsters,cfNoObstacles,cfNoItems,cfNoChangeRes]);
    Map[Coord.X,Coord.Y].Cell := portalID;
    if isVisible(Coord) then UI.Msg('A portal appears!');
    Game.Player.PortalLevel := Level;
  except
  end;
end;

function TLevel.Find(const ThingID: string) : TGameObject;
var Scan : TNode;
begin
  if ThingID = '' then exit(nil);
  if Child <> nil then
  begin
    Scan := Child;
    repeat
      if TGameObject(Scan).tid = ThingID then Exit( TGameObject(Scan) );
      Scan := Scan.Next;
    until Scan = Child;
  end;
  Exit(nil);
end;

function TLevel.CellCheck( c : TCoord2D; const cFlags : TFlags) : boolean;
begin
  if (cfNoMonsters  in cFlags) and (Map[c.x,c.y].NPC         <> nil)  then Exit(False);
  if (cfNoItems     in cFlags) and (Map[c.x,c.y].Item        <> nil)  then Exit(False);
  if (cfNoObstacles in cFlags) and (not Cells[Map[c.x,c.y].Cell].Pass)  then Exit(False);
  if (cfNoChangeRes in cFlags) and (not Cells[Map[c.x,c.y].Cell].canch) then Exit(False);
  if (cfNoBlock in cFlags) and (Cells[Map[c.x,c.y].Cell].bproj) then Exit(False);
  if (cfCorpse in cFlags)and (not (Map[c.x,c.y].Cell in [CELL_CORPSE, CELL_CORPSE_BLOOD])) then Exit(False);
  if (cfBones in cFlags) and (Cells[Map[c.x,c.y].Cell].name <> 'bones') then Exit(False);
  Exit(True);
end;

function TLevel.isVisible( c : TCoord2D ) : boolean;
begin
  Exit(lfLighted in Map[c.x,c.y].Light)
end;

function TLevel.isExplored( c : TCoord2D ) : boolean;
begin
  Exit(lfVisited in Map[c.x,c.y].Light);
end;

procedure TLevel.TimeFlow(time : LongInt);
begin
  // Do things.
  with Game.Player do
    if PortalCount > 0 then
       if PortalCount = 1 then
       begin
         PortalLevel := Game.Level.Level;
         PlacePortal(CELL_TOWN_PORTAL,Position);
         UI.PlaySound('sound/sfx/misc/sentinel.wav',Position.x,Position.y);
         PortalCount := 0;
       end else Dec(PortalCount);
  // Pass time flow to children:
  inherited TimeFlow(time);
end;

procedure TLevel.Clear;
var cx, cy : byte;
begin
  // First destroy all children.
  if Child <> nil then FreeAndNil(Child);

  // Now clear the map.
  for cx := 1 to MapSizeX do
    for cy := 1 to MapSizeY do
      with Map[cx,cy] do
      begin
        Cell   := CELL_FLOOR;
        Light  := [];
        NPC    := nil;
        Item   := nil;
      end;
end;

// Properly adds monster to a given square.
procedure TLevel.AddMonster(monster : string; c : TCoord2D );
var NPC : TNPC;
begin
  NPC := TNPC.Create(monster);
  Add(NPC);
  NPC.Displace(c);
  if not (NPC.isPlayer) then
    Inc(MonsterCount);
end;

// Properly adds item to a given square.
procedure TLevel.AddItem(item : TItem; c : TCoord2D );
begin
  Add(Item);
  Item.Displace(c);
  UI.PlaySound(Item.Sound2,c.x,c.y);
end;

// Adds an item to the nearest free square.
procedure TLevel.DropItem(item : TItem; c : TCoord2D );
begin
  try
    if Map[c.x,c.y].NPC = Game.Player
      then AddItem(item,Area.Drop(c,[cfNoItems,cfNoObstacles]))
      else AddItem(item,Area.Drop(c,[cfNoItems,cfNoMonsters,cfNoObstacles]));
  except
    FreeAndNil(item);
  end;
end;

// Properly adds monster to a given square.
procedure TLevel.AddMonster(npc : TNPC; c : TCoord2D );
begin
  Add(NPC);
  NPC.Displace(c);
  if not (NPC.isPlayer) then
    Inc(MonsterCount);
end;

// Adds a monster to the nearest free square.
procedure TLevel.DropMonster(npc : TNPC; c : TCoord2D );
begin
  try
    AddMonster(npc,Area.Drop(c,[cfNoMonsters,cfNoObstacles]));
  except
    FreeAndNil(npc);
  end;
end;

function TLevel.ProperCoord( c : TCoord2D ) : boolean;
begin
  Exit( Area.properCoord(c) )
end;

procedure TLevel.BroadcastEvent(origin : TCoord2D; radius : Word; event : String);
var npc : TNPC;
var x, y : Word;
begin
     for x := Max(1, origin.x - Radius) to Min(MapSizeX, origin.x + radius) do begin
         for y := Max(1, origin.y - Radius) to Min(MapSizeY, origin.y + radius) do begin
             npc := Map[x,y].npc;
             if npc <> nil then
                npc.RunHook(Hook_OnBroadcast,[event]);
         end;
     end;
end;

destructor TLevel.Destroy;
begin
  FreeAndNil( Area );
  // With TNode descendants ALWAYS call inherited Init and Done.
  inherited Destroy;
end;

{ TLevelArea }

constructor TLevelArea.Create(theLevel: TLevel);
begin
  inherited Create( NewArea( NewCoord2D( 1, 1 ), NewCoord2D( MapSizeX, MapSizeY ) ) );
  Level := theLevel;
end;

function TLevelArea.GetCell(const aWhere: TCoord2D): Byte;
begin
  Exit( Level.Map[ aWhere.X, aWhere.y ].Cell );
end;

procedure TLevelArea.PutCell(const aWhere: TCoord2D; const aWhat: Byte);
begin
  Level.Map[ aWhere.X, aWhere.y ].Cell := aWhat;
end;

function TLevelArea.isEmpty(const coord: TCoord2D; EmptyFlags: TFlags32): Boolean;
begin
  Exit( Level.CellCheck( coord, EmptyFlags ) );
end;

function TLevelArea.FindNearSpace(var Where: TCoord2D; Range: Byte; EmptyFlags: TFlags32): boolean;
const CriticalSize = 100;
var Critical : word;
    GC       : TCoord2D;
begin
  Critical := 0;
  repeat
    if Critical > CriticalSize then Exit(False);
    Inc(Critical);
    GC := Where.RandomShifted(Range);
  until ProperCoord(GC) and (GetCell(GC) = CELL_FLOOR) and (isEmpty(GC,EmptyFlags));
  Where := GC;
  Exit(true);
end;

function TLevelArea.FindEmptySquare: TCoord2D;
const CriticalSize = 6000;
var Critical : word;
begin
  Critical := 0;
  repeat
    Inc(Critical);
    Result := RanCoord( [CELL_FLOOR] );
  until (Around(Result,[CELL_FLOOR]) = 8) or (Critical > CriticalSize);
  if Critical > CriticalSize then
  begin
    Result := FindCell([CELL_FLOOR]);
    Log('FindBigEmptySpace -- Failed!');
  end;
end;


constructor TLevel.CreateFromStream(ISt: TStream);
var i,j, ThingsCount: word;
  aNPC: TNPC;
  aItem: TItem;
begin
  inherited CreateFromStream(ISt);
  Clear;
  for j:=1 to MapSizeY do
    for i:=1 to MapSizeX do
      with Map[i,j] do
      begin
        Cell  := ISt.ReadByte;
        ISt.Read(Light, SizeOf(Light));
        NPC  := nil;
        Item := nil;
      end;
  ThingsCount := ISt.ReadWord;
  Log('Loading '+inttostr(ThingsCount)+' objects');

  MonsterCount := 0;
  Area := TLevelArea.Create( Self );

  for i:=1 to ThingsCount do
  begin
    j := ISt.ReadByte;
    if j = 1 then
    begin
      aNPC := TNPC.CreateFromStream(ISt);
      AddMonster(aNPC, aNPC.Position);
    end
    else
    begin
      aItem := TItem.CreateFromStream(ISt);
      AddItem(aItem, aItem.Position);
    end
  end;

  Init;
end;

procedure TLevel.ToStream(OSt: TStream);
var i,j, ThingsCount: word;
    Scan: TNode;
begin
  inherited ToStream(OSt);
  ThingsCount :=0;
  for j:=1 to MapSizeY do
    for i:=1 to MapSizeX do
      with Map[i,j] do
      begin
        OSt.WriteByte(Cell);
        OSt.Write(Light, SizeOf(Light));
        if NPC<>nil then inc(ThingsCount);
        if Item<>nil then inc(ThingsCount);
      end;
  OSt.WriteWord(ThingsCount);

  if Child <> nil then
  begin
    Scan := Child;
    repeat
      if Scan is TNPC then
      begin
        OSt.WriteByte(1);
        TNPC(Scan).ToStream(OSt);
      end
      else
      begin
        OSt.WriteByte(0);
        TItem(Scan).ToStream(OSt);
      end;
      Scan := Scan.Next;
    until Scan = Child;
  end;

end;

function lua_level_raw_set_cell (L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
    Coord : TCoord2D;
begin
  State.Init( L );
  Coord := State.ToCoord(1);
  Game.Lua.Level.Map[Coord.x,Coord.y].Cell := State.ToInteger(2);
  Result := 0;
end;

function lua_level_raw_get_cell (L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
    Coord : TCoord2D;
begin
  State.Init( L );
  Coord := State.ToCoord(1);
  State.Push(Game.Lua.Level.Map[Coord.x,Coord.y].Cell);
  Result := 1;
end;

function lua_level_drop_npc(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
    NPC   : TNPC;
    tid   : AnsiString;
    c     : TCoord2D;
begin
  State.Init(L);
  //tid, x, y
  if State.StackSize < 2 then Exit(0);
  tid:=State.ToString(1);
  NPC := nil;
  c := State.ToCoord(2);
  if tid = '' then
  begin
    FreeAndNil(Game.Lua.Level.Map[c.x, c.y].NPC);
  end
  else
  begin
    NPC := TNPC.Create(tid);
    Game.Lua.Level.DropMonster(NPC, c);
  end;
  State.Push( NPC );
  Result := 1;
end;

function lua_level_drop_item(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
    Item  : TItem;
    Coord : TCoord2D;
    tid   : string;
begin
  State.Init(L);
  if State.StackSize < 2 then Exit(0);
  tid   := State.ToString(1);
  Coord := State.ToCoord(2);
  if tid = '' then
  begin
    FreeAndNil(Game.Lua.Level.Map[Coord.X, Coord.Y].NPC);
    exit(0);
  end;
  if State.StackSize = 3 then
    Item := TItem.Create(tid,State.ToInteger(3))
  else
    Item :=TItem.Create(tid);
  Game.Lua.Level.DropItem(Item, Coord);
  State.Push(Item);
  Result := 1;
end;

function lua_level_change (L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
begin
  State.Init(L);
  //Level, Cell
  LevelChange := True;
  Game.LevelNumber := State.ToInteger(1);
  Game.StairNumber := State.ToInteger(2);
  Result := 0;
end;

function lua_level_reveal (L: Plua_State): Integer; cdecl;
begin
  game.lua.level.reveal;
  Result := 0;
end;

// temp?
function lua_level_get_shop(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
    ID     : DWord;
begin
  State.Init(L);
  ID := State.ToInteger(1);
  if (ID = 0) or (ID > MaxShops)
     then State.PushNil
     else State.Push( Game.Shops[ ID ] );
  Result := 1;
end;

function lua_level_level_number (L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
begin
  State.Init(L);
  State.Push(game.lua.level.level);
  Result := 1;
end;

function lua_level_explosion (L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
begin
  State.Init(L);
  if State.StackSize < 1 then Exit(0);
  Game.Lua.Level.Explosion(
    State.ToCoord(1),
    State.ToInteger(2,RED),
    State.ToInteger(3,1),
    State.ToInteger(4,0),
    State.ToInteger(5,0),
    State.ToInteger(6,50)
  );
  Result := 0;
end;

function lua_level_broadcast_event (L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
begin
  State.Init(L);
  if State.StackSize <> 3 then Exit(0);
  Game.Lua.Level.BroadcastEvent(State.ToCoord(1),State.ToInteger(2),State.ToString(3));
  Result := 0;
end;


function lua_level_find(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
begin
  State.Init(L);
  State.Push( Game.Lua.Level.Find( State.ToString(1) ) );
  Result := 0;
end;

function lua_level_find_tile(L: Plua_State): Integer; cdecl;
var State  : TRLLuaState;
begin
  State.Init(L);
  try
    State.PushCoord( Game.Lua.Level.Area.FindCell( [CellID[State.ToString(1)]] ) );
    Result := 1;
  except
    Result := 0;
  end;
end;

function lua_level_find_nearest(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
    Flags : TFlags;
    Count : byte;
begin
  State.Init(L);
  Count := 2;
  Flags:=[];
  while Count <= State.StackSize do begin include(Flags,State.ToInteger(Count)); inc(Count); end;
  try
    State.PushCoord( Game.Lua.Level.Area.Drop(State.ToCoord(1), flags) );
    Result := 1;
  except
    Result := 0;
  end;
end;

function lua_level_find_empty_square(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
begin
  State.Init(L);
  try
    State.PushCoord( Game.Lua.Level.Area.FindEmptySquare );
    Result := 1;
  except
    Result := 0;
  end;
end;

function lua_level_find_empty_coord(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
    Flags : TFlags;
    Count : byte;
begin
  State.Init(L);
  Count := 2;
  Flags:=[];
  while Count <= State.StackSize do begin include(Flags,State.ToInteger(Count)); inc(Count); end;
  try
    State.PushCoord( Game.Lua.Level.Area.EmptyRanCoord([CellID[State.ToString(1)]],flags) );
    Result := 1;
  except
    Result := 0;
  end;
end;



function lua_level_find_near_coord(L: Plua_State): Integer; cdecl;
var State : TRLLuaState;
    Flags : TFlags;
    Count : byte;
    Coord : TCoord2D;
begin
  State.Init(L);
  Count := 3;
  Flags:=[];
  while Count <= State.StackSize do begin include(Flags,State.ToInteger(Count)); inc(Count); end;
  Coord := State.ToCoord(1);
  if Game.Lua.Level.Area.FindNearSpace(Coord, State.ToInteger(2),flags) then
  begin
    State.PushCoord( Coord );
    Result := 1;
  end
  else
    Result := 0;
end;



class procedure TLevel.RegisterLuaAPI(Lua: TLua);
begin
  Lua.SetTableFunction('level','raw_set_cell',     @lua_level_raw_set_cell);
  Lua.SetTableFunction('level','raw_get_cell',     @lua_level_raw_get_cell);

  Lua.SetTableFunction('level','drop_npc',         @lua_level_drop_npc);
  Lua.SetTableFunction('level','drop_item',        @lua_level_drop_item);
  Lua.SetTableFunction('level','change_to',        @lua_level_change);
  Lua.SetTableFunction('level','reveal',           @lua_level_reveal);
  Lua.SetTableFunction('level','get_shop',         @lua_level_get_shop);
  Lua.SetTableFunction('level','level',            @lua_level_level_number);
  Lua.SetTableFunction('level','explosion',        @lua_level_explosion);
  Lua.SetTableFunction('level','broadcast_event',  @lua_level_broadcast_event);

  Lua.SetTableFunction('level','find',             @lua_level_find);
  Lua.SetTableFunction('level','find_tile',        @lua_level_find_tile);
  Lua.SetTableFunction('level','find_nearest',     @lua_level_find_nearest);
  Lua.SetTableFunction('level','find_empty_square',@lua_level_find_empty_square);
  Lua.SetTableFunction('level','find_empty_coord', @lua_level_find_empty_coord);
  Lua.SetTableFunction('level','find_near_coord',  @lua_level_find_near_coord);
end;


end.
