{$MODE OBJFPC}
// @abstract(Item for RL Core)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(January 17, 2005)
// @lastmod(January 17, 2005)
//

unit rlitem;
interface
uses classes, vlua, rlthing, rlglobal;

type

{ TItem }

TItem = class(TThing)
       Volume     : Byte;
       IType      : Byte;
       Amount     : DWord;
       Spell      : byte;
       SpellLevel : byte;
       Charges    : Word;
       ChargesMax : Word;
       ChargesMod : byte;
       Str        : SmallInt;
       Dex        : SmallInt;
       Vit        : SmallInt;
       Mag        : SmallInt;
       HPMod      : SmallInt;
       MPMod      : SmallInt;
       AC         : SmallInt;
       StrReq     : Word;
       DexReq     : Word;
       MagReq     : Word;
       ACModPercent: Smallint;
       DmgMod     : Word;
       DmgModPercent:Integer;
       DurMax     : Word;
       DurMod     : Smallint;
       Dur        : Word;
       Effect     : Word;
       Price      : DWord;
       PriceMod   : DWord;
       PriceMul   : shortint;
       ToHitMod   : SmallInt;
       DmgTaken   : SmallInt;
       LightMod   : SmallInt; //+% light radius
       LifeSteal  : ShortInt;
       ManaSteal  : ShortInt;
       SpdAtk     : byte;
       SpdHit     : byte;
       SpdBlk     : byte;
       Prefix     : Word;
       PrefixValue: Integer;
       Suffix     : Word;
       SuffixValue: Integer;
       Sound2  : AnsiString;
       Sound1   : AnsiString;
       // Overriden constructor. Passed ilvl is only for the OnCreate script
       constructor Create(const thingID :string; ilvl : Word = 0);
       // Identifies an item
       procedure Identify;
       // Color for inventory windows
       function InvColor: byte;
       // Repair an item. RLevel is the repairer's level, 0 if ideal.
       procedure Repair( RLevel : Byte = 0 );
       // Recharge an item. RLevel is the recharger's level, 0 if ideal.
       procedure Recharge( RLevel : Byte = 0 );
       // TimeFlow. See TGameObject. Not Needed for Items in DiabloRL.
       procedure TimeFlow(time : LongInt); override;
       // Returns item name
       function GetName(outputType : TNameOutputType) : string; override;
       // Returns item price
       function GetPrice(PriceType: TPriceType): word;
       //Return Mag requirement for book
       function BookMagReq: Word;
       // Returns a property value
       function getProperty( PropertyID : Byte ) : Variant; override;
       // Sets a property value
       procedure setProperty( PropertyID : Byte; Value : Variant ); override;

       //Initialize before creation
       procedure Init;
       // Stream constructor
       constructor CreateFromStream( ISt : TStream ); override;
       // Stream writer
       procedure ToStream( OSt : TStream ); override;

       // register lua functions
       class procedure RegisterLuaAPI( Lua : TLua );
     private
       procedure MagicItemPrice;
     end;

implementation

uses sysutils, vmath, vutil,rlplayer, rlgobj, rllua, rlgame, variants;

procedure TItem.Init;
var LTable : TLuaTable;
begin
  Prefix := 0;
  Suffix := 0;
  PrefixValue := 0;
  SuffixValue := 0;

  Str := 0;
  Dex := 0;
  Vit := 0;
  Mag := 0;
  StrReq := 0;
  DexReq := 0;
  MagReq := 0;
  DurMax := 0;
  Dur    := 0;
  DurMod := 0;
  DmgMod:= 0; DmgModPercent := 0;
  ACModPercent := 0;
  Spell  := 0;
  Charges:= 0; ChargesMax:=0; ChargesMod := 1;
  Effect := 0;
  Price  := 0; PriceMod := 0; PriceMul:=0;
  ToHitMod := 0;
  LightMod := 0;
  LifeSteal:= 0;
  ManaSteal:= 0;
  Sound2 := '';
  Sound1 := '';
  LTable := TLuaTable.Create( Game.Lua, 'items', tID );
  with LTable do
  try
    Level  := GetNumber('level');
    Volume := GetNumber('volume');
    IType  := GetNumber('type');

    StrReq := GetNumber('minstr');
    MagReq := GetNumber('minmag');
    DexReq := GetNumber('mindex');

    Effect := GetNumber('effect');
    Price  := GetNumber('price');

    DurMax := GetNumber('durability');
    Dur    := RandRange(1+(DurMax div 4),(3*DurMax div 4));

    if Defined('spell')      then Spell := Game.Lua.IndexedTable[ 'spells', getString('spell'), 'nid' ];
    if Defined('str')        then str := getNumber('str');
    if Defined('dex')        then dex := getNumber('dex');
    if Defined('vit')        then vit := getNumber('vit');
    if Defined('mag')        then mag := getNumber('mag');
    if Defined('life')       then hpmod := getNumber('life');
    if Defined('mana')       then mpmod := getNumber('mana');
    if Defined('dmgmod')     then dmgmod := getNumber('dmgmod');
    if Defined('dmgmodpercent')then dmgmodpercent := getNumber('dmgmodpercent');
    if Defined('acmodpercent')then acmodpercent := getNumber('acmodpercent');
    if Defined('dmgtaken')   then dmgtaken := getNumber('dmgtaken');
    if Defined('tohit')      then tohitmod := getNumber('tohit');
    if Defined('durmod')     then durmod := getNumber('durmod');
    if Defined('spdatk')     then spdatk := getNumber('spdatk');
    if Defined('spdhit')     then spdhit := getNumber('spdhit');
    if Defined('spdblk')     then spdblk := getNumber('spdblk');
    if Defined('lightmod')   then lightmod := getNumber('lightmod');
    if Defined('lifesteal')  then lifesteal := getNumber('lifesteal');
    if Defined('manasteal')  then manasteal := getNumber('manasteal');
    if Defined('spelllevel') then spelllevel := getNumber('spelllevel');

    if Defined('sound1') then Sound1 := getString('sound1');
    if Defined('sound2') then Sound2 := getString('sound2');

    ReadHooks( LTable );
  finally
    Free;
  end;
end;

// Overriden constructor.
constructor TItem.Create(const thingID :string; ilvl : Word = 0);
var count    : byte;
begin
  // Most importantly set the section of the database by inheritance of
  // TThing.
  inherited Create(thingID);
  Init;
  DmgMin := 0; DmgMax := 0;
  Amount := 1;

  if not Game.Lua.Defined('items',thingID) then CritError('Item "'+thingID+'" not found!');

  with TLuaTable.Create( Game.Lua, 'items', thingID ) do
  try
    Name   := GetString('name');
    Flags  := GetFlags('flags');
    Pic    := GetString('pic')[1];
    Color  := GetNumber('color');

    DmgMin := GetNumber('damagemin');
    DmgMax := GetNumber('damagemax');

    AC := RandRange(GetNumber('armormin'),GetNumber('armormax'));
    if Defined('chargesmax') then
    begin
      chargesmax := getNumber('chargesmax');
      if Defined('charges') then charges:= getNumber('charges') else charges:=chargesmax;
    end;
    if ifUnique in flags then Include( Flags, ifUnknown );
  finally
    Free;
  end;

  if ilvl = 0 then ilvl := Level;
  RunHook( Hook_OnCreate, [ilvl]  )
end;

function TItem.BookMagReq : Word;
var i: word;
begin
  i := game.player.spells[spell];
  BookMagReq := MagReq;
  while i>0 do begin BookMagReq := (BookMagReq*12)div 10; dec(i); end;
  if BookMagReq > 213 then BookMagReq := 255;
end;

function TItem.InvColor: byte;
begin
  if Game.Player.ItemCheck(self) then
    if ifMagic in flags then exit(LightBlue)
    else if ifUnique in flags then exit(Yellow)
    else exit(LightGray)
  else exit(Red);
end;

procedure TItem.MagicItemPrice;
begin
  if PriceMul=0 then PriceMul:=1;
  if PriceMul>0 then
    Price:=Price*PriceMul+PriceMod
  else
    Price:=Price div abs(PriceMul)+PriceMod;
end;

procedure TItem.Identify;
var V : Variant;
begin
  if not (ifUnknown in Flags) then Exit;

  Exclude( Flags, ifUnknown );
  V := Game.Lua.IndexedTable[ 'items', tid, 'iname' ];
  if (not VarIsNull(V)) and ( V <> '' ) then Name := V;
  MagicItemPrice;
  if Prefix <> 0 then Name := Game.Lua.IndexedTable['prefixes',Prefix,'name'] + Name;
  if Suffix <> 0 then Name := Name + Game.Lua.IndexedTable['suffixes',Suffix,'name'];
  if (Spell  <> 0)and not (ifUnique in flags) then Name := Name + ' of ' + Game.Lua.IndexedTable['spells',Spell,'name'];
  if ifIndest in flags then
  begin
    Dur := 0;
    DurMax := 0;
  end;
end;

function TItem.GetPrice(PriceType: TPriceType): word;
begin
  if (Spell > 0) and ((ChargesMax div ChargesMod)> 0) then
    if PriceType = COST_RECHARGE then
      if ifSpellBound in flags then
        getPrice:=Game.Lua.IndexedTable['items',tid,'price']+Game.Lua.IndexedTable['spells',Spell,'charge_cost']*5
      else
        getPrice:=Game.Lua.IndexedTable['items',tid,'price']+Game.Lua.IndexedTable['spells',Spell,'charge_cost']*(5+(ChargesMax div ChargesMod))
    else
      if ifSpellBound in flags then
        getPrice:=Price
      else
        getPrice:=(Price*ChargesMax+Game.Lua.IndexedTable['spells',Spell,'charge_cost']*(Charges div ChargesMod))div ChargesMax
  else getPrice := Price;
  Case PriceType of
    COST_SELL: Result:= max(getPrice div 4,1);
    COST_REPAIR:
                begin
                  if (Dur = Durmax) or (DurMax = 0) then exit(0) else
                  if (ifUnknown in flags)or ([ifMagic, ifUnique]*flags=[]) then
                    getPrice:=(((100*(DurMax-Dur)) div DurMax)*getPrice(COST_BUY)) div 100
                  else  //unidentified or non-magic
                    getPrice:=((((100*(DurMax-Dur)) div DurMax)*getPrice(COST_BUY)*3) div 1000); //magical or unique
                  if getPrice > 0 then exit(max(1,getPrice div 2))
                  else if ([ifMagic, ifUnique]*flags = []) or (ifUnknown in flags) then exit(1)
                  else exit(0);
                end;
    COST_RECHARGE: if ChargesMax = 0 then exit(0) else
                getPrice:=(((100*(ChargesMax-Charges)) div ChargesMax) * getPrice) div 200;
  end;
end;

procedure TItem.Repair(RLevel: Byte);
begin
  if RLevel > 0 then
  repeat
    Dur    += RLevel + Random(RLevel);
    DurMax := Max( DurMax - Max(1, Round(DurMax / (RLevel + 9))), 1 );
  until Dur >= DurMax;
  Dur := DurMax;
end;

procedure TItem.Recharge(RLevel: Byte);
begin
  if RLevel > 0 then
  repeat
    Charges+= Random(RLevel div Game.Lua.IndexedTable['spells',Spell,'book_level'])+1;
    dec(ChargesMax);
  until Charges >= ChargesMax;
  Charges := ChargesMax;
end;

function TItem.GetName(outputType : TNameOutputType) : string;
var temp : string[70] = '';
    c: string[2] = '';
    _MagReq: word;
  function VS(l : LongInt) : string;
  begin if l > 0 then Exit('+'+IntToStr(l)) else Exit(IntToStr(l)); end;
begin

  if OutputType = SpellName then
  if Spell > 0 then
  begin
    with TLuaTable.Create(Game.Lua,'spells',Spell) do
    try
      temp := getString('name');
      if ifUnique in flags then
      _MagReq := 0
        else
      _MagReq:=getNumber('magic');
    finally
      Free;
    end;
    if _MagReq > game.player.getMag then begin temp := '@r'+temp; c:= '@r' end
    else if chargesmax > 0 then
    if (charges*100 div chargesmax) <= 50 then c:='@y'
    else if (charges*100 div chargesmax) <= 10 then c:='@r';
    if ChargesMax > 0 then temp := temp+' {'+c+inttostr(charges)+'@>'+'/'+c+inttostr(chargesmax)+'@>}';
    exit(temp);
  end;

  if (OutputType = Status2) then
  begin
    if IType in [TYPE_HELM, TYPE_WEAPON, TYPE_ARMOR, TYPE_SHIELD, TYPE_BOW, TYPE_STAFF] then
      if (ifIndest in flags) and ( not ( ifUnknown in Flags ) ) then
        temp:='Indestructible'
      else
        temp:='Durability: '+inttostr(Dur)+'/'+inttostr(DurMax)
    else
      temp:='';
    if itype = TYPE_BOOK then _MagReq := BookMagReq else _MagReq := MagReq;
    if ((StrReq<>0) or (DexReq<>0) or (_MagReq<>0)) then
    begin
      if not game.player.itemcheck(self) then
        temp:=temp+'@r';
      temp:=temp+' Requires ';
      if StrReq<>0 then
      begin
        if StrReq>Game.Player.GetStr then
          temp:=temp+'@r';
        temp:=temp+'STR:'+inttostr(StrReq);
        if (_MagReq<>0) or (DexReq<>0) then
          temp:=temp+'@> ';
      end;
      if _MagReq<>0 then
      begin
        if _MagReq>Game.Player.GetMag then
          temp:=temp+'@r';
        temp:=temp+'MAG:'+inttostr(_MagReq);
        if (DexReq<>0) then
          temp:=temp+'@> ';
      end;
      if DexReq<>0 then
      begin
        if DexReq>Game.Player.GetDex then
          temp:=temp+'@r';
        temp:=temp+'DEX:'+inttostr(DexReq)
      end;
      exit(temp);
    end
    else
      exit(temp);
  end;

  if (OutputType = Status1) then
  begin
    if ifUnknown in Flags then Exit('Unidentified');
    temp:='';
    case SpdAtk of
      SPD_QUICK: temp+='Quick Attack, ';
      SPD_FAST: temp+='Fast Attack, ';
      SPD_FASTER: temp+='Faster Attack, ';
      SPD_FASTEST: temp+='Fastest Attack, ';
    end;
    case SpdHit of
      SPD_FAST: temp+='Fast Recovery, ';
      SPD_FASTER: temp+='Faster Recovery, ';
      SPD_FASTEST: temp+='Fastest Recovery, ';
    end;
    case SpdBlk of
      SPD_FAST: temp+='Fast Blocking, ';
    end;
    if (Str = Dex) and (Dex = Mag) and
       (Mag = Vit) and (Vit <>0) then
       temp:=temp+('All Stats '+VS(Str))+', '
    else
    begin
      if Str  <> 0 then temp:=temp+('Strength '+VS(Str))+', ';
      if Mag  <> 0 then temp:=temp+('Magic '+VS(Mag))+', ';
      if Vit  <> 0 then temp:=temp+('Vitality '+VS(Vit))+', ';
      if Dex  <> 0 then temp:=temp+('Dexterity '+VS(Dex))+', ';
    end;
    if ifBaneDemon  in flags then temp:=temp+'+200% Dmg vs Demons,';
    if ifBaneUndead in flags then temp:=temp+'+200% Dmg vs Undead,';
    if ifBaneAnimal in flags then temp:=temp+'+200% Dmg vs Animals,';
    if HPMod <> 0 then temp:=temp+('Life '+VS(HPMod))+', ';
    if MPMod <> 0 then temp:=temp+('Mana '+VS(MPMod))+', ';
    if (AC <> 0) and not(IType in [TYPE_ARMOR, TYPE_HELM, TYPE_SHIELD]) then temp:=temp+VS(AC)+' AC, ';
    if DmgMod  <> 0 then temp:=temp+('Damage '+VS(DmgMod))+', ';
    if ACModPercent <> 0 then temp:=temp+('AC '+VS(ACModPercent)+'%, ');
    if ToHitMod <> 0 then temp:=temp+('ToHit '+VS(ToHitMod)+'%, ');
    if DmgModPercent <> 0 then temp:=temp+('Damage '+VS(DmgModPercent)+'%, ');
    if LightMod <> 0 then temp:=temp+('Light '+VS(LightMod)+'%, ');
    if DmgTaken<>0 then temp:=temp+VS(DmgTaken)+' Damage Taken, ';
    if DurMod>0 then temp:=temp+' High Durability, '
    else if DurMod<0 then temp:=temp+' Low Durability, ';
    if ChargesMod>1 then temp:=temp+' Extra Charges, ';
    if LifeSteal<>0 then temp:=temp+' Steal '+VS(LifeSteal div 10)+'% Life, ';
    if ManaSteal<>0 then temp:=temp+' Steal '+VS(ManaSteal div 10)+'% Mana, ';
    if SpellLevel>0 then temp:=temp+VS(SpellLevel)+' Spell Level, ';
    if ifVampiric in flags   then temp:=temp+' Steals Life, ';
    if ifTrapResist in flags then temp:=temp+' Half Trap Dmg, ';
    if ifKnockback in flags  then temp:=temp+' Knockback, ';

    if (length(temp) > 0) and (temp[length(temp)-1]=',') then delete(temp, length(temp)-1, 2);
    Exit(temp);
  end;

  temp := name;

  if Amount > 1 then
  case outputType of
    TheName    : Exit(IntToStr(Amount)+' '+temp);
    AName      : Exit('a pile of '+IntToStr(Amount)+' '+temp);
    PlainName  : Exit(IntToStr(Amount)+' '+temp);
    CTheName   : Exit(IntToStr(Amount)+' '+temp);
    CAName     : Exit('A pile of '+IntToStr(Amount)+' '+temp);
    InvName    : Exit(DPad(IntToStr(Amount)+' '+temp,35)+' '+IntToStr(Volume));
    Status1    : Exit(IntToStr(Amount)+' '+temp);
  end;

  if IType in [TYPE_ARMOR, TYPE_HELM, TYPE_SHIELD] then
    temp := temp+' ['+IntToStr(AC)+']';

  if IType in [TYPE_WEAPON, TYPE_BOW, TYPE_STAFF] then
    temp := temp+' ('+IntToStr(DmgMin)+'-'+IntToStr(DmgMax)+')';

  if (ifUnique in flags) and (OutputType in [CTheName, CAName]) then
    temp := CapLet(temp);
  if chargesmax > 0 then
    temp := temp+' {'+inttostr(charges)+'/'+inttostr(chargesmax)+'}';
  case outputType of
    TheName    : if ([ifUnique]*flags=[]) then temp := 'the '+temp;
    AName      : if ([ifUnique]*flags=[]) then temp := 'a '+temp;
    CTheName   : if ([ifUnique]*flags=[]) then temp := 'The '+temp;
    CAName     : if ([ifUnique]*flags=[]) then temp := 'A '+temp;
    InvName    : temp := DPad(temp,35)+' '+IntToStr(Volume);
    BuyName    : temp := DPad(temp,54)+' '+IntToStr(Price);
  end;

  Exit(temp);
end;

function TItem.getProperty(PropertyID: Byte): Variant;
begin
  case PropertyID of
    PROP_AC     : Exit( AC );
    PROP_ACMODPERCENT: Exit(ACModPercent);
    PROP_DMGMOD : Exit( DmgMod );
    PROP_DMGMODPERCENT: Exit(DmgModPercent);

    PROP_STR    : Exit( Str );
    PROP_MAG    : Exit( Mag );
    PROP_DEX    : Exit( Dex );
    PROP_VIT    : Exit( Vit );
    PROP_LIFE,
    PROP_HPMOD  : Exit( HPMod );
    PROP_MANA,
    PROP_MPMOD  : Exit( MPMod );

    PROP_STRREQ : Exit( StrReq );
    PROP_MAGREQ : Exit( MagReq );
    PROP_DEXREQ : Exit( DexReq );

    PROP_VOLUME : Exit( Volume );
    PROP_AMOUNT : Exit( Amount );
    PROP_EFFECT : Exit( Effect );
    PROP_DURMAX : Exit( DurMax );
    PROP_DUR    : Exit( Dur );
    PROP_PRICE  : Exit( Price );
    PROP_DURMOD : Exit( DurMod );
    PROP_PRICEM : Exit( PriceMod );
    PROP_PRICEMULT : Exit( PriceMul );

    PROP_ITYPE  : Exit(IType); //read-only
    PROP_SPELL  : Exit(Spell);
    PROP_SPELLLEVEL : Exit(SpellLevel);
    PROP_CHARGES: Exit(Charges);
    PROP_CHARGESMAX:Exit(ChargesMax);
    PROP_LIGHTMOD: Exit( LightMod );
    PROP_DMGTAKEN: Exit( DmgTaken );
    PROP_SPDATK : Exit(SpdAtk);
    PROP_SPDHIT : Exit(SpdHit);
    PROP_SPDBLK : Exit(SpdBlk);
    PROP_TOHIT  : Exit(ToHitMod);
    PROP_CHARGESMOD:Exit(ChargesMod);
    PROP_LIFESTEAL: Exit(LifeSteal);
    PROP_MANASTEAL: Exit(ManaSteal);

    PROP_PREFIX : Exit(Prefix);
    PROP_SUFFIX : Exit(Suffix);
    PROP_PREFIXVALUE : Exit(PrefixValue);
    PROP_SUFFIXVALUE : Exit(SuffixValue);
else Exit( inherited getProperty(PropertyID) )
  end;
end;

procedure TItem.setProperty(PropertyID: Byte; Value: Variant);
begin
  case PropertyID of
    PROP_AC     : AC := Value;
    PROP_ACMODPERCENT: ACModPercent:= Value;
    PROP_DMGMOD : DmgMod := Value;
    PROP_DMGMODPERCENT: DmgModPercent:= Value;

    PROP_STR    : Str := Value;
    PROP_MAG    : Mag := Value;
    PROP_DEX    : Dex := Value;
    PROP_VIT    : Vit := Value;
    PROP_LIFE,
    PROP_HPMOD  : HPMod := Value;
    PROP_MANA,
    PROP_MPMOD  : MPMod := Value;

    PROP_STRREQ : StrReq := Value;
    PROP_MAGREQ : MagReq := Value;
    PROP_DEXREQ : DexReq := Value;

    PROP_VOLUME : Volume := Value;
    PROP_AMOUNT : Amount := Value;
    PROP_EFFECT : Effect := Value;
    PROP_DURMAX : DurMax := Value;
    PROP_DUR    : Dur := Value;
    PROP_PRICE  : Price := Value;
    PROP_DURMOD : DurMod := Value;
    PROP_PRICEM : PriceMod := Value;
    PROP_PRICEMULT : PriceMul := Value;

    PROP_SPELL  : Spell := Game.Lua.IndexedTable[ 'spells', Value, 'nid' ];
    PROP_SPELLLevel  : SpellLevel := Value;
    PROP_CHARGES: Charges := Value;
    PROP_CHARGESMAX: ChargesMax := Value;
    PROP_LIGHTMOD: LightMod := Value;
    PROP_DMGTAKEN: DmgTaken := Value;
    PROP_SPDATK : SpdAtk := Value;
    PROP_SPDHIT : SpdHit := Value;
    PROP_SPDBLK : SpdBlk := Value;
    PROP_TOHIT  : ToHitMod:=Value;
    PROP_CHARGESMOD:ChargesMod:=Value;
    PROP_LIFESTEAL: LifeSteal := Value;
    PROP_MANASTEAL: ManaSteal := Value;
    PROP_PREFIX : Prefix := Value;
    PROP_SUFFIX : Suffix := Value;
    PROP_PREFIXVALUE : PrefixValue := Value;
    PROP_SUFFIXVALUE : SuffixValue := Value;
    else inherited setProperty(PropertyID, Value);
  end;
  end;

procedure TItem.TimeFlow(time : LongInt);
begin
  inherited TimeFlow(time);
end;

constructor TItem.CreateFromStream(ISt: TStream);
var i,j: word;
begin
  inherited CreateFromStream(ISt);
  Init;
  IType     := ISt.ReadByte;
  if IType = TYPE_GOLD then
    Amount := ISt.ReadDWord
  else
  begin
    Amount  := 1;
    i       := ISt.ReadWord;
    j       := ISt.ReadWord;
    AC      := ISt.ReadWord;
    Spell   := ISt.ReadByte;
    Charges := ISt.ReadWord;
    ChargesMax := ISt.ReadWord;
    Prefix  := ISt.ReadWord;
    Suffix  := ISt.ReadWord;
    PrefixValue:= ISt.ReadWord;
    SuffixValue:= ISt.ReadWord;
    StrReq := ISt.ReadWord;
    DexReq := ISt.ReadWord;
    MagReq := ISt.ReadWord;
  end;

  if (Prefix <> 0) or (Suffix<>0) then Game.Lua.RunHook('items',ID,'load_fix',[]);
  dur := i; durmax := j;
  if (not(ifUnknown in flags))and([ifMagic, ifUnique]*flags<>[]) then
    MagicItemPrice;
end;

procedure TItem.ToStream(OSt: TStream);
begin
  inherited ToStream(OSt);
  OSt.WriteByte(IType);
  if IType = TYPE_GOLD then
  begin
    OSt.WriteDWord(Amount);
    exit;
  end;
  OSt.WriteWord(Dur);
  OSt.WriteWord(DurMax);
  OSt.WriteWord(AC);
  OSt.WriteByte(Spell);
  OSt.WriteWord(Charges);
  OSt.WriteWord(ChargesMax);
  OSt.WriteWord(Prefix);
  OSt.WriteWord(Suffix);
  OSt.WriteWord(PrefixValue);
  OSt.WriteWord(SuffixValue);
  OSt.WriteWord(StrReq);
  OSt.WriteWord(DexReq);
  OSt.WriteWord(MagReq);
end;

function lua_item_new(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    it      : TItem;
begin
  State.Init(L);
  State.Push( TItem.Create( State.ToString(1), State.ToInteger(2) ) );
  Result := 1;
end;

function lua_item_identify(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    it      : TItem;
begin
  State.ObjectInit(L,it);
  it.Identify;
  Result := 0;
end;

class procedure TItem.RegisterLuaAPI(Lua: TLua);
begin
  Lua.SetTableFunction('item','identify',   @lua_item_identify);
  Lua.SetTableFunction('item','new',        @lua_item_new);
end;

end.
