{$MODE OBJFPC}
// @abstract(Game Object for RL Core)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(January 17, 2005)
// @lastmod(January 17, 2005)
//
// This unit holds the game's generic game object class : TGameObject.
// The purpose is for all the object to share a common Save/Load interface.

unit rlgobj;
interface
uses classes, vlua, vluastate, vnode, vutil, rlglobal;

type THook  = ( Hook_OnCreate, Hook_OnAttack, Hook_OnBroadcast, Hook_OnUse, Hook_OnTalk, Hook_OnAct, Hook_OnSpot, Hook_OnHit, Hook_OnPickUp, Hook_OnDrop, Hook_OnDie );
     THooks = set of THook;
const HookNames : array[ THook ] of string = ( 'OnCreate', 'OnAttack', 'OnBroadcast', 'OnUse', 'OnTalk', 'OnAct', 'OnSpot', 'OnHit', 'OnPickUp', 'OnDrop', 'OnDie' );

type

{ TGameObject }

TGameObject = class(TNode, TLuaReferencedObject)
       Flags    : TFlags;
       Name     : AnsiString;
       tid      : AnsiString;
       LuaIndex : Integer;
       Hooks    : THooks;
       // General constructor
       constructor Create(const thingID :string); virtual; reintroduce;
       // GetName returns the name of the thing -- to be overridden
       // GetName output MAY include colors -- for plain name use Name
       // directly.
       function GetName(outputType : TNameOutputType) : string; virtual;
       // This general procedure handles time-flow. Time is the amount
       // of time passed since last call. In GameObject it just propagates
       // TimeFlow to children and siblings.
       // Note: Due to this fact TGameObject siblings and children MUST
       // be TGameObjects.
       procedure TimeFlow(time : LongInt); virtual;
       // Returns a property value
       function getProperty( PropertyID : Byte ) : Variant; virtual;
       // Sets a property value
       procedure setProperty( PropertyID : Byte; Value : Variant ); virtual;
       // Runs a script with thing set to self
       function RunHook( Hook : THook; const Args : array of Const ) : Variant;

       // Stream constructor
       constructor CreateFromStream( ISt : TStream ); virtual;
       // Stream writer
       procedure ToStream( OSt : TStream ); virtual;

       // TLuaReferencedObject -> get Lua reference table index
       function GetLuaIndex   : Integer;
       // TLuaReferencedObject -> get object ID
       function GetID         : AnsiString;
       // TLuaReferencedObject -> get object prototype table name
       function GetProtoTable : AnsiString;
       // TLuaReferencedObject -> get object prototype name
       function GetProtoName : AnsiString;

       // Unregisters object
       destructor Destroy; override;

       // Check hooks in table
       procedure ReadHooks( Table : TLuaTable );

       // register lua functions
       class procedure RegisterLuaAPI( Lua : TLua );
     public
       property Properties[ PropertyID : Byte ] : Variant read getProperty write setProperty; default;
     end;

implementation
uses SysUtils, Variants, rllua, rlgame, rllevel, rlitem, rlplayer, rlnpc, rlshopinv;

constructor TGameObject.Create(const thingID :string);
var State : TRLLuaState;
begin
  State.Init(Game.Lua.LuaState);
  Hooks := [];
  inherited Create;
  tid := LowerCase(thingID);
  LuaIndex := State.RegisterNewObject( Self, GetProtoName );
  State.RegisterNewSubObject( LuaIndex, Self, 'flags', 'game_object_flags' );
end;

function TGameObject.GetName(outputType : TNameOutputType) : string;
begin
  case outputType of
    TheName    : Exit('the '+Name);
    AName      : Exit('a '+Name);
    CTheName   : Exit('The '+Name);
    CAName     : Exit('A '+Name);
    else Exit(name);
  end;
end;




procedure TGameObject.TimeFlow(time : LongInt);
var Scan1,Scan2 : TNode;
begin
  if Child <> nil then
  begin
    Scan1 := Child;
    repeat
      if Scan1<>nil then
      begin
        Scan2 := Scan1.Next;
        TGameObject(Scan1).TimeFlow(time);
        if Scan1.Parent <> Self then Scan1 := Scan2 else Scan1 := Scan1.Next;
      end;
    until (Scan1 = Child)or(Scan1=nil);
  end;
end;

function TGameObject.getProperty(PropertyID : Byte): Variant;
begin
  case PropertyID of
    PROP_NAME   : Exit( Name );
    PROP_ID     : Exit( tid );
    else
      CritError( 'Unknown property #@1 requested!', [PropertyID] );
  end;
  Exit( Null );
end;

procedure TGameObject.setProperty(PropertyID : Byte; Value: Variant);
begin
  case PropertyID of
    PROP_NAME   : Name := Value;
 // PROP_ID     : tid  := Value; // READ-ONLY
  else
    CritError( 'Unknown property #@1 requested!', [PropertyID] );
  end;
end;

function TGameObject.RunHook( Hook : THook; const Args: array of const) : Variant;
var State : TRLLuaState;
begin
  if Hook in Hooks then
  begin
    State.Init(Game.Lua.LuaState);
    Exit( State.RunHook( Self, HookNames[ Hook ], Args ) );
  end;
  Exit( false );
end;

constructor TGameObject.CreateFromStream(ISt: TStream);
var State : TRLLuaState;
begin
  State.Init(Game.Lua.LuaState);
  Clean;
  LuaIndex := 0;
  tid  := ISt.ReadAnsiString();
  ISt.Read( Flags, SizeOf(Flags));
  Name := ISt.ReadAnsiString();
  Log('Loading '+TID+'/'+Name);
  LuaIndex := State.RegisterNewObject( Self, GetProtoName );
  State.RegisterNewSubObject( LuaIndex, Self, 'flags', 'game_object_flags' );
end;

procedure TGameObject.ToStream(OSt: TStream);
begin
  OSt.WriteAnsiString(tid);
  Log('Saving '+TID+'/'+Name);
  OSt.Write( Flags, SizeOf(Flags) );
  OSt.WriteAnsiString(Name);
end;

function TGameObject.GetLuaIndex: Integer;
begin
  Exit(LuaIndex);
end;

function TGameObject.GetID: AnsiString;
begin
  Exit(TID);
end;

function TGameObject.GetProtoTable: AnsiString;
begin
  if Self is TItem  then Exit('items');
  if Self is TNPC   then Exit('npcs');
  if Self is TLevel then Exit('levels');
  if Self is TShopInventory then Exit('shops');
  CritError('Class "'+ClassName+'" GetProtoTable requested!');
end;

function TGameObject.GetProtoName: AnsiString;
begin
  if Self is TItem then Exit('item');
  if Self is TNPC then Exit('npc');
  if Self is TShopInventory then Exit('shop');
  Exit('thing');
end;

destructor TGameObject.Destroy;
var State : TRLLuaState;
begin
  State.Init(Game.Lua.LuaState);
  State.UnRegisterObject( Self );
  inherited Destroy;
end;

procedure TGameObject.ReadHooks(Table: TLuaTable);
var Hook : THook;
begin
  Hooks := [];
  for Hook := Low(THooks) to High(THooks) do
    if Table.isFunction(HookNames[Hook]) then
      Include( Hooks, Hook );
end;

function lua_game_object_property_set(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    go      : TGameObject;
begin
  State.ObjectInit(L,go);
  if State.StackSize < 3 then Exit(0);
  go.setProperty(State.ToInteger(2), State.ToVariant(3));
  Result := 0;
end;

function lua_game_object_property_get(L: Plua_State): Integer; cdecl;
var State   : TRLLuaState;
    go      : TGameObject;
begin
  State.ObjectInit(L,go);
  if State.StackSize < 2 then Exit(0);
  State.PushVariant( go.getProperty( State.ToInteger(2) ) );
  Result := 1;
end;

function lua_game_object_flags_get(L: Plua_State): Integer; cdecl;
var State   : TRLLuaState;
    go      : TGameObject;
begin
  State.ObjectInit(L,go);
  if State.StackSize < 2 then Exit(0);
  State.Push( State.ToInteger(2) in go.flags);
  Result := 1;
end;

function lua_game_object_flags_set(L: Plua_State): Integer; cdecl;
var State   : TRLLuaState;
    go      : TGameObject;
    Flag    : byte;
begin
  State.ObjectInit(L,go);
  if State.StackSize < 3 then Exit(0);
  Flag := State.ToInteger(2);
  if State.ToBoolean(3) then
    Include(go.Flags,Flag)
  else
    Exclude(go.Flags,Flag);
  Result := 0;
end;

function lua_game_object_is_player(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    go      : TGameObject;
begin
  State.ObjectInit(L,go);
  State.Push( go is TPlayer );
  Result := 1;
end;

function lua_game_object_is_npc(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    go      : TGameObject;
begin
  State.ObjectInit(L,go);
  State.Push( go is TNPC  );
  Result := 1;
end;

function lua_game_object_is_item(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    go      : TGameObject;
begin
  State.ObjectInit(L,go);
  State.Push( go is TItem  );
  Result := 1;
end;

function lua_game_object_is_level(L: Plua_State) : Integer; cdecl;
var State   : TRLLuaState;
    go      : TGameObject;
begin
  State.ObjectInit(L,go);
  State.Push( go is TLevel  );
  Result := 1;
end;


class procedure TGameObject.RegisterLuaAPI(Lua: TLua);
begin
  Lua.SetTableFunction('game_object','property_get',@lua_game_object_property_get);
  Lua.SetTableFunction('game_object','property_set',@lua_game_object_property_set);
  Lua.SetTableFunction('game_object','flags_get',   @lua_game_object_flags_get);
  Lua.SetTableFunction('game_object','flags_set',   @lua_game_object_flags_set);

  Lua.SetTableFunction('game_object','is_player',   @lua_game_object_is_player);
  Lua.SetTableFunction('game_object','is_npc',      @lua_game_object_is_npc);
  Lua.SetTableFunction('game_object','is_item',     @lua_game_object_is_item);
  Lua.SetTableFunction('game_object','is_level',    @lua_game_object_is_level);
end;


end.
