{$MODE OBJFPC}
// @abstract(Generators for RL Core)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(March 6, 2005)
// @lastmod(January 17, 2009)
//
// This unit hold's -cell data;

unit rlgen;
interface
uses vnode, vrltools, vdungen, vutil, rllevel, variants, vds;

type TCoordVector = specialize TVector<TCoord2D>;
     TStringVector = specialize TVector<AnsiString>;
var  PlayerC    : TCoord2D;


type

TTileData   = class;
TTileVector = specialize TVector<TTileData>;

{ TGenerator }

TGenerator = class(TDungeonBuilder)
  MapCoord : TCoord2D;
  constructor Create( nLevel : TLevel );
  function GetCell( const aWhere : TCoord2D ) : Byte; override;
  procedure PutCell( const aWhere : TCoord2D; const aWhat : Byte ); override;
  function isEmpty( const coord : TCoord2D; EmptyFlags : TFlags32 = []) : Boolean; override;
  procedure GenerateLevel(const ltype : string; const gtype : string; const MapCode : AnsiString; Translation : TPrintableCharToByte);
  destructor Destroy; override;
private
  procedure RoomLevel;
  procedure LoadTileset(TileSet : string);
private
  Level      : TLevel;
  TileData   : TTileVector;
  HotSpots   : TCoordVector;
  CellFloors : TCellSet;
  CellWalls : TCellSet;
  GenType    : string;
end;

{ TTileData }

TTileData = class(TDungeonTile)
  constructor Create( Generator : TGenerator; const TileCode : Ansistring; const Translation : TPrintableCharToByte );
  destructor  Destroy; override;
  procedure FlipX;
  procedure FlipY;
  procedure RandomFlip;
  procedure WriteToMap( Where : TCoord2D );
  function PickEntry(dir : char) : TCoord2D;
  function Scan( Where : TCoord2D; GoodTiles : TCellSet) : word;
private
  Gen : TGenerator;
end;

implementation
uses sysutils, vmath, vdebug, rlgame, rlglobal, vlua, rlcell;

const GenScanInvalid     = 50000;
      MaxDirExits        = 6;    {** max exits in one direction!}
      MonsterGroupNumber = 30;

function TTileData.PickEntry(dir : char) : TCoord2D;
var dx, dy : ShortInt;
    sx, sy : Byte;
    exlist : array[1..MaxDirExits] of TCoord2D;
    excnt  : byte;
function Good(ax,ay : Integer) : boolean;
begin
  if not Area.Contains(NewCoord2D(ax,ay)) then Exit(false);
  if (Map[ax,ay] <> DUNGEN_TILE_IGNORE) and
     (not (Map[ax,ay] in Gen.CellWalls)) then Exit(true);
  Exit(false);
end;
begin
  dx := 0; dy := 0;

  case LowerCase(dir) of
    's' : dy := +1;
    'n' : dy := -1;
    'e' : dx := +1;
    'w' : dx := -1;
  end;
  excnt := 0;
  for sx := 0 to SizeX-1 do
    for sy := 0 to SizeY-1 do
      if Map[sx,sy] = DUNGEN_TILE_HOTSPOT then
        if Good(sx+dx,sy+dy) then
          begin
            Inc(excnt);
            if excnt > MaxDirExits then CritError('TileData.PickEntry -- overflow of exits in one direction (MaxDirExits)');
            exlist[excnt].x := sx;
            exlist[excnt].y := sy;
          end;
  if excnt < 1 then CritError('TileData.PickEntry -- Tile "": no exit in "'+dir+'" direction.');
  Exit( exlist[Random(excnt)+1] );
end;

constructor TTileData.Create(Generator : TGenerator; const TileCode: Ansistring; const Translation: TPrintableCharToByte);
begin
  inherited Create( TileCode, Translation );
  Gen := Generator;
end;

destructor  TTileData.Destroy;
begin
  inherited Destroy;
end;

procedure TTileData.FlipX;
var TX,TY,T : Byte;
begin
  for TX := 0 to (SizeX div 2) do
    for TY := 0 to SizeY-1 do
    begin
      T:=Map[TX,TY];
      Map[TX,TY] := Map[SizeX-TX-1,TY];
      Map[SizeX-TX-1,TY] := T;
   end;
end;

procedure TTileData.FlipY;
var TX,TY,T : Byte;
begin
  for TX := 0 to SizeX-1 do
    for TY := 0 to (SizeY div 2) do
    begin
      T:=Map[TX,TY];
      Map[TX,TY] := Map[TX,SizeY-TY-1];
      Map[TX,SizeY-TY-1] := T;
    end;
end;

procedure TTileData.RandomFlip;
begin
  case Random(4) of
    1 : FlipX;
    2 : FlipY;
    3 : begin FlipX; FlipY; end;
  end;
end;

procedure TTileData.WriteToMap( Where : TCoord2D );
var a,b : byte;
    tc  : TCoord2D;
begin
  for a := 0 to SizeX-1 do
    for b := 0 to SizeY-1 do
      if Map[a,b] <> DUNGEN_TILE_IGNORE then
      begin
        tc.Create(Where.x+a,Where.y+b);
        if Gen.properCoord(tc) then
          if Map[a,b] = DUNGEN_TILE_HOTSPOT then
          begin
            Gen.PutCell(tc,CELL_WALL);
            Gen.HotSpots.Push(tc);
          end
             else
               Gen.PutCell(tc,Map[a,b]);
      end;
end;

function TTileData.Scan( Where : TCoord2D; GoodTiles : TCellSet):Word;
var Other : word;
    a,b   : Word;
begin
  if not Gen.Area.Shrinked.Contains( Where )          then Exit(GenScanInvalid);
  if not Gen.Area.Shrinked.Contains( Where + Area.B ) then Exit(GenScanInvalid);
  Other := 0;
  for a := Where.x to Where.x+SizeX-1 do
    for b := Where.y to Where.y+SizeY-1 do
      if Map[a-Where.x,b-Where.y] <> DUNGEN_TILE_IGNORE then
        if not (Gen.GetCell(NewCoord2D(a,b)) in GoodTiles) then Inc(Other);
  Scan := Other;
end;


{*********}


{ TGenerator }

constructor TGenerator.Create(nLevel: TLevel);
begin
  inherited Create( MapSizeX, MapSizeY );
  MapCoord := NewCoord2D(1,1);
  Level := nLevel;
  TileData := TTileVector.Create;
  HotSpots := TCoordVector.Create;

  CellFloors := [CELL_FLOOR];
  CellWalls  := [CELL_WALL,CELL_DOOR_CLOSED,CELL_DOOR_CLOSED_2];
end;

function TGenerator.GetCell(const aWhere: TCoord2D): Byte;
begin
  Exit( Level.Map[aWhere.X,aWhere.y].Cell );
end;

procedure TGenerator.PutCell(const aWhere: TCoord2D; const aWhat: Byte);
begin
  Level.Map[aWhere.X,aWhere.y].Cell := aWhat;
end;

function TGenerator.isEmpty(const coord: TCoord2D; EmptyFlags: TFlags32): Boolean;
begin
  Exit( Level.CellCheck(coord, EmptyFlags) );
end;

procedure TGenerator.GenerateLevel(const ltype: string; const gtype: string; const MapCode: AnsiString;
  Translation: TPrintableCharToByte);
var Where : TCoord2D;
    Count : Byte;
    StartTile : TTileData;
begin
  GenType := gtype;
  LoadTileset('tiles_'+gtype);
  Translation[ Ord('#') ] := CELL_WALL;
  Translation[ Ord('.') ] := CELL_FLOOR;
  Translation[ Ord('+') ] := CELL_DOOR_CLOSED;

  Fill(CELL_WALL);
  Where.X := Random(MapSizeX-50)+25;
  Where.Y := Random(MapSizeY-50)+20;

  if MapCode <> '' then
  begin
    if ltype <> 'map' then MapCoord := Where;
    StartTile := TTileData.Create(Self,MapCode,Translation);
    for Count := 1 to 10 do
      StartTile.WriteToMap(Where);
    FreeAndNil(StartTile);
  end
  else
  begin
    for Count := 1 to 20 do  HotSpots.Push(Where);
  end;
  RoomLevel;
  Game.Lua.Level := Level;
  with TLuaTable.Create( Game.Lua, 'generators', GenType ) do
  try
    if isFunction('OnPlaceItems')    then Execute('OnPlaceItems', [Level.Level] );
    if isFunction('OnPlaceMonsters') then Execute('OnPlaceMonsters', [Level.Level, MonsterGroupNumber] );
  finally
    Free;
  end;
  //PlaceItems;
  //PlaceMonsters;
end;

destructor TGenerator.Destroy;
var cnt : word;
begin
  if not TileData.Empty then
  for cnt := 0 to TileData.Size - 1 do
    TileData[cnt].Free;
  FreeAndNil( TileData );
  FreeAndNil( HotSpots );
  inherited Destroy;
end;

procedure TGenerator.RoomLevel;
var FCnt    : word;
    FLimit  : word;
    sca     : word;
    Entry   : TCoord2D;
    Where   : TCoord2D;
    GenOk   : Boolean;
    Dir     : TDirection;
    Doorway : byte;
    DoorPos : TCoord2D;

  procedure ClearDeadEnds;
  var GenX,GenY : byte;
  begin
    for GenX := 2 to MapSizeX-1 do
      for GenY := 2 to MapSizeY-1 do
        if (CrossAround(NewCoord2D(GenX,GenY),CellWalls) = 3) then
          PutCell(NewCoord2D(GenX,GenY),CELL_WALL);
  end;
  function RandomDoor : Byte;
  begin
    case Random(12) of
      0..7 : Exit(CELL_DOOR_CLOSED);
      8..9 : Exit(CELL_DOOR_CLOSED_2);
      else Exit(CELL_FLOOR);
    end;
  end;
  function RanDirection(Coord : TCoord2D) : char;
  begin
    if Area.isEdge(Coord) then Exit('n');
    if GetCell(Coord.ifIncY(+1)) in CellFloors then  Exit('n');
    if GetCell(Coord.ifIncY(-1)) in CellFloors then  Exit('s');
    if GetCell(Coord.ifIncX(+1)) in CellFloors then  Exit('w');
    if GetCell(Coord.ifIncX(-1)) in CellFloors then  Exit('e');
    Exit('n');
  end;

begin
  FLimit := 300;
  FCnt := 0;
  repeat
    Inc(FCnt);
    if HotSpots.Empty then Break;
    with TileData[Random(TileData.Size)] do
    begin
      GenOK := True;
      Where := HotSpots[ Random(HotSpots.Size) ];
      RandomFlip;
      Entry := PickEntry(RanDirection(Where));
      sca := Scan( Where - Entry, CellWalls );
      if sca = GenScanInvalid then Continue;
      if sca > 1 then GenOk := False;
      if not GenOk then Continue;
      WriteToMap(Where - Entry );
      //determine doorway bounds
      Entry := Where;
      if RanDirection(Where) in ['n','s'] then
      begin
        while (GetCell(Where.ifInc(-1,-1)) in CellFloors) and (GetCell(Where.ifInc(-1,1)) in CellFloors) do dec(Where.x);
        while (GetCell(Entry.ifInc(1,-1)) in CellFloors) and (GetCell(Entry.ifInc(1,1)) in CellFloors) do inc(Entry.x);
      end
      else
      begin
        while (GetCell(Where.ifInc(-1,-1)) in CellFloors) and (GetCell(Where.ifInc(1,-1)) in CellFloors) do dec(Where.y);
        while (GetCell(Entry.ifInc(-1,1)) in CellFloors) and (GetCell(Entry.ifInc(1,1)) in CellFloors) do inc(Entry.y);
      end;
      //roll a dooorway style
      if Level.Level < 5 then
        Doorway := random(6)
      else if Level.Level < 9 then
        Doorway := random(5)
      else
        Doorway := 0;
      case Doorway of
        0..2: Doorway := CELL_FLOOR;
        3,4 : Doorway := CELL_WALL;
        5   : Doorway := CELL_GRATE;
      end;
      if not(Doorway in CellFloors) then DoorPos := NewCoord2D(randRange(Where.x,Entry.x),randRange(Where.y,Entry.y));
      //fill doorway
      Dir:=newDirection(Where, Entry);
      while Where<>Entry do
      begin
        PutCell(Where,DoorWay);
        Where+=Dir;
      end;
      PutCell(Where,Doorway);
      //put a door if needed
      if not(Doorway in CellFloors) then PutCell(DoorPos, RandomDoor);
    end;
  until FCnt >= FLimit;

  for FCnt := 1 to 10 do ClearDeadEnds;
  RestoreWalls(CELL_WALL);
//  if Spec <> 1 then
end;

procedure TGenerator.LoadTileset(TileSet: string);
var cc    : byte;
    V     : Variant;
    TR    : TPrintableCharToByte;
begin
  TR := StandardTranslation;
  TR[ Ord('#') ] := CELL_WALL;
  TR[ Ord('.') ] := CELL_FLOOR;
  TR[ Ord('+') ] := CELL_DOOR_CLOSED;
  V := Game.Lua.Variables[TileSet];
  if not VarIsArray( V ) then  CritError(TileSet+' not an array!');
  for cc := VarArrayLowBound( V, 1 ) to VarArrayHighBound( V, 1 ) do
    TileData.Push( TTileData.Create( Self, V[cc], TR ) )
end;

end.

