{$MODE OBJFPC}
// @abstract(Core Game Unit for RL Core)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(January 17, 2005)
// @lastmod(May 10, 2010)
//
// This unit holds the game's main class : TGame.

unit rlgame;
interface
uses zstream, vsystem, vnode, rllevel, rlglobal, rlplayer, rllua, rlgen, rlui, rlshopinv, classes;

type

{ TGame }

TGame = class(TSystem)
       Player      : TPlayer;
       Level       : TLevel;
       LevelStore  : array[0..MaxLevels*2+1] of TLevel; //each level can has one sublevel
       LevelNumber : Word;
       StairNumber : Byte;
       Shops       : array[1..MaxShops] of TShopInventory;
       Lua         : TRLLua;
       TurnCount   : DWord;
       GraveYard   : TNode;
       constructor Create; override;
       procedure Prepare;
       procedure Run;
       destructor Destroy; override;
       procedure Load;
       procedure Save;
     end;

// TGame singleton.
const Game : TGame = nil;

implementation
uses sysutils, vutil, vinput, voutput, vtextut, vlua,
     rlcell, vsystems, vconsole, rlnpc;

constructor TGame.Create;
begin
  inherited Create;
  LevelNumber := 0;
  TurnCount   := 0;
  Systems.Add(UI, TGameUI.Create);
  if ParamStr(1) = '-god' then GodMode := True;
  Lua := TRLLua.Create;
  Systems.Add(Console,TLuaConsole.Create(TTextConsoleWindow.Create(),Lua));
  UI.ReadConfig;
  GraveYard:=TNode.Create;
  Add(GraveYard);
  if GodMode then begin
      Input.RegisterOverrideKey(VKEY_F1,Console);
  end;
end;

procedure TGame.Prepare;
var Count  : Word;
    s: string;

begin
  Output.HideCursor;
  LoadCells;

  for Count := 0 to MaxLevels*2+1 do LevelStore[Count] := nil;
  for Count := 1 to MaxShops  do Shops[Count]      := nil;

  repeat
    UI.Intro;
    Count := UI.MainMenu;
    case Count of
      1:
      begin
        if FileExists('save') then
        begin
          Output.DrawString(2,1,LightGray,'This will erase your previously saved game. Are you sure?(Y/n)');
          if UI.GetKey([ord('Y'),ord('n')]) = ord('Y') then
            break;
        end
        else
          break;
      end;
      2: begin Load; exit; end;
      3: UI.ShowManual;
      4: break;
    end;
  until false;
  if Count <> 4 then
  begin
  // We create a new player
  if FileExists('save') then DeleteFile('save');
  UI.SelectKlass;

  Output.Clear;
  Output.DrawString(36,12,Yellow,'Enter name');
  OutPut.ShowCursor;
  s := Trim(TextInputField(35,13,12));
  UI.PlaySound('sound/sfx/items/titlslct.wav');
  if s<>'' then
    Player.Name := s;

  for Count := 1 to Game.Lua.Table['shops','__counter'] do
    Shops[Count] := TShopInventory.Create(Game.Lua.IndexedTable['shops',Count,'id']);

  Output.HideCursor;
  Output.Clear;
  end
  else GameEnd:=true;
end;

procedure TGame.Run;
var Scan: TNode;
begin
  if not GameEnd then
  begin
  UI.Prepare;
  repeat
    if (Level<>nil) then
    begin
      Lua.TableExecute('achievment','OnExit',[Level.Level]);
    end;
    if LevelStore[LevelNumber] = nil then
    begin
       // We create a new level
       Level := TLevel.Create('level'+IntToStr(LevelNumber));
       // Makes Level a child of TGame, so we don't have to dispose of it manualy.
       Add(Level);
       LevelStore[LevelNumber] := Level;
       if (LevelNumber > Player.MaxDepth) and (LevelNumber <= MaxLevels) then
         Player.MaxDepth := LevelNumber;
    end
    else
    begin
      Level := LevelStore[LevelNumber];
      Lua.Level := Level;
      if Level.OnEnter then
        Lua.TableExecute('levels',LevelNumber,'OnEnter');

       begin
         PlayerC := Level.Area.FindCell([StairNumber]);

         // If player came throgh portal, destroy portal
         if StairNumber = CELL_TOWN_PORTAL then
           if (LevelNumber = 0) then
             inc(PlayerC.Y)
           else
           begin
             Level.Map[PlayerC.X,PlayerC.Y].Cell := CELL_FLOOR;
             Player.PortalLevel := 0;
           end;

         // If not portal level then destroy any portals on it.
         if (LevelNumber <> Player.PortalLevel)and(LevelNumber<>0) then
           Level.Area.Transmute([CELL_TOWN_PORTAL],CELL_FLOOR);

         if StairNumber = CELL_MARKER then Level.Map[PlayerC.X, PlayerC.Y].Cell := Player.Special;
       end;
    end;
    Lua.TableExecute('achievment','OnEnter',[LevelNumber]);
    UI.PlayMusic(Level.music);

    // Makes Player a child of TLevel. We don't have to dispose of it manualy.
    // And also it sets the player in his world ;-)
    if Player.Parent<>nil then
      Player.Querry := TNPC(TLevel(Player.Parent).Find('golem'));

    Player.Move(Level);

    // Now we can properly displace the player :D
    // GenX and GenY are taken from the generator
    Player.Displace( PlayerC );
    //drop player's golem here
    if Player.Querry <> nil then
    begin
      Level.Area.FindNearSpace(PlayerC,1,[cfNoMonsters, cfNoObstacles]);
      Player.Querry.Move( Level );
      Player.Querry.Displace( PlayerC );
    end;

    LevelChange := False;
    Player.Querry := nil;
    if nfManaShield in Player.flags then
      UI.Msg('Your protection worn off.');
    Player.Flags-=[nfInfravision, nfManaShield, nfReflect];
    repeat
      Level.TimeFlow(10);
      Inc(TurnCount);
      with Graveyard do
        while Child <> nil do
        begin
          Scan := Child;
          Child := Child.Next;
          Scan.Destroy;
        end;
    until GameEnd or LevelChange;
    Level.Map[Player.Position.x,Player.Position.y].NPC := nil;
  until GameEnd;
  if (Player.HP <= 0) or (LevelNumber=9) then
  begin
    if LevelNumber = 9 then
      Game.Lua.TableExecute('achievment', 'OnMortem', []);
    Game.Player.WriteMemorial;
    UI.ShowMortem;
  end;
  end;
  UI.Outro;
end;

destructor TGame.Destroy;
begin
  inherited Destroy;
  FreeAndNil(Lua);
  Log('Destroyed.');
end;

procedure TGame.Load;
var ISt: TGZFileStream;
    Count: byte;
    ver : string;
    Lvls: set of byte;
begin
  Output.HideCursor;
  LoadCells;
  StairNumber := CELL_MARKER;
  ISt := TGZFileStream.Create('save', gzOpenRead);
  try
    try
      ver := ISt.ReadAnsiString;
      if ver <> VERSION then CritError('Wrong save file version!');

      Player := TPlayer.CreateFromStream(ISt);
      ISt.Read(Lvls, sizeof(Lvls));
      for Count := 0 to MaxLevels*2+1 do
        if Count in Lvls then
          LevelStore[Count] := TLevel.CreateFromStream(ISt);

      LevelNumber := ISt.ReadWord;
      TurnCount := ISt.ReadWord;

      for Count := 1 to Game.Lua.Table['shops','__counter'] do
        Shops[Count] := TShopInventory.CreateFromStream(ISt);
    except
      FreeAndNil(ISt);
      DeleteFile('save');
      Log('save file corrupt!');
      Prepare;
    end;
  finally
    FreeAndNil(ISt);
    DeleteFile('save');
  end;
  Game.Lua.Execute('load_quest_maps',[]);
end;

procedure TGame.Save;
var OSt: TGZFileStream;
    Count: byte;
    Lvls: set of byte;
begin
  OSt := TGZFileStream.Create('save', gzOpenWrite);
  OSt.WriteAnsiString(VERSION);
  Player.Special := Level.Map[Player.Position.X, Player.Position.Y].Cell;
  Level.Map[Player.Position.X, Player.Position.Y].Cell := CELL_MARKER;
  Player.Detach;
  Level.Map[Player.Position.X, Player.Position.Y].NPC := nil;
  Player.ToStream(OSt);

  Lvls := [];
  for Count := 0 to MaxLevels*2+1 do
    if LevelStore[Count]<>nil then include(Lvls, Count);
  OSt.Write(Lvls, sizeof(Lvls));

  for Count := 0 to MaxLevels*2+1 do
    if LevelStore[Count]<>nil then LevelStore[Count].ToStream(OSt);

  Ost.WriteWord(LevelNumber);
  Ost.WriteWord(TurnCount);

  for Count := 1 to Game.Lua.Table['shops','__counter'] do
    Shops[Count].ToStream(OSt);

  FreeAndNil(OSt);
  Level.Add(Player);
  Log('Saving done!');
end;

end.
