require('diablorl:functions')

-- main namespace --
DiabloRL = {}

-- klasses array --
klasses = {}

-- npcs array --
npcs = {}

-- skeletons --
skeletons = {}

-- cells array --
cells = {}

-- items array --
items = {}

-- spells array --
spells = {}

-- shops array --
shops = {}

-- prototypes array --
prototypes = {}

-- suffixes array --
suffixes = {}

-- prefixes array --
prefixes = {}

-- levels array --
levels = {}

-- level generators --
generators = {}

-- quests array --
quests = {}

-- achievments array --
achievments = {}

-- Metatable hack that disallows usage of undeclared variables --
setmetatable(_G, {
  __newindex = function (_, n)
  error("attempt to write to undeclared variable "..n, 2)
  end,
  __index = function (_, n)
  error("attempt to read undeclared variable "..n, 2)
  end,
})

-- Declare constructors --
declare("Prototype")
declare("Cell")
declare("NPC")
declare("Klass")
declare("Player")
declare("Item")
declare("Level")
declare("Suffix")
declare("Prefix")
declare("Quest")
declare("Achievment")
declare("Spell")
declare("Generator")
declare("Shop")

declare( "game_object_flags", {} )
setmetatable(game_object_flags, {
	__newindex = function (self, key, value)
		game_object.flags_set(self, key, value )
	end,
	__index = function (self, key)
		return game_object.flags_get(self, key )
	end,
})

function game_object:set_property(key, value)
	key = _G["PROP_"..string.upper(key)]
	game_object.property_set( self, key, value )
end

function game_object:get_property(key)
	key = _G["PROP_"..string.upper(key)]
	return game_object.property_get( self, key )
end

setmetatable(game_object, {
  __newindex = game_object.set_property,
  __index = game_object.get_property
})

table.merge( thing, game_object )

setmetatable(thing,getmetatable(game_object))

declare( "shop_item", {} )

setmetatable(shop_item, {
	__newindex = function (self, key, value)
		shop.item_set( self, key, value )
	end,
	__index = function (self, key)
		return shop.item_get( self, key )
	end,
})

function shop:add( it )
	local i = 0
	repeat
		i = i + 1
		if i > ITEMS_SHOP then return nil end
	until self:item_get( i ) == nil
	if type( it ) == "string" then it = item.new( it ) end
	self:item_set( i, it )
	self:sort()
	return it
end

function shop:clear()
	for i = 1, ITEMS_SHOP do
		self:item_set(i,nil)
	end
end

function shop:items()
	local i = 0
	return function ()
		while i < ITEMS_SHOP do
			i = i + 1
			local it = self:item_get( i )
			if it then return it end
		end
		return nil
	end
end

table.merge( shop, game_object )
setmetatable(shop,getmetatable(game_object))

function item:is_wearable()
	local itype = self.itype
	return  itype == TYPE_HELM   or itype == TYPE_ARMOR or
			itype == TYPE_SHIELD or itype == TYPE_WEAPON or
			itype == TYPE_RING   or itype == TYPE_AMULET or
			itype == TYPE_STAFF  or itype == TYPE_SSTAFF or
			itype == TYPE_BOW
end

table.merge( item, thing )
setmetatable(item,getmetatable(thing))

function npc:msg(message)
	if self:is_player() then
		ui.msg(message)
	end
end

function npc:seek_away(c)
	self:seek(c,-1)
end

function npc:circle(c)
	self:seek(c,0)
end

function npc:distance(c)
	return coord.distance(self:get_position(),c)
end

table.merge( npc, thing )
setmetatable(npc,getmetatable(thing))

declare("player", {})
table.merge( player, npc )

function player:salutation()
	if self.klass == KlassRogue then
		return "lass"
	else
		return "lad"
	end
end


-- this array allows browsing through all items
-- the order is 40 slots of inventory, then 7 slots of equipment and then quickslots
player.item = {}
setmetatable(player.item, {
	__newindex = function (_, key, value)
		player:item_set(key, value)
	end,
	__index = function (_, key)
		return player:item_get( key )
	end,
})

player.eq  = {}
setmetatable(player.eq, {
	__newindex = function (_, key, value)
		if type(key) == "string" then key = _G["SLOT_"..string.upper(key)] end
		key = key + ITEMS_INV
		player:item_set(key, value)
	end,
	__index = function (_, key)
		if type(key) == "string" then key = _G["SLOT_"..string.upper(key)] end
		key = key + ITEMS_INV
		return player:item_get( key )
	end,
})

player.qs  = {}
setmetatable(player.qs, {
	__newindex = function (_, key, value)
		key = key + ITEMS_INV + ITEMS_EQ
		player:item_set(key, value)
	end,
	__index = function (_, key)
		key = key + ITEMS_INV + ITEMS_EQ
		return player:item_get( key )
	end,
})


function player:items()
	local i = 0
	return function ()
		while i < ITEMS_ALL do
			i = i + 1
			local item = player:item_get(i)
			if item then return item end
		end
	end
end

player.quest = {}
setmetatable(player.quest, {
	__newindex = function ( _, key, value )
		if type(key) == "string" then key = quests[key].nid end
		player:quest_set( key, value )
	end,
	__index = function (_, key)
		if type(key) == "string" then key = quests[key].nid end
		return player:quest_get( key )
	end,
})

player.spells  = {}
setmetatable(player.spells, {
	__newindex = function (_, key, value)
		if type(key) == "string" then key = spells[key].nid end
		player:spell_set( key, value )
	end,
	__index = function (_, key)
		if type(key) == "string" then key = spells[key].nid end
		return player:spell_get( key )
	end,
})



setmetatable(player,getmetatable(npc))

-- temporary function -- will be removed
function level.apply_magic( ilvl, mag_chance )
	utils.try_roll_magic( this, ilvl, mag_chance )
end

function level.move_walls(s)
	while true do
		local c = level.find_tile(s)
		if not c then break	end
		level.set_cell( c, "floor" )
	end
end

function level.lever_action(x,y,n)
	ui.play_sound("sound/sfx/items/lever.wav",x,y)
	level.move_walls("moving_wall_" .. n)
	level.move_walls("moving_grate_" .. n)
end

function level.random_item_table(ilvl)
	if ilvl <  1 then ilvl = 1  end
	if ilvl > 25 then ilvl = 25 end

	if items["__table"..ilvl] then
		return items["__table"..ilvl]
	end

	local result = {}
	for _,v in ipairs(items) do
		if v.level <= ilvl then
			if not v.no_random then
				table.insert( result, v.id )
				if v.double_random then
					table.insert( result, v.id )
				end
			end
		end
	end

	items["__table"..ilvl] = result
	return result
end

function level.random_item_id(ilvl, itypes)
	local item_table = level.random_item_table(ilvl)

	itypes = itypes or {TYPE_HELM,TYPE_ARMOR,TYPE_SHIELD,TYPE_WEAPON,
						TYPE_POTION,TYPE_SCROLL,TYPE_RING,TYPE_AMULET,
						TYPE_BOOK,TYPE_STAFF,TYPE_BOW}

	-- make itype set
	local itype_filter = {}
	for _,v in ipairs(itypes) do
		itype_filter[v] = true
	end

	-- filter list
	local list = {}
	for _,item_id in ipairs(item_table) do
		if ( itype_filter[ items[item_id].type ] ) and not (items[item_id].is_unique) then
			table.insert(list,item_id)
		end
	end

	-- sanity check
	if #list == 0 then
		error("random_item_id("..ilvl..", itypes generated an empty set!")
	end

	-- return random item
	return table.random(list)
end

function level.random_item_proto(ilvl, iproto)

	local item_table = level.random_item_table(ilvl)

	-- filter list
	local list = {}
	for _,item_id in ipairs(item_table) do
		if ( items[item_id].pid == iproto ) and not (items[item_id].is_unique) then
			table.insert(list,item_id)
		end
	end

	-- sanity check
	if #list == 0 then
		error("random_item_proto("..ilvl..", itypes generated an empty set!")
	end

	-- return random item
	return table.random(list)
end

function level.random_unique_item_id(ilvl)

	local list = {}
	local lvl = ilvl
	local item_table = nil
	repeat
		item_table = level.random_item_table(ilvl)
		for _,item_id in ipairs(item_table) do
			if (items[item_id].level == lvl) and (items[item_id].is_unique) then
				table.insert(list,item_id)
			end
		end
		lvl = lvl - 1
	until ( #list > 0 ) or (ilvl == 0)
	-- sanity check
	if #list == 0 then
		error("random_item_id("..ilvl..", itypes generated an empty set!")
	end

	-- return random item
	return table.random(list)
end

function level.random_shrine()
	local shrines = {}
	for _,n in ipairs(cells) do
		if n.special == 1 then
			if n.minl <= level.level() and n.maxl >= level.level() then
				table.insert( shrines, n.id )
			end
		end
	end
	return( table.random( shrines ) )
end

function level.drop_gold( x, y, drop_level )
	drop_level = drop_level or levels[level.level()].depth
	local item = level.drop_item( "gold", coord.new( x, y ) )
	item.amount = math.random( 5 * drop_level, 15 * drop_level ) - 1
	if drop_level > 12 then
		item.amount = item.amount * 1.125
	end
end

-- item drop rules described in Jarulf 3.8.1
function level.drop_random_base(id,x,y,ilvl,mag_chance,unique_drop)
	-- Is it magical?
	local itype = items[id].type
	if (itype == TYPE_AMULET or itype == TYPE_RING) then
		mag_chance = 100
	elseif (itype == TYPE_SCROLL or itype == TYPE_POTION or itype == TYPE_BOOK) then
		mag_chance = 0
	elseif (not mag_chance) then
		mag_chance = math.min( 11+(ilvl+1)*0.89, 100 )
	end

	local bMagic = math.random(100) <= mag_chance
	if (bMagic) then
		local unique_chance = 2
		if (unique_drop) then
			ilvl = ilvl + 4
			unique_chance = 16
		end
		if (math.random(100) <= unique_chance) then
			-- try to find unique item of selected base type
			local item_table = level.random_item_table(ilvl)
			for _,item_id in ipairs(item_table) do
				if ((items[item_id].name == items[id].name) and (items[item_id].is_unique)) then
					-- TODO: If multiple matches found, choose highest qlvl item that hasn't been created yet
					id = item_id
					break
				end
			end
		end
	end

	local item = level.drop_item( id, coord.new(x,y), ilvl )
	if items[id].is_unique then return end

	if ((itype == TYPE_STAFF) and (item.spell == 0)) then
		-- Jarulf 3.8.2 "Staves are always magical if they have no spell"
		bMagic = true
	end
	if (bMagic) then
		-- TODO: Enforce minimum [ilvl/2] for chosen prefix & suffix
		utils.roll_magic( item, ilvl )
	end
end

function level.drop_random_item(x,y,ilvl,mag_chance,unique_drop)
	-- Determination of base item
	local id
	if unique_drop or mag_chance == 100 then
		-- unique monster always drops magical item
		id = level.random_item_id(ilvl, { TYPE_BOOK, TYPE_WEAPON, TYPE_BOW, TYPE_STAFF, TYPE_HELM, TYPE_SHIELD, TYPE_ARMOR, TYPE_RING, TYPE_AMULET })
	else
		id = level.random_item_id(ilvl)
	end

	level.drop_random_base(id,x,y,ilvl,mag_chance,unique_drop)
end

-- validate_items: remove invalid items from a list
-- item_list - table of item ids
-- ilvl - maximum item level
function level.validate_items(item_list,ilvl)
	-- iterate in reverse so that remove doesn't mess it up
	for i = #item_list, 1, -1 do
		local id = item_list[i]
		if ((items[id] == nil) or (items[id].level > ilvl)) then
			table.remove(item_list,i)
		end
	end
end

function level.drop_random_scroll(x,y)
	-- Jarulf 3.8.1: scroll drops from library book and skeleton tome are limited to this set
	local scroll_set = {'scroll_apocalypse','scroll_healing','scroll_identify',
		'scroll_infravision','scroll_nova','scroll_mana_shield','scroll_phasing',
		'scroll_teleport','scroll_town_portal'}
	level.validate_items(scroll_set,level.level() * 2)
	if (#scroll_set > 0) then
		level.drop_item(table.random(scroll_set),coord.new(x,y))
	end
end

function level.drop_random_potion(x,y)
	local potion_set = {'potion_healing', 'potion_mana', 'potion_rejuvenation',
			'potion_full_healing', 'potion_full_mana', 'potion_full_rejuvenation'}
	level.validate_items(potion_set,level.level() * 2)
	if (#potion_set > 0) then
		level.drop_item(table.random(potion_set),coord.new(x,y))
	end
end

function level.drop_special(x,y)
	-- "Special Items" Jarulf 3.8 footnote 6; 3.8.1; and 3.8.3
	local item_set = {"potion_healing","potion_mana","scroll_town_portal"}
	level.validate_items(item_set,level.level())
	if (#item_set > 0) then
		level.drop_item(table.random(item_set),coord.new(x,y))
	end
end

function level.chest_drop(x, y, maxitems, magic)
	-- Jarulf 3.8.1 "Chest"
	ui.msg("You open the chest.")
	ui.play_sound('sound/sfx/items/chest.wav',x,y)
	local itemcount
	if (magic) then
		itemcount = maxitems
	else
		itemcount = math.random(0,maxitems)
	end
	if (not magic and (math.random(8) == 1)) then
		while (itemcount > 0) do
			level.drop_special(x, y)
			itemcount = itemcount - 1
		end
	else
		while (itemcount > 0) do
			if (not magic and (math.random(4) == 1)) then
				level.drop_gold(x, y)
			else
				if (magic) then
					magic = 100
				end
				level.drop_random_item(x, y, levels[level.level()].depth * 2,magic)
			end
			itemcount = itemcount - 1
		end
	end
end

function level.change( lev, cell )
	if type(cell) == "string" then cell = cells[cell].nid end
	level.change_to(lev, cell)
end

function level.get_cell( c )
	return cells[level.raw_get_cell( c )].id
end

function level.set_cell(c,cell)
	if not c then return end
	if type(cell) == "string" then cell = cells[cell].nid end
	level.raw_set_cell(c,cell)
end

function level.refresh_shops()
	for nid,shop in ipairs(shops) do
		shop.OnRefill( level.get_shop(nid) )
	end
end

function level.shop( id )
	shops[id].OnEnter( level.get_shop( shops[id].nid ) )
end

function level.drop_npcs( n, x, y )
	local N = levels[n].npc_count
	x = x or 0
	y = y or 0
	if N == 0 then return end
	for _,i in pairs(levels[n].npc) do
		level.drop_npc( '', coord.new( i.x + x, i.y + y ) )
		level.drop_npc( i.id, coord.new( i.x + x, i.y + y ) )
	end
end

function level.drop_skeleton(x,y)
	if skeletons[level.level()] then
		local skel_table = skeletons[level.level()]
		if (#skel_table > 0) then
			local skeleton = level.drop_npc(table.random(skel_table), coord.new( x,y ) )
			skeleton.scount = 20
		end
	end
end

setmetatable(level, {
	__newindex = function (_, key, value)
		old_thing_push_level()
		old_thing[key] = value
	end,
	__index = function (_, key)
		old_thing_push_level()
		return old_thing[key]
	end,
})

declare("achievment", {})

achievment.OnKill_List   = {}
achievment.OnMortem_List = {}
achievment.OnEnter_List	 = {}
achievment.OnExit_List 	 = {}
achievment.OnEquip_List	 = {}
achievment.OnUse_List	 = {}
achievment.OnPickUp_List = {}

achievment.OnKill = function( being )
	for _,a in ipairs(achievment.OnKill_List) do
		if achievments[a].active then achievments[a].OnKill( being ) end
	end
end

achievment.OnEnter = function( lvl )
	for _,a in ipairs(achievment.OnEnter_List) do
		if achievments[a].active then achievments[a].OnEnter( lvl ) end
	end
end

achievment.OnExit = function( lvl )
	for _,a in ipairs(achievment.OnExit_List) do
		if achievments[a].active then achievments[a].OnExit( lvl ) end
	end
end

achievment.OnEquip = function( itm )
	for _,a in ipairs(achievment.OnEquip_List) do
		if achievments[a].active then achievments[a].OnEquip( itm ) end
	end
end

achievment.OnUse = function( itm )
	for _,a in ipairs(achievment.OnUse_List) do
		if achievments[a].active then achievments[a].OnUse( itm ) end
	end
end

achievment.OnPickUp = function( itm )
	for _,a in ipairs(achievment.OnPickUp_List) do
		if achievments[a].active then achievments[a].OnPickUp( itm ) end
	end
end

achievment.OnMortem = function()
	for _,a in ipairs(achievment.OnMortem_List) do
		if achievments[a].active then achievments[a].OnMortem() end
	end
end

declare( 'apply_fix', false )
function apply_fix( it, fix, value )
	if (not value) and type(fix.min) == "number" and type(fix.max) == "number" then
		value = math.random(fix.min,fix.max)
	end

	if fix.OnApply then fix.OnApply( it, value ) end
	if #fix.flags > 0 then
		for _,f in ipairs(fix.flags) do
			it.flags[ f ] = true
		end
	end

    if value and fix.min ~= fix.max then
      it.pricem = it.pricem + math.floor( ((value-fix.min) / (fix.max-fix.min))
	                                           * (fix.price_max-fix.price_base ) )
    end
    it.pricem = it.pricem + fix.price_base
	if math.abs(fix.multiplier) > 1 then
		it.pricemult = it.pricemult + fix.multiplier
	end

	it.flags[ ifMagic ]   = true
	it.flags[ ifUnknown ] = true
	it.color = LIGHTBLUE
	value = value or 0
	return value
end

function Suffix(b)
  b.level = b.level or 1
  b.occurence = b.occurence or { TYPE_ARMOR, TYPE_SHIELD, TYPE_WEAPON, TYPE_STAFF, TYPE_BOW, TYPE_RING, TYPE_AMULET }
  b.multiplier = b.multiplier or 1
  b.price_base = b.price_base or 0
  b.price_max  = b.price_max or 0
  b.flags = b.flags or {}
  register( suffixes, b )
  b.apply = function( item )
	item.suffixvalue = apply_fix( item, b )
	item.suffix = b.nid
  end
  return b.nid
end

function Prefix(b)
  b.level = b.level or 1
  b.occurence = b.occurence or { TYPE_ARMOR, TYPE_SHIELD, TYPE_WEAPON, TYPE_STAFF, TYPE_BOW, TYPE_RING, TYPE_AMULET }
  b.multiplier = b.multiplier or 1
  b.price_base = b.price_base or 0
  b.price_max  = b.price_max or 0
  b.flags = b.flags or {}
  register( prefixes, b )
  b.apply = function( item )
	item.prefixvalue = apply_fix( item, b )
	item.prefix = b.nid
  end
  return b.nid
end


function Prototype(b)
  b.pid = b.pid or ""
  prototypes[b.pid] = b
  return b.pid
end


function Cell(cell)
  if cell.proto then
    apply_prototype( prototypes[cell.proto], cell )
  end
  cell.piclow = cell.piclow or cell.pic
  cell.special = cell.special or 0
  register( cells, cell )
  return cell.id
end

function Shop(shop)
  register( shops, shop )
  return shop.id
end


function Item(b)
  if b.proto then
    apply_prototype( prototypes[b.proto], b )
  end
  b.iname = b.iname or ""
  b.type = b.type or TYPE_OTHER
  b.flags = b.flags or {}
  b.color = b.color or LIGHTGRAY
  b.is_unique = b.is_unique or false
  for _,i in pairs(b.flags) do
	if i == ifUnique then
		b.is_unique = true
		b.color = YELLOW
  end end
  b.price = b.price or 0
  b.level = b.level or 1
  b.volume = b.volume or 1
  b.pic = b.pic or '?'
  b.piclow = b.piclow or b.pic

  b.load_fix = function(self)
    local unk = self.flags[ifUnknown]
	if self.prefix > 0 then apply_fix(self, prefixes[self.prefix], self.prefixvalue) end
	if self.suffix > 0 then apply_fix(self, suffixes[self.suffix], self.suffixvalue) end
	self.flags[ifUnknown] = unk
  end

  register( items, b )

  if items["level"..b.level] == nil then
	items["level"..b.level] = {}
  end
  table.insert(items["level"..b.level],b.id)

  return b.id
end


function NPC(b)
  if b.proto then
	if prototypes[b.proto] then
	  apply_prototype( prototypes[b.proto], b )
	elseif npcs[b.proto] then
	  apply_prototype( npcs[b.proto], b )
	else
	  error(b.name .. ": monster missing proto " .. b.proto)
	end
  end
  b.iname = b.iname or ""
  b.ai = b.ai or AISkeleton
  b.size = b.size or 5000
  b.hpmin = b.hpmin or 1
  b.hpmax = b.hpmax or 1
  b.flags = b.flags or {}
  b.level = b.level or 1
  b.color = b.color or LIGHTGRAY
  b.pic = b.pic or '?'
  b.item = b.item or ''
  
  register( npcs, b )
  -- make it easier to test flags
  b.flags = table.to_set( b.flags )
  
  -- this is a hack, but at least a clean one
  local old_act = b.OnAct
  b.OnAct = function(self)
	if old_act then 
		old_act(self) 
	else
		if ai_scripts[ self.ai ].OnAct then 
			ai_scripts[ self.ai ].OnAct(self)
		end
	end
  end
  
  -- rest is brutally copied
  for k,v in pairs(ai_scripts[ b.ai ]) do
	if not b[k] then
		b[k] = v
	end
  end

  if b.flags[nfUnique] then
	--[[
	if (b.mob) then
		if (npcs[b.proto] == nil) then
			error("missing base monster class " .. b.proto.pid)
		end
	end]]
	if npcs["unique_list"] == nil then
		npcs["unique_list"] = {}
	end
	table.insert(npcs["unique_list"],b.id)
  else
	if b.group then
		for k,v in ipairs(b.group) do
			if npcs["level"..v] == nil then
				npcs["level"..v] = {}
			end
			table.insert(npcs["level"..v],b.id)
			-- separate list for skeletons
			if b.flags[nfSkeleton] then
				if skeletons[v] == nil then
					skeletons[v] = {}
				end
				table.insert(skeletons[v], b.id)
			end
			-- todo - this should be without duplicates
			if npcs["list"] == nil then
				npcs["list"] = {}
			end
			table.insert(npcs["list"],b.id)
		end
	end
  end

  b.group = b.group or {}

  return b.id
end


function Klass(b)
  b.name = b.name
  b.dname = b.dname
  b.klassid = b.klassid
  b.ai = b.ai or AIPLAYER
  b.str = b.str
  b.dex = b.dex
  b.mag = b.mag
  b.vit = b.vit
  b.life = b.life
  b.mana = b.mana
  b.lifelev = b.lifelev
  b.manalev = b.manalev
  b.hpmin = b.hpmin or 1
  b.hpmax = b.hpmax or 1
  b.spdmov = b.spdmov
  b.spdatk = b.spdatk
  b.spdhit = b.spdhit
  b.spdblk = b.spdblk
  b.spdmag = b.spdmag
  b.corpse = b.corpse
  b.flags = b.flags or {}
  b.level = b.level or 1
  b.color = b.color
  b.pic = b.pic or '@'
  b.desc = b.desc or 'Unavailable.'

  register( klasses, b )

  if b.klassid ~= b.nid then
	error("Wrong klassid! Got "..b.klassid.." should be "..b.nid)
  end
end

declare("LevelInitMap")
function LevelInitMap(b)
	b.npc_count = 0

	if b.map_key then
		local __key = b.map_key
		local __npc = {}
		local npcstr = ""

		b.map_key = {}
		for i,j in pairs(__key) do
			b.map_key[i] = j[1]
			if j[2] then
				__npc[i] = j[2]
				npcstr = npcstr..i
			end
		end

		if npcstr ~= '' then
			local len = string.len( b.map )
			local width = string.find( b.map, '\n' ) or len
			local x
			b.npc = {}
			for i = 1,len do
				x = string.find( npcstr, string.sub( b.map, i, i ), 1, true )
				if x then
					x = string.sub( npcstr, x, x )
					b.npc_count = b.npc_count + 1
					b.npc[b.npc_count] = {}
					b.npc[b.npc_count].x = i%width
					b.npc[b.npc_count].y = math.ceil(i/width)
					b.npc[b.npc_count].id = __npc[x]
				end
			end
		end
	end
end

function Level( lev, b )

	if type(lev) ~= "number" then
		for i = 17,33 do
			if not (levels[i]) then
				lev = i
				break
			end
		end
	end
	if (b.special) then
		-- link special stair
		levels[b.special].special = lev
		b.depth = b.special
	else
		b.depth = lev
	end

	b.level = b.level or lev
	b.type = b.type or "room"
	b.id = b.id or "level"..lev

	LevelInitMap(b)

	register( levels, b )
	return b.id
end


function Quest( quest )
  if quest.enabled == nil then
	quest.enabled = true
  end
  quest.score = quest.score or 0
  quest.message = quest.message or 'completed "'..quest.name..'"'
  quest.failed = quest.failed or 255
  register( quests, quest )
  return quest.id
end

function Achievment( b )
	b.name = b.name or 'error'
	b.score = b.score or 0
	b.message = b.message or 'completed "'..b.name..'" challenge'
	b.value = b.value or 0
	register( achievments, b )
	if b.OnEnter 	then table.insert(achievment.OnEnter_List	,b.id) end
	if b.OnExit 	then table.insert(achievment.OnExit_List	,b.id) end
	if b.OnKill 	then table.insert(achievment.OnKill_List	,b.id) end
	if b.OnPickUp 	then table.insert(achievment.OnPickUp_List	,b.id) end
	if b.OnEquip 	then table.insert(achievment.OnEquip_List	,b.id) end
	if b.OnUse 		then table.insert(achievment.OnUse_List		,b.id) end
	if b.OnMortem 	then table.insert(achievment.OnMortem_List	,b.id) end
	return b.id
end

function Generator( generator )
  register( generators, generator )
  return generator.id
end


function Spell( b )

  b.name     = b.name or "Unknown"
  b.effect   = b.effect or 0
  b.type     = b.type or DAMAGE_GENERAL
  b.cost     = b.cost or 0
  b.dmin     = b.dmin or 0
  b.dmax     = b.dmax or 0
  b.target   = b.target or 0 -- auto
  b.townsafe = b.townsafe or false

  register( spells, b )

  if b.scroll_level then
    if b.townsafe then
	  b.scroll_flags = {}
	else
	  b.scroll_flags = { ifNotInTown }
	end
    Item{
      id = "scroll_"..b.id,
      name = "scroll of "..string.lower(b.name),
      type = TYPE_SCROLL,
      pic = "?",
      volume = 1,
      flags = b.scroll_flags,
      color = b.color,
      price = b.scroll_price,
      level = b.scroll_level,
	  minmag = b.scroll_mag or 0,
	  spell = b.id,
	  sound2 = "sound/sfx/items/flipscrl.wav",
	  sound1 = "sound/sfx/items/invscrol.wav"
	}
  end

  return b.id
end

ui.msg_enter = function(msg)
	ui.msg(msg.." Press <Enter>...")
	ui.get_key(KEY_ENTER)
end

ui.msg_confirm = function(msg)
	ui.msg(msg.." [y/n]")
	while true do
		local key = ui.get_key()
		if key == string.byte("y") or key == string.byte("Y") then return true end
		if key == string.byte("n") or key == string.byte("N") then return false end
	end
end

--TODO: remove randomness
function ui.gossip( id, topic )
  if not topic then
	if npcs[id] and npcs[id].gossip then
		local i = math.random(#npcs[id].gossip)
		if (npcs[id].gossip[i].wav) then
			ui.play_sound("sound/sfx/towners/" .. npcs[id].gossip[i].wav)
		end
		ui.plot_talk( npcs[id].gossip[i].text )
	end
  else
	if quests[topic] and quests[topic].gossip and quests[topic].gossip[id] then
		if (quests[topic].gossipwav[id]) then
			ui.play_sound("sound/sfx/towners/" .. quests[topic].gossipwav[id])
		end
		ui.plot_talk( quests[topic].gossip[id] )
	end
  end
end

function ui.shop( id, desc )
	return ui.shop_run( shops[id].nid, desc )
end


function ui.talk( id, talk_table )
	local topics  = {}
	local actions = {}
	for _,v in ipairs(talk_table) do
		table.insert(topics,v[1])
		table.insert(actions,v[2])
	end
	while true do
		local result = ui.talk_run( npcs[id].name, unpack(topics) )
		if (result == 0) or (actions[result] == false) then break end
		ui.play_sound('sound/sfx/items/titlslct.wav')
		actions[result]()
	end
end

function ui.talk_topics( id )
	local talk_table = {}
	if npcs[id].gossip then
		table.insert(talk_table,{ "Gossip", function() ui.gossip(id) end })
	end
	for _,q in ipairs(quests) do
		if q.gossip and q.gossip[id] then
			local qv = player.quest[q.id]
			if qv ~= 0 and qv < q.completed then
				table.insert(talk_table,{ q.name, function() ui.gossip(id,q.id) end })
			end
		end
	end
	table.insert(talk_table,{ "Back", false })
	ui.talk( id, talk_table )
end

declare("utils", {})

function utils.random_fix(fixes, itype, minl, maxl)
	local list = {}
	for _,v in ipairs(fixes) do
		if v.level >= minl and v.level <= maxl then
			if table.ifind( v.occurence, itype ) then
				table.insert(list,v)
			end
		end
	end
	if #list == 0 then
		return false
	else
		return table.random( list )
	end
end

function utils.roll_magic(it, minl, maxl)
	if maxl == nil then
		maxl = minl
		minl = math.min(math.floor(maxl / 2),25)
	end

	local prefix = false
	local suffix = false
	local itype = items[it.id].type
	local roll = math.random(100)

	if ( itype == TYPE_STAFF ) and ( it.spell ~= 0 ) then
		prefix = utils.random_fix( prefixes, TYPE_SSTAFF, minl, maxl )
	elseif roll <= 21 then
		prefix = utils.random_fix( prefixes, itype, minl, maxl )
	elseif roll <= 84 then
		suffix = utils.random_fix( suffixes, itype, minl, maxl )
	else
		prefix = utils.random_fix( prefixes, itype, minl, maxl )
		suffix = utils.random_fix( suffixes, itype, minl, maxl )
	end

	if prefix then prefix.apply( it ) end
	if suffix then suffix.apply( it ) end
end


function utils.try_roll_magic( item, ilvl, chance )
	local itype = items[item.id].type

	if ( itype == TYPE_STAFF ) and ( item.spell ~= 0 ) then
		chance = 10
	elseif itype == TYPE_AMULET or itype == TYPE_RING then
		chance = 100
	end
	if math.random(100) <= chance then
		utils.roll_magic( item, ilvl )
	end
end

declare("load_quest_maps")

function load_quest_maps()
	  -- Adjust level maps for applicable quests
	for _,q in ipairs(quests) do
		if (q.enabled and q.map) then
			local l = levels[q.level]
			l.map = q.map
			l.map_key = q.map_key
			LevelInitMap(l)
		end
	end
end
