/****************************************************************************
**
** Copyright (C) 2010 Mikko Mattila
** Contact: (Gmail: mattila.mikko)
**
** This file is part of DGScores.
**
** GNU General Public License Usage
** This file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact the
** original author.
**
****************************************************************************/


#include "dgsscorewindow.h"
#include "ui_dgsscorewindow.h"

DGSScoreWindow::DGSScoreWindow(DGSData *data, QWidget *parent) :
        QMainWindow(parent),
        ui(new Ui::DGSScoreWindow)
{
    ui->setupUi(this);
    m_data = data;

    setupUi();

#if defined(Q_WS_MAEMO_5)
    setupMaemoUi();
#endif

    show();
}

DGSScoreWindow::~DGSScoreWindow()
{
    delete ui;
    qDebug() << "Score Window Destroyed";
}

void DGSScoreWindow::setupUi()
{
    QString allCourses = tr("<Show All Rounds>");
    ui->comboBoxCourses->setEditable(false);
    ui->comboBoxCourses->setModel(m_data->getCourseModel());
    ui->comboBoxCourses->setModelColumn(1);
    ui->comboBoxCourses->insertItem(0, allCourses);
    ui->comboBoxCourses->setCurrentIndex(0);
    m_selectedCourse = allCourses;

    ui->tableViewRounds->setModel(m_data->getRoundViewModel());
    ui->tableViewRounds->hideColumn(0);
    ui->tableViewRounds->hideColumn(1);
    ui->tableViewRounds->verticalHeader()->hide();

    connect(m_data, SIGNAL(error(QString)), this, SLOT(close()), Qt::QueuedConnection);

#if !defined(Q_WS_MAEMO_5)
    connect(ui->tableViewRounds, SIGNAL(clicked(QModelIndex)), this, SLOT(refreshRound()));
    connect(ui->comboBoxCourses, SIGNAL(currentIndexChanged(QString)), this, SLOT(refreshCourseAndRound()));
    connect(ui->pushButtonOK, SIGNAL(clicked()), this, SLOT(exitScoreWindow()));
    connect(ui->pushButtonRemoveRound, SIGNAL(clicked()), this, SLOT(removeRound()));
#endif
}

void DGSScoreWindow::setupMaemoUi()
{
#if defined(Q_WS_MAEMO_5)
    //Hide stuff that maemo doesn't want.
    ui->groupBoxFilters->hide();
    ui->frameBottomOk->hide();
    ui->groupBoxRoundView->setTitle("");

    //Add menubar to remove rounds
    QAction *actionRemoveRound = new QAction(tr("Remove Selected Round"), this);
    ui->menuBar->addAction(actionRemoveRound);
    connect(actionRemoveRound, SIGNAL(triggered()), this, SLOT(removeRound()));

    MaemoHorizontalDivider *hLineCourses = new MaemoHorizontalDivider(tr("Course Details"), this);
    ui->gridLayoutRoundView->addWidget(hLineCourses, 0, 0, 1, ui->gridLayoutRoundView->columnCount());
    MaemoHorizontalDivider *hLineRounds = new MaemoHorizontalDivider(tr("Round Details"), this);
    ui->gridLayoutRoundView->addWidget(hLineRounds, 3, 0, 1, ui->gridLayoutRoundView->columnCount());

    ui->lineRounds->hide();

    m_courseSelector = new QMaemo5ListPickSelector(this);
    m_courseSelector->setModel(m_data->getCourseModel());
    m_courseSelector->setModelColumn(1);

    QMaemo5ValueButton *courseButton = new QMaemo5ValueButton(tr("Course Selection"), this);
    courseButton->setValueLayout(QMaemo5ValueButton::ValueUnderText);
    courseButton->setPickSelector(m_courseSelector);

    m_courseSelector->setCurrentIndex(0);
    ui->horizontalLayoutMaemoBanner->addWidget(courseButton);

    m_roundSelector = new QMaemo5ListPickSelector(this);
    m_roundSelector->setModel(m_data->getRoundViewModel());
    m_roundSelector->setModelColumn(3);

    QMaemo5ValueButton *roundButton = new QMaemo5ValueButton(tr("Round Selection"), this);
    roundButton->setValueLayout(QMaemo5ValueButton::ValueUnderText);
    roundButton->setPickSelector(m_roundSelector);

    ui->horizontalLayoutMaemoBanner->addWidget(roundButton);

    connect(m_courseSelector, SIGNAL(selected(QString)), this, SLOT(refreshCourseAndRound()));
    connect(m_roundSelector, SIGNAL(selected(QString)), this, SLOT(refreshRound()));

#endif

}


void DGSScoreWindow::exitScoreWindow()
{
    emit exited();
    this->close();
}

void DGSScoreWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void DGSScoreWindow::closeEvent(QCloseEvent *event)
{
    QString item = ui->comboBoxCourses->itemData(0,Qt::DisplayRole).toString();

    if (item.contains("<") && item.contains(">") )
    {
        qDebug() << "Deleting Item <Show All Rounds>";
        ui->comboBoxCourses->removeItem(0);
    }
    event->accept();
    this->deleteLater();
}

//Updates the whole ScoreWindow UI.
void DGSScoreWindow::refreshCourseAndRound()
{
#if defined(Q_WS_MAEMO_5)
    QString courseName = m_courseSelector->currentValueText();
#else
    QString courseName = ui->comboBoxCourses->currentText();
#endif
    //If course wasn't changed
    if (courseName != m_selectedCourse)
    {
        m_selectedCourse = courseName;
        refreshRoundListing();
        refreshCourseDetails();
    }
    refreshRound();
}

// Updates the information of the selected round to the UI
void DGSScoreWindow::refreshRound()
{
#if defined(Q_WS_MAEMO_5)
    QString courseName = m_courseSelector->currentValueText();
#else
    QString courseName = ui->comboBoxCourses->currentText();
#endif

    //If all courses are being show, update the course information also
    if (m_selectedCourse != courseName || m_selectedCourse.contains('<'))
    {
        qDebug() << "here";
        refreshRoundDetails();
        m_selectedCourse = courseName;
        refreshCourseDetails();
    }
    refreshRoundDetails();
    refreshScoreLabels();
}

void DGSScoreWindow::refreshRoundListing()
{
    if (m_selectedCourse.contains('<'))
    {
        //Disables filtering in the roundViewModel
        qDebug() << "Disabling course filter";
        setRoundFiltering(m_data->getRoundViewModel(0));
    }
    else if (m_selectedCourse.length() > 0)
    {
        qDebug() << __FUNCTION__ << "setting current course:" << m_selectedCourse;
        m_data->setCurrentCourse(m_selectedCourse);

        qDebug() << "Setting filter for course: " << m_selectedCourse;
        setRoundFiltering(m_data->getRoundViewModel(m_selectedCourse));
    }
}

void DGSScoreWindow::setRoundFiltering(QSqlQueryModel *filteredModel)
{

#if defined(Q_WS_MAEMO_5)
    m_roundSelector->setModel(filteredModel);
    m_roundSelector->setCurrentIndex(0);
#else
    ui->tableViewRounds->setModel(filteredModel);
    ui->tableViewRounds->selectRow(0);
#endif
}

void DGSScoreWindow::refreshCourseDetails()
{
    ui->labelCourseLocationValue->setText(m_data->courseLocation());
    ui->labelScoreViewCourseName->setText(m_data->courseName());
    int record = m_data->courseRecord(); //Returns INT_MAX if there is no valid record

    if (record == INT_MAX)
    {
        ui->labelRecordValue->setText("N/A");
    }
    else if (record > 0)
    {
        ui->labelRecordValue->setText("Par +" + QString().setNum(record));
    }
    else
    {
        ui->labelRecordValue->setText("Par " + QString().setNum(record));
    }
}

void DGSScoreWindow::refreshRoundDetails()
{
    int curRow = -1;
#if defined(Q_WS_MAEMO_5)
    curRow = m_roundSelector->currentIndex();
#else
    QModelIndexList indexes = ui->tableViewRounds->selectionModel()->selectedIndexes();
    if (indexes.count() > 0)
    {
        curRow = indexes.at(0).row();
    }
#endif
    int round_id = -1;
    if (curRow != -1)
    {
        //Becomes 0 or -1 if invalid
        round_id = m_data->getRoundViewModel(m_selectedCourse)->index(curRow, 0).data(0).toInt();
    }

    if (round_id > 0)
    {
        qDebug() << "Refreshing Round Scores";
        m_data->setCurrentRound(round_id);
    }
    else
    {
        qDebug() << "No round selected. Clearing scores.";
        m_data->setCurrentRound(-1);
    }
}

void DGSScoreWindow::clearRoundDetails()
{
    //Remove old groupboxes and dividers
    foreach (QGroupBox *gb, m_scoreGroupBoxes)
    {
        gb->deleteLater();
    }
    m_scoreGroupBoxes.clear();
#if defined(Q_WS_MAEMO_5)
    foreach (MaemoHorizontalDivider *div, m_dividers)
    {
        div->deleteLater();
    }
    m_dividers.clear();
#endif

    ui->labelScoreValue->setText("Incomplete round");
    ui->labelScoreValue->setStyleSheet("color: "+QSettings().value("style/badColor").toString() +";");
    ui->labelDurationValue->setText(tr("N/A"));
}

void DGSScoreWindow::refreshScoreLabels()
{
    clearRoundDetails();

    //If invalid round, stop.
    if (!m_data->roundIsValid())
    {
        qDebug() << "Invalid round selected";
        return;
    }

    QList<int> pars = m_data->coursePars();
    QList<int> scores = m_data->roundScores();

    // Check if the course is complete, or if it even has scores at all
    bool hasScore = false;
    bool isComplete = true;
    foreach (int s, scores)
    {
        if (s > 0)
        {
            hasScore = true;
        }
        else
        {
            isComplete = false;
            break;
        }
    }

    if (isComplete)
    {
        //Show round score value
        int scoreInt = m_data->roundScore();
        QString score(QString().setNum(scoreInt));
        if (scoreInt > 0)
        {
            score.prepend("+");
        }
        ui->labelScoreValue->setText(tr("Par ") + score);

        //Show round duration
        ui->labelDurationValue->setText(m_data->roundDuration());

        //Set colors for round score label
        if (scoreInt <= 0)
        {
            ui->labelScoreValue->setStyleSheet("color: "+QSettings().value("style/goodColor").toString() +";");
        }
        else
        {
            ui->labelScoreValue->setStyleSheet("color: "+QSettings().value("style/badColor").toString() +";");
        }
    }

    if (hasScore)
    {
        drawScores(scores, pars);
    }
    else
    {
        qDebug() << "Round has no scores";
    }
    ui->gridLayoutRoundView->removeItem(ui->verticalSpacer);
    ui->gridLayoutRoundView->addItem(ui->verticalSpacer, ui->gridLayoutRoundView->rowCount(), 0);
}

void DGSScoreWindow::drawScores(QList<int> scores, QList<int> pars)
{
    QSettings settings;
    int holesPerGb = settings.value("score_view/scoresPerGroupBox").toInt();
    int lastStaticRow = ui->gridLayoutRoundView->rowCount()-1;
    int gbCount = (m_data->courseHoleCount()/9.0) + 0.999999999999999;

    //Draw groupboxes and their contents
    for(int i=1; i<=gbCount; i++)
    {
        QGroupBox *gb = new QGroupBox(this);
        gb->setTitle(tr("Holes ") + QString().setNum(1+(i-1)*holesPerGb) + " - " + QString().setNum(i*holesPerGb));
        gb->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);

        int columnSpan = ui->gridLayoutRoundView->columnCount();
        QGridLayout *gbLayout = new QGridLayout(gb);
        gb->setLayout(gbLayout);

#if defined(Q_WS_MAEMO_5)
        MaemoHorizontalDivider *holeNumberDivider = new MaemoHorizontalDivider(gb->title(), this);
        m_dividers.append(holeNumberDivider);
        ui->gridLayoutRoundView->addWidget(holeNumberDivider, lastStaticRow+2*i,0, 1, columnSpan);
        gb->setTitle("");
#endif

        ui->gridLayoutRoundView->addWidget(gb, lastStaticRow+2*i+1, 0, 1, columnSpan);
        m_scoreGroupBoxes.append(gb);

        for(int j=1; j<=holesPerGb; j++)
        {
            int holeNr = (i-1)*holesPerGb +j;
            if (pars.count() < holeNr )
            {
                QSpacerItem *spacer = new QSpacerItem(20, 20, QSizePolicy::Expanding, QSizePolicy::Minimum);
                gbLayout->addItem(spacer, 0, j);
                break;
            }
            QLabel *score = new QLabel(gb);
            score->setText(QString().setNum(scores.at(holeNr-1)));
            score->setAlignment(Qt::AlignCenter | Qt::AlignVCenter);

            QFont font = score->font();
            font.setPointSize(font.pointSize()+4);
            score->setFont(font);

            if (scores.at(holeNr-1) <= pars.at(holeNr-1) && scores.at(holeNr-1) > 0 )
            {
                score->setStyleSheet("color: "+settings.value("style/goodColor").toString() +";");
            }
            else if (scores.at(holeNr-1) > 0)
            {
                score->setStyleSheet("color: "+settings.value("style/badColor").toString() +";");
            }
            QLabel *par = new QLabel(gb);
            par->setText(QString().setNum(pars.at(holeNr-1)));
            par->setAlignment(score->alignment());
            par->setFont(font);
            par->setMinimumWidth(30);
            int scoreRow = 0;
            gbLayout->addWidget(par,scoreRow+1,j-1);
            gbLayout->addWidget(score, scoreRow,j-1);
        }
    }
}

void DGSScoreWindow::removeRound()
{
    //Don't do anything if a round isn't selected.
#if defined(Q_WS_MAEMO_5)
    if (m_roundSelector->currentValueText().length() < 1)
        return;
#else
    if (ui->tableViewRounds->selectionModel()->selectedRows(0).count() < 1)
        return;
#endif

    m_data->removeSelectedRound();
    refreshRoundListing();
    refreshRound();
}
