/****************************************************************************
**
** Copyright (C) 2010 Mikko Mattila
** Contact: (Gmail: mattila.mikko)
**
** This file is part of DGScores.
**
** GNU General Public License Usage
** This file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact the 
** original author.
**
****************************************************************************/

#include "dgsplayview.h"
#include "ui_dgsplayview.h"

DGSPlayView::DGSPlayView(DGSData *data, QWidget *parent)
    : QMainWindow(parent), ui(new Ui::DGSPlayView)
{
    ui->setupUi(this);
    this->m_data = data;
    this->m_mapView = NULL;

    // Get course details and show them
    QString courseName = m_data->courseName();
    QString courseLocation = m_data->courseLocation();
    ui->labelPlayCourseDetails->setText(courseName + ", " + courseLocation);
    ui->labelPlayHoleCount->setText(QString().setNum(m_data->courseHoleCount()) + tr(" holes"));
    m_pars = m_data->coursePars();

    if (this->width() > 640)
    {
        ui->pushButtonPlayLeft->setMinimumWidth(150);
        ui->pushButtonPlayRight->setMinimumWidth(150);
    }
    // Add the hole score widgets and their labels
    addScoreSpinBoxes();

    //Move the bottom spacer to right position
    ui->gridLayoutPlayViewHoles->removeItem(ui->verticalSpacerBottom);
    ui->gridLayoutPlayViewHoles->addItem(ui->verticalSpacerBottom, 3, 0);
    ui->scrollAreaPlay->setProperty("FingerScrollable", true);

    ui->scrollAreaPlay->setMinimumWidth(this->width());
    ui->gridLayout->removeWidget(ui->scrollAreaPlay);
    ui->gridLayout->addWidget(ui->scrollAreaPlay, 0, 0, ui->gridLayout->rowCount(), ui->gridLayout->columnCount(), Qt::AlignJustify);
    ui->pushButtonPlayLeft->raise();
    ui->pushButtonPlayRight->raise();

    ui->frameBannerGradient->setMinimumWidth(ui->pushButtonPlayLeft->minimumWidth());
    ui->frameBannerGradient->lower();

    //Add and connect menu actions
    QAction *actionSaveAndExit = new QAction(tr("Save Results and Return"), this);
    ui->menubar->addAction(actionSaveAndExit);
    QAction *actionDiscard = new QAction(tr("Discard Results and Return"), this);
    ui->menubar->addAction(actionDiscard);
    connect(actionSaveAndExit, SIGNAL(triggered()), this, SLOT(saveScoresAndExit()));
    connect(actionDiscard, SIGNAL(triggered()), this, SLOT(exitPlayView()));

    m_completionTimer = new QTimer(this);
    connect(m_completionTimer, SIGNAL(timeout()), this, SLOT(showEndRoundDialog()));
    connect(ui->pushButtonPlayRight, SIGNAL(clicked()), this , SLOT(scrollContentsRight()));
    connect(ui->pushButtonPlayLeft, SIGNAL(clicked()), this , SLOT(scrollContentsLeft()));
    connect(ui->toolButtonCourseMap, SIGNAL(clicked()), this, SLOT(showCourseMap()));

    //Animation for scrolling contents
    m_animation = new QPropertyAnimation(ui->scrollAreaPlay->horizontalScrollBar(), "value");

    this->setWindowTitle(tr("DGScores - Play"));
    this->show();

#if defined(Q_WS_MAEMO_5)
    FullScreenExitButton *fsButton = new FullScreenExitButton(this);
    toggleFullScreen();
    qDebug() << "Going to Full Screen Mode";
    this->resize(this->size());
#endif

    //Set margins so that one spinbox is always in the center.
    int margin = ui->scrollAreaPlay->width()/2 - m_playHoles.first()->width()/2
                 - 2*ui->gridLayoutPlayViewHoles->contentsMargins().left();
    ui->gridLayoutPlayViewHoles->setContentsMargins(margin,11,margin,11);

    ui->scrollAreaPlay->horizontalScrollBar()->setValue(ui->scrollAreaPlay->horizontalScrollBar()->minimum());

    //Tell data controller that playing has started.
    m_data->setPlayStartTime();
    m_saved = false;

    //Disable course info button if no info is available.
    if (m_data->courseNotes().isEmpty() && m_data->courseMapPath().isEmpty() )
    {
        ui->toolButtonCourseMap->setDisabled(true);
    }

    qDebug() << "Play View initialized!";
}

DGSPlayView::~DGSPlayView()
{
    delete ui;
    qDebug() << "Play View Destroyed";
}

void DGSPlayView::addScoreSpinBoxes()
{
    int holeCount = m_data->courseHoleCount();
    if ( holeCount < 1 )
    {
        emit error("No holes were found for the selected course. Something has probably been corrupted. \nTry editing the current course to fix the issue.");
        return;
    }

    QList<int> lengths = m_data->courseLengths();
    QList<int> pars = m_data->coursePars();
    for (int i=1; i<=holeCount; i++)
    {
        DGSHoleSpinBox *sb = new DGSHoleSpinBox(this);
        connect(sb, SIGNAL(valueChanged(int)), this, SLOT(holeScoreChanged()));
        QLabel *holeLabel = new QLabel(sb);
        holeLabel->setText(tr("Hole " ) + QString().setNum(i));
        holeLabel->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
        holeLabel->setStyleSheet("QLabel{font-size: 18pt;}");
        QString par = tr("Par ") + QString().setNum(pars.at(i-1));
        QString length = "\n";
        if (lengths.at(i-1) > 0)
        {
            length = QString("\n") + QString().setNum(lengths.at(i-1)) + tr("m");
        }
        QLabel *holeInfoLabel = new QLabel(sb);
        holeInfoLabel->setText(par + length);
        holeInfoLabel->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
        ui->gridLayoutPlayViewHoles->addWidget(holeLabel, 1, i);
        ui->gridLayoutPlayViewHoles->addWidget(sb, 2, i);
        ui->gridLayoutPlayViewHoles->addWidget(holeInfoLabel, 3, i);
        m_playHoles.append(sb);

    }
}

void DGSPlayView::refreshPlayScore()
{
    int total = 0;
    int parScore = 0;
    for (int i=0; i<m_playHoles.count(); i++)
    {
        if (m_playHoles.at(i)->value() > 0)
        {
            total = total + m_playHoles.at(i)->value();
            parScore = parScore + m_pars.at(i);
        }
    }
    QString sign;
    if (total-parScore > 0)
    {
        sign = "+";
    }
    ui->labelPlayCurrentScoreValue->setText("Par " + sign + QString().setNum(total - parScore));
}

void DGSPlayView::saveScoresAndExit()
{
    saveScores();
    exitPlayView();
}

void DGSPlayView::saveScores()
{
    m_completionTimer->stop();
    QList<int> scores;
    foreach (QSpinBox *box, m_playHoles)
    {
        scores.append(box->value());
    }
    m_data->saveScores(scores);
    m_saved = true;
}

void DGSPlayView::exitPlayView()
{
    emit exited();
    this->close();
    this->deleteLater();
}


void DGSPlayView::showCourseMap()
{
    if (m_mapView)
    {
        m_mapView->show();
    }
    else
    {
        m_mapView = new DGSCourseMapView(m_data, this);
    }
#if defined(Q_WS_MAEMO_5)
    m_mapView->setAttribute(Qt::WA_Maemo5StackedWindow);
    connect(m_mapView, SIGNAL(exited()), this, SLOT(mapViewClosed()));
#endif
}


void DGSPlayView::mapViewClosed()
{
    this->show();
    this->update();
}

void DGSPlayView::scrollContentsRight()
{
    scrollContents(true);
}

void DGSPlayView::scrollContentsLeft()
{
    scrollContents(false);
}

//False value scrolls left, true right.
void DGSPlayView::scrollContents(bool toRight)
{
    if (m_playHoles.count() < 2)
    {
        return;
    }
    int travelDistance = m_playHoles.at(1)->pos().x() - m_playHoles.at(0)->pos().x(); //Distance between two spinboxes
    if (!toRight)
    {
        travelDistance = -travelDistance;
    }

    int startPosition = ui->scrollAreaPlay->horizontalScrollBar()->value();

    if (m_animation->state() == QAbstractAnimation::Running)
    {
        int stopPosition = m_animation->endValue().toInt() + travelDistance;
        m_animation->stop();
        m_animation->setEasingCurve(QEasingCurve::OutCubic);
        m_animation->setDuration(800);
        m_animation->setStartValue(m_animation->currentValue().toInt());
        m_animation->setEndValue(stopPosition);
        m_animation->start();
    }
    else
    {
        m_animation->setDuration(800);
        m_animation->setStartValue(startPosition);
        m_animation->setEndValue(startPosition+travelDistance);
        m_animation->setEasingCurve(QEasingCurve::OutCubic);
        m_animation->start();
    }
}


//Also styles the holes to be the right color.
int DGSPlayView::completedHolesCount()
{
    QSettings settings;
    int completedHoles = 0;
    int i = 0;
    foreach (DGSHoleSpinBox *sb, m_playHoles)
    {
        if (sb->value() > 0)
        {
            completedHoles++;
            if (sb->value() > m_pars.at(i))
            {
                sb->setFontColor(settings.value("style/badColor").toString());
            }
            else
            {
                sb->setFontColor(settings.value("style/goodColor").toString());
            }
        }
        else
        {
            sb->setFontColor(settings.value("style/neutralColor").toString());
        }
        i++;
    }
    return completedHoles;
}

void DGSPlayView::holeScoreChanged()
{
    if (completedHolesCount() == m_playHoles.count())
    {
        m_completionTimer->setSingleShot(true);
        m_completionTimer->start(2000);
        qDebug() << "Completion Timer reset";

        //Disable course info button
        ui->toolButtonCourseMap->setDisabled(true);
    }
    else
    {
        //Enable course info button
        if (!m_data->courseNotes().isEmpty() || !m_data->courseMapPath().isEmpty() )
        {
            ui->toolButtonCourseMap->setDisabled(false);
        }
        m_completionTimer->stop();
    }
    refreshPlayScore();
}

void DGSPlayView::showEndRoundDialog()
{
    QMessageBox msgBox;
    QString message(tr("It looks like you have completed the course.\nWould you like to save the score and exit to Main Menu?"));
    msgBox.setText(message);
    msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Cancel);
    msgBox.setDefaultButton(QMessageBox::Save);
    msgBox.setWindowTitle(tr("Save and exit?"));
    int ret = msgBox.exec();
    //If Yes
    if (ret == QMessageBox::Save)
    {
        saveScoresAndExit();
    }
}

void DGSPlayView::closeEvent(QCloseEvent *event)
{
    //If all holes finished already, but not saved yet
    if (m_completionTimer->isActive())
    {
        saveScores();
        event->accept();
        this->deleteLater();
        return;
    }
    // The scores have been saved already, and the window is closing.
    else if (m_saved || completedHolesCount() == 0)
    {
        event->accept();
        this->deleteLater();
        return;
    }

    QMessageBox msgBox;
    QString message(tr("You haven't completed the course yet. \nIf you're sure you want to exit, would you like your scores to be saved?"));
    msgBox.setText(message);
    msgBox.setStandardButtons(QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);
    msgBox.setDefaultButton(QMessageBox::Cancel);
    msgBox.setWindowTitle(tr("Course not completed yet. Save scores?"));
    int ret = msgBox.exec();
    //If Save
    if (ret == QMessageBox::Save)
    {
        saveScores();
        event->accept();
        this->deleteLater();
        return;
    }
    else if (ret == QMessageBox::Discard)
    {
        event->accept();
        this->deleteLater();
        return;
    }

    //QMessageBox::Cancel
    m_saved = false;
    event->ignore();
}

void DGSPlayView::changeEvent(QEvent *event)
{
    if (event->type() == QEvent::WindowStateChange)
    {
        const bool isFullScreen = windowState() & Qt::WindowFullScreen;
        if (isFullScreen)
        {
            ui->frameLogo->setMinimumSize(QSize(117, 111));
            foreach (DGSHoleSpinBox *sb, m_playHoles)
            {
                sb->setFontSize(48);
            }
        }
        else
        {
            ui->frameLogo->setMinimumSize(QSize(100, 96));
            foreach (DGSHoleSpinBox *sb, m_playHoles)
            {
                sb->setFontSize(36);
            }
        }
    }
    QWidget::changeEvent(event);
}

void DGSPlayView::toggleFullScreen()
{
    if(this->isFullScreen())
        showNormal();
    else
        showFullScreen();
}
