/****************************************************************************
**
** Copyright (C) 2010 Mikko Mattila
** Contact: (Gmail: mattila.mikko)
**
** This file is part of DGScores.
**
** GNU General Public License Usage
** This file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact the 
** original author.
**
****************************************************************************/

#include "dgsmainwindow.h"
#include "ui_dgsmainwindow.h"


DGSMainWindow::DGSMainWindow(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::DGSMainWindow)
{

    QCoreApplication::setOrganizationName("DiscGolf");
    QCoreApplication::setOrganizationDomain("dgscores.garage.maemo.org");
    QCoreApplication::setApplicationName("DGScores");

    //if (QSettings().value("setupDone").toInt() != 1)
    //{
    firstTimeSetup();
    //}

    // UI part
    ui->setupUi(this);
    setWindowTitle(tr("DGScores - Main Menu"));

#if defined(Q_WS_MAEMO_5)
    setAttribute(Qt::WA_Maemo5StackedWindow);
    ui->horizontalLayout->setContentsMargins(9,0,9,9);
#else
    resize(800, 480);
#endif

    m_lineEditFilter = new AutoClearingLineEdit(tr("Filter Courses"), this);
    ui->horizontalLayoutFilter->addWidget(m_lineEditFilter);
    connect(m_lineEditFilter, SIGNAL(textChanged(QString)), this, SLOT(filterCourses(QString)));


    initModelsAndMappers();
    show();

    //Main menu UI
    connect(ui->pushButtonAddCourse, SIGNAL(clicked()), this, SLOT(addCourseClicked()));
    connect(ui->pushButtonEditCourse, SIGNAL(clicked()), this, SLOT(editCourseClicked()));
    connect(ui->pushButtonDeleteCourse, SIGNAL(clicked()), this, SLOT(deleteCourse()));
    connect(ui->pushButtonScores, SIGNAL(clicked()), this, SLOT(initScoreView()));
    connect(ui->pushButtonPlay, SIGNAL(clicked()), this, SLOT(initPlayView()));
    connect(ui->listViewCourses, SIGNAL(clicked(QModelIndex)), this, SLOT(courseSelectionChanged(QModelIndex )));

    //Disable some of the menu buttons until something is selected.
    disableMenuButtons(true);
}

DGSMainWindow::~DGSMainWindow()
{
    qDebug() << "Deleting UI";
    delete ui;
}

void DGSMainWindow::firstTimeSetup()
{
    QSettings settings;
    settings.setValue("style/goodColor", "#7ab900");
    settings.setValue("style/neutralColor", "#FFFFFF");
    settings.setValue("style/badColor", "#ee7600");
    settings.setValue("score_view/scoresPerGroupBox", 9);
    settings.setValue("defaults/NewCourseHoleCount", 9);
    settings.setValue("defaults/NewCourseDefaultPar", 3);
    settings.setValue("defaults/NewCourseDefaultLength", 0);
    settings.setValue("setupDone", 1);
    settings.setValue("defaults/dateFormat", "dd/MM/yyyy hh:mm:ss");

#if defined(Q_OS_LINUX)
    QString dataDir = QDir::homePath() + "/.dgscores/";
#else
    QString dataDir = QDir::homePath() + "/dgscores/";
#endif
    QString mapDir = dataDir + "maps/";

    QDir().mkdir(dataDir);
    QDir().mkdir(mapDir);

    settings.setValue("defaults/DataDirectory", dataDir);
    settings.setValue("defaults/MapsDirectory", mapDir);
}

void DGSMainWindow::disableMenuButtons(bool setDisabled)
{
    ui->pushButtonDeleteCourse->setDisabled(setDisabled);
    ui->pushButtonEditCourse->setDisabled(setDisabled);
    ui->pushButtonPlay->setDisabled(setDisabled);
}

void DGSMainWindow::initModelsAndMappers()
{
    qDebug() << "Initializing the data controller";
    m_data = new DGSData(this);
    if (!m_data->isValid())
    {
        showFatalError("Failed to initialize data model!");
    }
    //Enable error reporting from m_data
    connect(m_data, SIGNAL(error(QString )), this, SLOT(showFatalError(QString )), Qt::QueuedConnection);

    qDebug() << "Initializing ListView";
    QSqlTableModel *courseModel = m_data->getCourseModel();
    ui->listViewCourses->setModel(courseModel);
    ui->listViewCourses->setModelColumn(1);
}

void DGSMainWindow::initScoreView()
{
    DGSScoreWindow *scoreView = new DGSScoreWindow(m_data, this);
#if defined(Q_WS_MAEMO_5)
    scoreView->setAttribute(Qt::WA_Maemo5StackedWindow);
#else
    scoreView->move(this->pos());
    scoreView->resize(this->size());
    this->hide();
#endif
    connect(scoreView, SIGNAL(error(QString)), this, SLOT(showFatalError(QString)));
    connect(scoreView, SIGNAL(exited()), this, SLOT(show()));
}

void DGSMainWindow::initPlayView()
{
    DGSPlayView *playView = new DGSPlayView(m_data, this);
#if defined(Q_WS_MAEMO_5)
    playView->setAttribute(Qt::WA_Maemo5StackedWindow);
#else
    playView->move(this->pos());
    playView->resize(this->size());
    this->hide();
#endif
    courseSelectionChanged(ui->listViewCourses->currentIndex());
    connect(playView, SIGNAL(error(QString)), this, SLOT(showFatalError(QString)));
    connect(playView, SIGNAL(exited()), this, SLOT(show()));
}

void DGSMainWindow::addCourseClicked()
{
    editCourse(-1);
}

void DGSMainWindow::editCourseClicked()
{
    editCourse(m_selectedRow);
}

//Use courseRow = -1 for new course
void DGSMainWindow::editCourse(int courseRow) //defaults to false
{
    DGSCourseSettingsDialog *settingsDialog = new DGSCourseSettingsDialog(m_data, courseRow, this);
}

void DGSMainWindow::deleteCourse()
{
    QMessageBox msgBox;
    QString message = tr("Are you sure you want to delete the course: ") + m_data->courseName() + "?\nAll related scores will also be deleted." ;
    msgBox.setText(message);
    msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
    msgBox.setDefaultButton(QMessageBox::No);
    msgBox.setWindowTitle(tr("Delete course?"));
    int ret = msgBox.exec();
    //If Yes
    if (ret == QMessageBox::Yes)
    {
        m_data->removeSelectedCourse();
        disableMenuButtons(true);
    }
}

void DGSMainWindow::courseSelectionChanged(QModelIndex index)
{
    if (ui->listViewCourses->selectionModel()->selectedIndexes().count() < 1)
    {
        disableMenuButtons(true);
    }
    else
    {
        m_data->setCurrentCourse(index.data(0).toString());
        disableMenuButtons(false);
        m_selectedRow = index.row();
    }
}


void DGSMainWindow::filterCourses(QString filter)
{
    if (m_lineEditFilter->hasValidFilter())
    {
        m_data->getCourseModel()->setFilter("name LIKE \"%" + filter + "%\"");
    }
    else
    {
        m_data->getCourseModel()->setFilter("course_id > 0");
    }
}

void DGSMainWindow::showFatalError(QString error)
{
    QMessageBox msgBox;
    QString message(tr("A fatal error has been encountered and DGScores needs to close. \nSome data may have been lost.\n") + error);
    msgBox.setText(message);
    msgBox.setWindowTitle("Fatal Error");
    msgBox.exec();
    this->close();
}
