#include "dgscoursesettingsdialog.h"
#include "ui_dgscoursesettingsdialog.h"

DGSCourseSettingsDialog::DGSCourseSettingsDialog(DGSData *data, int mapperRow, QWidget *parent) :
        QDialog(parent), ui(new Ui::DGSCourseSettingsDialog)
{
    qDebug() << "Initializing Course Settings Dialog";
    m_data = data;
    if ( !m_data->isValid() )
    {
        emit error("Received an invalid data model!");
    }
    ui->setupUi(this);
    initMapper(mapperRow);

#if defined(Q_WS_MAEMO_5)
    QPushButton *saveButton = new QPushButton("Save Course", this);
    ui->gridLayoutFrame->addWidget(saveButton, 0, 2);
#else
    QPushButton *saveButton = new QPushButton("Save", this);
    ui->buttonBox->addButton(saveButton, QDialogButtonBox::ApplyRole);
#endif
    QPushButton *cancelButton = new QPushButton("Cancel", this);
    ui->buttonBox->addButton(cancelButton, QDialogButtonBox::RejectRole);

    ui->lineEditFilePath->hide();

    //Course Changes UI
    connect(ui->spinBoxHoleCount, SIGNAL(valueChanged(int)), this, SLOT(refreshHoleCount(int)));
    connect(saveButton, SIGNAL(clicked()), this, SLOT(saveSettings()));
    connect(this, SIGNAL(rejected()), this, SLOT(discardSettings()));
    connect(ui->pushButtonBrowseMap, SIGNAL(clicked()), this, SLOT(selectLocalFile()));
    connect(ui->pushButtonHTTPRetreiveMap, SIGNAL(clicked()), this, SLOT(selectCourseMapFromUrl()));

    if (mapperRow < 0)
    {
        addNewCourse();
    }
    else
    {
        modifyCourse();
    }
    
    this->setWindowModality(Qt::ApplicationModal);

    qDebug() << "Showing Course Settings Dialog";

    //Setup Maemo UI
#if defined(Q_WS_MAEMO_5)
    ui->gridLayoutFrame->setContentsMargins(3,9,0,3);
    ui->gridLayoutPars->setContentsMargins(0,0,0,0);
    ui->gridLayoutLengths->setContentsMargins(0,0,0,0);
    ui->groupBoxHoleLengths->setTitle("");
    ui->groupBoxHolePars->setTitle("");

    MaemoHorizontalDivider *hLinePars = new MaemoHorizontalDivider("Hole Pars", this);
    ui->gridLayoutPars->addWidget(hLinePars, 0, 0, 1, ui->gridLayoutPars->columnCount());

    MaemoHorizontalDivider *hLineLengths = new MaemoHorizontalDivider("Hole Lengths", this);
    ui->gridLayoutLengths->addWidget(hLineLengths, 0, 0, 1, ui->gridLayoutLengths->columnCount());
#endif
    show();

}

DGSCourseSettingsDialog::~DGSCourseSettingsDialog()
{
    qDebug() << "Course Settings Dialog Destroyed";
    delete ui;
}

void DGSCourseSettingsDialog::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void DGSCourseSettingsDialog::initMapper(int mapperRow)
{
    m_mapper = new QDataWidgetMapper(this);
    DGSCourseModel *courseModel = m_data->getCourseModel();
    m_mapper->setModel(courseModel);

    //Init Modify view forms
    m_mapper->addMapping(ui->lineEditCourseName, courseModel->fieldIndex("name"));
    m_mapper->addMapping(ui->lineEditCourseLocation, courseModel->fieldIndex("location"));
    m_mapper->addMapping(ui->plainTextEditNotes, courseModel->fieldIndex("notes"));
    m_mapper->addMapping(ui->lineEditFilePath, courseModel->fieldIndex("map_image"));
    if (mapperRow >= 0)
    {
        m_mapper->setCurrentIndex(mapperRow);
    }
}

void DGSCourseSettingsDialog::addNewCourse()
{
    this->setWindowTitle(tr("DGScores - Add Course"));
    // Add a new row and select it for editing.
    int row = m_data->addCourseRow();
    m_mapper->setCurrentIndex(row);

    QSettings settings;
    int defaultHoleCount =  settings.value("defaults/NewCourseHoleCount").toInt();
    //Set up the UI
    ui->spinBoxHoleCount->setValue(defaultHoleCount);
    ui->lineEditCourseName->setText("");
}

void DGSCourseSettingsDialog::modifyCourse()
{
    //Initialize Edit Course view
    this->setWindowTitle(tr("DGScores - Edit Course"));
    int holeCount = m_data->courseHoleCount();
    ui->spinBoxHoleCount->setValue(holeCount);
    QList<int> pars = m_data->coursePars();
    QList<int> lengths = m_data->courseLengths();
    if (m_data->courseMapPath() != "" && QFile().exists(m_data->courseMapPath()))
    {
        ui->labelMapFileName->setText(tr("Map OK"));
    }

    for (int i=0; i < m_parSpinBoxes.size(); i++)
    {
        if (pars.size() > i)
        {
            m_parSpinBoxes.at(i)->setValue(pars.at(i));
        }
        if (lengths.size() > i)
        {
            m_lengthSpinBoxes.at(i)->setValue(lengths.at(i));
        }
    }
}


void DGSCourseSettingsDialog::refreshHoleCount(int holeCount)
{
    while(m_parSpinBoxes.count()>holeCount)
    {
        removeHoleFromUI();
    }

    while(m_parSpinBoxes.count()<holeCount)
    {
        addHoleToUI();
    }
    ui->lineEditCourseName->setFocus();
}

void DGSCourseSettingsDialog::removeHoleFromUI()
{
    m_parSpinBoxes.last()->deleteLater();
    m_parSpinBoxes.removeLast();
    m_lengthSpinBoxes.last()->deleteLater();
    m_lengthSpinBoxes.removeLast();
    m_parLabels.last()->deleteLater();
    m_parLabels.removeLast();
    m_lengthLabels.last()->deleteLater();
    m_lengthLabels.removeLast();
}

void DGSCourseSettingsDialog::addHoleToUI()
{
    QString labelText(QString("Hole ") + QString().setNum(m_parSpinBoxes.count()+1));

    //Add spinbox and set a default value
    QSpinBox *par = new QSpinBox(this);
    int defaultPar = QSettings().value("defaults/NewCourseDefaultPar").toInt();
    par->setValue(defaultPar);
    par->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);

    QSpinBox *length = new QSpinBox(this);
    int defaultLength = QSettings().value("defaults/NewCourseDefaultLength").toInt();
    length->setValue(defaultLength);
    length->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
    length->setMaximum(1000);

    QLabel *parLabel = new QLabel(labelText, this);
    parLabel->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
    QLabel *lengthLabel = new QLabel(labelText, this);
    lengthLabel->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);

    int xpos = 1 + 2*(ui->gridLayoutPars->count()/2 % 2);
    int ypos = m_parSpinBoxes.count()/2;

#if defined(Q_WS_MAEMO_5)
    //For Maemo, horizontal dividers are added. See constructor.
    ypos++;
#endif

    m_parSpinBoxes.append(par);
    m_lengthSpinBoxes.append(length);
    m_parLabels.append(parLabel);
    m_lengthLabels.append(lengthLabel);
    ui->gridLayoutPars->addWidget(par, ypos, xpos+1);
    ui->gridLayoutPars->addWidget(parLabel, ypos, xpos);

    ui->gridLayoutLengths->addWidget(length, ypos, xpos+1);
    ui->gridLayoutLengths->addWidget(lengthLabel, ypos, xpos);
}

void DGSCourseSettingsDialog::saveSettings()
{
    qDebug() << "Saving Course Changes";
    //Save changes and go to Main Menu
    QString courseName = ui->lineEditCourseName->text();
    int courseId = m_data->currentCourse();

    //Check that there are no illegal characters
    if (courseName.contains(QRegExp("[<>]")) || courseName.toInt() > 0)
    {
        qDebug() << "The course name contains invalid characters or is a number";
        QMessageBox msgBox;
        QString charList(" < > ");
        msgBox.setText(tr("The course name must not contain any of the following characters:")+ charList + tr("\nIt must also not consist entirely of numbers. \n\nPlease specify a different name."));
        msgBox.setStandardButtons(QMessageBox::Ok);
        msgBox.setWindowTitle(tr("Invalid Course Name"));
        msgBox.exec();
        return;
    }

    //We're adding a new course, and the course name already exists in db
    //Or we're editing an existing course, and the name already exists in db, and it's not the course we're editing.
    if ( (courseId <= 0 && m_data->courseId(courseName) > 0) ||
         (courseId > 0 && m_data->courseId(courseName) > 0 && m_data->courseName(courseId).compare(courseName, Qt::CaseInsensitive) != 0 ))
    {
        qDebug() << "The course name already exists.";
        QMessageBox msgBox;
        msgBox.setText(tr("The course name already exists. \n\nPlease specify a different name."));
        msgBox.setStandardButtons(QMessageBox::Ok);
        msgBox.setWindowTitle(tr("Course Name already exists!"));
        msgBox.exec();
        return;
    }

    if (!m_mapper->submit())
    {
        error("Failed to submit changes from mapper." + m_data->getCourseModel()->lastError().text());
    }
    QList<int> pars;
    QList<double> lengths;
    QList<QString> positions;

    for (int i=0; i<ui->spinBoxHoleCount->value(); i++)
    {
        pars.append(m_parSpinBoxes.at(i)->value());
        lengths.append(m_lengthSpinBoxes.at(i)->value());
        //Not implemented
        //positions.append(m_positionBoxes.at(i)->value());
    }
    m_data->saveCourseChanges(pars, lengths, positions);
    this->close();
}

void DGSCourseSettingsDialog::discardSettings()
{
    //Revert changes
    qDebug() << "Discarding changes";
    m_mapper->revert();
    m_data->cancelCourseChanges();
    this->close();
}

void DGSCourseSettingsDialog::selectLocalFile()
{
    QString sourcePath;
    QString defaultDir = QDir::homePath();

#if defined(Q_WS_MAEMO_5)
    defaultDir = "/home/user/MyDocs/";
#endif

    sourcePath = QFileDialog::getOpenFileName(this, tr("Open Image"), defaultDir, tr("Image Files (*.png *.jpg *.bmp)"));
    if (sourcePath != "")
    {
        setCourseMap(sourcePath);
    }
}

void DGSCourseSettingsDialog::selectCourseMapFromUrl()
{
    QString mapPath = QDir::tempPath();
    bool ok;
    QString mapUrl = QInputDialog::getText(this, tr("Course Map URL?"), tr("Url:"), QLineEdit::Normal, "http://", &ok);
    if (ok && !mapUrl.isEmpty())
    {
        DownloadManager *mgr = new DownloadManager(this);
        connect (mgr, SIGNAL(fileSaved(QString)), this, SLOT(setCourseMap(QString)));
        mgr->download(mapUrl, mapPath);
    }



}

void DGSCourseSettingsDialog::setCourseMap(QString sourcePath)
{
    QString mapDir = QSettings().value("defaults/MapsDirectory").toString();
    if (QFile::exists(sourcePath))
    {
        QString extension = sourcePath.split(QChar('.'), QString::KeepEmptyParts).last();
        QString mapPath = mapDir + ui->lineEditCourseName->text() + QString().setNum(qrand()) + "." + extension;
        qDebug() << mapPath;
        if (QFile::exists(mapPath))
        {
            QFile().remove(mapPath);
        }
        if (!QFile::copy(sourcePath, mapPath))
        {
            QMessageBox::warning(this, tr("Failed to copy file"),
                                 tr("There was a problem copying the map file. \nPlease try again."),
                                 QMessageBox::Ok, QMessageBox::Ok);
            ui->labelMapFileName->setText(tr("No map selected"));
            return;
        }
        ui->lineEditFilePath->setText(mapPath);
        ui->labelMapFileName->setText(tr("Map OK"));
    }
    else
    {
        QMessageBox::warning(this, tr("Problem retreiving course map file"),
                             tr("There was a problem retreiving the file. \nPlease try again."),
                             QMessageBox::Ok, QMessageBox::Ok );
        ui->labelMapFileName->setText(tr("No map selected"));
        return;
    }
}
