/****************************************************************************
**
** Copyright (C) 2010 Mikko Mattila
** Contact: (Gmail: mattila.mikko)
**
** This file is part of DGScores.
**
** GNU General Public License Usage
** This file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact the 
** original author.
**
****************************************************************************/

#include "dgsscoremodel.h"

DGSScoreModel::DGSScoreModel(QObject *parent, QSqlDatabase db) :
        QSqlTableModel(new QSqlTableModel(parent, db))
{
    qDebug() << "Initializing the scores model";
    this->setTable("scores");
    this->setSort(1, Qt::AscendingOrder);

    if (!this->select())
    {
        QString errMsg = QString("") + __FUNCTION__ + "Scores model:" + db.lastError().text() + this->lastError().text();
        qDebug() << errMsg;
    }
}

bool DGSScoreModel::saveScores(int round_id, QList<int> hole_ids, QList<int> scores)
{
    //Save the scores.
    qDebug() << "Saving scores for round_id:"<< round_id;
    for (int i = 0; i<scores.count(); i++)
    {
        if ( !this->database().isOpen() )
        {
            qDebug() << "DB error @" << "DGSScoreModel::" <<__FUNCTION__;
            return false;
        }
        QSqlQuery query;
        query.prepare(QString("INSERT INTO scores(round_id, hole_id, score) VALUES(?,?,?)"));
        query.bindValue(0, round_id);
        query.bindValue(1, hole_ids.at(i));
        query.bindValue(2, scores.at(i));
        query.exec();

        if (query.lastError().isValid())
        {
            qDebug() << "Failure saving scores at DGSScoreModel::" <<__FUNCTION__ << query.lastError().text();
            return false;
        }
    }
    select();
    qDebug() << "Scores saved";
    return true;
}

int DGSScoreModel::getRoundScore(int round_id) const
{
    int score = 0;
    QList<int> scores = getRoundScores(round_id);
    foreach (int s, scores)
    {
        score = score + s;
    }
    return score;
}

//TODO: It's not guaranteed that the scores are given in the same order as the holes are
QList<int> DGSScoreModel::getRoundScores(int round_id) const
{
    //Get scores for round_id and sum them
    QList<int> scores;

    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSScoreModel::" <<__FUNCTION__;
        return scores;
    }
    QSqlQuery query;
    query.prepare("SELECT score FROM scores WHERE round_id = :parameter");
    query.bindValue(":parameter", round_id);
    query.exec();
    if (query.lastError().isValid())
    {
        qDebug() << "Error getting scores at" << "DGSScoreModel::" <<__FUNCTION__ << query.lastError();
    }
    while (query.next())
    {
        scores.append(query.value(0).toInt());
    }

    return scores;
}

bool DGSScoreModel::deleteRoundScores(int round_id)
{
    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSScoreModel::" <<__FUNCTION__;
        return false;
    }
    QSqlQuery query;
    query.prepare(QString("DELETE FROM scores WHERE round_id = ? "));
    query.bindValue(0, round_id);
    query.exec();

    this->select();
    if (query.lastError().isValid())
    {
        qDebug() << "Deleting scores failed at DGSScoreModel::" <<__FUNCTION__ << query.lastError();
        return false;
    }
    return true;
}
