/****************************************************************************
**
** Copyright (C) 2010 Mikko Mattila
** Contact: (Gmail: mattila.mikko)
**
** This file is part of DGScores.
**
** GNU General Public License Usage
** This file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact the 
** original author.
**
****************************************************************************/

#include "dgsholemodel.h"

DGSHoleModel::DGSHoleModel(QObject *parent, QSqlDatabase db) :
        QSqlRelationalTableModel(new QSqlRelationalTableModel(parent, db))
{
    qDebug() << "Initializing the holes model";
    this->setTable("holes");
    this->setSort(1, Qt::AscendingOrder);
    this->setHeaderData(1, Qt::Horizontal, tr("Course") ,0);
    this->setHeaderData(2, Qt::Horizontal, tr("Score") ,0);
    this->setRelation(1, QSqlRelation("courses", "course_id", "name"));

    if (!this->select())
    {
        QString errMsg = QString("") + __FUNCTION__ + "Scores model:" + db.lastError().text() + this->lastError().text();
        qDebug() << errMsg;
    }
}

int DGSHoleModel::getHoleCount(int course_id) const
{
    QList<int> pars = getPars(course_id);
    return pars.count();
}

int DGSHoleModel::getCourseParScore(int course_id) const
{
    int totalPar = 0;
    QList<int> pars = getPars(course_id);
    for(int i=0; i<pars.size();i++)
    {
        totalPar = totalPar + pars.at(i);
    }
    return totalPar;
}

QList<int> DGSHoleModel::getPars(int course_id) const
{
    return getHoleDetails(course_id, QString("par"));
}

QList<int> DGSHoleModel::getLengths(int course_id) const
{
    return getHoleDetails(course_id, QString("length"));
}

QList<int> DGSHoleModel::getHoleDetails(int course_id, QString fieldName) const
{
    QList<int> pars;
    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSHoleModel::" <<__FUNCTION__;
        return pars;
    }
    QSqlQuery query;
    query.prepare("SELECT "+ fieldName +" FROM holes WHERE course_id = :parameter");
    query.bindValue(":parameter", course_id);
    query.exec();
    while(query.next())
    {
        int par = (int)query.value(0).toDouble();
        pars.append(par);
    }
    return pars;
}

bool DGSHoleModel::replaceHoleDetails(int course_id, QList<int> pars, QList<double> lengths, QList<QString> positions)
{
    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSHoleModel::" <<__FUNCTION__;
        return false;
    }
    QSqlQuery query;
    query.prepare("DELETE FROM holes WHERE course_id = ?");
    query.bindValue(0, course_id);
    if (!query.exec())
    {
        qDebug() << "Deleting old holes failed at DGSHoleModel::" <<__FUNCTION__ << query.lastError();
        return false;
    }
    query.finish();

    qDebug() << "Adding holes to course_id:" << course_id;

    for(int i=0; i<pars.count(); i++)
    {
        query.prepare("INSERT INTO holes(course_id, par, length) VALUES(?, ?, ?)");
        query.bindValue(0, course_id);
        query.bindValue(1, pars.at(i));
        query.bindValue(2, lengths.at(i));
        if (!query.exec())
        {
            qDebug() << "Inserting holes failed at DGSHoleModel::" <<__FUNCTION__ << query.lastError();
            return false;
        }
    }
    return this->select();
}

QList<int> DGSHoleModel::getHoleIds(int course_id) const
{
    QList <int> list;
    list = getHoleDetails(course_id, "hole_id");
    return list;
}
