/****************************************************************************
**
** Copyright (C) 2010 Mikko Mattila
** Contact: (Gmail: mattila.mikko)
**
** This file is part of DGScores.
**
** GNU General Public License Usage
** This file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact the 
** original author.
**
****************************************************************************/

#include "dgsroundmodel.h"

DGSRoundModel::DGSRoundModel(QObject *parent, QSqlDatabase db) :
        QSqlRelationalTableModel(new QSqlRelationalTableModel(parent, db))
{

    qDebug() << "Initializing the round model";
    this->setTable("rounds");
    this->setSort(1, Qt::AscendingOrder);
    this->setHeaderData(1, Qt::Horizontal, tr("Course") ,0);
    this->setHeaderData(2, Qt::Horizontal, tr("Start Time") ,0);
    this->setHeaderData(3, Qt::Horizontal, tr("End Time") ,0);
    this->setRelation(1, QSqlRelation("courses", "course_id", "name"));

    if (!this->select())
    {
        QString errMsg = QString("") + __FUNCTION__ + "Rounds model:" + db.lastError().text() + this->lastError().text();
        qDebug() << errMsg;
    }
}

int DGSRoundModel::saveRound(int course_id, QString startTime, QString endTime)
{
    QSqlQuery query;
    query.prepare("INSERT INTO rounds(course_id, start_time, end_time) VALUES(?, ?, ?)");
    query.bindValue(0, course_id);
    query.bindValue(1, startTime);
    query.bindValue(2, endTime);
    if (query.exec())
    {
        int round_id = query.lastInsertId().toInt();
        qDebug() << "Round saved, round_id:" << round_id;
        if (round_id > 0)
        {
            this->select();
            return round_id;
        }
    }
    qDebug() << "Saving round failed at DGSRoundModel::" <<__FUNCTION__ << query.lastError();
    return -1;
}

bool DGSRoundModel::deleteRound(int round_id)
{
    qDebug() << "Deleting round" << round_id  << "from db...";
    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSRoundModel::" <<__FUNCTION__;
        return false;
    }
    QSqlQuery query;
    query.prepare(QString("DELETE FROM rounds WHERE round_id = ? "));
    query.bindValue(0, round_id);
    query.exec();

    this->select();
    if (query.lastError().isValid())
    {
        qDebug() << "Deleting course failed at DGSHoleModel::" <<__FUNCTION__ << query.lastError();
        return false;
    }
    return true;
}

QList<int> DGSRoundModel::getRounds(int course_id) const
{
    QList<int> rounds;
    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSRoundModel::" <<__FUNCTION__;
        return rounds;
    }
    QSqlQuery query;
    query.prepare("SELECT round_id FROM rounds WHERE course_id = :parameter");
    query.bindValue(":parameter", course_id);
    query.exec();
    while(query.next())
    {
        int round = (int)query.value(0).toDouble();
        rounds.append(round);
    }
    return rounds;
}

int DGSRoundModel::getCourseId(int round_id) const
{
    int course_id = -1;
    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSRoundModel::" <<__FUNCTION__;
        return -1;
    }
    QSqlQuery query;
    query.prepare("SELECT course_id FROM rounds WHERE round_id = :parameter");
    query.bindValue(":parameter", round_id);
    query.exec();
    query.next();
    if (query.isValid())
    {
        course_id = query.value(0).toInt();
    }

    if (course_id > 0)
    {
        return course_id;
    }
    qDebug() << "Cannot find valid course for round_id" << round_id;
    return -1;
}

QDateTime DGSRoundModel::getStartTime(int round_id) const
{
    QDateTime time;
    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSRoundModel::" <<__FUNCTION__;
        return time;
    }
    QSqlQuery query;
    query.prepare("SELECT start_time FROM rounds WHERE round_id = :parameter");
    query.bindValue(":parameter", round_id);
    query.exec();
    query.next();
    if (query.isValid())
    {
        time = query.value(0).toDateTime();
    }
    return time;
}

QDateTime DGSRoundModel::getEndTime(int round_id) const
{
    QDateTime time;
    if ( !this->database().isOpen() )
    {
        qDebug() << "DB error @" << "DGSRoundModel::" <<__FUNCTION__;
        return time;
    }
    QSqlQuery query;
    query.prepare("SELECT end_time FROM rounds WHERE round_id = :parameter");
    query.bindValue(":parameter", round_id);
    query.exec();
    query.next();
    if (query.isValid())
    {
        time = query.value(0).toDateTime();
    }
    return time;
}
