/****************************************************************************
**
** Copyright (C) 2010 Mikko Mattila
** Contact: (Gmail: mattila.mikko)
**
** This file is part of DGScores.
**
** GNU General Public License Usage
** This file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact the 
** original author.
**
****************************************************************************/

#include "dgsdata.h"

DGSData::DGSData(QWidget *parent)
{
    m_isValid = true;
    
    // If the database file is not where it's supposed to be,
    // use the template from a resource file
    
    QString dbFile = "dgs.db";
    QString dbPath = "";
    
#if defined(Q_WS_MAEMO_5)
    dbPath = QDir::homePath() + "/.dgscores/";
#endif
    
    QFile file(dbPath + dbFile);
    if (!file.exists(dbPath + dbFile))
    {
        QDir dir(dbPath);
        if (!dir.exists(dbPath))
        {
            qDebug() << "Creating Directory" << dbPath;
            dir.mkdir(dbPath);
        }
        QFile::copy(":/db/dgs.db", dbPath + dbFile);
        QFile::setPermissions(dbPath + dbFile,QFile::ReadUser | QFile::WriteUser);
        qDebug() << "DB not found. Using a template DB.";
        qDebug() << "Saving DB file to" << dbPath;
    }
    else
    {
        qDebug() << "Found a DB file at " << dbPath + dbFile;
    }
    createDbConnection();
    initModels();

    //    for (int i=0; i<10; i++)
    //    {
    //      createRandomCourse();
    //    }
    //    for (int i=0; i<50; i++)
    //    {
    //      createRandomRound();
    //    }
}

DGSData::~DGSData()
{
    m_courses->submitAll();
    m_scores->submitAll();
    m_rounds->submitAll();
    m_holes->submitAll();
    removeDbConnection();
}

bool DGSData::createDbConnection()
{
    QString dbPath = "./dgs.db";
    
#if defined(Q_WS_MAEMO_5)
    dbPath = QDir::homePath() + "/.dgscores/dgs.db";
#endif
    
    m_db = QSqlDatabase::addDatabase("QSQLITE");
    m_db.setDatabaseName(dbPath);
    if (!m_db.open()) {
        qDebug() << "createDbConnection() FAIL:" << m_db.lastError();
        emit error(m_db.lastError().text());
        return false;
    }
    qDebug() << "DB connection up";
    return true;
}

void DGSData::removeDbConnection()
{
    
    m_db.close();
    m_db.removeDatabase("QSQLITE");
    qDebug() << "DB connection down with lastError:" << m_db.lastError().text();
}

void DGSData::createRandomCourse()
{
    QStringList locations = (QStringList() << "Sauvo" << "Pälkäne" << "Kaarina" << "Vaasa" << "Rovaniemi" << "Rauma" << "Eckerö" << "Sastamala" << "Tampere" << "Kolari" << "Keuruu" << "Töysä" << "Hamina" << "Kotka" << "Turku" << "Hattula" << "Tampere" << "Kalajoki" << "Oulu" << "Karttula" << "Sastamala" << "Nurmijärvi" << "Ähtäri" << "Porvoo" << "Kauhava" << "Ikaalinen" << "Inkoo" << "Masku" << "Tampere" << "Kristiinankaupunki" << "Kirkkonummi" << "Tuusula" << "Seinäjoki" << "Joutsa");
    QStringList names = (QStringList() << "Nokia DiscGolfPark" << "Nummelanharjun frisbeegolfrata" << "Oittaan ulkoilualue" << "Otanlahti" << "Oulunsalo" << "Pahkavuori" << "Paimio DiscGolfPark" << "Pajulahti" << "Pajuniitty" << "Pappilanmetsä" << "Parainen Disc Golf Park" << "Peltosalmen urheilupuisto" << "Petäjämäen urheilukeskus" << "Peurunka" << "Piispalan matkailukeskus" << "Pikisaaren frisbeegolfrata" << "Pitkämö Camping" << "Päiväkumpu" << "Rajamaanrannan lähiliikuntapaikka" << "Rajamäen liikuntapuisto" << "Rantakylän virkistysalue" << "Rantapirtti" << "Raudaskylä" << "Ristijärvi DiscGolfPark" << "Ruka Adventures" << "Rämäkkävuori" << "Sampolan lähiliikuntapaikka" << "Sappee DiscGolfPark" << "Satamanpuisto" << "Savelanpuisto" << "Siilinlahden frisbeegolfrata" << "Siltamäen frisbeegolfrata" << "Simonmetsä" << "Siuntion kylpylä");

    int number = (int) ((double)locations.count() * (qrand() / (RAND_MAX + 1.0)));
    QString name = names[number];
    QString location = locations[number];
    QString notes = "";
    QString map_image = "";

    QSqlQuery query;
    query.prepare("INSERT INTO courses(name, location, notes, map_image) VALUES(?, ?, ?, ?)");
    query.bindValue(0, name);
    query.bindValue(1, location);
    query.bindValue(2, notes);
    query.bindValue(3, map_image);
    int course_id = 0;
    if (query.exec())
    {
        course_id = query.lastInsertId().toInt();
        qDebug() << "Course saved, course_id:" << course_id;
    }
    else
    {
        qDebug() << "Fail";
        return;
    }

    int holeCount = 1 + (int) (18 * (qrand() / (RAND_MAX + 1.0)));

    QList<int> pars;
    QList<double> lengths;
    QList<QString> positions;

    for (int i=0; i<holeCount; i++)
    {
        pars.append(2 + (int) (6 * (qrand() / (RAND_MAX + 1.0))));
        lengths.append(0 + (int) (400 * (qrand() / (RAND_MAX + 1.0))));
        positions.append("");
    }
    //Add holes
    m_holes->replaceHoleDetails(course_id, pars, lengths, positions);

}

void DGSData::createRandomRound()
{
    int duration = 240 + (int) (18000 * (qrand() / (RAND_MAX + 1.0)));
    int course_id = 29;
    QList<int> courses;
    for (int i=0; i<m_courses->rowCount(); i++)
    {
        courses.append(m_courses->index(i, 0).data(0).toInt());
    }
    course_id = courses.at((int) (courses.count() * (qrand() / (RAND_MAX + 1.0))));
    QDateTime startTime = QDateTime().currentDateTime().addSecs(-duration*1000);
    QDateTime endTime = startTime.addSecs(duration);
    int round_id = m_rounds->saveRound(course_id, startTime.toString(QSettings().value("defaults/dateFormat").toString()), endTime.toString(QSettings().value("defaults/dateFormat").toString()));
    QList<int> hole_ids = m_holes->getHoleIds(course_id);
    QList<int> scores;

    for(int i=0; i<hole_ids.count(); i++)
    {
        scores.append(1 + (int) (11 * (qrand() / (RAND_MAX + 1.0))));
    }
    m_scores->saveScores(round_id, hole_ids, scores);

}

void DGSData::initModels()
{
    if ( !m_db.isOpen() )
    {
        QString errMsg = QString("DB error @") + __FUNCTION__ + m_db.lastError().text();
        qDebug() << errMsg;
        emit error(errMsg);
        return;
    }
    m_courses = new DGSCourseModel(this, m_db);
    if (m_courses->lastError().isValid())
    {
        m_isValid = false;
        emit error(m_courses->lastError().text());
        return;
    }
    m_scores = new DGSScoreModel(this, m_db);
    if (m_scores->lastError().isValid())
    {
        m_isValid = false;
        emit error(m_scores->lastError().text());
        return;
    }
    m_holes = new DGSHoleModel(this, m_db);
    if (m_holes->lastError().isValid())
    {
        m_isValid = false;
        emit error(m_holes->lastError().text());
        return;
    }
    m_rounds = new DGSRoundModel(this, m_db);
    if (m_rounds->lastError().isValid())
    {
        m_isValid = false;
        emit error(m_rounds->lastError().text());
        return;
    }
}

bool DGSData::isValid()
{
    return m_isValid;
}

void DGSData::setCurrentCourse(int id)
{
    if (id == -1)
    {
        qDebug() << "Current course unset in DGSData";
    }
    else if (!courseIsValid(id))
    {
        qDebug() << "Invalid course id received at " << __FUNCTION__ << ":" << id;
        emit error(QString("Invalid course id received at ") + __FUNCTION__ + ":" + id);
        return;
    }
    else
    {
        m_currentCourseId = id;
        qDebug() << "Course selected with id" << id;        
    }
}

void DGSData::setCurrentCourse(QString name)
{
    setCurrentCourse(m_courses->getCourseId(name));
}

void DGSData::setCurrentRound(int id)
{
    if (id == -1)
    {
        qDebug() << "Current round unset in DGSData";
        m_currentRoundId = id;
    }
    else if (!roundIsValid(id))
    {
        qDebug() << "Invalid round id received at " << __FUNCTION__ << ":" << id;
        emit error(QString("Invalid round id received at ") + __FUNCTION__ + ":" + id);
        return;
    }
    //If id is valid and not -1
    else
    {
        m_currentRoundId = id;
        qDebug() << "Round selected with id" << m_currentRoundId;
        setCurrentCourse(m_rounds->getCourseId(id));
    }
}

 int DGSData::currentCourse()
{
    return m_currentCourseId;
}

 int DGSData::currentRound()
{
    return m_currentRoundId;
}

bool DGSData::roundIsValid(int id)
{
    //If function was called without the parameter, we'll use the current round.
    if (id == -1)
    {
        id = m_currentRoundId;
    }
    if ( !m_rounds->getStartTime(id).isNull() )
    {
        return true;
    }
    return false;
}

bool DGSData::courseIsValid(int id)
{
    //If function was called without the parameter, we'll use the current course.
    if (id == -1)
    {
        id = m_currentCourseId;
    }
    if (m_courses->getCourseName(id) != "")
    {
        return true;
    }
    return false;
}

void DGSData::setPlayStartTime()
{
    m_playStartTime = QDateTime::currentDateTime();
}

int DGSData::addCourseRow()
{
    int row = 0;
    if (!m_courses->insertRow(row))
    {
        qDebug() << "Failed to insert row: " << row;
    }
    
    m_courses->setData(m_courses->index(row, 1), "<course name>");
    //Do submit, because we need to get the course_id.
    m_courses->submitAll();
    
    int course_id = m_courses->data(m_courses->index(row, 0),0).toInt();
    setCurrentCourse(course_id);
    qDebug() << "Added new course with id:" << m_currentCourseId;
    return row;
}

void DGSData::removeSelectedCourse()
{
    qDebug() << "Removing the course map";
    if (courseMapPath() != "")
    {
        QFile().remove(courseMapPath());
    }
    qDebug() << "Removing the selected course";
    //First delete all the scores that relate to rounds for the current course.
    QList<int> rounds = m_rounds->getRounds(m_currentCourseId);
    foreach (int round_id, rounds)
    {
        if (!m_scores->deleteRoundScores(round_id))
        {
            emit error(m_scores->lastError().text());
        }
    }
    if (!m_courses->removeCourse(m_currentCourseId))
    {
        emit error(m_courses->lastError().text());
    }
    setCurrentCourse(-1);

}

void DGSData::removeSelectedRound()
{
    qDebug() << "Removing the round" << m_currentRoundId << "from the course: " << courseName();
    if (!m_scores->deleteRoundScores(m_currentRoundId))
    {
        emit error(m_scores->lastError().text());
    }
    if (!m_rounds->deleteRound(m_currentRoundId))
    {
        emit error(m_scores->lastError().text());
    }
    setCurrentRound(-1);
}

void DGSData::saveCourseChanges(QList<int> pars, QList<double> lengths, QList<QString> positions)
{
    if(!m_courses->submitAll())
    {
        emit error(m_courses->lastError().text());
    }
    qDebug() << "Saving details for course_id:" << m_currentCourseId;
    if (!m_holes->replaceHoleDetails(m_currentCourseId, pars, lengths, positions))
    {
        emit error(m_holes->lastError().text());
    }
}

void DGSData::cancelCourseChanges()
{
    if (courseName() == tr("<course name>") || courseName() == "")
    {
        removeSelectedCourse();
    }
    m_courses->select();
}

bool DGSData::saveScores(QList<int> scores)
{
    QString format = QSettings().value("defaults/dateFormat").toString();
    int round_id = m_rounds->saveRound(m_currentCourseId, m_playStartTime.toString(format), QDateTime::currentDateTime().toString(format));
    if (round_id < 1)
    {
        emit error("Failed to save the round" + m_rounds->lastError().text());
        return false;
    }
    //Get hole ids and save scores
    if (!m_scores->saveScores(round_id, m_holes->getHoleIds(m_currentCourseId), scores))
    {
        emit error("Failed to save scores" + m_scores->lastError().text());
        return false;
    }
    return true;
}

DGSCourseModel* DGSData::getCourseModel()
{
    return m_courses;
}

QSqlQueryModel* DGSData::getScoreViewModel(QModelIndex index)
{
    int row = index.row();
    QModelIndex round = m_roundViewModel->index(row, 0);
    int round_id = m_roundViewModel->data(round,0).toInt();
    
    QSqlQueryModel *model = new QSqlQueryModel;
    QString query = QString("SELECT score FROM scores WHERE round_id = ") + QString().setNum(round_id);
    model->setQuery(query);
    if (model->lastError().isValid())
    {
        emit error(QString("Failure at ") + __FUNCTION__ + model->lastError().text());
    }
    return model;
}

QSqlQueryModel* DGSData::getHoleViewModel()
{    
    qDebug() << "Getting pars for course id: " << m_currentCourseId;
    
    QSqlQueryModel *model = new QSqlQueryModel;
    QString query = QString("SELECT par FROM holes WHERE course_id = ") + QString().setNum(m_currentCourseId);
    model->setQuery(query);
    if (model->lastError().isValid())
    {
        emit error(QString("Failure at ") + __FUNCTION__ + model->lastError().text());
    }
    return model;
}

QSqlQueryModel* DGSData::getRoundViewModel(int course_id)
{
    qDebug() << "Getting rounds for course id: " << course_id;
    QSqlQueryModel *model = new QSqlQueryModel;
    QString query = QString("SELECT rounds.round_id, rounds.course_id, courses.name, rounds.start_time FROM courses, rounds WHERE courses.course_id=rounds.course_id");
    if (course_id > 0)
    {
        query.append(" AND rounds.course_id =");
        query.append(QString().setNum(course_id));
    }
    model->setQuery(query);
    model->setHeaderData(2, Qt::Horizontal, "Course", Qt::DisplayRole);
    model->setHeaderData(3, Qt::Horizontal, "Start Time", Qt::DisplayRole);
    
    m_roundViewModel = model;
    return model;
}

QSqlQueryModel* DGSData::getRoundViewModel(QString courseName)
{
    if(!courseName.contains("<"))
    {
        return getRoundViewModel(m_courses->getCourseId(courseName));
    }
    return getRoundViewModel();
}

int DGSData::courseRecord() const
{
    int record = INT_MAX; //At least 32767
    
    //Find all round_ids for course_id
    QList<int> rounds = m_rounds->getRounds(m_currentCourseId);
    
    //Return the smallest value
    foreach (int round_id, rounds)
    {
        int currentRoundScore = m_scores->getRoundScore(round_id);
        
        if(currentRoundScore < record)
        {
            record = currentRoundScore;
        }
    }
    
    int par = m_holes->getCourseParScore(m_currentCourseId);
    // If there is a valid record.
    if (record != INT_MAX)
    {
        return record - par;
    }
    else
    {
        return INT_MAX;
    }
}

QList<int> DGSData::coursePars()
{
    QList<int> pars = m_holes->getPars(m_currentCourseId);
    if (pars.isEmpty())
    {
        emit error("Error getting course pars" + m_holes->lastError().text());
    }
    return pars;
}

QList<int> DGSData::courseLengths()
{
    QList<int> lengths = m_holes->getLengths(m_currentCourseId);
    if (lengths.isEmpty())
    {
        emit error("Error getting course lengths" + m_holes->lastError().text());
    }
    return lengths;
}

int DGSData::courseHoleCount() const
{
    return m_holes->getHoleCount(m_currentCourseId);
}

QString DGSData::courseName() const
{
    return m_courses->getCourseName(m_currentCourseId);
}

QString DGSData::courseName(int course_id) const
{
    return m_courses->getCourseName(course_id);
}

int DGSData::courseId(QString courseName) const
{
    return m_courses->getCourseId(courseName);
}

QString DGSData::courseLocation() const
{
    return m_courses->getCourseLocation(m_currentCourseId);
}

QString DGSData::courseNotes() const
{
    return m_courses->getCourseNotes(m_currentCourseId);
}

QString DGSData::courseMapPath() const
{
    return m_courses->getCourseMapPath(m_currentCourseId);
}

int DGSData::courseParScore()
{
    int total = 0;
    QList<int> pars = this->coursePars();
    for (int i=0; i<pars.count(); i++)
    {
        if (pars.at(i) > 0)
        {
            total = total + pars.at(i);
        }
    }
    return total;
}

QList<int> DGSData::roundScores() const
{
    QList<int> scores = m_scores->getRoundScores(m_currentRoundId);
    // If scores are missing, add them so
    // that the result has the same amount of holes, as the real hole count
    for (int i = scores.count(); i<courseHoleCount(); i++)
    {
        scores.append(0);
    }
    return scores;
}

int DGSData::roundScore()
{
    int score = 0;
    QList<int> scores = this->roundScores();
    foreach (int s, scores)
    {
        score = score + s;
    }
    score = score - courseParScore();

    return score;
}

QString DGSData::roundDuration() const
{
    QDateTime endTime = m_rounds->getEndTime(m_currentRoundId);
    QDateTime startTime = m_rounds->getStartTime(m_currentRoundId);
    
    int total_seconds = startTime.secsTo(endTime);
    
    int seconds = total_seconds % 60;
    int total_minutes = total_seconds / 60;
    int minutes = total_minutes % 60;
    int total_hours = total_minutes / 60;
    
    QString hh = QString().setNum(total_hours);
    QString mm = QString().setNum(minutes);
    QString ss = QString().setNum(seconds);
    
    QString duration = QString("%1:%2:%3").arg(hh, 2, QChar('0')).arg(mm, 2, QChar('0')).arg(ss, 2, QChar('0'));
    return duration;
}
