#include <iostream>
#include <limits>

#include <Lum/Base/DateTime.h>
#include <Lum/Base/L10N.h>
#include <Lum/Base/Path.h>
#include <Lum/Base/Size.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Calendar.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Time.h>

#include <Lum/OS/Main.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/DateSelect.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/String.h>
#include <Lum/Tab.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/TimeSelect.h>

#include "config.h"

static Lum::Def::AppInfo info;

class MainWindow : public Lum::Dialog
{
private:
  Lum::Model::ActionRef    convertUnix;
  Lum::Model::ActionRef    convertLocal;
  Lum::Model::ActionRef    about;

  Lum::Model::StringRef    unixInput;
  Lum::Model::StringRef    utcResult;
  Lum::Model::StringRef    localResult;

  Lum::Model::CalendarRef  dateInput;
  Lum::Model::TimeRef      timeInput;
  Lum::Model::StringRef    unixResult;

public:
  MainWindow()
  : convertUnix(new Lum::Model::Action()),
    convertLocal(new Lum::Model::Action()),
    about(new Lum::Model::Action()),
    unixInput(new Lum::Model::String(Lum::Base::NumberToWString(Lum::Base::SystemTime().GetTime()))),
    utcResult(new Lum::Model::String(L"")),
    localResult(new Lum::Model::String(L"")),
    dateInput(new Lum::Model::Calendar()),
    timeInput(new Lum::Model::Time()),
    unixResult(new Lum::Model::String(L""))
  {
    dateInput->SetToCurrent();
    timeInput->SetToNow();

    ConvertUnix();
    ConvertLocal();

    Observe(convertUnix);
    Observe(convertLocal);
    Observe(about);
  }

  void PreInit()
  {
    Lum::DateSelect       *dateSel;
    Lum::Label            *label;
    Lum::Panel            *panel;
    Lum::String           *string;
    Lum::Tab              *tab;
    Lum::Text             *text;
    Lum::TextValue        *value;
    Lum::TimeSelect       *timeSel;

    Lum::Base::SystemTime now;
    std::wstring          tmp;

    tab=Lum::Tab::Create(true,true);

    panel=Lum::VPanel::Create(true,false);

    label=Lum::Label::Create(true,false);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,Lum::Base::NumberDigits(std::numeric_limits<time_t>::max()));
    string->SetModel(unixResult);
    string->SetAlignment(Lum::String::right);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeDec));
    label->AddLabel(_(L"UNIX_TIME",L"Unix time:"),string);

    value=Lum::TextValue::Create(utcResult,Lum::TextValue::right);
    value->SetMinWidth(Lum::Base::Size::stdCharWidth,now.GetUTCLocaleDateTime().length());
    label->AddLabel(_(L"UTC_TIME",L"UTC time:"),value);

    value=Lum::TextValue::Create(localResult,Lum::TextValue::right);
    value->SetMinWidth(Lum::Base::Size::stdCharWidth,now.GetLocalLocaleDateTime().length());
    label->AddLabel(_(L"LOCAL_TIME",L"Local time:"),value);

    panel->Add(label);

    panel->AddSpace();
    panel->Add(Lum::ButtonRow::Create(true,false)
               ->Add(Lum::Button::Create(_(L"CONVERT",L"_Convert"),convertUnix,true,false)));

    tab->Add(_(L"FROM_UNIX",L"From Unix"),panel);

    panel=Lum::VPanel::Create(true,false);

    label=Lum::Label::Create(true,false);

    dateSel=new Lum::DateSelect();
    dateSel->SetModel(dateInput);
    label->AddLabel(_(L"LOCAL_DATE",L"Local date:"),dateSel);

    timeSel=new Lum::TimeSelect();
    timeSel->SetModel(timeInput);
    label->AddLabel(_(L"LOCAL_TIMESTAMP",L"Local time:"),timeSel);

    value=Lum::TextValue::Create(unixResult,Lum::TextValue::right);
    value->SetMinWidth(Lum::Base::Size::stdCharWidth,Lum::Base::NumberDigits(std::numeric_limits<time_t>::max()));
    label->AddLabel(_(L"UNIX_TIME",L"Unix time:"),value);

    panel->Add(label);
    panel->AddSpace();
    panel->Add(Lum::ButtonRow::Create(true,false)
               ->Add(Lum::Button::Create(_(L"CONVERT",L"_Convert"),convertLocal,true,false)));

    tab->Add(_(L"TO_UNIX",L"To Unix"),panel);

    tmp=_(L"DIFFERENCE",L"Difference between local time and UTC: ");
    tmp.append(Lum::Base::NumberToWString(now.GetDifferenceToUTC()));
    tmp.append(L" ");
    tmp.append(_(L"SECONDS",L" seconds"));
    tmp.append(L"\n\n");
    if (now.IsDSTActive()) {
      tmp.append(_(L"DST_ACTIVE",L"DST is active"));
      tmp.append(L" (");
      tmp.append(Lum::Base::NumberToWString(now.GetDST()));
      tmp.append(L" ");
      tmp.append(_(L"SECONDS",L" seconds"));
      tmp.append(L")");
    }
    else {
      tmp.append(_(L"DST_INACTIVE",L"is not active"));
    }

    text=new Lum::Text(tmp);
    text->SetFlex(true,true);
    tab->Add(_(L"INFO",L"Info"),text);

    SetMain(Lum::VPanel::Create(true,true)
            ->Add(tab)
            ->AddSpace()
            ->Add(Lum::ButtonRow::Create(true,false)
                  ->AddOptional(Lum::Button::Create(_ld(dlgButtonAbout),about,true,false))
                  ->Add(Lum::Button::CreateClose(GetClosedAction(),true,false))));

    Dialog::PreInit();
  }

  void ConvertUnix()
  {
    if (!unixInput.Valid() || unixInput->IsNull()) {
      utcResult->Set(L"No valid input");
      localResult->Set(L"No valid input");
      return;
    }

    time_t t;

    if (!Lum::Base::WStringToNumber(unixInput->Get(),t)) {
      utcResult->Set(L"No valid number");
      localResult->Set(L"No valid number");
      return;
    }

    Lum::Base::SystemTime sysTime(t);

    utcResult->Set(sysTime.GetUTCLocaleDateTime());
    localResult->Set(sysTime.GetLocalLocaleDateTime());
  }

  void ConvertLocal()
  {
    Lum::Base::SystemTime time;

    try {
      time.SetLocalDateTime(timeInput->Get().GetHour(),timeInput->Get().GetMinute(),timeInput->Get().GetSecond(),0,
                            dateInput->Get().GetDayOfMonth(),dateInput->Get().GetMonth(),dateInput->Get().GetYear());
    }
    catch (...) {
      Lum::Dlg::Msg::ShowOk(this,
                            L"Conversion error!",
                            L"The given date and time could not be converted!\n"
                            L"\n"
                            L"This is likely a problem with the date being out side the range\n"
                            L"of the OS internal system time data type!");
      time.SetToNow();
    }

    unixResult->Set(Lum::Base::NumberToWString(time.GetTime()));
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==convertUnix && convertUnix->IsFinished()) {
      ConvertUnix();
    }
    else if (model==convertLocal && convertLocal->IsFinished()) {
      ConvertLocal();
    }
    else if (model==about && about->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::OS::MainDialog<MainWindow>
{
public:
  Main()
  {
#if defined(APP_DATADIR)
    Lum::Base::Path::SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"Some date conversion routines...");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License");
  }
};

LUM_MAIN(Main,L"DateJinni")

