#include "urlgrabber.h"
#include "cookiejar.h"
#include "json.h"
#include "utils.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QStringList>
#include <QRegExp>
#include <QScriptEngine>
#include <QDebug>

using namespace QtJson;

UrlGrabber* grabberInstance = 0;
QScriptEngine* youtubeDecryptionEngine = 0;
QHash<QUrl, QScriptValue> youtubeDecryptionCache;

UrlGrabber::UrlGrabber(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_format(Videos::Unknown),
    m_busy(false),
    m_cancelled(false)
{
    if (!grabberInstance) {
        grabberInstance = this;
        youtubeDecryptionEngine = new QScriptEngine(this);
    }
}

UrlGrabber* UrlGrabber::instance() {
    return grabberInstance;
}

void UrlGrabber::setYouTubeFormats(QSet<int> formats) {
    m_youtubeFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::LQ:
            videoFormat.value = 18;
            videoFormat.displayName = QString("%1 AVC1").arg(tr("Normal (upto 360P)"));
            break;
        case Videos::Normal:
            videoFormat.value = 34;
            videoFormat.displayName = "360P FLV";
            break;
        case Videos::HQ:
            videoFormat.value = 35;
            videoFormat.displayName = "480P FLV";
            break;
        case Videos::HD:
            videoFormat.value = 22;
            videoFormat.displayName = "720P AVC1";
            break;
        case Videos::SuperHD:
            videoFormat.value = 37;
            videoFormat.displayName = "1080P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setYouTubeFormats(): Invalid video format";
            break;
        }

        m_youtubeFormats.append(videoFormat);
    }
}

void UrlGrabber::setDailymotionFormats(QSet<int> formats) {
    m_dailymotionFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::Normal:
            videoFormat.value = "stream_h264_url";
            videoFormat.displayName = QString("%1 AVC1").arg(tr("Normal"));
            break;
        case Videos::HQ:
            videoFormat.value = "stream_h264_hq_url";
            videoFormat.displayName = "480P AVC1";
            break;
        case Videos::HD:
            videoFormat.value = "stream_h264_hd_url";
            videoFormat.displayName = "720P AVC1";
            break;
        case Videos::SuperHD:
            videoFormat.value = "stream_h264_hd1080_url";
            videoFormat.displayName = "1080P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setDailymotionFormats(): Invalid video format";
            break;
        }

        m_dailymotionFormats.append(videoFormat);
    }
}

void UrlGrabber::setVimeoFormats(QSet<int> formats) {
    m_vimeoFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::LQ:
            videoFormat.value = "mobile";
            videoFormat.displayName = tr("Mobile");
            break;
        case Videos::Normal:
            videoFormat.value = "sd";
            videoFormat.displayName = "360P AVC1";
            break;
        case Videos::HD:
            videoFormat.value = "hd";
            videoFormat.displayName = "720P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setVimeoFormats(): Invalid video format";
            break;
        }

        m_vimeoFormats.append(videoFormat);
    }
}

void UrlGrabber::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            this->setCancelled(false);
            emit busy(message, numberOfOperations);
        }
        else if (!this->cancelled()) {
            emit busyProgressChanged(numberOfOperations);
        }

        emit busyChanged(isBusy);
    }
}

void UrlGrabber::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), 0, 0);
    emit currentOperationCancelled();
}

void UrlGrabber::getVideoUrl(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getYouTubeVideoUrl(id);
        break;
    case Services::Dailymotion:
        this->getDailymotionVideoUrl(id);
        break;
    case Services::Vimeo:
        this->getVimeoVideoUrl(id);
        break;
    default:
        qWarning() << "UrlGrabber::getVideoUrl(): No/invalid service specied";
        return;
    }
}

QScriptValue UrlGrabber::getYouTubeDecryptionFunction(const QUrl &playerUrl) {
    if (youtubeDecryptionCache.contains(playerUrl)) {
        return youtubeDecryptionCache.value(playerUrl);
    }

    QNetworkRequest request(playerUrl);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addYouTubeDecryptionFunctionToCache()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));

    return QScriptValue();
}

void UrlGrabber::addYouTubeDecryptionFunctionToCache() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), 0, 0);
        return;
    }

    QString response(reply->readAll());
    QString funcName = response.section("signature=", 1, 1).section('(', 0, 0);
    response = QString("function %1%2").arg(funcName).arg(response.section("function " + funcName, 1, 1).section(";function", 0, 0));
    youtubeDecryptionEngine->evaluate(response);

    QScriptValue global = youtubeDecryptionEngine->globalObject();
    QScriptValue decryptionFunction = global.property(funcName);

    if (decryptionFunction.isFunction()) {
        youtubeDecryptionCache[reply->request().url()] = decryptionFunction;
        emit youtubeDecryptionFunctionReady(decryptionFunction);
    }
    else {
        emit error(tr("Unable to decrypt video signature"));
        this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), 0, 0);
    }

    reply->deleteLater();
}

QMap<int, QUrl> UrlGrabber::getYouTubeVideoUrlMap(QString page, QScriptValue decryptionFunction) {
    QMap<int, QUrl> urlMap;
    QStringList parts = page.split(',', QString::SkipEmptyParts);

    if (decryptionFunction.isFunction()) {
        foreach (QString part, parts) {
            part = Utils::unescape(part);
            part.replace(QRegExp("(^|&)s="), "&signature=");
            QString oldSig = part.section("signature=", 1, 1).section('&', 0, 0);
            part.replace(oldSig, decryptionFunction.call(QScriptValue(), QScriptValueList() << oldSig).toString());
            QStringList splitPart = part.split("url=");

            if (!splitPart.isEmpty()) {
                QUrl url(splitPart.last());

                if (!url.hasQueryItem("signature")) {
                    url.addQueryItem("signature", splitPart.first().section("signature=", 1, 1).section('&', 0, 0));
                }

                urlMap[url.queryItemValue("itag").toInt()] = url;
            }
        }
    }
    else {
        foreach (QString part, parts) {
            part = Utils::unescape(part);
            part.replace(QRegExp("(^|&)sig="), "&signature=");
            QStringList splitPart = part.split("url=");

            if (!splitPart.isEmpty()) {
                QUrl url(splitPart.last());

                if (!url.hasQueryItem("signature")) {
                    url.addQueryItem("signature", splitPart.first().section("signature=", 1, 1).section('&', 0, 0));
                }

                urlMap[url.queryItemValue("itag").toInt()] = url;
            }
        }
    }

    return urlMap;
}

void UrlGrabber::getYouTubeVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    this->getYouTubeVideoInfoPage(id, SLOT(checkYouTubeVideoInfoPage()));
}

void UrlGrabber::getYouTubeVideoInfoPage(const QString &id, const char *slot) {
    QUrl url("http://www.youtube.com/get_video_info");
    url.addQueryItem("video_id", id);
    url.addQueryItem("el", "detailpage");
    url.addQueryItem("ps", "default");
    url.addQueryItem("eurl", "gl");
    url.addQueryItem("gl", "US");
    url.addQueryItem("hl", "en");
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, slot);
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkYouTubeVideoInfoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map=")) {
        qDebug() << "No format map in YouTube video info page. Retrieving the web page";
        this->getYouTubeVideoWebPage(reply->request().url().queryItemValue("video_id"), SLOT(checkYouTubeWebPage()));
    }
    else {
        response = response.section("url_encoded_fmt_stream_map=", 1, 1);
        QString separator = response.left(response.indexOf('%'));

        if ((separator == "s") || (response.contains("%26s%3D"))) {
            qDebug() << "YouTube video has encrypted signatures. Retrieving the web page";
            this->getYouTubeVideoWebPage(reply->request().url().queryItemValue("video_id"), SLOT(checkYouTubeWebPage()));
        }
        else {
            qDebug() << "YouTube video info OK. Parsing the page";
            response.replace("%2C" + separator, "," + separator);
            this->parseYouTubeVideoPage(QScriptValue(), response);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getYouTubeVideoWebPage(const QString &id, const char *slot) {
    QUrl url("https://www.youtube.com/watch");
    url.addQueryItem("v", id);
    url.addQueryItem("gl", "US");
    url.addQueryItem("hl", "en");
    url.addQueryItem("has_verified", "1");
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, slot);
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkYouTubeWebPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map\":")) {
        qDebug() << response;

        this->setBusy(false);
        QString message = response.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
        emit error(message.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : message);
    }
    else {
        QVariantMap assets = Json::parse(QString("%1}").arg(response.section("\"assets\": ", 1, 1).section('}', 0, 0))).toMap();
        QUrl playerUrl = assets.value("js").toUrl();
        response = response.section("url_encoded_fmt_stream_map\": \"", 1, 1).section(", \"", 0, 0).trimmed().replace("\\u0026", "&").remove(QRegExp("itag=\\d+"));

        bool encryptedSignatures = !response.contains("sig=");

        if (encryptedSignatures) {
            if (playerUrl.isValid()) {
                QScriptValue decryptionFunction = this->getYouTubeDecryptionFunction(playerUrl);

                if (decryptionFunction.isFunction()) {
                    this->parseYouTubeVideoPage(decryptionFunction, response);
                }
                else {
                    m_youtubePage = response;
                    this->connect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), this, SLOT(parseYouTubeVideoPage(QScriptValue)));
                }
            }
            else {
                this->setBusy(false);
                emit error(tr("Unable to decrypt video signature"));
            }
        }
        else {
            this->parseYouTubeVideoPage(QScriptValue(), response);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::parseYouTubeVideoPage(QScriptValue decryptionFunction, QString page) {
    this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), this, SLOT(parseYouTubeVideoPage(QScriptValue)));
    this->setBusy(false);

    if (page.isEmpty()) {
        page = m_youtubePage;
    }

    QMap<int, QUrl> urlMap = this->getYouTubeVideoUrlMap(page, decryptionFunction);
    QUrl videoUrl;
    int index = 0;

    while ((videoUrl.isEmpty()) && (index < m_youtubeFormats.size())) {
        videoUrl = urlMap.value(m_youtubeFormats.at(index).value.toInt());
        m_format = m_youtubeFormats.at(index).format;
        index++;
    }

    if (videoUrl.isEmpty()) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        emit gotVideoUrl(videoUrl, m_format);
    }
}

void UrlGrabber::getDailymotionVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("http://www.dailymotion.com/embed/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseDailymotionVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseDailymotionVideoPage() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantMap info = Json::parse(response.section("var info = ", 1, 1).section(";\n", 0, 0)).toMap();
    QUrl url;
    int i = 0;

    while ((url.isEmpty()) && (i < m_dailymotionFormats.size())) {
        url.setUrl(info.value(m_dailymotionFormats.at(i).value.toString(), "").toString());
        m_format = m_dailymotionFormats.at(i).format;
        i++;
    }

    if (url.isEmpty()) {
        qDebug() << response;
        QString errorString = info.value("error").toMap().value("message").toString();
        emit error(errorString.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : errorString);
    }
    else {
        emit gotVideoUrl(url, m_format);
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("http://player.vimeo.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVimeoVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseVimeoVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    QVariantMap paramMap = Json::parse(params).toMap();

    if (paramMap.isEmpty()) {
        qDebug() << response;
        this->setBusy(false);
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");
        QString quality;

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QVariantList qualities = formatMap.value(codec).toList();
            int i = 0;

            while ((quality.isEmpty()) && (i < m_vimeoFormats.size())) {
                if (qualities.contains(m_vimeoFormats.at(i).value)) {
                    quality = m_vimeoFormats.at(i).value.toString();
                    m_format = m_vimeoFormats.at(i).format;
                }

                i++;
            }

            QString timeStamp = requestMap.value("timestamp").toString();
            QString signature = requestMap.value("signature").toString();
            QString id = videoMap.value("id").toString();
            QUrl url("http://player.vimeo.com/play_redirect");
            url.addQueryItem("quality", quality);
            url.addQueryItem("clip_id", id);
            url.addQueryItem("time", timeStamp);
            url.addQueryItem("sig", signature);

            if (quality != "mobile") {
                url.addQueryItem("codecs", codec);
            }

            this->getVimeoVideoRedirect(url);
        }
        else {
            this->setBusy(false);
            qDebug() << response;
            emit error(tr("Unable to retrieve video. Access may be restricted"));
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoRedirect(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoVideoRedirect()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkVimeoVideoRedirect() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (redirect.isEmpty()) {
        emit gotVideoUrl(reply->request().url(), m_format);
    }
    else {
        emit gotVideoUrl(redirect, m_format);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVideoFormats(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getAvailableYouTubeVideoFormats(id);
        break;
    case Services::Dailymotion:
        this->getAvailableDailymotionVideoFormats(id);
        break;
    case Services::Vimeo:
        this->getAvailableVimeoVideoFormats(id);
        break;
    default:
        return;
    }
}

void UrlGrabber::getAvailableYouTubeVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    this->getYouTubeVideoInfoPage(id, SLOT(checkYouTubeVideoInfoFormats()));
}

void UrlGrabber::checkYouTubeVideoInfoFormats() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map=")) {
        qDebug() << "No format map in YouTube video info page. Retrieving the web page";
        this->getYouTubeVideoWebPage(reply->request().url().queryItemValue("video_id"), SLOT(checkYouTubeWebPageFormats()));
    }
    else {
        response = response.section("url_encoded_fmt_stream_map=", 1, 1);
        QString separator = response.left(response.indexOf('%'));

        if ((separator == "s") || (response.contains("%26s%3D"))) {
            qDebug() << "YouTube video has encrypted signatures. Retrieving the web page";
            this->getYouTubeVideoWebPage(reply->request().url().queryItemValue("video_id"), SLOT(checkYouTubeWebPageFormats()));
        }
        else {
            qDebug() << "YouTube video info OK. Parsing the page";
            response.replace("%2C" + separator, "," + separator);
            this->parseYouTubeVideoFormats(QScriptValue(), response);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::checkYouTubeWebPageFormats() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map\":")) {
        qDebug() << response;

        QString message = response.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
        emit error(message.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : message);
    }
    else {
        QVariantMap assets = Json::parse(QString("%1}").arg(response.section("\"assets\": ", 1, 1).section('}', 0, 0))).toMap();
        QUrl playerUrl = assets.value("js").toUrl();
        response = response.section("url_encoded_fmt_stream_map\": \"", 1, 1).section(", \"", 0, 0).trimmed().replace("\\u0026", "&").remove(QRegExp("itag=\\d+"));

        bool encryptedSignatures = !response.contains("sig=");

        if (encryptedSignatures) {
            if (playerUrl.isValid()) {
                QScriptValue decryptionFunction = this->getYouTubeDecryptionFunction(playerUrl);

                if (decryptionFunction.isFunction()) {
                    this->parseYouTubeVideoFormats(decryptionFunction, response);
                }
                else {
                    m_youtubePage = response;
                    this->connect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), this, SLOT(parseYouTubeVideoFormats(QScriptValue)));
                }
            }
            else {
                this->setBusy(false);
                emit error(tr("Unable to decrypt video signature"));
            }
        }
        else {
            this->parseYouTubeVideoFormats(QScriptValue(), response);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::parseYouTubeVideoFormats(QScriptValue decryptionFunction, QString page) {
    this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), this, SLOT(parseYouTubeVideoFormats(QScriptValue)));
    this->setBusy(false);

    if (page.isEmpty()) {
        page = m_youtubePage;
    }

    QVariantList formats;
    QList<int> keys;
#ifdef SYMBIAN_OS
    keys << 37 << 22 << 18;
#else
    keys << 37 << 22 << 35 << 34 << 18;
#endif
    QMap<int, QUrl> urlMap = this->getYouTubeVideoUrlMap(page, decryptionFunction);

    foreach (int key, keys) {
        if (urlMap.contains(key)) {
            QVariantMap format;
            format["value"] = key;
            format["url"] = urlMap.value(key);

            switch (key) {
            case 37:
                format["name"] = "1080P AVC1";
                format["service"] = Services::YouTube;
                formats << format;
                break;
            case 22:
                format["name"] = "720P AVC1";
                format["service"] = Services::YouTube;
                formats << format;
                break;
    #ifndef SYMBIAN_OS
            case 35:
                format["name"] = "480P FLV";
                format["service"] = Services::YouTube;
                formats << format;
                break;
            case 34:
                format["name"] = "360P AVC1";
                format["service"] = Services::YouTube;
                formats << format;
                break;
    #endif
            case 18:
                format["name"] = QString("%1 AVC1").arg(tr("Normal"));
                format["service"] = Services::YouTube;
                formats << format;
            default:
                break;
            }
        }
    }

    if (formats.isEmpty()) {
        emit error(tr("Unable to retrieve formats. Access may be restricted"));
    }
    else {
        emit gotVideoFormats(formats);
    }
}

void UrlGrabber::getAvailableDailymotionVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("http://www.dailymotion.com/embed/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkDailymotionVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantMap info = Json::parse(response.section("var info = ", 1, 1).section(";\n", 0, 0)).toMap();
    QVariantList formats;
    int i = 0;

    QMap<QString, QString> formatNames;
    formatNames["stream_h264_url"] = QString("%1 AVC1").arg(tr("Normal"));
    formatNames["stream_h264_hq_url"] = "480P AVC1";
    formatNames["stream_h264_hd_url"] = "720P AVC1";
    formatNames["stream_h264_hd1080_url"] = "1080P AVC1";

    foreach (QString formatName, formatNames.keys()) {
        QUrl url = info.value(formatName).toUrl();

        if (!url.isEmpty()) {
            QVariantMap format;
            format["name"] = formatNames.value(formatName);
            format["service"] = Services::Dailymotion;
            format["url"] = url;
            formats << format;
        }

        i++;
    }

    if (formats.isEmpty()) {
        qDebug() << response;
        QString errorString = info.value("error").toMap().value("message").toString();
        emit error(errorString.isEmpty() ? tr("Unable to retrieve video formats. Access may be restricted") : errorString);
    }
    else {
        emit gotVideoFormats(formats);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVimeoVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("http://player.vimeo.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkVimeoVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("config:", 1, 1).section("};", 0, 0);
    QVariantMap paramMap = Json::parse(params).toMap();

    if (paramMap.isEmpty()) {
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QVariantMap requestMap = paramMap.value("request").toMap();
        QVariantMap videoMap = paramMap.value("video").toMap();
        QVariantMap formatMap = videoMap.value("files").toMap();
        QString codec("h264");

        if (!formatMap.isEmpty()) {
            codec = formatMap.keys().first();

            QStringList qualities = formatMap.value(codec).toStringList();
            QVariantList formats;
            QMap<QString, QString> formatNames;
            formatNames["mobile"] = tr("Mobile");
            formatNames["sd"] = "360P AVC1";
            formatNames["hd"] = "720P AVC1";

            while (!qualities.isEmpty()) {
                QString quality = qualities.takeFirst();
                QString timeStamp = requestMap.value("timestamp").toString();
                QString signature = requestMap.value("signature").toString();
                QString id = videoMap.value("id").toString();

                QVariantMap format;
                format["name"] = formatNames[quality];
                format["service"] = Services::Vimeo;
                QUrl url("http://player.vimeo.com/play_redirect");
                url.addQueryItem("quality", quality);
                url.addQueryItem("clip_id", id);
                url.addQueryItem("time", timeStamp);
                url.addQueryItem("sig", signature);

                if (quality != "mobile") {
                    url.addQueryItem("codecs", codec);
                }

                format["url"] = url;
                formats << format;
            }

            emit gotVideoFormats(formats);
        }
        else {
            qDebug() << response;
            emit error(tr("Unable to retrieve video formats. Access may be restricted"));
        }
    }

    reply->deleteLater();
}
