#include "dailymotionvideolistwindow.h"
#include "dailymotionvideoinfowindow.h"
#include "dailymotionplaylistdialog.h"
#include "videolistdelegate.h"
#include "dailymotionvideoselectionwindow.h"
#include "videoplaybackwindow.h"
#include "../base/dailymotionvideolistmodel.h"
#include "../base/session.h"
#include "listview.h"
#include <QAction>
#include <QLabel>
#include <QMenuBar>
#include <QTimer>

DailymotionVideoListWindow::DailymotionVideoListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_model(new DailymotionVideoListModel(m_session, this)),
    m_group(0),
    m_groupAction(this->menuBar()->addAction(tr("Join group"), this, SLOT(toggleGroupMembership()))),
    m_deleteFromUploadsAction(this->menuBar()->addAction(tr("Delete from uploads"), this, SLOT(deleteVideosFromUploads()))),
    m_addToFavouritesAction(this->menuBar()->addAction(tr("Add to favourites"), this, SLOT(addVideosToFavourites()))),
    m_deleteFromFavouritesAction(this->menuBar()->addAction(tr("Delete from favourites"), this, SLOT(deleteVideosFromFavourites()))),
    m_addToPlaylistAction(this->menuBar()->addAction(tr("Add to playlist"), this, SLOT(addVideosToPlaylist()))),
    m_addToQueueAction(this->menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(this->menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(this->menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_deleteFromUploadsContextAction(m_contextMenu->addAction(tr("Delete from uploads"), this, SLOT(deleteVideoFromUploads()))),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_deleteFromFavouritesContextAction(m_contextMenu->addAction(tr("Delete from favourites"), this, SLOT(deleteVideoFromFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this))
{
    this->setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    this->connect(m_model, SIGNAL(loadingChanged(bool)), this, SLOT(onLoadingChanged(bool)));
    this->connect(m_model, SIGNAL(countChanged(int)), this, SLOT(onCountChanged(int)));
    this->connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    this->connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    this->connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    this->connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));

    this->onOrientationChanged();
}

void DailymotionVideoListWindow::showEvent(QShowEvent *event) {
    StackedWindow::showEvent(event);

    if ((m_model) && (m_model->refreshRequired())) {
        QTimer::singleShot(1000, m_model, SLOT(reset()));
    }
}

void DailymotionVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void DailymotionVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void DailymotionVideoListWindow::setVideoFeed(const QString &feed) {
    m_model->getVideos(feed);
    this->setMenuActions();
}

void DailymotionVideoListWindow::getGroupVideos(QSharedPointer<GroupItem> group) {
    m_group = group;
    m_model->getGroupVideos(m_group.data()->id());
    this->setMenuActions();

    this->connect(m_group.data(), SIGNAL(memberChanged(bool)), this, SLOT(setMenuActions()));
    m_session->dailymotion()->memberOfGroup(m_group.data()->id());
}

void DailymotionVideoListWindow::search(const QString &query, Queries::QueryOrder order, const QString &language) {
    this->setWindowTitle(tr("Search") + QString(" ('%1')").arg(query));
    m_model->search(query, order, language);
    this->setMenuActions();
}

void DailymotionVideoListWindow::setMenuActions() {
    bool signedIn = m_session->dailymotion()->userSignedIn();

    m_groupAction->setEnabled((signedIn) && (!m_group.isNull()));

    if (m_groupAction->isEnabled()) {
        m_groupAction->setText(m_group.data()->member() ? tr("Leave group") : tr("Join group"));
    }

    m_deleteFromUploadsAction->setEnabled((signedIn) && (m_model->feed() == DAILYMOTION_UPLOADS_FEED));
    m_addToFavouritesAction->setEnabled((signedIn) && (m_model->feed() != DAILYMOTION_FAVOURITES_FEED));
    m_deleteFromFavouritesAction->setEnabled((signedIn) && (m_model->feed() == DAILYMOTION_FAVOURITES_FEED));
    m_addToPlaylistAction->setEnabled(signedIn);
    m_deleteFromUploadsContextAction->setVisible((signedIn) && (m_model->feed() == DAILYMOTION_UPLOADS_FEED));
    m_addToFavouritesContextAction->setVisible((signedIn) && (m_model->feed() != DAILYMOTION_FAVOURITES_FEED));
    m_deleteFromFavouritesContextAction->setVisible((signedIn) && (m_model->feed() == DAILYMOTION_FAVOURITES_FEED));
    m_addToPlaylistContextAction->setVisible(signedIn);
}

void DailymotionVideoListWindow::onLoadingChanged(bool loading) {
    this->toggleBusy(loading);
    m_noResultsLabel->setVisible((!loading) && (m_model->rowCount() == 0));
}

void DailymotionVideoListWindow::onCountChanged(int count) {
    m_noResultsLabel->setVisible((count == 0) && (!m_model->loading()));
}

void DailymotionVideoListWindow::toggleGroupMembership() {
    if (m_group.data()->member()) {
        m_session->dailymotion()->leaveGroup(m_group.data()->id());
    }
    else {
        m_session->dailymotion()->joinGroup(m_group.data()->id());
    }
}

void DailymotionVideoListWindow::deleteVideosFromUploads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Delete from uploads"), SLOT(deleteVideosFromUploads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::addVideosToFavourites() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Add to favourites"), SLOT(addVideosToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::deleteVideosFromFavourites() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Delete from favourites"), SLOT(deleteVideosFromFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::addVideosToPlaylist() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Add to playlist"), SLOT(showPlaylistDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::addVideoDownloads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::addAudioDownloads() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::queueVideos() {
    DailymotionVideoSelectionWindow *window = new DailymotionVideoSelectionWindow(m_session, m_model, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void DailymotionVideoListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteFromUploadsContextAction);
}

void DailymotionVideoListWindow::deleteVideoFromUploads() {
    m_session->dailymotion()->deleteFromUploads(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionVideoListWindow::addVideoToFavourites() {
    m_session->dailymotion()->addToFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionVideoListWindow::deleteVideoFromFavourites() {
    m_session->dailymotion()->deleteFromFavourites(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionVideoListWindow::showPlaylistDialog() {
    DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(m_session, this);
    dialog->open();

    this->connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    this->connect(dialog, SIGNAL(addToNewPlaylist(QVariantMap)), this, SLOT(addVideoToNewPlaylist(QVariantMap)));
}

void DailymotionVideoListWindow::addVideoToPlaylist(const QString &id) {
    m_session->dailymotion()->addToPlaylist(QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()), id);
}

void DailymotionVideoListWindow::addVideoToNewPlaylist(const QVariantMap &playlist) {
    m_session->dailymotion()->createPlaylist(playlist, QStringList(m_model->data(m_listView->currentIndex(), AbstractVideoListModel::VideoIdRole).toString()));
}

void DailymotionVideoListWindow::addVideoDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListWindow::addAudioDownload() {
    m_session->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void DailymotionVideoListWindow::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void DailymotionVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        this->showVideoInfo(m_model->get(index.row()));
    }
}

void DailymotionVideoListWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    DailymotionVideoInfoWindow *window = new DailymotionVideoInfoWindow(m_session, video, this);
    window->show();
}

void DailymotionVideoListWindow::playVideo(const QModelIndex &index) {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
        window->show();
    }
    else {
        m_session->urlGrabber()->getDailymotionVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}
