#include "transferlistmodel.h"
#include "session.h"
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

TransferListModel::TransferListModel(Session *session, QObject *parent) :
    QAbstractListModel(parent),
    m_session(session)
{
    QHash<int, QByteArray> roles;
    roles[FileNameRole] = "fileName";
    roles[TitleRole] = "title";
    roles[DownloadPathRole] = "downloadPath";
    roles[StatusRole] = "status";
    roles[StatusTextRole] = "statusText";
    roles[StatusInfoRole] = "statusInfo";
    roles[ProgressRole] = "progress";
    roles[SizeRole] = "size";
    roles[PriorityRole] = "priority";
    roles[PriorityTextRole] = "priorityText";
    roles[ConvertibleToAudioRole] = "convertibleToAudio";
    roles[SaveAsAudioRole] = "saveAsAudio";
    roles[ServiceRole] = "service";
    roles[TransferTypeRole] = "transferType";
    roles[SelectedRole] = "selected";

    this->setRoleNames(roles);

    if (this->session()) {
        for (int i = 0; i < this->session()->transferList()->size(); i++) {
            QSharedPointer<TransferItem> transfer = this->session()->transferList()->at(i);
            this->connect(transfer.data(), SIGNAL(progressChanged(int)), this, SLOT(onTransferProgressChanged()));
            this->connect(transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged()));
            this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged()));
            this->connect(transfer.data(), SIGNAL(statusInfoChanged(QString)), this, SLOT(onTransferStatusChanged()));
            this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged()));
            this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::TransferPriority)), this, SLOT(onTransferPriorityChanged()));
        }

        this->connect(this->session(), SIGNAL(transferRemoved(int)), this, SLOT(onTransferRemoved(int)));
    }

    emit countChanged(this->rowCount());
}

TransferListModel::~TransferListModel() {}

void TransferListModel::setSession(Session *session) {
    m_session = session;

    if (this->session()) {
        for (int i = 0; i < this->session()->transferList()->size(); i++) {
            QSharedPointer<TransferItem> transfer = this->session()->transferList()->at(i);
            this->connect(transfer.data(), SIGNAL(progressChanged(int)), this, SLOT(onTransferProgressChanged()));
            this->connect(transfer.data(), SIGNAL(sizeChanged(qint64)), this, SLOT(onTransferSizeChanged()));
            this->connect(transfer.data(), SIGNAL(statusChanged(Transfers::TransferStatus)), this, SLOT(onTransferStatusChanged()));
            this->connect(transfer.data(), SIGNAL(statusInfoChanged(QString)), this, SLOT(onTransferStatusChanged()));
            this->connect(transfer.data(), SIGNAL(fileNameChanged(QString)), this, SLOT(onTransferFileNameChanged()));
            this->connect(transfer.data(), SIGNAL(priorityChanged(Transfers::TransferPriority)), this, SLOT(onTransferPriorityChanged()));
        }

        this->connect(this->session(), SIGNAL(transferRemoved(int)), this, SLOT(onTransferRemoved(int)));
    }

    emit countChanged(this->rowCount());
}

int TransferListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return !this->session() ? 0 : this->session()->transferList()->size();
}

QVariant TransferListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case FileNameRole:
        return this->session()->transferList()->at(index.row()).data()->fileName();
    case TitleRole:
        return this->session()->transferList()->at(index.row()).data()->title();
    case StatusRole:
        return this->session()->transferList()->at(index.row()).data()->status();
    case StatusTextRole:
        return this->session()->transferList()->at(index.row()).data()->statusText();
    case StatusInfoRole:
        return this->session()->transferList()->at(index.row()).data()->statusInfo();
    case ProgressRole:
        return this->session()->transferList()->at(index.row()).data()->progress();
    case SizeRole:
        return this->session()->transferList()->at(index.row()).data()->size();
    case PriorityRole:
        return this->session()->transferList()->at(index.row()).data()->priority();
    case PriorityTextRole:
        return this->session()->transferList()->at(index.row()).data()->priorityText();
    case ConvertibleToAudioRole:
        return this->session()->transferList()->at(index.row()).data()->convertibleToAudio();
    case SaveAsAudioRole:
        return this->session()->transferList()->at(index.row()).data()->saveAsAudio();
    case ServiceRole:
        return this->session()->transferList()->at(index.row()).data()->service();
    case TransferTypeRole:
        return this->session()->transferList()->at(index.row()).data()->transferType();
    case SelectedRole:
        return QVariant(m_selectedRows.contains(index.row()));
    default:
        return QVariant();
    }
}

QVariant TransferListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

bool TransferListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    if (!index.isValid()) {
        return false;
    }

    switch (role) {
    case FileNameRole:
        this->session()->transferList()->at(index.row()).data()->setFileName(value.toString());
        break;
    case DownloadPathRole:
        this->session()->transferList()->at(index.row()).data()->setDownloadPath(value.toString());
        break;
    case StatusRole:
        this->session()->transferList()->at(index.row()).data()->setStatus(static_cast<Transfers::TransferStatus>(value.toInt()));
        break;
    case PriorityRole:
        this->session()->transferList()->at(index.row()).data()->setPriority(static_cast<Transfers::TransferPriority>(value.toInt()));
        break;
    case SaveAsAudioRole:
        this->session()->transferList()->at(index.row()).data()->setSaveAsAudio(value.toBool());
        break;
    default:
        return false;
    }

    return true;
}

bool TransferListModel::setData(int row, const QVariant &value, const QByteArray &role) {
    return setData(this->index(row, 0), value, this->roleNames().key(role));
}

void TransferListModel::onTransferStatusChanged() {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        int row = this->indexOf(transfer);
        emit dataChanged(this->index(row), this->index(row));
    }
}

void TransferListModel::onTransferProgressChanged() {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        int row = this->indexOf(transfer);
        emit dataChanged(this->index(row), this->index(row));
    }
}

void TransferListModel::onTransferSizeChanged() {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        int row = this->indexOf(transfer);
        emit dataChanged(this->index(row), this->index(row));
    }
}

void TransferListModel::onTransferFileNameChanged() {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        int row = this->indexOf(transfer);
        emit dataChanged(this->index(row), this->index(row));
    }
}

void TransferListModel::onTransferPriorityChanged() {
    if (TransferItem *transfer = qobject_cast<TransferItem*>(sender())) {
        int row = this->indexOf(transfer);
        emit dataChanged(this->index(row), this->index(row));
    }
}

void TransferListModel::onTransferRemoved(int row) {
    this->beginRemoveRows(QModelIndex(), row, row);
    this->endRemoveRows();
    emit countChanged(this->rowCount());
}

QSharedPointer<TransferItem> TransferListModel::get(int row) const {
    if ((row >= 0) && (row < this->rowCount())) {
        return this->session()->transferList()->at(row);
    }

    return QSharedPointer<TransferItem>();
}

TransferItem* TransferListModel::getFromQML(int row) const {
#ifdef QML_USER_INTERFACE
    if (TransferItem *transfer = this->get(row).data()) {
        QDeclarativeEngine::setObjectOwnership(transfer, QDeclarativeEngine::CppOwnership);
        return transfer;
    }
#else
    Q_UNUSED(row)
#endif
    return 0;
}

void TransferListModel::selectRow(int row) {
    m_selectedRows.append(row);

    emit dataChanged(this->index(row), this->index(row));
    emit itemsSelectedChanged(this->itemsSelected());
}

void TransferListModel::unselectRow(int row) {
    m_selectedRows.removeOne(row);

    emit dataChanged(this->index(row), this->index(row));
    emit itemsSelectedChanged(this->itemsSelected());
}

void TransferListModel::toggleSelected(int row) {
    if (m_selectedRows.contains(row)) {
        this->unselectRow(row);
    }
    else {
        this->selectRow(row);
    }

    emit itemsSelectedChanged(this->itemsSelected());
}

void TransferListModel::selectAll() {
    m_selectedRows.clear();

    for (int i = 0; i < this->rowCount(); i++) {
        m_selectedRows.append(i);
    }

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    emit itemsSelectedChanged(true);
}

void TransferListModel::selectNone() {
    m_selectedRows.clear();

    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
    emit itemsSelectedChanged(false);
}

void TransferListModel::pauseSelectedTransfers() {
    QList<int> rows = m_selectedRows;
    qSort(rows.begin(), rows.end(), qLess<int>());

    foreach (int i, rows) {
        this->setData(this->index(i), Transfers::Paused, StatusRole);
    }
}

void TransferListModel::resumeSelectedTransfers() {
    QList<int> rows = m_selectedRows;
    qSort(rows.begin(), rows.end(), qLess<int>());

    foreach (int i, rows) {
        this->setData(this->index(i), Transfers::Queued, StatusRole);
    }
}

void TransferListModel::cancelSelectedTransfers() {
    QList<int> rows = m_selectedRows;
    qSort(rows.begin(), rows.end(), qGreater<int>());

    foreach (int i, rows) {
        this->setData(this->index(i), Transfers::Cancelled, StatusRole);
    }
}

void TransferListModel::moveSelectedTransfers(int destinationRow) {
    Q_ASSERT((destinationRow >= 0) || (destinationRow < this->rowCount()));

    QList<int> rows = m_selectedRows;
    qSort(rows.begin(), rows.end(), qLess<int>());

    int movedUp = 0;
    int movedDown = 0;

    foreach (int sourceRow, rows) {
        if (sourceRow < destinationRow) {
            this->moveTransfer(sourceRow - movedDown, destinationRow);
            movedDown++;
        }
        else if (sourceRow > destinationRow){
            this->moveTransfer(sourceRow, destinationRow + movedUp);
            movedUp++;
        }
    }
}

void TransferListModel::moveTransfer(int sourceRow, int destinationRow) {
    Q_ASSERT((sourceRow >= 0) && (sourceRow < this->rowCount()) && (destinationRow >= 0) && (destinationRow < this->rowCount()));

    if (sourceRow == destinationRow) {
        return;
    }

    if (sourceRow < destinationRow) {
        destinationRow--;
        emit layoutAboutToBeChanged();
        this->session()->transferList()->move(sourceRow, destinationRow);
        emit layoutChanged();
        this->session()->transferList()->at(destinationRow).data()->setRow(destinationRow);

        for (int i = sourceRow; i < destinationRow; i++) {
            this->session()->transferList()->at(i).data()->setRow(i - 1);
        }
    }
    else {
        emit layoutAboutToBeChanged();
        this->session()->transferList()->move(sourceRow, destinationRow);
        emit layoutChanged();
        this->session()->transferList()->at(destinationRow).data()->setRow(destinationRow);

        for (int i = destinationRow + 1; i <= sourceRow; i++) {
            this->session()->transferList()->at(i).data()->setRow(i + 1);
        }
    }
}

int TransferListModel::indexOf(TransferItem *transfer) const {
    for (int i = 0; i < this->rowCount(); i++) {
        if (this->session()->transferList()->at(i).data() == transfer) {
            return i;
        }
    }

    return -1;
}
