#include "youtubeplaylistvideoswindow.h"
#include "videolistdelegate.h"
#include "youtubevideoinfowindow.h"
#include "youtubechannelwindow.h"
#include "videoselectionwindow.h"
#include "youtubeplaylistdialog.h"
#include "videoplaybackwindow.h"
#include <QMenuBar>
#include <QGridLayout>
#include <QScrollBar>

YouTubePlaylistVideosWindow::YouTubePlaylistVideosWindow(Session *session, QSharedPointer<PlaylistItem> playlist, QWidget *parent) :
    StackedWindow(session, parent),
    m_playlist(playlist),
    m_listView(new ListView(this)),
    m_model(new YouTubeVideoListModel(m_session, this)),
    m_addToFavouritesAction(menuBar()->addAction(tr("Add to favourites"), this, SLOT(addVideosToFavourites()))),
    m_addToPlaylistAction(menuBar()->addAction(tr("Add to playlist"), this, SLOT(addVideosToPlaylist()))),
    m_watchLaterAction(menuBar()->addAction(tr("Watch later"), this, SLOT(addVideosToWatchLaterPlaylist()))),
    m_addToQueueAction(menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_addToFavouritesContextAction(m_contextMenu->addAction(tr("Add to favourites"), this, SLOT(addVideoToFavourites()))),
    m_addToPlaylistContextAction(m_contextMenu->addAction(tr("Add to playlist"), this, SLOT(showPlaylistDialog()))),
    m_watchLaterContextAction(m_contextMenu->addAction(tr("Watch later"), this, SLOT(addVideoToWatchLaterPlaylist()))),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_titleLabel(new QLabel(QString("<p style='font-size:32px;'>%1</p>").arg(m_playlist.data()->title()))),
    m_descriptionLabel(new QLabel(m_playlist.data()->description().isEmpty() ? tr("No description") : m_playlist.data()->description(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_playlist.data()->date()).arg(m_playlist.data()->artist()), this)),
    m_thumbnail(new ThumbnailLabel(this)),
    m_avatar(new ThumbnailLabel(this)),
    m_playButton(new QPushButton(tr("Play all"), this)),
    m_separator1(new SeparatorLabel(tr("Description"), this)),
    m_separator2(new SeparatorLabel(QString("-- %1").arg(tr("videos")), this)),
    m_scrollArea(new QScrollArea(this))
{
    setWindowTitle(tr("Playlist"));
    setCentralWidget(m_scrollArea);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_descriptionLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    m_thumbnail->setInteractive(false);
    m_thumbnail->setFixedSize(160, 120);
    m_avatar->setFixedSize(60, 60);
    m_playButton->setFixedWidth(150);

    grid->setContentsMargins(0, 0, 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 0, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 1, 0);
    grid->addItem(new QSpacerItem(10, 10, QSizePolicy::Maximum), 3, 0);
    grid->addWidget(m_titleLabel, 0, 1, 1, 3, Qt::AlignTop);
    grid->addWidget(m_thumbnail, 1, 1, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 2, Qt::AlignTop);
    grid->addWidget(m_dateLabel, 1, 3, Qt::AlignTop);
    grid->addWidget(m_playButton, 2, 3, Qt::AlignRight);
    grid->addWidget(m_separator1, 3, 0, 1, 4);
    grid->addWidget(m_descriptionLabel, 4, 1, 1, 3);
    grid->addWidget(m_separator2, 5, 0, 1, 4);
    grid->addWidget(m_listView, 6, 0, 1, 4);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    connect(m_model, SIGNAL(loadingChanged()), this, SLOT(onLoadingChanged()));
    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_avatar, SIGNAL(clicked()), this, SLOT(showUserChannel()));
    connect(m_playButton, SIGNAL(clicked()), this, SLOT(playPlaylist()));

    m_model->getPlaylistVideos(m_playlist.data()->playlistId());

    setThumbnails();
    setMenuActions();
    onOrientationChanged();
}

void YouTubePlaylistVideosWindow::setLandscapeLayout() {
    m_scrollArea->widget()->setMaximumWidth(800);
}

void YouTubePlaylistVideosWindow::setPortraitLayout() {
    m_scrollArea->widget()->setMaximumWidth(480);
}

void YouTubePlaylistVideosWindow::setThumbnails() {
    QImage thumbnail = m_session->thumbnailCache()->thumbnail(m_playlist.data()->thumbnailUrl(), QSize(160, 120));
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_playlist.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void YouTubePlaylistVideosWindow::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_addToFavouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled(signedIn);
}

void YouTubePlaylistVideosWindow::onLoadingChanged() {
    toggleBusy(m_model->loading());
    m_playButton->setEnabled((!m_model->loading()) && (m_model->rowCount() > 0) && (m_session->settings()->mediaPlayer() == "cutetube"));

    if (!m_model->loading()) {
        m_separator2->setText(QString("%1 %2").arg(m_model->rowCount()).arg(tr("videos")));

        if (m_model->rowCount() > 0) {
            m_listView->setFixedHeight(m_model->rowCount() * 150);
        }
    }
}

void YouTubePlaylistVideosWindow::addVideosToFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to favourites"), SLOT(addVideosToFavourites()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void YouTubePlaylistVideosWindow::addVideosToPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to playlist"), SLOT(showPlaylistDialog()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void YouTubePlaylistVideosWindow::addVideosToWatchLaterPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Watch later"), SLOT(addVideosToWatchLaterPlaylist()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void YouTubePlaylistVideosWindow::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void YouTubePlaylistVideosWindow::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void YouTubePlaylistVideosWindow::queueVideos() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, m_scrollArea->verticalScrollBar()->value())));
    window->show();
}

void YouTubePlaylistVideosWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToFavouritesContextAction);
}

void YouTubePlaylistVideosWindow::addVideoToFavourites() {
    m_session->youtube()->addToFavourites(m_model->get(m_listView->currentIndex().row()));
}

void YouTubePlaylistVideosWindow::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(m_session, this);
    dialog->open();

    connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideoToNewPlaylist(NewPlaylist)));
}

void YouTubePlaylistVideosWindow::addVideoToPlaylist(const QString &id) {
    m_session->youtube()->addToPlaylist(m_model->get(m_listView->currentIndex().row()), id);
}

void YouTubePlaylistVideosWindow::addVideoToNewPlaylist(const NewPlaylist &playlist) {
    m_session->youtube()->createPlaylist(playlist, m_model->get(m_listView->currentIndex().row()));
}

void YouTubePlaylistVideosWindow::addVideoToWatchLaterPlaylist() {
    m_session->youtube()->addToWatchLaterPlaylist(m_model->get(m_listView->currentIndex().row()));
}

void YouTubePlaylistVideosWindow::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void YouTubePlaylistVideosWindow::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void YouTubePlaylistVideosWindow::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void YouTubePlaylistVideosWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        showVideoInfo(m_model->get(index.row()));
    }
}

void YouTubePlaylistVideosWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(m_session, video, this);
    window->show();
}

void YouTubePlaylistVideosWindow::playVideo(const QModelIndex &index) {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
        window->show();
    }
    else {
        m_session->urlGrabber()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}

void YouTubePlaylistVideosWindow::playPlaylist() {
    QList< QSharedPointer<VideoItem> > videos;

    for (int i = 0; i < m_model->rowCount(); i++) {
        videos.append(m_model->get(i));
    }

    VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, videos, this);
    window->show();
}

void YouTubePlaylistVideosWindow::showUserChannel() {
    YouTubeChannelWindow *window = new YouTubeChannelWindow(m_session, m_playlist.data()->artistId(), this);
    window->show();
}
