#include "youtubeaccountslistmodel.h"
#include <QSqlRecord>
#include <QSqlField>

YouTubeAccountsListModel::YouTubeAccountsListModel(YouTube *youtube, CookieJar *cookieJar, QObject *parent) :
    QSqlTableModel(parent, QSqlDatabase::database()),
    m_youtube(youtube),
    m_jar(cookieJar)
{
    setTable("youtubeAccounts");
    setEditStrategy(QSqlTableModel::OnManualSubmit);
    select();

    connect(m_youtube, SIGNAL(signedIn(QString,QString,QString)), this, SLOT(addAccount(QString,QString,QString)));
    connect(this, SIGNAL(accountSelected(QString,QString,QString)), m_youtube, SLOT(setAccount(QString,QString,QString)));
    connect(this, SIGNAL(accountAdded(QString,QString,QString)), m_youtube, SLOT(setAccount(QString,QString,QString)));
    connect(this, SIGNAL(accountDeleted()), m_youtube, SLOT(setAccount()));

    connect(this, SIGNAL(gotCookies(QByteArray)), m_jar, SLOT(setYouTubeCookies(QByteArray)));
    connect(this, SIGNAL(accountAdded(QString,QString,QString)), m_jar, SLOT(clearYouTubeCookies()));
    connect(this, SIGNAL(accountDeleted()), m_jar, SLOT(clearYouTubeCookiesFromDB()));
}

YouTubeAccountsListModel::~YouTubeAccountsListModel() {
    clear();
}

void YouTubeAccountsListModel::setActiveAccount(int row) {
    database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField refreshField("refresh", QVariant::String);
    QSqlField cookiesField("cookies", QVariant::ByteArray);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(data(index(row, 0), DisplayNameRole));
    usernameField.setValue(data(index(row, 0), UsernameRole));
    tokenField.setValue(data(index(row, 0), AccessTokenRole));
    refreshField.setValue(data(index(row, 0), RefreshTokenRole));
    cookiesField.setValue(data(index(row, 0), CookiesRole));
    activeField.setValue(1);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(refreshField);
    rec.append(cookiesField);
    rec.append(activeField);

    setRecord(row, rec);

    for (int i = 0; i < rowCount(); i++) {
        if (i != row) {
            nameField.setValue(data(index(i, 0), DisplayNameRole));
            usernameField.setValue(data(index(i, 0), UsernameRole));
            tokenField.setValue(data(index(i, 0), AccessTokenRole));
            refreshField.setValue(data(index(i, 0), RefreshTokenRole));
            cookiesField.setValue(data(index(i, 0), CookiesRole));
            activeField.setValue(0);

            rec.clear();
            rec.append(nameField);
            rec.append(usernameField);
            rec.append(tokenField);
            rec.append(refreshField);
            rec.append(cookiesField);
            rec.append(activeField);

            setRecord(i, rec);
        }
    }

    submitAll();
}

void YouTubeAccountsListModel::switchAccount(int row) {
    setActiveAccount(row);

    database().open();
    QString user = data(index(row, 0), UsernameRole).toString();
    QString token = data(index(row, 0), AccessTokenRole).toString();
    QString refresh = data(index(row, 0), RefreshTokenRole).toString();
    QByteArray cookieString = data(index(row, 0), CookiesRole).toByteArray();

    emit accountSelected(user, token, refresh);
    emit gotCookies(cookieString);
}

QVariant YouTubeAccountsListModel::data(const QModelIndex &idx, int role) const {
    return QSqlTableModel::data(index(idx.row(), role - (Qt::DisplayRole)));
}

void YouTubeAccountsListModel::addAccount(const QString &displayName, const QString &token, const QString &refresh) {
    database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField refreshField("refresh", QVariant::String);
    QSqlField cookiesField("cookies", QVariant::ByteArray);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(displayName);
    usernameField.setValue(QVariant::String);
    tokenField.setValue(token);
    refreshField.setValue(refresh);
    cookiesField.setValue(QVariant::ByteArray);
    activeField.setValue(0);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(refreshField);
    rec.append(cookiesField);
    rec.append(activeField);

    insertRecord(-1, rec);
    submitAll();

    setActiveAccount(rowCount() -1);

    emit accountAdded(QString(), token, refresh);
}

void YouTubeAccountsListModel::deleteAccount(int row) {
    database().open();
    bool signOut = data(index(row, 0), ActiveRole).toBool();

    if ((removeRow(row)) && (submitAll())) {
        if (signOut) {
            emit accountDeleted();
        }

        emit info(tr("Account deleted. Please visit the YouTube website to revoke access"));
    }
    else {
        emit warning(tr("Database error. Unable to delete account"));
    }
}
