#include "dailymotionvideodescriptiontab.h"
#include "dailymotionplaylistdialog.h"
#include "dailymotionmetadatadialog.h"
#include "sharingdialog.h"
#include "sharetofacebookdialog.h"
#include "sharetotwitterdialog.h"
#include "facebookauthdialog.h"
#include "twitterauthdialog.h"
#include "dailymotionvideolistwindow.h"
#include "dailymotionchannelwindow.h"
#include "videoplaybackwindow.h"

DailymotionVideoDescriptionTab::DailymotionVideoDescriptionTab(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_video(video),
    m_editAction(new QAction(tr("Edit"), this)),
    m_favouritesAction(new QAction(m_video.data()->favourite() ? tr("Delete from favourites") : tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_queueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_shareAction(new QAction(tr("Share"), this)),
    m_thumbnail(new ThumbnailLabel(m_video.data()->duration(), this)),
    m_avatar(new ThumbnailLabel(this)),
    m_ratingIndicator(new RatingIndicator(m_video.data()->rating(), m_video.data()->ratingCount(), this)),
    m_titleLabel(new QLabel(m_video.data()->title(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_video.data()->date()).arg(m_video.data()->artist()), this)),
    m_viewsLabel(new QLabel(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Views")).arg(m_video.data()->viewCount()), this)),
    m_descriptionLabel(new QLabel(QString("Description<br><font color='#4d4d4d'>%1</font>").arg(m_video.data()->description().isEmpty() ? tr("No description") : m_video.data()->description()), this)),
    m_tagsLabel(new QLabel(tagsToString(m_video.data()->tags()), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    addAction(m_editAction);
    addAction(m_favouritesAction);
    addAction(m_addToPlaylistAction);
    addAction(m_queueAction);
    addAction(m_videoDownloadAction);
    addAction(m_audioDownloadAction);
    addAction(m_shareAction);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);
    QFont smallFont;
    smallFont.setPixelSize(20);
    m_avatar->setFixedSize(60, 60);
    m_titleLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);
    m_dateLabel->setFont(smallFont);
    m_viewsLabel->setFont(smallFont);
    m_descriptionLabel->setWordWrap(true);
    m_descriptionLabel->setFont(smallFont);
    m_tagsLabel->setWordWrap(true);
    m_tagsLabel->setFont(smallFont);

    grid->addWidget(m_titleLabel, 0, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 0);
    grid->addWidget(m_dateLabel, 1, 1, Qt::AlignTop);
    grid->addWidget(m_ratingIndicator, 2, 0, 1, 2);
    grid->addWidget(m_viewsLabel, 3, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_descriptionLabel, 4, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_tagsLabel, 5, 0, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    connect(m_thumbnail, SIGNAL(clicked()), this, SLOT(playVideo()));
    connect(m_avatar, SIGNAL(clicked()), this, SLOT(showUserChannel()));
    connect(m_editAction, SIGNAL(triggered()), this, SLOT(showEditMetadataDialog()));
    connect(m_favouritesAction, SIGNAL(triggered()), this, SLOT(toggleFavourite()));
    connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(showPlaylistDialog()));
    connect(m_queueAction, SIGNAL(triggered()), this, SLOT(addVideoToPlaybackQueue()));
    connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownload()));
    connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownload()));
    connect(m_shareAction, SIGNAL(triggered()), this, SLOT(showShareDialog()));
    connect(m_tagsLabel, SIGNAL(linkActivated(QString)), this, SLOT(search(QString)));
    connect(m_video.data(), SIGNAL(favouriteChanged()), this, SLOT(onFavouriteChanged()));

    setMenuActions();
    setThumbnails();
    onOrientationChanged();
}

void DailymotionVideoDescriptionTab::setLandscapeLayout() {
    m_thumbnail->setFixedSize(280, 210);
    m_grid->setContentsMargins(10, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
}

void DailymotionVideoDescriptionTab::setPortraitLayout() {
    m_thumbnail->setFixedSize(460, 345);
    m_grid->setContentsMargins(0, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
}

void DailymotionVideoDescriptionTab::setMenuActions() {
    bool signedIn = m_session->dailymotion()->userSignedIn();

    m_editAction->setEnabled((signedIn) && (m_video.data()->artistId() == m_session->dailymotion()->username()));
    m_favouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
}

QString DailymotionVideoDescriptionTab::tagsToString(QStringList tags) {
    QString tagString("Tags<br>");

    if (tags.isEmpty()) {
        tagString.append(QString("<p style='color:#4d4d4d;'>%1</p>").arg(tr("No tags")));
    }
    else {
        while (!tags.isEmpty()) {
            tagString.append(QString("<a href='%1'>%1</a>   ").arg(tags.takeFirst()));
        }
    }

    return tagString;
}

void DailymotionVideoDescriptionTab::setThumbnails() {
    QImage thumbnail = m_session->thumbnailCache()->thumbnail(m_video.data()->largeThumbnailUrl());
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_video.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void DailymotionVideoDescriptionTab::showEditMetadataDialog() {
    DailymotionMetadataDialog *dialog = new DailymotionMetadataDialog(m_session, m_video, this);
    dialog->open();
}

void DailymotionVideoDescriptionTab::toggleFavourite() {
    if (m_video.data()->favourite()) {
        m_session->dailymotion()->deleteFromFavourites(m_video);
    }
    else {
        m_session->dailymotion()->addToFavourites(m_video);
    }
}

void DailymotionVideoDescriptionTab::onFavouriteChanged() {
    m_favouritesAction->setText(m_video.data()->favourite() ? tr("Delete from favourites") : tr("Add to favourites"));
}

void DailymotionVideoDescriptionTab::showPlaylistDialog() {
    DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(m_session, this);
    dialog->open();
    connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideoToNewPlaylist(NewPlaylist)));
}

void DailymotionVideoDescriptionTab::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(this);
    dialog->open();

    connect(dialog, SIGNAL(valueChanged(Services::SharingService)), this, SLOT(shareVideo(Services::SharingService)));
}

void DailymotionVideoDescriptionTab::shareVideo(Services::SharingService service) {
    switch (service) {
    case Services::Facebook:
        showFacebookDialog();
        break;
    case Services::Twitter:
        showTwitterDialog();
        break;
    case Services::Email:
        m_session->sharing()->shareViaEmail(m_video);
        break;
    case Services::Clipboard:
        m_session->sharing()->copyToClipboard(m_video.data()->url());
        break;
    default:
        return;
    }
}

void DailymotionVideoDescriptionTab::showFacebookDialog() {
    if (m_session->sharing()->userSignedInToFacebook()) {
        ShareToFacebookDialog *dialog = new ShareToFacebookDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        FacebookAuthDialog *dialog = new FacebookAuthDialog(m_session, this);
        dialog->open();
    }
}

void DailymotionVideoDescriptionTab::showTwitterDialog() {
    if (m_session->sharing()->userSignedInToTwitter()) {
        ShareToTwitterDialog *dialog = new ShareToTwitterDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        TwitterAuthDialog *dialog = new TwitterAuthDialog(m_session, this);
        dialog->open();
    }
}

void DailymotionVideoDescriptionTab::addVideoToPlaylist(const QString &id) {
    m_session->dailymotion()->addToPlaylist(m_video, id);
}

void DailymotionVideoDescriptionTab::addVideoToNewPlaylist(const NewPlaylist &playlist) {
    m_session->dailymotion()->createPlaylist(playlist, m_video);
}

void DailymotionVideoDescriptionTab::addVideoToPlaybackQueue() {
    m_session->playbackQueue()->addVideo(m_video);
}

void DailymotionVideoDescriptionTab::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_video);
}

void DailymotionVideoDescriptionTab::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_video, true);
}

void DailymotionVideoDescriptionTab::playVideo() {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_video, m_session->currentWindow());
        window->show();
    }
    else {
        m_session->urlGrabber()->getDailymotionVideoUrl(m_video.data()->videoId());
    }
}

void DailymotionVideoDescriptionTab::showUserChannel() {
    DailymotionChannelWindow *window = new DailymotionChannelWindow(m_session, m_video.data()->artistId(), m_session->currentWindow());
    window->show();
}

void DailymotionVideoDescriptionTab::search(const QString &query) {
    DailymotionVideoListWindow *window = new DailymotionVideoListWindow(m_session, m_session->currentWindow());
    window->search(query, m_session->settings()->searchOrder(), m_session->settings()->languageFilter());
    window->show();
}
