#include "dailymotionprofiletab.h"

DailymotionProfileTab::DailymotionProfileTab(Session *session, const QString &userId, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_avatar(new AvatarLabel(this)),
    m_channelLabel(new QLabel(this)),
    m_viewsLabel(new QLabel(this)),
    m_ageLabel(new QLabel(this)),
    m_genderLabel(new QLabel(this)),
    m_locationLabel(new QLabel(this)),
    m_descriptionLabel(new QLabel(this)),
    m_noProfileLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No profile found")), this)),
    m_subscriptionButton(new QPushButton(tr("Subscribe"))),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->addWidget(m_channelLabel, 0, 0, 1, 3);
    grid->addWidget(m_avatar, 1, 0, Qt::AlignTop);
    grid->addWidget(m_viewsLabel, 1, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_subscriptionButton, 2, 2);
    grid->addWidget(new SeparatorLabel(tr("Profile"), this), 3, 0, 1, 3);
    grid->addWidget(m_ageLabel, 4, 0, 1, 3);
    grid->addWidget(m_genderLabel, 5, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(m_locationLabel, 6, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(new SeparatorLabel(tr("Description"), this), 7, 0, 1, 3);
    grid->addWidget(m_descriptionLabel, 8, 0, 1, 3);

    m_grid->setContentsMargins(0, 0 ,0, 0);
    m_grid->addWidget(m_scrollArea);
    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_descriptionLabel->setWordWrap(true);
    m_scrollArea->hide();
    m_noProfileLabel->hide();
    m_viewsLabel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);
    m_subscriptionButton->setFixedWidth(150);

    connect(m_session->dailymotion(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    connect(m_session->dailymotion(), SIGNAL(warning(QString)), this, SLOT(onProfileError()));
    connect(m_subscriptionButton, SIGNAL(clicked()), this, SLOT(toggleSubscription()));

    if ((userId == "me") || (userId == m_session->dailymotion()->username())) {
        m_session->dailymotion()->getCurrentUserProfile();
        m_subscriptionButton->hide();
    }
    else {
        m_session->dailymotion()->getUserProfile(userId);
    }

    m_session->toggleBusy(true);
    onOrientationChanged();
}

void DailymotionProfileTab::setLandscapeLayout() {
    m_noProfileLabel->setFixedSize(800, 400);
}

void DailymotionProfileTab::setPortraitLayout() {
    m_noProfileLabel->setFixedSize(480, 700);
}

void DailymotionProfileTab::setUser(QSharedPointer<UserItem> user) {
    m_user = user;
    m_avatar->setName(m_user.data()->username());
    m_channelLabel->setText(QString("<p style='font-size:32px;'>%1</p>").arg(m_user.data()->channelName()));
    m_viewsLabel->setText(!m_user.data()->viewCount() ? tr("No views") : QString("%1 %2").arg(m_user.data()->viewCount()).arg(tr("views")));
    m_ageLabel->setText(QString("%1: %2").arg(tr("Age")).arg(!m_user.data()->age() ? tr("Unknown") : QString::number(m_user.data()->age())));
    m_genderLabel->setText(QString("%1: %2").arg(tr("Gender")).arg(m_user.data()->gender()));
    m_locationLabel->setText(QString("%1: %2").arg(tr("Location")).arg(m_user.data()->location().isEmpty() ? tr("Unknown") : m_user.data()->location()));
    m_descriptionLabel->setText(m_user.data()->description().isEmpty() ? tr("No description") : m_user.data()->description());
    m_subscriptionButton->setText(m_user.data()->subscribed() ? tr("Unsubscribe") : tr("Subscribe"));

    m_scrollArea->show();

    setAvatar();
    m_session->toggleBusy(false);

    connect(m_user.data(), SIGNAL(subscribedChanged()), this, SLOT(onSubscribedChanged()));
    disconnect(m_session->dailymotion(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    disconnect(m_session->dailymotion(), SIGNAL(warning(QString)), this, SLOT(onProfileError()));
}

void DailymotionProfileTab::setAvatar() {
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_user.data()->avatarUrl(), QSize(128, 128));

    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
        disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
    else {
        connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
}

void DailymotionProfileTab::onProfileError() {
    m_noProfileLabel->show();
}

void DailymotionProfileTab::toggleSubscription() {
    if (m_user.data()->subscribed()) {
        m_session->dailymotion()->unsubscribe(m_user);
    }
    else {
        m_session->dailymotion()->subscribe(m_user);
    }
}

void DailymotionProfileTab::onSubscribedChanged() {
    m_subscriptionButton->setText(m_user.data()->subscribed() ? tr("Unsubscribe") : tr("Subscribe"));
}
