#include "dailymotionplaylistlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include <QtNetwork/QNetworkReply>
#include <QtNetwork/QNetworkRequest>

using namespace QtJson;

DailymotionPlaylistListModel::DailymotionPlaylistListModel(Session *session, QObject *parent) :
    AbstractPlaylistListModel(session->newThumbnailCache(), parent),
    m_session(session),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void DailymotionPlaylistListModel::setFeed(const QString &afeed) {
    m_feed = afeed + "&fields=" + DAILYMOTION_PLAYLIST_FIELDS;
}

void DailymotionPlaylistListModel::getPlaylists(const QString &afeed) {
    if (!afeed.isEmpty()) {
        setFeed(afeed);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->dailymotion()->createReply(feed(), offset());
    connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void DailymotionPlaylistListModel::search(const QString &aquery, Queries::QueryOrder anorder) {
    if (!aquery.isEmpty()) {
        setQuery(aquery);
        setOrder(anorder);
    }

    setLoading(true);
    QNetworkReply *reply = m_session->dailymotion()->createSearchReply(Queries::Playlists, query(), offset(), order());
    connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void DailymotionPlaylistListModel::addPlaylists() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        setLoading(false);
        setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(m_session->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        connect(m_session->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
        m_session->dailymotion()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            setLoading(false);
            setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                appendPlaylist(QSharedPointer<PlaylistItem>(new PlaylistItem(entries.at(i).toMap())));
            }

            setLoading(false);
            setMoreResults(res.value("has_more").toBool());
            setOffset(res.value("page").toInt() + 1);
        }

        disconnect(m_session->dailymotion(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        disconnect(m_session->dailymotion(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    reply->deleteLater();
}

void DailymotionPlaylistListModel::getMorePlaylists() {
    if ((moreResults()) && (!loading())) {
        if (!query().isEmpty()) {
            search();
        }
        else {
            getPlaylists();
        }
    }
}
