#include "youtubevideodescriptiontab.h"
#include "youtubeplaylistdialog.h"
#include "youtubemetadatadialog.h"
#include "sharingdialog.h"
#include "sharetofacebookdialog.h"
#include "sharetotwitterdialog.h"
#include "facebookauthdialog.h"
#include "twitterauthdialog.h"
#include "youtubevideolistwindow.h"
#include "youtubechannelwindow.h"

YouTubeVideoDescriptionTab::YouTubeVideoDescriptionTab(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_video(video),
    m_editAction(new QAction(tr("Edit"), this)),
    m_favouritesAction(new QAction(video->favourite() ? tr("Delete from favourites") : tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_watchLaterAction(new QAction(tr("Watch later"), this)),
    m_queueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_shareAction(new QAction(tr("Share"), this)),
    m_likeButton(new QPushButton(QIcon(":/images/like.png"), "", this)),
    m_dislikeButton(new QPushButton(QIcon(":/images/dislike.png"), "", this)),
    m_thumbnail(new ThumbnailLabel(m_video.data()->duration(), this)),
    m_avatar(new ThumbnailLabel(this)),
    m_titleLabel(new QLabel(m_video.data()->title(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_video.data()->date()).arg(m_video.data()->artist()), this)),
    m_likesLabel(new QLabel(QString::number(m_video.data()->likes()), this)),
    m_dislikesLabel(new QLabel(QString::number(m_video.data()->dislikes()), this)),
    m_viewsLabel(new QLabel(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Views")).arg(m_video.data()->viewCount()), this)),
    m_descriptionLabel(new QLabel(QString("Description<br><font color='#4d4d4d'>%1</font>").arg(m_video.data()->description().isEmpty() ? tr("No description") : m_video.data()->description()), this)),
    m_tagsLabel(new QLabel(tagsToString(m_video.data()->tags()), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    addAction(m_editAction);
    addAction(m_favouritesAction);
    addAction(m_addToPlaylistAction);
    addAction(m_watchLaterAction);
    addAction(m_queueAction);
    addAction(m_videoDownloadAction);
    addAction(m_audioDownloadAction);
    addAction(m_shareAction);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);
    QFont smallFont;
    smallFont.setPixelSize(20);
    m_avatar->setFixedSize(60, 60);
    m_titleLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);
    m_dateLabel->setFont(smallFont);
    m_likesLabel->setFont(smallFont);
    m_dislikesLabel->setFont(smallFont);
    m_viewsLabel->setFont(smallFont);
    m_descriptionLabel->setWordWrap(true);
    m_descriptionLabel->setFont(smallFont);
    m_tagsLabel->setWordWrap(true);
    m_tagsLabel->setFont(smallFont);
    m_likeButton->setFixedSize(70, 70);
    m_likeButton->setEnabled((m_session->youtube()->userSignedIn()) && (m_video.data()->artistId() != m_session->youtube()->username()));
    m_dislikeButton->setFixedSize(70, 70);
    m_dislikeButton->setEnabled(m_likeButton->isEnabled());
    m_likesLabel->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred);

    grid->addWidget(m_titleLabel, 0, 0, 1, 4, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 0);
    grid->addWidget(m_dateLabel, 1, 1, 1, 3, Qt::AlignTop);
    grid->addWidget(m_likeButton, 2, 0);
    grid->addWidget(m_likesLabel, 2, 1);
    grid->addWidget(m_dislikeButton, 2, 2);
    grid->addWidget(m_dislikesLabel, 2, 3);
    grid->addWidget(m_viewsLabel, 3, 0, 1, 4, Qt::AlignTop);
    grid->addWidget(m_descriptionLabel, 4, 0, 1, 4, Qt::AlignTop);
    grid->addWidget(m_tagsLabel, 5, 0, 1, 4);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    connect(m_likeButton, SIGNAL(clicked()), this, SLOT(likeVideo()));
    connect(m_dislikeButton, SIGNAL(clicked()), this, SLOT(dislikeVideo()));
    connect(m_thumbnail, SIGNAL(clicked()), this, SLOT(playVideo()));
    connect(m_avatar, SIGNAL(clicked()), this, SLOT(showUserChannel()));
    connect(m_video.data(), SIGNAL(likedChanged()), this, SLOT(onVideoLiked()));
    connect(m_video.data(), SIGNAL(dislikedChanged()), this, SLOT(onVideoDisliked()));
    connect(m_video.data(), SIGNAL(favouriteChanged()), this, SLOT(onFavouriteChanged()));
    connect(m_editAction, SIGNAL(triggered()), this, SLOT(showEditMetadataDialog()));
    connect(m_favouritesAction, SIGNAL(triggered()), this, SLOT(toggleFavourite()));
    connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(showPlaylistDialog()));
    connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(addVideoToWatchLaterPlaylist()));
    connect(m_queueAction, SIGNAL(triggered()), this, SLOT(addVideoToPlaybackQueue()));
    connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownload()));
    connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownload()));
    connect(m_shareAction, SIGNAL(triggered()), this, SLOT(showShareDialog()));
    connect(m_tagsLabel, SIGNAL(linkActivated(QString)), this, SLOT(search(QString)));

    setMenuActions();
    setThumbnails();
    onOrientationChanged();
}

void YouTubeVideoDescriptionTab::setLandscapeLayout() {
    m_thumbnail->setFixedSize(280, 210);
    m_grid->setContentsMargins(10, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
}

void YouTubeVideoDescriptionTab::setPortraitLayout() {
    m_thumbnail->setFixedSize(460, 345);
    m_grid->setContentsMargins(0, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
}

void YouTubeVideoDescriptionTab::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_editAction->setEnabled((signedIn) && (m_video.data()->artistId() == m_session->youtube()->username()));
    m_favouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled(signedIn);
}

QString YouTubeVideoDescriptionTab::tagsToString(QStringList tags) {
    QString tagString("Tags<br>");

    if (tags.isEmpty()) {
        tagString.append(QString("<p style='color:#4d4d4d;'>%1</p>").arg(tr("No tags")));
    }
    else {
        while (!tags.isEmpty()) {
            tagString.append(QString("<a href='%1'>%1</a>   ").arg(tags.takeFirst()));
        }
    }

    return tagString;
}

void YouTubeVideoDescriptionTab::setThumbnails() {
    QImage thumbnail = m_session->thumbnailCache()->thumbnail(m_video.data()->largeThumbnailUrl());
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_video.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void YouTubeVideoDescriptionTab::showEditMetadataDialog() {
    YouTubeMetadataDialog *dialog = new YouTubeMetadataDialog(m_session, m_video, this);
    dialog->open();
}

void YouTubeVideoDescriptionTab::toggleFavourite() {
    if (m_video.data()->favourite()) {
        m_session->youtube()->deleteFromFavourites(m_video);
    }
    else {
        m_session->youtube()->addToFavourites(m_video);
    }
}

void YouTubeVideoDescriptionTab::onFavouriteChanged() {
    m_favouritesAction->setText(m_video.data()->favourite() ? tr("Delete from favourites") : tr("Add to favourites"));
}

void YouTubeVideoDescriptionTab::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(m_session, this);
    dialog->open();

    connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideoToNewPlaylist(NewPlaylist)));
}

void YouTubeVideoDescriptionTab::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(this);
    dialog->open();

    connect(dialog, SIGNAL(valueChanged(Services::SharingService)), this, SLOT(shareVideo(Services::SharingService)));
}

void YouTubeVideoDescriptionTab::shareVideo(Services::SharingService service) {
    switch (service) {
    case Services::Facebook:
        showFacebookDialog();
        break;
    case Services::Twitter:
        showTwitterDialog();
        break;
    case Services::Email:
        m_session->sharing()->shareViaEmail(m_video);
        break;
    case Services::Clipboard:
        m_session->sharing()->copyToClipboard(m_video.data()->url());
        break;
    default:
        return;
    }
}

void YouTubeVideoDescriptionTab::showFacebookDialog() {
    if (m_session->sharing()->userSignedInToFacebook()) {
        ShareToFacebookDialog *dialog = new ShareToFacebookDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        FacebookAuthDialog *dialog = new FacebookAuthDialog(m_session, this);
        dialog->open();
    }
}

void YouTubeVideoDescriptionTab::showTwitterDialog() {
    if (m_session->sharing()->userSignedInToTwitter()) {
        ShareToTwitterDialog *dialog = new ShareToTwitterDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        TwitterAuthDialog *dialog = new TwitterAuthDialog(m_session, this);
        dialog->open();
    }
}

void YouTubeVideoDescriptionTab::addVideoToPlaylist(const QString &id) {
    m_session->youtube()->addToPlaylist(m_video, id);
}

void YouTubeVideoDescriptionTab::addVideoToNewPlaylist(const NewPlaylist &playlist) {
    m_session->youtube()->createPlaylist(playlist, m_video);
}

void YouTubeVideoDescriptionTab::addVideoToWatchLaterPlaylist() {
    m_session->youtube()->addToWatchLaterPlaylist(m_video);
}

void YouTubeVideoDescriptionTab::addVideoToPlaybackQueue() {
    m_session->playbackQueue()->addVideo(m_video);
}

void YouTubeVideoDescriptionTab::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_video);
}

void YouTubeVideoDescriptionTab::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_video, true);
}

void YouTubeVideoDescriptionTab::likeVideo() {
    if ((!m_video.data()->liked()) && (!m_video.data()->disliked())) {
        m_session->youtube()->rateVideo(m_video, "like");
    }
}

void YouTubeVideoDescriptionTab::dislikeVideo() {
    if ((!m_video.data()->liked()) && (!m_video.data()->disliked())) {
        m_session->youtube()->rateVideo(m_video, "dislike");
    }
}

void YouTubeVideoDescriptionTab::onVideoLiked() {
    m_likesLabel->setText(QString::number(m_video.data()->likes()));
}

void YouTubeVideoDescriptionTab::onVideoDisliked() {
    m_dislikesLabel->setText(QString::number(m_video.data()->dislikes()));
}

void YouTubeVideoDescriptionTab::playVideo() {
    m_session->urlGrabber()->getYouTubeVideoUrl(m_video.data()->videoId());
}

void YouTubeVideoDescriptionTab::showUserChannel() {
    YouTubeChannelWindow *window = new YouTubeChannelWindow(m_session, m_video.data()->artistId(), m_session->currentWindow());
    window->show();
}

void YouTubeVideoDescriptionTab::search(const QString &query) {
    YouTubeVideoListWindow *window = new YouTubeVideoListWindow(m_session, m_session->currentWindow());
    window->search(query, m_session->settings()->searchOrder(), m_session->settings()->searchTime(), m_session->settings()->searchDuration(), m_session->settings()->languageFilter());
    window->show();
}
