#include "vimeo.h"
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QStringList>
#include <QRegExp>
#include <QtWebKit/QWebPage>
#include <QtWebKit/QWebFrame>
#include <QtWebKit/QWebElementCollection>

Vimeo::Vimeo(QObject *parent) :
    QObject(parent),
    m_nam(0)
{
    m_queryOrders[Queries::Relevance] = "relevant";
    m_queryOrders[Queries::Date] = "date";
    m_queryOrders[Queries::Views] = "plays";
    m_queryOrders[Queries::Rating] = "likes";
}

void Vimeo::search(QString query, int offset, Queries::QueryOrder order) {
    QUrl url(QString("http://vimeo.com/search/page:%1/sort:%2/format:detail").arg(offset).arg(m_queryOrders[order]));
    url.addQueryItem("q", query.simplified());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(parseVimeoPage()));
}

void Vimeo::parseVimeoPage() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QList< QSharedPointer<VideoItem> > videos;

    QString response(reply->readAll());
    response.remove(QRegExp("<head>.*</head>"));
    QWebPage page;
    page.mainFrame()->setHtml(response);
    QWebElement doc = page.mainFrame()->documentElement();
    QWebElementCollection elements = doc.findAll("li.clearfix");

    for (int i = 0; i < elements.count(); i++) {
        QWebElement el = elements.at(i);
        QString id = el.attribute("id").section('_', -1);
        QString image = el.findFirst("a").findFirst("img").attribute("src");
        QWebElement data = el.findFirst("div.data");
        QString title = data.findFirst("p.title").toPlainText();
        QString durationString = data.findFirst("div.duration").toPlainText();
        int mins = durationString.section(':', 0, 0).toInt();
        int secs = durationString.section(':', -1).toInt();
        int duration = mins * 60 + secs;
        QWebElement meta = data.findFirst("p.meta");
        QString artist = meta.findFirst("a").toPlainText();
        QString date = Utils::localDateTimeFromString(meta.findFirst("time").attribute("datetime"));
        QWebElement count = data.findFirst("p.count");
        QString playsString = count.findFirst("span.plays").toPlainText();
        int plays = 0;

        if (playsString.contains('K')) {
            playsString = playsString.section('K', 0, 0);

            if (playsString.contains('.')) {
                plays = playsString.section('.', 0, 0).toInt() * 1000 + playsString.section('.', 1, 1).toInt() * 100;
            }
            else {
                plays = playsString.section('.', 0, 0).toInt() * 1000;
            }
        }
        else {
            plays = playsString.section(' ', 0, 0).remove(',').toInt();
        }

        QString likesString = count.findFirst("span.likes").toPlainText();
        int likes = 0;

        if (likesString.contains('K')) {
            likesString = likesString.section('K', 0, 0);

            if (likesString.contains('.')) {
                likes = likesString.section('.', 0, 0).toInt() * 1000 + likesString.section('.', 1, 1).toInt() * 100;
            }
            else {
                likes = likesString.section('.', 0, 0).toInt() * 1000;
            }
        }
        else {
            likes = likesString.section(' ', 0, 0).remove(',').toInt();
        }

        QSharedPointer<VideoItem> video(new VideoItem);
        video.data()->setId(id);
        video.data()->setVideoId(id);
        video.data()->setUrl(QUrl("http://vimeo.com/" + id));
        video.data()->setTitle(title);
        video.data()->setArtist(artist);
        video.data()->setDate(date);
        video.data()->setThumbnailUrl(QUrl(image));
        video.data()->setDuration(duration);
        video.data()->setLikes(likes);
        video.data()->setRating(likes);
        video.data()->setViewCount(plays);
        video.data()->setService(Services::Vimeo);
        videos.append(video);
    }

    emit gotVideos(videos);
}

void Vimeo::getVideoInfo(QSharedPointer<VideoItem> video) {
    if (!video.isNull()) {
        m_videoAction = video;

        QUrl url(QString("http://vimeo.com/api/v2/video/%1.xml").arg(m_videoAction.data()->videoId()));
        QNetworkRequest request(url);
        request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
        QNetworkReply *reply = networkAccessManager()->get(request);
        connect(reply, SIGNAL(finished()), this, SLOT(checkVideoInfo()));
    }
}

void Vimeo::checkVideoInfo() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (!m_videoAction.isNull()) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode video = doc.namedItem("videos").namedItem("video");
        m_videoAction.data()->setDescription(video.firstChildElement("description").text());
        m_videoAction.data()->setArtistAvatarUrl(QUrl(video.firstChildElement("user_portrait_medium").text()));
        m_videoAction.data()->setTags(video.firstChildElement("tags").text().split(",", QString::SkipEmptyParts));
        m_videoAction.data()->setLargeThumbnailUrl(QUrl(video.firstChildElement("thumbnail_large").text()));

        emit gotVideoInfo();
    }
}

void Vimeo::getFullVideo(QString id) {
    emit busy(tr("Retrieving video details"));

    if (id.contains('/')) {
        id = id.section('/', -1);
    }

    QUrl url(QString("http://vimeo.com/api/v2/video/%1.xml").arg(id));
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:7.0.1) Gecko/20100101 Firefox/7.0.1");
    QNetworkReply *reply = networkAccessManager()->get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideo()));
}

void Vimeo::checkFullVideo() {
    emit busyProgressChanged(1);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QSharedPointer<VideoItem> videoItem(new VideoItem);

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomNode video = doc.namedItem("videos").namedItem("video");
    videoItem.data()->setId(video.firstChildElement("id").text());
    videoItem.data()->setVideoId(videoItem.data()->id());
    videoItem.data()->setUrl(QUrl(video.firstChildElement("url").text()));
    videoItem.data()->setDate(Utils::localDateTimeFromString(video.firstChildElement("upload_date").text()));
    videoItem.data()->setThumbnailUrl(QUrl(video.firstChildElement("thumbnail_medium").text()));
    videoItem.data()->setLargeThumbnailUrl(QUrl(video.firstChildElement("thumbnail_large").text()));
    videoItem.data()->setTitle(video.firstChildElement("title").text());
    videoItem.data()->setDescription(video.firstChildElement("description").text());
    videoItem.data()->setDuration(video.firstChildElement("duration").text().toInt());
    videoItem.data()->setArtist(video.firstChildElement("user_name").text());
    videoItem.data()->setArtistAvatarUrl(QUrl(video.firstChildElement("user_portrait_medium").text()));
    videoItem.data()->setTags(video.firstChildElement("tags").text().split(",", QString::SkipEmptyParts));
    videoItem.data()->setLikes(video.firstChildElement("stats_number_of_likes").text().toInt());
    videoItem.data()->setRating(videoItem.data()->likes());
    videoItem.data()->setViewCount(video.firstChildElement("stats_number_of_plays").text().toInt());

    emit gotVideo(videoItem);
}
