#include "archivewindow.h"
#include "archivevideolistdelegate.h"
#include "querydialog.h"
#include <QMenuBar>

ArchiveWindow::ArchiveWindow(Session *session, StackedWindow *parent) :
    StackedWindow(session, parent),
    m_model(new ArchiveListModel(m_session, this)),
    m_listView(new ListView(this)),
    m_filterBox(new FilterBox(this)),
    m_contextMenu(new QMenu(this)),
    m_deleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(showConfirmDeleteDialog()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this)),
    m_vbox(0)
{
    setWindowTitle(tr("Archive"));
    setCentralWidget(new QWidget());

    m_vbox = new QVBoxLayout(centralWidget());
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_listView);
    m_vbox->addWidget(m_filterBox);
    m_filterBox->hide();
    m_noResultsLabel->hide();
    m_listView->setModel(m_model);
    m_listView->setRootIndex(m_model->rootIndex());
    m_listView->setItemDelegate(new ArchiveVideoListDelegate(m_listView));

    connect(m_model, SIGNAL(directoryLoaded(QString)), this, SLOT(onDirectoryLoaded()));
    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_filterBox, SIGNAL(cleared()), m_filterBox, SLOT(hide()));
    connect(m_filterBox, SIGNAL(filterTextChanged(QString)), m_model, SLOT(setQuery(QString)));

    onOrientationChanged();
}

void ArchiveWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void ArchiveWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void ArchiveWindow::onDirectoryLoaded() {
    m_noResultsLabel->setVisible(m_model->rowCount() == 0);
}

void ArchiveWindow::onItemClicked(const QModelIndex &index) {
    m_session->videoLauncher()->launchVideo(index.data(ArchiveListModel::FilePathRole).toString());
}

void ArchiveWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_deleteAction);
}

void ArchiveWindow::showConfirmDeleteDialog() {
    QueryDialog *dialog = new QueryDialog(tr("Do you wish to delete '%1'?").arg(m_listView->currentIndex().data(ArchiveListModel::TitleRole).toString()), this);
    dialog->setWindowTitle(tr("Delete?"));
    dialog->open();

    connect(dialog, SIGNAL(accepted()), this, SLOT(deleteVideo()));
}

void ArchiveWindow::deleteVideo() {
    m_model->deleteVideo(m_listView->currentIndex());
}

void ArchiveWindow::keyPressEvent(QKeyEvent *event) {
    if ((m_filterBox->isHidden()) && (event->key() >= Qt::Key_0) && (event->key() <= Qt::Key_Z)) {
        showFilterBox();
        m_filterBox->setFilterText(event->text());
    }
}

void ArchiveWindow::showFilterBox() {
    m_filterBox->show();
    m_filterBox->setFocus(Qt::OtherFocusReason);
}
