#include "youtubecommentdialog.h"
#include "../base/thumbnailcache.h"
#include "../base/youtube.h"
#include "thumbnaillabel.h"
#include <QTextEdit>
#include <QLabel>
#include <QPushButton>
#include <QDialogButtonBox>
#include <QGridLayout>
#include <QScrollArea>
#include <QMaemo5Style>

YouTubeCommentDialog::YouTubeCommentDialog(QSharedPointer<VideoItem> video, QSharedPointer<CommentItem> initialComment, QWidget *parent) :
    AbstractDialog(parent),
    m_video(video),
    m_initialComment(initialComment),
    m_commentEdit(new QTextEdit(this)),
    m_avatar(!m_initialComment ? 0 : new ThumbnailLabel(this)),
    m_initialCommentLabel(m_initialComment.isNull() ? 0 : new QLabel(QString("<p style='font-size:20px;color:%1;'>%2</p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(m_initialComment.data()->body()), this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_doneButton(m_buttonBox->addButton(tr("Done"), QDialogButtonBox::AcceptRole)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Add comment"));

    QLabel *label = new QLabel(tr("Your comment"), this);
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);
    grid->setContentsMargins(0, 0, 0, 0);
    grid->addWidget(label, 0, 0, 1, 2);
    grid->addWidget(m_commentEdit, 1, 0, 1, 2);

    if (!m_initialComment.isNull()) {
        m_avatar->setFixedSize(60, 60);
        m_avatar->setInteractive(false);
        m_initialCommentLabel->setWordWrap(true);

        QLabel *label2 = new QLabel(tr("In reply to"), this);

        grid->addWidget(label2, 2, 0, 1, 2);
        grid->addWidget(m_avatar, 3, 0, Qt::AlignTop);
        grid->addWidget(m_initialCommentLabel, 3, 1, Qt::AlignTop);
    }

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_doneButton->setEnabled(false);

    this->connect(m_commentEdit, SIGNAL(textChanged()), this, SLOT(onCommentTextChanged()));
    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(submitComment()));

    if (!m_initialComment.isNull()) {
        this->setAvatar();
    }

    this->onOrientationChanged();
}

void YouTubeCommentDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void YouTubeCommentDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void YouTubeCommentDialog::onCommentTextChanged() {
    m_doneButton->setEnabled(!m_commentEdit->toPlainText().isEmpty());
}

void YouTubeCommentDialog::setAvatar() {
    QImage avatar = ThumbnailCache::instance()->thumbnail(m_initialComment.data()->artistAvatarUrl(), QSize(60, 60));

    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
    else {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
}

void YouTubeCommentDialog::submitComment() {
    QVariantMap comment;
    comment.insert("videoId", m_video.data()->videoId());
    comment.insert("body", m_commentEdit->toPlainText());

    if (m_initialComment) {
        comment.insert("replyId", m_initialComment.data()->id());
        YouTube::instance()->replyToComment(comment);
    }
    else {
        YouTube::instance()->addComment(comment);
    }

    this->accept();
}
